<?php
/**
 * Shortcode: [digital_rise_price_filter]
 *
 * Attributes:
 *   categories  – Comma-separated category slugs OR IDs for THIS page.
 *   min         – Override default min price.
 *   max         – Override default max price.
 *
 * @package Digital_Rise_WooCommerce_Filter
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

add_shortcode( 'digital_rise_price_filter', 'drwf_render_price_filter' );

/**
 * Resolve category term IDs from a mixed slugs/IDs string.
 *
 * @param  string $raw Comma-separated slugs or IDs.
 * @return array       Array of integer term IDs.
 */
function drwf_resolve_category_ids( $raw ) {
    if ( empty( $raw ) ) {
        return array();
    }

    $items = array_map( 'trim', explode( ',', $raw ) );
    $ids   = array();

    foreach ( $items as $item ) {
        if ( is_numeric( $item ) ) {
            $ids[] = absint( $item );
        } else {
            $term = get_term_by( 'slug', sanitize_title( $item ), 'product_cat' );
            if ( $term && ! is_wp_error( $term ) ) {
                $ids[] = (int) $term->term_id;
            }
        }
    }

    return array_unique( array_filter( $ids ) );
}

/**
 * Shortcode callback.
 *
 * @param  array  $atts Shortcode attributes.
 * @return string       HTML output.
 */
function drwf_render_price_filter( $atts ) {

    if ( ! class_exists( 'WooCommerce' ) ) {
        return '<p>' . esc_html__( 'WooCommerce is required for this filter.', 'digital-rise-woocommerce-filter' ) . '</p>';
    }

    static $instance_counter = 0;
    $instance_counter++;
    $uid = 'drwf-' . $instance_counter;

    wp_enqueue_style( 'drwf-filter-style' );
    wp_enqueue_script( 'drwf-filter-script' );

    $options      = get_option( 'drwf_settings', drwf_get_defaults() );
    $default_min  = isset( $options['default_min'] ) ? floatval( $options['default_min'] ) : 0;
    $default_max  = isset( $options['default_max'] ) ? floatval( $options['default_max'] ) : 1000;
    $filter_title = isset( $options['filter_title'] ) && '' !== $options['filter_title']
        ? $options['filter_title']
        : __( 'Filter by Price', 'digital-rise-woocommerce-filter' );
    $button_label = isset( $options['button_label'] ) && '' !== $options['button_label']
        ? $options['button_label']
        : __( 'Open', 'digital-rise-woocommerce-filter' );
    $collapsed    = isset( $options['collapsed_default'] ) ? $options['collapsed_default'] : '1';

    $atts = shortcode_atts( array(
        'min'        => $default_min,
        'max'        => $default_max,
        'categories' => '',
    ), $atts, 'digital_rise_price_filter' );

    $currency_symbol = function_exists( 'get_woocommerce_currency_symbol' )
        ? get_woocommerce_currency_symbol()
        : '$';

    // Determine effective categories.
    $shortcode_cat_ids = drwf_resolve_category_ids( $atts['categories'] );
    $admin_cat_ids     = isset( $options['allowed_categories'] ) ? array_map( 'intval', (array) $options['allowed_categories'] ) : array();

    if ( ! empty( $shortcode_cat_ids ) ) {
        $effective_cat_ids = $shortcode_cat_ids;
    } elseif ( ! empty( $admin_cat_ids ) ) {
        $effective_cat_ids = $admin_cat_ids;
    } else {
        $effective_cat_ids = array();
    }

    $cat_args = array(
        'taxonomy'   => 'product_cat',
        'orderby'    => 'name',
        'order'      => 'ASC',
        'hide_empty' => true,
    );
    if ( ! empty( $effective_cat_ids ) ) {
        $cat_args['include'] = $effective_cat_ids;
    }
    $categories = get_terms( $cat_args );

    $is_collapsed = ( '1' === $collapsed );

    // Formatted price labels for initial display.
    $formatted_min = drwf_format_price( $atts['min'] );
    $formatted_max = drwf_format_price( $atts['max'] );

    ob_start();
    ?>
    <div class="drwf-filter-wrapper drwf-instance"
         id="<?php echo esc_attr( $uid ); ?>"
         data-min="<?php echo esc_attr( $atts['min'] ); ?>"
         data-max="<?php echo esc_attr( $atts['max'] ); ?>"
         data-currency="<?php echo esc_attr( $currency_symbol ); ?>"
         data-scope-cats="<?php echo esc_attr( implode( ',', $effective_cat_ids ) ); ?>">

        <!-- ── Collapsible Filter Panel ── -->
        <div class="drwf-filter-form">

            <!-- Clickable Header -->
            <div class="drwf-filter-header" role="button" tabindex="0"
                 aria-expanded="<?php echo $is_collapsed ? 'false' : 'true'; ?>"
                 aria-controls="<?php echo esc_attr( $uid ); ?>-body">
                <h3 class="drwf-filter-title"><?php echo esc_html( $filter_title ); ?></h3>
                <span class="drwf-toggle-icon <?php echo $is_collapsed ? '' : 'drwf-open'; ?>">
                    <svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M3 5L7 9L11 5" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                </span>
            </div>

            <!-- Collapsible Body -->
            <div class="drwf-filter-body <?php echo $is_collapsed ? 'drwf-collapsed' : ''; ?>"
                 id="<?php echo esc_attr( $uid ); ?>-body">

                <!-- Price Range Slider -->
                <div class="drwf-range-section">
                    <div class="drwf-range-labels">
                        <span class="drwf-range-label-min"><?php echo esc_html( $currency_symbol ); ?><span class="drwf-val"><?php echo esc_html( $formatted_min ); ?></span></span>
                        <span class="drwf-range-label-max"><?php echo esc_html( $currency_symbol ); ?><span class="drwf-val"><?php echo esc_html( $formatted_max ); ?></span></span>
                    </div>
                    <div class="drwf-range-track-wrapper">
                        <div class="drwf-range-track">
                            <div class="drwf-range-fill"></div>
                        </div>
                        <input type="range" class="drwf-range-input drwf-range-input--min"
                               min="<?php echo esc_attr( $atts['min'] ); ?>"
                               max="<?php echo esc_attr( $atts['max'] ); ?>"
                               value="<?php echo esc_attr( $atts['min'] ); ?>"
                               step="1"
                               aria-label="<?php esc_attr_e( 'Minimum price', 'digital-rise-woocommerce-filter' ); ?>" />
                        <input type="range" class="drwf-range-input drwf-range-input--max"
                               min="<?php echo esc_attr( $atts['min'] ); ?>"
                               max="<?php echo esc_attr( $atts['max'] ); ?>"
                               value="<?php echo esc_attr( $atts['max'] ); ?>"
                               step="1"
                               aria-label="<?php esc_attr_e( 'Maximum price', 'digital-rise-woocommerce-filter' ); ?>" />
                    </div>
                </div>

                <?php if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) : ?>
                <div class="drwf-category-section">
                    <h4 class="drwf-category-title"><?php esc_html_e( 'Categories', 'digital-rise-woocommerce-filter' ); ?></h4>
                    <div class="drwf-category-list">
                        <?php foreach ( $categories as $cat ) : ?>
                            <label class="drwf-category-item">
                                <input type="checkbox" class="drwf-cat-checkbox" value="<?php echo esc_attr( $cat->term_id ); ?>" checked />
                                <span class="drwf-cat-name"><?php echo esc_html( $cat->name ); ?></span>
                                <span class="drwf-cat-count">(<?php echo esc_html( $cat->count ); ?>)</span>
                            </label>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>

                <div class="drwf-filter-actions">
                    <button type="button" class="drwf-filter-button drwf-btn-filter">
                        <?php esc_html_e( 'Filter', 'digital-rise-woocommerce-filter' ); ?>
                    </button>
                    <button type="button" class="drwf-reset-button drwf-btn-reset">
                        <?php esc_html_e( 'Reset', 'digital-rise-woocommerce-filter' ); ?>
                    </button>
                </div>
            </div>
        </div>

        <!-- Loading -->
        <div class="drwf-loading" style="display:none;">
            <span class="drwf-spinner"></span>
            <span><?php esc_html_e( 'Loading products…', 'digital-rise-woocommerce-filter' ); ?></span>
        </div>

        <!-- Products Grid -->
        <div class="drwf-products-grid">
            <?php
            $initial_cat_ids = array();
            if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
                $initial_cat_ids = wp_list_pluck( $categories, 'term_id' );
            }
            echo drwf_get_filtered_products( $atts['min'], $atts['max'], $initial_cat_ids, $button_label, $effective_cat_ids ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            ?>
        </div>

    </div>
    <?php
    return ob_get_clean();
}

/**
 * Build product HTML for the given price range and categories.
 *
 * @param  float  $min_price      Minimum price.
 * @param  float  $max_price      Maximum price.
 * @param  array  $category_ids   Visitor-selected category IDs.
 * @param  string $button_label   CTA button text.
 * @param  array  $scope_cats     Allowed scope (shortcode or admin).
 * @return string                 Product cards HTML.
 */
function drwf_get_filtered_products( $min_price = 0, $max_price = 1000, $category_ids = array(), $button_label = '', $scope_cats = array() ) {

    $min_price = floatval( $min_price );
    $max_price = floatval( $max_price );

    if ( '' === $button_label ) {
        $options      = get_option( 'drwf_settings', drwf_get_defaults() );
        $button_label = isset( $options['button_label'] ) && '' !== $options['button_label']
            ? $options['button_label']
            : __( 'Open', 'digital-rise-woocommerce-filter' );
    }

    // Clamp visitor categories to allowed scope.
    if ( ! empty( $scope_cats ) && ! empty( $category_ids ) ) {
        $category_ids = array_intersect( array_map( 'intval', $category_ids ), array_map( 'intval', $scope_cats ) );
    } elseif ( ! empty( $scope_cats ) && empty( $category_ids ) ) {
        $category_ids = $scope_cats;
    }

    $args = array(
        'post_type'      => 'product',
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'meta_query'     => array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
            array(
                'key'     => '_price',
                'value'   => array( $min_price, $max_price ),
                'compare' => 'BETWEEN',
                'type'    => 'NUMERIC',
            ),
        ),
        'orderby'  => 'meta_value_num',
        'meta_key' => '_price', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
        'order'    => 'ASC',
    );

    if ( ! empty( $category_ids ) ) {
        $args['tax_query'] = array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
            array(
                'taxonomy' => 'product_cat',
                'field'    => 'term_id',
                'terms'    => array_map( 'intval', $category_ids ),
                'operator' => 'IN',
            ),
        );
    }

    $query = new WP_Query( $args );

    ob_start();

    if ( $query->have_posts() ) {
        while ( $query->have_posts() ) {
            $query->the_post();
            global $product;

            if ( ! $product instanceof WC_Product ) {
                $product = wc_get_product( get_the_ID() );
            }
            if ( ! $product ) {
                continue;
            }

            $image     = $product->get_image( 'woocommerce_thumbnail' );
            $title     = $product->get_name();
            $price     = $product->get_price_html();
            $permalink = $product->get_permalink();
            ?>
            <div class="drwf-product-card">
                <a href="<?php echo esc_url( $permalink ); ?>" class="drwf-product-link">
                    <div class="drwf-product-image">
                        <?php echo wp_kses_post( $image ); ?>
                    </div>
                    <div class="drwf-product-info">
                        <h4 class="drwf-product-title"><?php echo esc_html( $title ); ?></h4>
                        <span class="drwf-product-price"><?php echo wp_kses_post( $price ); ?></span>
                    </div>
                </a>
                <div class="drwf-product-actions">
                    <a href="<?php echo esc_url( $permalink ); ?>"
                       class="drwf-cta-button"
                       aria-label="<?php echo esc_attr( sprintf( __( 'Open %s', 'digital-rise-woocommerce-filter' ), $title ) ); ?>">
                        <?php echo esc_html( $button_label ); ?>
                    </a>
                </div>
            </div>
            <?php
        }
        wp_reset_postdata();
    } else {
        ?>
        <p class="drwf-no-products"><?php esc_html_e( 'No products found matching your criteria.', 'digital-rise-woocommerce-filter' ); ?></p>
        <?php
    }

    return ob_get_clean();
}
