<?php
/**
 * Admin settings page for Digital Rise WooCommerce Filter Solutions.
 *
 * WooCommerce → Digital Rise Filter
 * WordPress Settings API + wp-color-picker.
 *
 * @package Digital_Rise_WooCommerce_Filter
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRWF_Admin_Settings {

    private $option_group = 'drwf_settings_group';
    private $option_name  = 'drwf_settings';

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_filter( 'plugin_action_links_' . DRWF_PLUGIN_BASENAME, array( $this, 'plugin_action_links' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
    }

    public function enqueue_admin_assets( $hook_suffix ) {
        if ( 'woocommerce_page_drwf-settings' !== $hook_suffix ) {
            return;
        }

        wp_enqueue_style( 'wp-color-picker' );
        wp_enqueue_script( 'wp-color-picker' );

        wp_add_inline_script( 'wp-color-picker', '
            jQuery(function($){
                $(".drwf-color-field").wpColorPicker();
                $(".drwf-select-all").on("click",function(e){
                    e.preventDefault();
                    $(".drwf-cat-grid input[type=checkbox]").prop("checked",true);
                });
                $(".drwf-select-none").on("click",function(e){
                    e.preventDefault();
                    $(".drwf-cat-grid input[type=checkbox]").prop("checked",false);
                });

                /* Live preview for number format */
                function updateFormatPreview(){
                    var thou = $("[name=\"drwf_settings[thousand_separator]\"]").val() || ".";
                    var dec  = $("[name=\"drwf_settings[decimal_separator]\"]").val() || ",";
                    var decs = parseInt($("[name=\"drwf_settings[decimals]\"]").val()) || 0;
                    var num  = 1234567.89;
                    var parts = num.toFixed(decs).split(".");
                    parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thou);
                    var result = decs > 0 ? parts[0] + dec + (parts[1]||"") : parts[0];
                    $("#drwf-format-preview").text(result);
                }
                $("[name=\"drwf_settings[thousand_separator]\"], [name=\"drwf_settings[decimal_separator]\"], [name=\"drwf_settings[decimals]\"]").on("input change",updateFormatPreview);
                updateFormatPreview();
            });
        ' );

        wp_add_inline_style( 'wp-color-picker', '
            .drwf-admin-wrap { max-width: 860px; }
            .drwf-admin-header {
                background: linear-gradient(135deg, #1d2327 0%, #2c3338 100%);
                color: #fff; border-radius: 8px; padding: 28px 32px; margin-bottom: 20px;
            }
            .drwf-admin-header h1 { margin: 0 0 6px; font-size: 22px; color: #fff; }
            .drwf-admin-header p { margin: 0; color: #c3c4c7; font-size: 14px; }
            .drwf-info-box {
                background: #f0f6fc; border: 1px solid #c3c4c7; border-radius: 8px;
                padding: 18px 22px; margin-bottom: 20px;
            }
            .drwf-info-box code {
                background: #1d2327; color: #50d7a9; padding: 5px 12px;
                border-radius: 4px; font-size: 13px; display: inline-block; margin: 4px 2px;
            }
            .drwf-info-box h4 { margin: 0 0 8px; }
            .drwf-donate-box {
                background: #fef9e7; border: 1px solid #f0c674; border-radius: 8px;
                padding: 20px 24px; margin-bottom: 20px;
            }
            .drwf-donate-box h3 { margin: 0 0 8px; }
            .drwf-donate-box .button-primary { background: #0073aa; border-color: #0073aa; margin-top: 8px; }
            .drwf-color-grid {
                display: grid; grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
                gap: 14px 24px;
            }
            .drwf-color-grid .drwf-color-item { display: flex; flex-direction: column; gap: 4px; }
            .drwf-color-grid .drwf-color-item label { font-weight: 500; font-size: 13px; }
            .drwf-cat-grid {
                display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
                gap: 6px 16px; max-height: 260px; overflow-y: auto; padding: 10px;
                border: 1px solid #dcdcde; border-radius: 4px; background: #fff;
            }
            .drwf-cat-grid label { display: flex; align-items: center; gap: 6px; font-size: 13px; cursor: pointer; }
            .drwf-cat-grid label:hover { color: #2271b1; }
            .drwf-cat-grid .count { color: #8c8f94; font-size: 12px; }
            .drwf-select-helpers { margin-bottom: 8px; }
            .drwf-select-helpers a { cursor: pointer; margin-right: 12px; font-size: 13px; }
            .drwf-format-preview-box {
                display: inline-flex; align-items: center; gap: 8px;
                background: #1d2327; color: #50d7a9; font-weight: 700; font-size: 18px;
                padding: 8px 18px; border-radius: 6px; margin-top: 6px; font-family: monospace;
            }
            .drwf-format-fields { display: flex; gap: 16px; flex-wrap: wrap; align-items: flex-end; }
            .drwf-format-fields .drwf-fmt-group { display: flex; flex-direction: column; gap: 4px; }
            .drwf-format-fields .drwf-fmt-group label { font-weight: 500; font-size: 13px; }
            .drwf-format-fields .drwf-fmt-group select,
            .drwf-format-fields .drwf-fmt-group input { min-width: 90px; }
            .drwf-admin-footer { margin-top: 24px; color: #646970; font-size: 13px; }
            .drwf-admin-footer a { text-decoration: none; }
        ' );
    }

    public function add_menu_page() {
        add_submenu_page(
            'woocommerce',
            __( 'Digital Rise Filter Settings', 'digital-rise-woocommerce-filter' ),
            __( 'Digital Rise Filter', 'digital-rise-woocommerce-filter' ),
            'manage_woocommerce',
            'drwf-settings',
            array( $this, 'render_settings_page' )
        );
    }

    public function register_settings() {
        register_setting( $this->option_group, $this->option_name, array( $this, 'sanitize_settings' ) );

        /* ── General ── */
        add_settings_section( 'drwf_general', __( 'General Settings', 'digital-rise-woocommerce-filter' ), array( $this, 'section_general_cb' ), 'drwf-settings' );
        add_settings_field( 'ajax_enabled',      __( 'Enable AJAX Filtering', 'digital-rise-woocommerce-filter' ),  array( $this, 'field_ajax_enabled' ),      'drwf-settings', 'drwf_general' );
        add_settings_field( 'collapsed_default',  __( 'Start Filter Collapsed', 'digital-rise-woocommerce-filter' ), array( $this, 'field_collapsed_default' ),  'drwf-settings', 'drwf_general' );
        add_settings_field( 'filter_title',      __( 'Filter Title Text', 'digital-rise-woocommerce-filter' ),      array( $this, 'field_filter_title' ),      'drwf-settings', 'drwf_general' );
        add_settings_field( 'button_label',      __( 'Product Button Label', 'digital-rise-woocommerce-filter' ),   array( $this, 'field_button_label' ),      'drwf-settings', 'drwf_general' );
        add_settings_field( 'default_min',       __( 'Default Min Price', 'digital-rise-woocommerce-filter' ),      array( $this, 'field_default_min' ),       'drwf-settings', 'drwf_general' );
        add_settings_field( 'default_max',       __( 'Default Max Price', 'digital-rise-woocommerce-filter' ),      array( $this, 'field_default_max' ),       'drwf-settings', 'drwf_general' );

        /* ── Number Format ── */
        add_settings_section( 'drwf_number_format', __( 'Price Number Format', 'digital-rise-woocommerce-filter' ), array( $this, 'section_number_format_cb' ), 'drwf-settings' );
        add_settings_field( 'number_format_fields', __( 'Format Options', 'digital-rise-woocommerce-filter' ), array( $this, 'field_number_format' ), 'drwf-settings', 'drwf_number_format' );

        /* ── Categories ── */
        add_settings_section( 'drwf_categories', __( 'Default Allowed Categories', 'digital-rise-woocommerce-filter' ), array( $this, 'section_category_cb' ), 'drwf-settings' );
        add_settings_field( 'allowed_categories', __( 'Visible Categories', 'digital-rise-woocommerce-filter' ), array( $this, 'field_allowed_categories' ), 'drwf-settings', 'drwf_categories' );

        /* ── Colors ── */
        add_settings_section( 'drwf_colors', __( 'Color Customization', 'digital-rise-woocommerce-filter' ), array( $this, 'section_colors_cb' ), 'drwf-settings' );
        add_settings_field( 'color_fields', __( 'Colors', 'digital-rise-woocommerce-filter' ), array( $this, 'field_colors' ), 'drwf-settings', 'drwf_colors' );
    }

    /**
     * Sanitize all settings.
     */
    public function sanitize_settings( $input ) {
        $clean    = array();
        $defaults = drwf_get_defaults();

        $clean['ajax_enabled']      = isset( $input['ajax_enabled'] ) ? '1' : '0';
        $clean['collapsed_default'] = isset( $input['collapsed_default'] ) ? '1' : '0';
        $clean['filter_title']      = isset( $input['filter_title'] ) ? sanitize_text_field( $input['filter_title'] ) : $defaults['filter_title'];
        $clean['button_label']      = isset( $input['button_label'] ) ? sanitize_text_field( $input['button_label'] ) : $defaults['button_label'];
        $clean['default_min']       = isset( $input['default_min'] ) ? abs( floatval( $input['default_min'] ) ) : 0;
        $clean['default_max']       = isset( $input['default_max'] ) ? abs( floatval( $input['default_max'] ) ) : 1000;

        if ( $clean['default_max'] < $clean['default_min'] ) {
            $clean['default_max'] = $clean['default_min'];
        }

        // Number format.
        $allowed_separators          = array( '.', ',', ' ', '' );
        $clean['thousand_separator'] = isset( $input['thousand_separator'] ) && in_array( $input['thousand_separator'], $allowed_separators, true )
            ? $input['thousand_separator']
            : '.';
        $clean['decimal_separator']  = isset( $input['decimal_separator'] ) && in_array( $input['decimal_separator'], array( '.', ',' ), true )
            ? $input['decimal_separator']
            : ',';
        $clean['decimals']           = isset( $input['decimals'] ) ? absint( $input['decimals'] ) : 0;
        if ( $clean['decimals'] > 4 ) {
            $clean['decimals'] = 4;
        }

        // Prevent same separator for thousands and decimals when decimals > 0.
        if ( $clean['decimals'] > 0 && $clean['thousand_separator'] === $clean['decimal_separator'] ) {
            $clean['decimal_separator'] = ( '.' === $clean['thousand_separator'] ) ? ',' : '.';
        }

        // Categories.
        $clean['allowed_categories'] = array();
        if ( isset( $input['allowed_categories'] ) && is_array( $input['allowed_categories'] ) ) {
            $clean['allowed_categories'] = array_filter( array_map( 'absint', $input['allowed_categories'] ) );
        }

        // Colors.
        $color_keys = array(
            'color_primary', 'color_primary_hover', 'color_title', 'color_label',
            'color_card_bg', 'color_card_border', 'color_price', 'color_price_range',
            'color_cta_bg', 'color_cta_text', 'color_cta_hover',
            'color_filter_bg', 'color_filter_border',
            'color_reset_bg', 'color_reset_text',
        );
        foreach ( $color_keys as $key ) {
            $clean[ $key ] = isset( $input[ $key ] ) ? sanitize_hex_color( $input[ $key ] ) : $defaults[ $key ];
            if ( empty( $clean[ $key ] ) ) {
                $clean[ $key ] = $defaults[ $key ];
            }
        }

        return $clean;
    }

    /* ── Section Callbacks ── */

    public function section_general_cb() {
        echo '<p>' . esc_html__( 'Configure the price filter shortcode behaviour and appearance.', 'digital-rise-woocommerce-filter' ) . '</p>';
    }

    public function section_number_format_cb() {
        echo '<p>' . esc_html__( 'Control how prices are displayed on the range slider labels and admin fields. Example: 1000000 → 1.000.000 or 1,000,000.', 'digital-rise-woocommerce-filter' ) . '</p>';
    }

    public function section_category_cb() {
        echo '<p>' . esc_html__( 'These are the default allowed categories (global). Override per page with the shortcode categories attribute.', 'digital-rise-woocommerce-filter' ) . '</p>';
    }

    public function section_colors_cb() {
        echo '<p>' . esc_html__( 'Customise all filter and product card colors. Changes apply site-wide instantly.', 'digital-rise-woocommerce-filter' ) . '</p>';
    }

    /* ── Field Callbacks ── */

    public function field_ajax_enabled() {
        $options = get_option( $this->option_name, drwf_get_defaults() );
        $checked = isset( $options['ajax_enabled'] ) ? $options['ajax_enabled'] : '1';
        ?>
        <label>
            <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[ajax_enabled]" value="1" <?php checked( $checked, '1' ); ?> />
            <?php esc_html_e( 'Filter products without page reload', 'digital-rise-woocommerce-filter' ); ?>
        </label>
        <?php
    }

    public function field_collapsed_default() {
        $options = get_option( $this->option_name, drwf_get_defaults() );
        $checked = isset( $options['collapsed_default'] ) ? $options['collapsed_default'] : '1';
        ?>
        <label>
            <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[collapsed_default]" value="1" <?php checked( $checked, '1' ); ?> />
            <?php esc_html_e( 'Filter panel starts collapsed (visitor clicks to expand)', 'digital-rise-woocommerce-filter' ); ?>
        </label>
        <?php
    }

    public function field_filter_title() {
        $options = get_option( $this->option_name, drwf_get_defaults() );
        $value   = isset( $options['filter_title'] ) ? $options['filter_title'] : 'Filter by Price';
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[filter_title]"
               value="<?php echo esc_attr( $value ); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e( 'The heading displayed above the price range slider.', 'digital-rise-woocommerce-filter' ); ?></p>
        <?php
    }

    public function field_button_label() {
        $options = get_option( $this->option_name, drwf_get_defaults() );
        $value   = isset( $options['button_label'] ) ? $options['button_label'] : 'Open';
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[button_label]"
               value="<?php echo esc_attr( $value ); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e( 'CTA text on each product card (e.g. "Open", "Ouvrir", "Visit").', 'digital-rise-woocommerce-filter' ); ?></p>
        <?php
    }

    public function field_default_min() {
        $options = get_option( $this->option_name, drwf_get_defaults() );
        $value   = isset( $options['default_min'] ) ? $options['default_min'] : '0';
        ?>
        <input type="number" name="<?php echo esc_attr( $this->option_name ); ?>[default_min]"
               value="<?php echo esc_attr( $value ); ?>" min="0" step="any" class="small-text" />
        <span class="description">&rarr; <?php echo esc_html( drwf_format_price( $value ) ); ?></span>
        <?php
    }

    public function field_default_max() {
        $options = get_option( $this->option_name, drwf_get_defaults() );
        $value   = isset( $options['default_max'] ) ? $options['default_max'] : '1000';
        ?>
        <input type="number" name="<?php echo esc_attr( $this->option_name ); ?>[default_max]"
               value="<?php echo esc_attr( $value ); ?>" min="0" step="any" class="small-text" />
        <span class="description">&rarr; <?php echo esc_html( drwf_format_price( $value ) ); ?></span>
        <?php
    }

    /**
     * Number format fields with live preview.
     */
    public function field_number_format() {
        $options  = get_option( $this->option_name, drwf_get_defaults() );
        $thou_sep = isset( $options['thousand_separator'] ) ? $options['thousand_separator'] : '.';
        $dec_sep  = isset( $options['decimal_separator'] ) ? $options['decimal_separator'] : ',';
        $decimals = isset( $options['decimals'] ) ? intval( $options['decimals'] ) : 0;
        $n        = esc_attr( $this->option_name );
        ?>
        <div class="drwf-format-fields">

            <div class="drwf-fmt-group">
                <label for="drwf-thou-sep"><?php esc_html_e( 'Thousand Separator', 'digital-rise-woocommerce-filter' ); ?></label>
                <select id="drwf-thou-sep" name="<?php echo $n; ?>[thousand_separator]">
                    <option value="." <?php selected( $thou_sep, '.' ); ?>><?php esc_html_e( 'Dot ( . )', 'digital-rise-woocommerce-filter' ); ?></option>
                    <option value="," <?php selected( $thou_sep, ',' ); ?>><?php esc_html_e( 'Comma ( , )', 'digital-rise-woocommerce-filter' ); ?></option>
                    <option value=" " <?php selected( $thou_sep, ' ' ); ?>><?php esc_html_e( 'Space (  )', 'digital-rise-woocommerce-filter' ); ?></option>
                    <option value="" <?php selected( $thou_sep, '' ); ?>><?php esc_html_e( 'None', 'digital-rise-woocommerce-filter' ); ?></option>
                </select>
            </div>

            <div class="drwf-fmt-group">
                <label for="drwf-dec-sep"><?php esc_html_e( 'Decimal Separator', 'digital-rise-woocommerce-filter' ); ?></label>
                <select id="drwf-dec-sep" name="<?php echo $n; ?>[decimal_separator]">
                    <option value="," <?php selected( $dec_sep, ',' ); ?>><?php esc_html_e( 'Comma ( , )', 'digital-rise-woocommerce-filter' ); ?></option>
                    <option value="." <?php selected( $dec_sep, '.' ); ?>><?php esc_html_e( 'Dot ( . )', 'digital-rise-woocommerce-filter' ); ?></option>
                </select>
            </div>

            <div class="drwf-fmt-group">
                <label for="drwf-decimals"><?php esc_html_e( 'Decimal Places', 'digital-rise-woocommerce-filter' ); ?></label>
                <input type="number" id="drwf-decimals" name="<?php echo $n; ?>[decimals]"
                       value="<?php echo esc_attr( $decimals ); ?>" min="0" max="4" step="1" class="small-text" />
            </div>

        </div>

        <p style="margin-top:10px;">
            <?php esc_html_e( 'Preview:', 'digital-rise-woocommerce-filter' ); ?>
            <span class="drwf-format-preview-box" id="drwf-format-preview"><?php echo esc_html( drwf_format_price( 1234567.89 ) ); ?></span>
        </p>
        <p class="description"><?php esc_html_e( 'This format is applied to the range slider labels on the front end and the admin price previews.', 'digital-rise-woocommerce-filter' ); ?></p>
        <?php
    }

    /**
     * Category checkboxes grid.
     */
    public function field_allowed_categories() {
        $options      = get_option( $this->option_name, drwf_get_defaults() );
        $allowed_cats = isset( $options['allowed_categories'] ) ? (array) $options['allowed_categories'] : array();

        $categories = get_terms( array(
            'taxonomy'   => 'product_cat',
            'orderby'    => 'name',
            'order'      => 'ASC',
            'hide_empty' => false,
        ) );

        if ( empty( $categories ) || is_wp_error( $categories ) ) {
            echo '<p>' . esc_html__( 'No product categories found. Create some in Products → Categories first.', 'digital-rise-woocommerce-filter' ) . '</p>';
            return;
        }
        ?>
        <div class="drwf-select-helpers">
            <a href="#" class="drwf-select-all"><?php esc_html_e( 'Select All', 'digital-rise-woocommerce-filter' ); ?></a>
            <a href="#" class="drwf-select-none"><?php esc_html_e( 'Select None (show all)', 'digital-rise-woocommerce-filter' ); ?></a>
        </div>
        <div class="drwf-cat-grid">
            <?php foreach ( $categories as $cat ) : ?>
                <label>
                    <input type="checkbox"
                           name="<?php echo esc_attr( $this->option_name ); ?>[allowed_categories][]"
                           value="<?php echo esc_attr( $cat->term_id ); ?>"
                           <?php checked( in_array( (int) $cat->term_id, array_map( 'intval', $allowed_cats ), true ) ); ?> />
                    <?php echo esc_html( $cat->name ); ?>
                    <span class="count">(<?php echo esc_html( $cat->count ); ?>)</span>
                </label>
            <?php endforeach; ?>
        </div>
        <p class="description"><?php esc_html_e( 'Global default. Override per page: [digital_rise_price_filter categories="shoes,bags"]', 'digital-rise-woocommerce-filter' ); ?></p>
        <?php
    }

    /**
     * Color pickers grid.
     */
    public function field_colors() {
        $options  = get_option( $this->option_name, drwf_get_defaults() );
        $defaults = drwf_get_defaults();

        $color_fields = array(
            'color_primary'        => __( 'Primary / Slider / Filter Button', 'digital-rise-woocommerce-filter' ),
            'color_primary_hover'  => __( 'Primary Hover', 'digital-rise-woocommerce-filter' ),
            'color_title'          => __( 'Title Text', 'digital-rise-woocommerce-filter' ),
            'color_label'          => __( 'Label / Description Text', 'digital-rise-woocommerce-filter' ),
            'color_filter_bg'      => __( 'Filter Panel Background', 'digital-rise-woocommerce-filter' ),
            'color_filter_border'  => __( 'Filter Panel Border', 'digital-rise-woocommerce-filter' ),
            'color_price_range'    => __( 'Price Range Labels (slider)', 'digital-rise-woocommerce-filter' ),
            'color_card_bg'        => __( 'Product Card Background', 'digital-rise-woocommerce-filter' ),
            'color_card_border'    => __( 'Product Card Border', 'digital-rise-woocommerce-filter' ),
            'color_price'          => __( 'Product Price Text', 'digital-rise-woocommerce-filter' ),
            'color_cta_bg'         => __( 'CTA Button Background', 'digital-rise-woocommerce-filter' ),
            'color_cta_text'       => __( 'CTA Button Text', 'digital-rise-woocommerce-filter' ),
            'color_cta_hover'      => __( 'CTA Button Hover', 'digital-rise-woocommerce-filter' ),
            'color_reset_bg'       => __( 'Reset Button Background', 'digital-rise-woocommerce-filter' ),
            'color_reset_text'     => __( 'Reset Button Text', 'digital-rise-woocommerce-filter' ),
        );

        echo '<div class="drwf-color-grid">';
        foreach ( $color_fields as $key => $label ) {
            $val = isset( $options[ $key ] ) && '' !== $options[ $key ] ? $options[ $key ] : $defaults[ $key ];
            ?>
            <div class="drwf-color-item">
                <label><?php echo esc_html( $label ); ?></label>
                <input type="text"
                       name="<?php echo esc_attr( $this->option_name ); ?>[<?php echo esc_attr( $key ); ?>]"
                       value="<?php echo esc_attr( $val ); ?>"
                       class="drwf-color-field"
                       data-default-color="<?php echo esc_attr( $defaults[ $key ] ); ?>" />
            </div>
            <?php
        }
        echo '</div>';
    }

    /* ── Settings Page ── */

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }
        ?>
        <div class="wrap drwf-admin-wrap">

            <div class="drwf-admin-header">
                <h1><?php esc_html_e( 'Digital Rise WooCommerce Filter Solutions', 'digital-rise-woocommerce-filter' ); ?></h1>
                <p><?php esc_html_e( 'A lightweight, shortcode-based AJAX price range & category filter for WooCommerce.', 'digital-rise-woocommerce-filter' ); ?></p>
            </div>

            <div class="drwf-info-box">
                <h4><?php esc_html_e( 'Shortcode Usage', 'digital-rise-woocommerce-filter' ); ?></h4>
                <?php
                // Fetch real categories for dynamic examples.
                $example_cats = get_terms( array(
                    'taxonomy'   => 'product_cat',
                    'orderby'    => 'count',
                    'order'      => 'DESC',
                    'hide_empty' => true,
                    'number'     => 10,
                    'exclude'    => array( get_option( 'default_product_cat', 0 ) ), // exclude "Uncategorized"
                ) );

                $slug_examples = '';
                $id_examples   = '';

                if ( ! empty( $example_cats ) && ! is_wp_error( $example_cats ) ) {
                    // Pick up to 3 slugs for the slug example.
                    $slug_list = array();
                    $id_list   = array();
                    $count     = 0;
                    foreach ( $example_cats as $ecat ) {
                        if ( $count >= 3 ) {
                            break;
                        }
                        $slug_list[] = $ecat->slug;
                        $id_list[]   = $ecat->term_id;
                        $count++;
                    }
                    $slug_examples = implode( ',', $slug_list );
                    $id_examples   = implode( ',', $id_list );
                }

                // Fallback if no categories exist.
                if ( empty( $slug_examples ) ) {
                    $slug_examples = 'shoes,bags';
                    $id_examples   = '12,34';
                }
                ?>
                <p><strong><?php esc_html_e( 'Basic:', 'digital-rise-woocommerce-filter' ); ?></strong>
                    <code>[digital_rise_price_filter]</code></p>
                <p><strong><?php esc_html_e( 'Per-page categories (slug):', 'digital-rise-woocommerce-filter' ); ?></strong>
                    <code>[digital_rise_price_filter categories="<?php echo esc_html( $slug_examples ); ?>"]</code></p>
                <p><strong><?php esc_html_e( 'Per-page categories (ID):', 'digital-rise-woocommerce-filter' ); ?></strong>
                    <code>[digital_rise_price_filter categories="<?php echo esc_html( $id_examples ); ?>"]</code></p>
                <p><strong><?php esc_html_e( 'Custom price range:', 'digital-rise-woocommerce-filter' ); ?></strong>
                    <code>[digital_rise_price_filter min="50" max="5000" categories="<?php echo esc_html( isset( $slug_list[0] ) ? $slug_list[0] : 'electronics' ); ?>"]</code></p>

                <?php if ( ! empty( $example_cats ) && ! is_wp_error( $example_cats ) ) : ?>
                <div style="margin-top:12px; padding:10px 14px; background:#fff; border:1px solid #dcdcde; border-radius:6px;">
                    <strong><?php esc_html_e( 'Your available product categories:', 'digital-rise-woocommerce-filter' ); ?></strong>
                    <table class="widefat striped" style="margin-top:8px;">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Name', 'digital-rise-woocommerce-filter' ); ?></th>
                                <th><?php esc_html_e( 'Slug', 'digital-rise-woocommerce-filter' ); ?></th>
                                <th><?php esc_html_e( 'ID', 'digital-rise-woocommerce-filter' ); ?></th>
                                <th><?php esc_html_e( 'Products', 'digital-rise-woocommerce-filter' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $example_cats as $ecat ) : ?>
                            <tr>
                                <td><?php echo esc_html( $ecat->name ); ?></td>
                                <td><code><?php echo esc_html( $ecat->slug ); ?></code></td>
                                <td><code><?php echo esc_html( $ecat->term_id ); ?></code></td>
                                <td><?php echo esc_html( $ecat->count ); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>

            <div class="drwf-donate-box">
                <h3>&#x2764; <?php esc_html_e( 'Support This Plugin', 'digital-rise-woocommerce-filter' ); ?></h3>
                <p><?php esc_html_e( 'If you find this plugin helpful, please consider supporting its development with a small donation.', 'digital-rise-woocommerce-filter' ); ?></p>
                <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/"
                   target="_blank" rel="noopener noreferrer" class="button button-primary">
                    <?php esc_html_e( 'Donate Now', 'digital-rise-woocommerce-filter' ); ?>
                </a>
            </div>

            <form method="post" action="options.php">
                <?php
                settings_fields( $this->option_group );
                do_settings_sections( 'drwf-settings' );
                submit_button();
                ?>
            </form>

            <div class="drwf-admin-footer">
                <p>
                    <?php
                    printf(
                        esc_html__( 'Developed by %s', 'digital-rise-woocommerce-filter' ),
                        '<a href="https://digital-rise-solutions.com/" target="_blank" rel="noopener noreferrer">Digital Rise Solutions</a>'
                    );
                    ?>
                    &middot; <?php echo esc_html( 'v' . DRWF_VERSION ); ?>
                </p>
            </div>

        </div>
        <?php
    }

    public function plugin_action_links( $links ) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            esc_url( admin_url( 'admin.php?page=drwf-settings' ) ),
            esc_html__( 'Settings', 'digital-rise-woocommerce-filter' )
        );
        array_unshift( $links, $settings_link );
        return $links;
    }
}

new DRWF_Admin_Settings();
