<?php
/**
 * Plugin Name:       Digital Rise WooCommerce Filter Solutions
 * Plugin URI:        https://digital-rise-solutions.com/
 * Description:       A professional shortcode-based AJAX price range & category filter for WooCommerce. Collapsible UI, full color customization, per-page category control, thousand-separated prices.
 * Version:           1.0.0
 * Author:            Digital Rise Solutions
 * Author URI:        https://digital-rise-solutions.com/
 * License:           GPL-2.0+
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       digital-rise-woocommerce-filter
 * Domain Path:       /languages
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * WC requires at least: 6.0
 * WC tested up to:   9.5
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'DRWF_VERSION', '1.0.0' );
define( 'DRWF_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'DRWF_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'DRWF_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Return default plugin options.
 *
 * @return array
 */
function drwf_get_defaults() {
    return array(
        'ajax_enabled'         => '1',
        'default_min'          => '0',
        'default_max'          => '1000',
        'filter_title'         => 'Filter by Price',
        'button_label'         => 'Open',
        'allowed_categories'   => array(),
        'collapsed_default'    => '1',
        // Number formatting.
        'thousand_separator'   => '.',
        'decimal_separator'    => ',',
        'decimals'             => '0',
        // Colors.
        'color_primary'        => '#2271b1',
        'color_primary_hover'  => '#135e96',
        'color_title'          => '#1d2327',
        'color_label'          => '#50575e',
        'color_card_bg'        => '#ffffff',
        'color_card_border'    => '#dcdcde',
        'color_price'          => '#2271b1',
        'color_price_range'    => '#2271b1',
        'color_cta_bg'         => '#2271b1',
        'color_cta_text'       => '#ffffff',
        'color_cta_hover'      => '#135e96',
        'color_filter_bg'      => '#ffffff',
        'color_filter_border'  => '#dcdcde',
        'color_reset_bg'       => '#f0f0f1',
        'color_reset_text'     => '#50575e',
    );
}

/**
 * Class Digital_Rise_WooCommerce_Filter
 */
final class Digital_Rise_WooCommerce_Filter {

    private static $instance = null;

    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'plugins_loaded', array( $this, 'check_woocommerce' ) );
        add_action( 'init', array( $this, 'load_textdomain' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
        $this->includes();
        add_action( 'before_woocommerce_init', array( $this, 'declare_hpos_compatibility' ) );
    }

    public function declare_hpos_compatibility() {
        if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
        }
    }

    public function check_woocommerce() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            add_action( 'admin_notices', array( $this, 'woocommerce_missing_notice' ) );
        }
    }

    public function woocommerce_missing_notice() {
        ?>
        <div class="notice notice-error">
            <p>
                <?php
                printf(
                    esc_html__( '%1$sDigital Rise WooCommerce Filter Solutions%2$s requires %1$sWooCommerce%2$s to be installed and active.', 'digital-rise-woocommerce-filter' ),
                    '<strong>',
                    '</strong>'
                );
                ?>
            </p>
        </div>
        <?php
    }

    public function load_textdomain() {
        load_plugin_textdomain(
            'digital-rise-woocommerce-filter',
            false,
            dirname( DRWF_PLUGIN_BASENAME ) . '/languages'
        );
    }

    private function includes() {
        require_once DRWF_PLUGIN_DIR . 'includes/shortcode.php';
        require_once DRWF_PLUGIN_DIR . 'includes/ajax-handler.php';

        if ( is_admin() ) {
            require_once DRWF_PLUGIN_DIR . 'includes/admin-settings.php';
        }
    }

    /**
     * Enqueue front-end assets + inject CSS custom properties + number format config.
     */
    public function enqueue_assets() {
        wp_register_style(
            'drwf-filter-style',
            DRWF_PLUGIN_URL . 'assets/css/filter.css',
            array(),
            DRWF_VERSION
        );

        wp_register_script(
            'drwf-filter-script',
            DRWF_PLUGIN_URL . 'assets/js/filter.js',
            array( 'jquery' ),
            DRWF_VERSION,
            true
        );

        $options  = get_option( 'drwf_settings', drwf_get_defaults() );
        $defaults = drwf_get_defaults();

        // Build CSS custom properties.
        $color_vars = '';
        $color_keys = array(
            'color_primary', 'color_primary_hover', 'color_title', 'color_label',
            'color_card_bg', 'color_card_border', 'color_price', 'color_price_range',
            'color_cta_bg', 'color_cta_text', 'color_cta_hover',
            'color_filter_bg', 'color_filter_border',
            'color_reset_bg', 'color_reset_text',
        );
        foreach ( $color_keys as $key ) {
            $val     = isset( $options[ $key ] ) && '' !== $options[ $key ] ? $options[ $key ] : $defaults[ $key ];
            $css_var = '--drwf-' . str_replace( '_', '-', str_replace( 'color_', '', $key ) );
            $color_vars .= $css_var . ':' . esc_attr( $val ) . ';';
        }

        wp_add_inline_style( 'drwf-filter-style', ':root{' . $color_vars . '}' );

        wp_localize_script( 'drwf-filter-script', 'drwf_ajax', array(
            'ajax_url'           => admin_url( 'admin-ajax.php' ),
            'nonce'              => wp_create_nonce( 'drwf_filter_nonce' ),
            'ajax_enabled'       => isset( $options['ajax_enabled'] ) ? $options['ajax_enabled'] : '1',
            'collapsed'          => isset( $options['collapsed_default'] ) ? $options['collapsed_default'] : '1',
            'thousand_separator' => isset( $options['thousand_separator'] ) ? $options['thousand_separator'] : '.',
            'decimal_separator'  => isset( $options['decimal_separator'] ) ? $options['decimal_separator'] : ',',
            'decimals'           => isset( $options['decimals'] ) ? intval( $options['decimals'] ) : 0,
        ) );
    }
}

/**
 * Format a price number using admin settings (thousand separator, decimal separator, decimals).
 *
 * @param  float  $number The price number.
 * @return string         Formatted string e.g. "1.000.000" or "1,000,000.00".
 */
function drwf_format_price( $number ) {
    $options   = get_option( 'drwf_settings', drwf_get_defaults() );
    $decimals  = isset( $options['decimals'] ) ? intval( $options['decimals'] ) : 0;
    $dec_sep   = isset( $options['decimal_separator'] ) ? $options['decimal_separator'] : ',';
    $thou_sep  = isset( $options['thousand_separator'] ) ? $options['thousand_separator'] : '.';

    return number_format( floatval( $number ), $decimals, $dec_sep, $thou_sep );
}

/**
 * Activation hook.
 */
function drwf_activate() {
    if ( false === get_option( 'drwf_settings' ) ) {
        add_option( 'drwf_settings', drwf_get_defaults() );
    }
}
register_activation_hook( __FILE__, 'drwf_activate' );

function drwf_deactivate() {}
register_deactivation_hook( __FILE__, 'drwf_deactivate' );

Digital_Rise_WooCommerce_Filter::get_instance();
