/**
 * Digital Rise WooCommerce Filter Solutions – Front-end JS
 *
 * Supports multiple shortcode instances per page.
 * Collapsible panel, dual-range slider, category checkboxes, AJAX filter.
 * Price formatting uses admin-configured thousand/decimal separators.
 *
 * @package Digital_Rise_WooCommerce_Filter
 */

/* global jQuery, drwf_ajax */
(function ($) {
    'use strict';

    /* ──────────────────────────
       Number formatting from admin settings
       ────────────────────────── */
    var thousandSep = (typeof drwf_ajax !== 'undefined' && drwf_ajax.thousand_separator !== undefined)
        ? drwf_ajax.thousand_separator
        : '.';
    var decimalSep  = (typeof drwf_ajax !== 'undefined' && drwf_ajax.decimal_separator !== undefined)
        ? drwf_ajax.decimal_separator
        : ',';
    var decimals    = (typeof drwf_ajax !== 'undefined' && drwf_ajax.decimals !== undefined)
        ? parseInt(drwf_ajax.decimals, 10)
        : 0;

    /**
     * Format a number with admin-configured thousand/decimal separators.
     *
     * Example: formatPrice(1000000) → "1.000.000" when thou="." dec="," decs=0
     *          formatPrice(1000000) → "1,000,000.00" when thou="," dec="." decs=2
     *
     * @param  {number} num
     * @return {string}
     */
    function formatPrice(num) {
        num = parseFloat(num);
        if (isNaN(num)) {
            return '0';
        }

        // Fixed decimal places.
        var fixed = num.toFixed(decimals);
        var parts = fixed.split('.');
        var intPart = parts[0];
        var decPart = parts.length > 1 ? parts[1] : '';

        // Add thousand separator.
        if (thousandSep !== '') {
            intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, thousandSep);
        }

        if (decimals > 0 && decPart !== '') {
            return intPart + decimalSep + decPart;
        }

        return intPart;
    }

    /* ──────────────────────────
       Init each instance
       ────────────────────────── */
    $('.drwf-instance').each(function () {
        var $wrap      = $(this);
        var $header    = $wrap.find('.drwf-filter-header');
        var $body      = $wrap.find('.drwf-filter-body');
        var $icon      = $wrap.find('.drwf-toggle-icon');
        var $rangeMin  = $wrap.find('.drwf-range-input--min');
        var $rangeMax  = $wrap.find('.drwf-range-input--max');
        var $fill      = $wrap.find('.drwf-range-fill');
        var $labelMin  = $wrap.find('.drwf-range-label-min .drwf-val');
        var $labelMax  = $wrap.find('.drwf-range-label-max .drwf-val');
        var $filterBtn = $wrap.find('.drwf-btn-filter');
        var $resetBtn  = $wrap.find('.drwf-btn-reset');
        var $products  = $wrap.find('.drwf-products-grid');
        var $loading   = $wrap.find('.drwf-loading');

        var absMin     = parseFloat($wrap.data('min')) || 0;
        var absMax     = parseFloat($wrap.data('max')) || 1000;
        var scopeCats  = $wrap.data('scope-cats') ? String($wrap.data('scope-cats')) : '';

        /* ── Collapsible Toggle ── */
        $header.on('click keydown', function (e) {
            if (e.type === 'keydown' && e.which !== 13 && e.which !== 32) {
                return;
            }
            e.preventDefault();

            var isCollapsed = $body.hasClass('drwf-collapsed');
            if (isCollapsed) {
                $body.removeClass('drwf-collapsed');
                $icon.addClass('drwf-open');
                $header.attr('aria-expanded', 'true');
            } else {
                $body.addClass('drwf-collapsed');
                $icon.removeClass('drwf-open');
                $header.attr('aria-expanded', 'false');
            }
        });

        /* ── Slider UI ── */
        function updateSliderUI() {
            var min = parseFloat($rangeMin.val());
            var max = parseFloat($rangeMax.val());

            if (min > max) {
                $rangeMin.val(max);
                min = max;
            }

            var range   = absMax - absMin;
            var leftPc  = range ? ((min - absMin) / range) * 100 : 0;
            var rightPc = range ? ((max - absMin) / range) * 100 : 100;

            $fill.css({
                left:  leftPc + '%',
                width: (rightPc - leftPc) + '%'
            });

            $labelMin.text(formatPrice(min));
            $labelMax.text(formatPrice(max));
        }

        $rangeMin.on('input', function () {
            if (parseFloat($(this).val()) > parseFloat($rangeMax.val())) {
                $(this).val($rangeMax.val());
            }
            updateSliderUI();
        });

        $rangeMax.on('input', function () {
            if (parseFloat($(this).val()) < parseFloat($rangeMin.val())) {
                $(this).val($rangeMin.val());
            }
            updateSliderUI();
        });

        updateSliderUI();

        /* ── Loading ── */
        function toggleLoading(show) {
            if (show) {
                $products.css('opacity', '0.35');
                $loading.show();
                $filterBtn.prop('disabled', true);
            } else {
                $products.css('opacity', '1');
                $loading.hide();
                $filterBtn.prop('disabled', false);
            }
        }

        /* ── Categories ── */
        function getSelectedCategories() {
            var cats = [];
            $wrap.find('.drwf-cat-checkbox:checked').each(function () {
                cats.push(parseInt($(this).val(), 10));
            });
            return cats;
        }

        /* ── AJAX Filter ── */
        function filterProducts() {
            var minPrice   = parseFloat($rangeMin.val()) || 0;
            var maxPrice   = parseFloat($rangeMax.val()) || absMax;
            var categories = getSelectedCategories();

            if (typeof drwf_ajax === 'undefined' || drwf_ajax.ajax_enabled === '0') {
                var url = new URL(window.location.href);
                url.searchParams.set('drwf_min', minPrice);
                url.searchParams.set('drwf_max', maxPrice);
                url.searchParams.set('drwf_cats', categories.join(','));
                window.location.href = url.toString();
                return;
            }

            toggleLoading(true);

            $.ajax({
                url:  drwf_ajax.ajax_url,
                type: 'POST',
                data: {
                    action:     'drwf_filter_products',
                    nonce:      drwf_ajax.nonce,
                    min_price:  minPrice,
                    max_price:  maxPrice,
                    categories: categories,
                    scope_cats: scopeCats
                },
                success: function (response) {
                    if (response.success && response.data && response.data.html) {
                        $products.html(response.data.html);
                    } else {
                        $products.html(
                            '<p class="drwf-no-products">No products found matching your criteria.</p>'
                        );
                    }
                },
                error: function () {
                    $products.html(
                        '<p class="drwf-no-products">Something went wrong. Please try again.</p>'
                    );
                },
                complete: function () {
                    toggleLoading(false);
                }
            });
        }

        /* ── Reset ── */
        function resetFilter() {
            $rangeMin.val(absMin);
            $rangeMax.val(absMax);
            updateSliderUI();
            $wrap.find('.drwf-cat-checkbox').prop('checked', true);
            filterProducts();
        }

        /* ── Bind ── */
        $filterBtn.on('click', filterProducts);
        $resetBtn.on('click', resetFilter);
    });

})(jQuery);
