<?php
/**
 * Plugin Name:       Digital Rise WooCommerce Badge
 * Plugin URI:        https://digital-rise-solutions.com/
 * Description:       Display customizable out-of-stock badges on WooCommerce product images based on product category, with per-product custom text override. Unlimited configurations, overlay mask, all positions, forced display.
 * Version:           3.0.0
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * Author:            Digital Rise Solutions
 * Author URI:        https://digital-rise-solutions.com/
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       dr-woo-badge
 * WC requires at least: 6.0
 * WC tested up to:   9.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

define( 'DRWB_VER', '3.0.0' );
define( 'DRWB_FILE', __FILE__ );
define( 'DRWB_URL', plugin_dir_url( __FILE__ ) );

final class DRWB {

    private static $i = null;
    private $done = [];
    private $lock = false;

    // DB option keys.
    const OPT     = 'drwb_settings';
    const BADGES  = 'drwb_badges';
    const META    = '_drwb_custom_badge';  // Per-product meta key.
    const MIGRATED = 'drwb_v3_migrated';

    public static function go(): self {
        if ( ! self::$i ) self::$i = new self();
        return self::$i;
    }

    private function __construct() {
        add_action( 'plugins_loaded', [ $this, 'boot' ] );
    }

    /* ================================================================
     *  BOOT
     * ==============================================================*/
    public function boot(): void {
        if ( ! class_exists( 'WooCommerce' ) ) {
            add_action( 'admin_notices', function() {
                echo '<div class="notice notice-error"><p><strong>Digital Rise WooCommerce Badge:</strong> WooCommerce is required.</p></div>';
            });
            return;
        }

        add_action( 'before_woocommerce_init', function() {
            if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) )
                \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', DRWB_FILE, true );
        });

        $this->migrate();

        if ( is_admin() ) {
            add_action( 'admin_menu', [ $this, 'menu' ] );
            add_action( 'admin_enqueue_scripts', [ $this, 'admin_assets' ] );
            add_action( 'wp_ajax_drwb_save_settings', [ $this, 'ax_save_settings' ] );
            add_action( 'wp_ajax_drwb_save_badge',    [ $this, 'ax_save_badge' ] );
            add_action( 'wp_ajax_drwb_delete_badge',   [ $this, 'ax_delete_badge' ] );
            add_action( 'wp_ajax_drwb_purge_all',      [ $this, 'ax_purge_all' ] );

            // Per-product meta box.
            add_action( 'add_meta_boxes', [ $this, 'add_product_metabox' ] );
            add_action( 'woocommerce_process_product_meta', [ $this, 'save_product_meta' ] );
        }

        // Frontend.
        add_action( 'wp_enqueue_scripts', [ $this, 'front_css' ] );

        // Hook-based rendering (non-force mode).
        add_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'hook_loop' ], 10 );
        add_action( 'woocommerce_before_single_product_summary', [ $this, 'hook_single' ], 11 );

        // Force mode: filter product images.
        add_filter( 'woocommerce_product_get_image', [ $this, 'filter_img' ], 10, 5 );
    }

    /* ================================================================
     *  MIGRATION
     * ==============================================================*/
    private function migrate(): void {
        if ( get_option( self::MIGRATED ) ) return;

        $badges = get_option( self::BADGES, [] );
        if ( ! is_array( $badges ) || empty( $badges ) ) {
            update_option( self::MIGRATED, '1' );
            return;
        }

        $clean = [];
        foreach ( $badges as $old_id => $b ) {
            if ( ! is_array( $b ) ) continue;
            $new_id = 'b' . substr( md5( $old_id . wp_rand() ), 0, 10 );

            // Map old field names to new.
            if ( isset( $b['overlay_opacity'] ) && ! isset( $b['opacity'] ) )
                $b['opacity'] = $b['overlay_opacity'];
            if ( isset( $b['font_size'] ) && ! isset( $b['fontsize'] ) )
                $b['fontsize'] = $b['font_size'];

            unset( $b['overlay_opacity'], $b['font_size'] );
            $clean[ $new_id ] = $b;
        }

        update_option( self::BADGES, $clean, 'yes' );
        update_option( self::MIGRATED, '1' );
    }

    /* ================================================================
     *  DATA
     * ==============================================================*/
    private function opt(): array {
        static $c = null;
        if ( $c ) return $c;
        return $c = wp_parse_args( get_option( self::OPT, [] ), [
            'enabled' => '1', 'force_badge' => '1', 'custom_css' => '',
        ]);
    }

    private function badges(): array {
        $r = get_option( self::BADGES, [] );
        return is_array( $r ) ? $r : [];
    }

    private function on(): bool  { return ! empty( $this->opt()['enabled'] ); }
    private function frc(): bool { return ! empty( $this->opt()['force_badge'] ); }

    private function mkid(): string {
        return 'b' . substr( md5( uniqid( wp_rand(), true ) ), 0, 10 );
    }

    private function cleanid( string $s ): string {
        return preg_replace( '/[^a-z0-9_]/', '', strtolower( trim( $s ) ) );
    }

    /* Position keys ↔ labels */
    private function positions(): array {
        return [
            'top-left'      => 'Top Left',
            'top-center'    => 'Top Center',
            'top-right'     => 'Top Right',
            'center'        => 'Center',
            'bottom-left'   => 'Bottom Left',
            'bottom-center' => 'Bottom Center',
            'bottom-right'  => 'Bottom Right',
            'full-overlay'  => 'Full Image Overlay',
        ];
    }

    /* Position key → inline CSS */
    private function pos_css( string $p ): string {
        $m = [
            'top-left'      => 'top:8px;left:8px;',
            'top-center'    => 'top:8px;left:50%;transform:translateX(-50%);',
            'top-right'     => 'top:8px;right:8px;',
            'center'        => 'top:50%;left:50%;transform:translate(-50%,-50%);',
            'bottom-left'   => 'bottom:8px;left:8px;',
            'bottom-center' => 'bottom:8px;left:50%;transform:translateX(-50%);',
            'bottom-right'  => 'bottom:8px;right:8px;',
            'full-overlay'  => 'top:0;left:0;width:100%;height:100%;border-radius:0;',
        ];
        return $m[ $p ] ?? $m['top-left'];
    }

    private function san( array $d ): array {
        $pk = array_keys( $this->positions() );
        return [
            'enabled'  => ! empty( $d['enabled'] ) ? '1' : '',
            'category' => absint( $d['category'] ?? 0 ),
            'text'     => sanitize_text_field( $d['text'] ?? '' ),
            'bg'       => sanitize_hex_color( $d['bg'] ?? '#e74c3c' ) ?: '#e74c3c',
            'color'    => sanitize_hex_color( $d['color'] ?? '#ffffff' ) ?: '#ffffff',
            'position' => in_array( $d['position'] ?? '', $pk, true ) ? $d['position'] : 'top-left',
            'opacity'  => max( 0, min( 100, absint( $d['opacity'] ?? 55 ) ) ),
            'fontsize' => max( 8, min( 60, absint( $d['fontsize'] ?? 14 ) ) ),
        ];
    }

    private function hex2rgb( string $h ): array {
        $h = ltrim( $h, '#' );
        if ( 3 === strlen( $h ) ) $h = $h[0].$h[0].$h[1].$h[1].$h[2].$h[2];
        return [ hexdec( substr($h,0,2) ), hexdec( substr($h,2,2) ), hexdec( substr($h,4,2) ) ];
    }

    /* ================================================================
     *  PER-PRODUCT META BOX
     * ==============================================================*/
    public function add_product_metabox(): void {
        add_meta_box(
            'drwb_product_badge',
            __( 'Custom Out-of-Stock Badge', 'dr-woo-badge' ),
            [ $this, 'render_product_metabox' ],
            'product',
            'side',
            'default'
        );
    }

    public function render_product_metabox( $post ): void {
        $meta = get_post_meta( $post->ID, self::META, true );
        $meta = wp_parse_args( is_array( $meta ) ? $meta : [], [
            'enabled'  => '',
            'text'     => '',
            'bg'       => '#e74c3c',
            'color'    => '#ffffff',
            'position' => 'center',
            'opacity'  => 55,
            'fontsize' => 16,
        ]);
        wp_nonce_field( 'drwb_product_meta', 'drwb_product_nonce' );
        ?>
        <p class="description" style="margin-bottom:12px;">
            <?php esc_html_e( 'Override the category badge with custom text for this product only. Leave disabled to use category badge.', 'dr-woo-badge' ); ?>
        </p>
        <p>
            <label><input type="checkbox" name="drwb_meta[enabled]" value="1" <?php checked( $meta['enabled'], '1' ); ?> />
            <strong><?php esc_html_e( 'Enable custom badge for this product', 'dr-woo-badge' ); ?></strong></label>
        </p>
        <p>
            <label><?php esc_html_e( 'Badge Text:', 'dr-woo-badge' ); ?></label><br>
            <input type="text" name="drwb_meta[text]" value="<?php echo esc_attr( $meta['text'] ); ?>" class="widefat"
                   placeholder="<?php esc_attr_e( 'e.g. Waiting for product proof from ministry', 'dr-woo-badge' ); ?>" />
        </p>
        <p>
            <label><?php esc_html_e( 'Position:', 'dr-woo-badge' ); ?></label><br>
            <select name="drwb_meta[position]" class="widefat">
                <?php foreach ( $this->positions() as $k => $l ) : ?>
                    <option value="<?php echo esc_attr( $k ); ?>" <?php selected( $meta['position'], $k ); ?>><?php echo esc_html( $l ); ?></option>
                <?php endforeach; ?>
            </select>
        </p>
        <p>
            <label><?php esc_html_e( 'Background:', 'dr-woo-badge' ); ?></label>
            <input type="color" name="drwb_meta[bg]" value="<?php echo esc_attr( $meta['bg'] ); ?>" />
            &nbsp;
            <label><?php esc_html_e( 'Text:', 'dr-woo-badge' ); ?></label>
            <input type="color" name="drwb_meta[color]" value="<?php echo esc_attr( $meta['color'] ); ?>" />
        </p>
        <p>
            <label><?php esc_html_e( 'Font Size:', 'dr-woo-badge' ); ?></label>
            <input type="number" name="drwb_meta[fontsize]" value="<?php echo esc_attr( $meta['fontsize'] ); ?>" min="8" max="60" style="width:60px" />px
            &nbsp;&nbsp;
            <label><?php esc_html_e( 'Opacity:', 'dr-woo-badge' ); ?></label>
            <input type="number" name="drwb_meta[opacity]" value="<?php echo esc_attr( $meta['opacity'] ); ?>" min="0" max="100" step="5" style="width:60px" />%
        </p>
        <?php
    }

    public function save_product_meta( int $post_id ): void {
        if ( ! isset( $_POST['drwb_product_nonce'] ) || ! wp_verify_nonce( $_POST['drwb_product_nonce'], 'drwb_product_meta' ) ) return;
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;
        if ( ! current_user_can( 'edit_product', $post_id ) ) return;

        $raw = isset( $_POST['drwb_meta'] ) && is_array( $_POST['drwb_meta'] ) ? wp_unslash( $_POST['drwb_meta'] ) : [];

        $meta = [
            'enabled'  => ! empty( $raw['enabled'] ) ? '1' : '',
            'text'     => sanitize_text_field( $raw['text'] ?? '' ),
            'bg'       => sanitize_hex_color( $raw['bg'] ?? '#e74c3c' ) ?: '#e74c3c',
            'color'    => sanitize_hex_color( $raw['color'] ?? '#ffffff' ) ?: '#ffffff',
            'position' => in_array( $raw['position'] ?? '', array_keys( $this->positions() ), true ) ? $raw['position'] : 'center',
            'opacity'  => max( 0, min( 100, absint( $raw['opacity'] ?? 55 ) ) ),
            'fontsize' => max( 8, min( 60, absint( $raw['fontsize'] ?? 16 ) ) ),
        ];

        if ( ! empty( $meta['enabled'] ) && ! empty( $meta['text'] ) ) {
            update_post_meta( $post_id, self::META, $meta );
        } else {
            delete_post_meta( $post_id, self::META );
        }
    }

    /* ================================================================
     *  ADMIN AJAX
     * ==============================================================*/
    public function ax_save_settings(): void {
        check_ajax_referer( 'drwb_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_woocommerce' ) ) wp_send_json_error();
        $s = [
            'enabled'     => ! empty( $_POST['enabled'] ) ? '1' : '',
            'force_badge' => ! empty( $_POST['force_badge'] ) ? '1' : '',
            'custom_css'  => wp_strip_all_tags( wp_unslash( $_POST['custom_css'] ?? '' ) ),
        ];
        update_option( self::OPT, $s, 'yes' );
        wp_send_json_success( [ 'settings' => $s ] );
    }

    public function ax_save_badge(): void {
        check_ajax_referer( 'drwb_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_woocommerce' ) ) wp_send_json_error();
        $id  = isset( $_POST['badge_id'] ) ? $this->cleanid( wp_unslash( $_POST['badge_id'] ) ) : '';
        $raw = isset( $_POST['badge'] ) && is_array( $_POST['badge'] ) ? wp_unslash( $_POST['badge'] ) : [];
        if ( ! $raw ) wp_send_json_error( 'No data' );
        $bid = $id ?: $this->mkid();
        $all = $this->badges();
        $all[ $bid ] = $this->san( $raw );
        update_option( self::BADGES, $all, 'yes' );
        wp_send_json_success( [ 'badge_id' => $bid, 'badge' => $all[$bid], 'all' => $all ?: new \stdClass ] );
    }

    public function ax_delete_badge(): void {
        check_ajax_referer( 'drwb_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_woocommerce' ) ) wp_send_json_error();
        $id  = isset( $_POST['badge_id'] ) ? $this->cleanid( wp_unslash( $_POST['badge_id'] ) ) : '';
        if ( ! $id ) wp_send_json_error( 'No ID' );
        $all = $this->badges();
        unset( $all[ $id ] );
        update_option( self::BADGES, $all, 'yes' );
        wp_send_json_success( [ 'all' => $all ?: new \stdClass ] );
    }

    public function ax_purge_all(): void {
        check_ajax_referer( 'drwb_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_woocommerce' ) ) wp_send_json_error();
        delete_option( self::BADGES );
        wp_send_json_success( [ 'all' => new \stdClass ] );
    }

    /* ================================================================
     *  ADMIN PAGE
     * ==============================================================*/
    public function menu(): void {
        add_submenu_page( 'woocommerce', 'Out of Stock Badges', 'Out of Stock Badges', 'manage_woocommerce', 'drwb-settings', [ $this, 'admin_page' ] );
    }

    public function admin_assets( string $hook ): void {
        if ( 'woocommerce_page_drwb-settings' !== $hook ) return;
        wp_enqueue_style( 'wp-color-picker' );
        wp_enqueue_script( 'drwb-admin', DRWB_URL.'assets/js/admin.js', ['jquery','wp-color-picker'], DRWB_VER, true );
        wp_enqueue_style( 'drwb-admin', DRWB_URL.'assets/css/admin.css', [], DRWB_VER );

        $terms = get_terms( [ 'taxonomy'=>'product_cat', 'hide_empty'=>false, 'orderby'=>'name' ] );
        $cats = [];
        if ( ! is_wp_error( $terms ) ) foreach ( $terms as $t ) $cats[] = [ 'id'=>$t->term_id, 'name'=>$t->name ];

        $b = get_option( self::BADGES, [] );
        wp_localize_script( 'drwb-admin', 'DRWB', [
            'ajax_url'   => admin_url('admin-ajax.php'),
            'nonce'      => wp_create_nonce('drwb_nonce'),
            'categories' => $cats,
            'positions'  => $this->positions(),
            'badges'     => $b ?: new \stdClass,
            'settings'   => $this->opt(),
        ]);
    }

    public function admin_page(): void {
        if ( ! current_user_can( 'manage_woocommerce' ) ) return;
        $raw = get_option( self::BADGES, [] );
        $cnt = is_array($raw) ? count($raw) : 0;
        ?>
        <div class="wrap drwb-wrap">
            <div class="drwb-header">
                <div class="drwb-header__title">
                    <h1><span class="dashicons dashicons-tag"></span> Out of Stock Badges</h1>
                    <p class="drwb-header__sub">Category-based badges for out-of-stock products &mdash; <a href="https://digital-rise-solutions.com/" target="_blank">Digital Rise Solutions</a></p>
                </div>
                <span class="drwb-ver">v<?php echo DRWB_VER; ?></span>
            </div>

            <details style="margin-bottom:14px;font-size:11px;color:#888;"><summary>Debug: <?php echo $cnt; ?> badge(s) in DB</summary><pre style="background:#f5f5f5;padding:8px;max-height:150px;overflow:auto;margin-top:6px;"><?php echo esc_html(print_r($raw,true)); ?></pre></details>

            <div class="drwb-card">
                <div class="drwb-card__hd"><h2>Global Settings</h2><button id="drwb-save-global" class="button button-primary">Save Settings</button></div>
                <div class="drwb-card__bd">
                    <div class="drwb-row"><label class="drwb-tog"><input type="checkbox" id="drwb-global-enabled" value="1"/><span class="drwb-tog__sl"></span><span>Enable Plugin</span></label></div>
                    <div class="drwb-row"><label class="drwb-tog"><input type="checkbox" id="drwb-force-badge" value="1"/><span class="drwb-tog__sl"></span><span>Force Badge Display</span></label><p class="description">Injects badge into the product image HTML for maximum theme compatibility.</p></div>
                    <div class="drwb-row"><label><strong>Custom CSS</strong></label><textarea id="drwb-custom-css" rows="3" class="large-text code" placeholder=".drwb-badge { }"></textarea></div>
                </div>
                <div class="drwb-card__st" id="drwb-global-status"></div>
            </div>

            <div class="drwb-card" style="background:#fff8e1;border-color:#ffe082;">
                <div class="drwb-card__bd" style="padding:12px 20px;">
                    <strong>💡 Per-Product Custom Badge:</strong> Edit any product → see the <em>"Custom Out-of-Stock Badge"</em> box on the right sidebar. Add custom text like <em>"Waiting for product proof from ministry"</em> for that specific product. This overrides the category badge.
                </div>
            </div>

            <div class="drwb-badges-hd">
                <h2>Badge Configurations</h2>
                <div>
                    <button id="drwb-purge" class="button" style="color:#d63638;border-color:#d63638;margin-right:8px;">Delete All</button>
                    <button id="drwb-add-badge" class="button button-primary"><span class="dashicons dashicons-plus-alt2"></span> Add New Badge</button>
                </div>
            </div>
            <div id="drwb-list"></div>
            <div id="drwb-empty" class="drwb-empty" style="display:none;"><span class="dashicons dashicons-tag"></span><p>No badges configured. Click "Add New Badge" to start.</p></div>
        </div>
        <?php
    }

    /* ================================================================
     *  FRONTEND CSS
     * ==============================================================*/
    public function front_css(): void {
        if ( ! $this->on() ) return;
        wp_enqueue_style( 'drwb', DRWB_URL.'assets/css/drwb-badge.css', [], DRWB_VER.'.'.time() );
        $c = $this->opt()['custom_css'] ?? '';
        if ( $c ) wp_add_inline_style( 'drwb', wp_strip_all_tags( $c ) );
    }

    /* ================================================================
     *  FRONTEND RENDERING
     * ==============================================================*/

    /**
     * Hook: shop/archive loop (non-force mode only).
     */
    public function hook_loop(): void {
        if ( $this->frc() ) return;
        global $product;
        if ( $product instanceof WC_Product ) $this->emit( $product );
    }

    /**
     * Hook: single product page – ONLY for the main product, NOT related.
     */
    public function hook_single(): void {
        // Only fire on the main product, not related/upsell.
        if ( ! is_product() ) return;
        global $product;
        if ( ! $product instanceof WC_Product ) return;

        // Verify this is the main queried product.
        $main_id = get_queried_object_id();
        if ( $product->get_id() !== $main_id ) return;

        $this->emit( $product );
    }

    /**
     * Filter: woocommerce_product_get_image (force mode).
     * Only fires for out-of-stock products in shop/archive context.
     */
    public function filter_img( $html, $product, $size = '', $attr = [], $ph = false ): string {
        if ( $this->lock || ! $this->on() || ! $this->frc() ) return $html;
        if ( ! $product instanceof WC_Product ) return $html;
        if ( $product->is_in_stock() ) return $html;

        // CRITICAL: Skip if we're on a single product page rendering related products.
        // Only allow filter for the MAIN product on single pages.
        if ( is_product() ) {
            $main_id = get_queried_object_id();
            if ( $product->get_id() !== $main_id ) {
                // This is a related/upsell product on the single page — skip filter, use hook instead.
                return $html;
            }
        }

        $badge = $this->build( $product );
        if ( ! $badge ) return $html;
        return '<span class="drwb-wrap">' . $html . $badge . '</span>';
    }

    /**
     * Emit badge HTML (with duplicate guard).
     */
    private function emit( WC_Product $p ): void {
        if ( ! $this->on() || $p->is_in_stock() ) return;
        $k = $p->get_id();
        if ( isset( $this->done[$k] ) ) return;
        $h = $this->build( $p );
        if ( $h ) { echo $h; $this->done[$k] = 1; } // phpcs:ignore
    }

    /**
     * Build badge HTML. Priority:
     *  1. Per-product custom badge (meta box)
     *  2. Category-based badge (admin settings)
     *
     * ALL visual properties are inline styles — no CSS class dependency.
     */
    private function build( WC_Product $p ): string {
        // --- 1. Per-product override ---
        $meta = get_post_meta( $p->get_id(), self::META, true );
        if ( is_array( $meta ) && ! empty( $meta['enabled'] ) && ! empty( $meta['text'] ) ) {
            return $this->render_badge(
                $meta['text'],
                $meta['position'] ?? 'center',
                $meta['bg'] ?? '#e74c3c',
                $meta['color'] ?? '#ffffff',
                absint( $meta['fontsize'] ?? 16 ),
                absint( $meta['opacity'] ?? 55 )
            );
        }

        // --- 2. Category badge ---
        $badges = $this->badges();
        $pcats  = wc_get_product_cat_ids( $p->get_id() );
        if ( ! $badges || ! $pcats ) return '';

        foreach ( $badges as $b ) {
            if ( empty( $b['enabled'] ) || empty( $b['category'] ) || empty( $b['text'] ) ) continue;
            if ( ! in_array( absint( $b['category'] ), $pcats, true ) ) continue;

            return $this->render_badge(
                $b['text'],
                $b['position'] ?? 'top-left',
                $b['bg'] ?? '#e74c3c',
                $b['color'] ?? '#ffffff',
                max( 8, absint( $b['fontsize'] ?? 14 ) ),
                max( 0, min( 100, absint( $b['opacity'] ?? 55 ) ) )
            );
        }

        return '';
    }

    /**
     * Render a single badge <span> with full inline styles.
     */
    private function render_badge( string $text, string $pos, string $bg, string $col, int $fs, int $opacity ): string {
        $pcss  = $this->pos_css( $pos );
        $is_ov = ( 'full-overlay' === $pos );
        $is_ct = ( 'center' === $pos );

        // --- Full Overlay ---
        if ( $is_ov ) {
            $rgb = $this->hex2rgb( $bg );
            $a   = round( $opacity / 100, 2 );
            $os  = 'position:absolute;'.$pcss
                . 'display:flex;align-items:center;justify-content:center;'
                . 'background-color:rgba('.$rgb[0].','.$rgb[1].','.$rgb[2].','.$a.');'
                . 'z-index:10;pointer-events:none;box-sizing:border-box;';
            $ts  = 'font-size:'.$fs.'px;color:'.esc_attr($col).';'
                . 'font-weight:800;letter-spacing:2px;text-transform:uppercase;'
                . 'text-shadow:0 2px 12px rgba(0,0,0,.5),0 0 4px rgba(0,0,0,.3);'
                . 'padding:16px 32px;line-height:1.3;text-align:center;'
                . 'font-family:-apple-system,BlinkMacSystemFont,Segoe UI,Roboto,sans-serif;';
            return '<span class="drwb-badge" style="'.esc_attr($os).'">'
                 . '<span style="'.esc_attr($ts).'">'.esc_html($text).'</span></span>';
        }

        // --- Center (bigger, with backdrop) ---
        $base = 'position:absolute;'.$pcss
            . 'background-color:'.esc_attr($bg).';color:'.esc_attr($col).';'
            . 'font-size:'.$fs.'px;font-weight:700;line-height:1.4;'
            . 'letter-spacing:.5px;text-transform:uppercase;text-align:center;'
            . 'pointer-events:none;box-sizing:border-box;z-index:10;'
            . 'font-family:-apple-system,BlinkMacSystemFont,Segoe UI,Roboto,sans-serif;'
            . 'display:inline-flex;align-items:center;justify-content:center;white-space:nowrap;';

        if ( $is_ct ) {
            $base .= 'padding:12px 28px;border-radius:6px;box-shadow:0 4px 20px rgba(0,0,0,.3);';
        } else {
            $base .= 'padding:6px 16px;border-radius:4px;box-shadow:0 2px 8px rgba(0,0,0,.15);';
        }

        return '<span class="drwb-badge" style="'.esc_attr($base).'">'.esc_html($text).'</span>';
    }
}

DRWB::go();
