<?php
/**
 * Frontend Class
 *
 * @package Digital_Rise_WhatsApp_Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Frontend Class
 */
class DRWS_Frontend {

	/**
	 * Constructor.
	 */
	public function __construct() {
		// Only run on frontend.
		if ( is_admin() ) {
			return;
		}

		// Disable WooCommerce purchasing functionality.
		$this->disable_woocommerce_purchasing();

		// Add WhatsApp button.
		$this->add_whatsapp_button();

		// Enqueue frontend assets.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ) );
	}

	/**
	 * Disable WooCommerce purchasing functionality.
	 */
	private function disable_woocommerce_purchasing() {
		// Remove add to cart buttons - multiple priorities to catch all themes.
		remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart', 30 );
		remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart', 10 );
		
		// Also remove at different priorities in case theme re-adds them.
		add_action( 'init', array( $this, 'remove_cart_actions_init' ), 99 );

		// Disable cart functionality - reasonable priority.
		add_filter( 'woocommerce_is_purchasable', '__return_false', 99 );
		add_filter( 'woocommerce_variation_is_purchasable', '__return_false', 99 );

		// Disable checkout.
		add_filter( 'woocommerce_cart_needs_payment', '__return_false', 99 );

		// Remove cart and checkout pages from WooCommerce.
		add_filter( 'woocommerce_cart_item_removed_title', '__return_empty_string' );
		
		// Hide cart widget.
		add_filter( 'woocommerce_widget_cart_is_hidden', '__return_true' );

		// Redirect cart and checkout to shop.
		add_action( 'template_redirect', array( $this, 'redirect_cart_checkout' ), 5 );

		// Remove cart menu item count.
		add_filter( 'woocommerce_add_to_cart_fragments', '__return_empty_array', 99 );

		// Disable cart in REST API.
		add_filter( 'woocommerce_rest_check_permissions', array( $this, 'disable_cart_rest_api' ), 10, 4 );
		
		// Hide add to cart buttons with CSS (backup method).
		add_action( 'wp_head', array( $this, 'hide_cart_buttons_css' ), 99 );
	}

	/**
	 * Remove cart actions on init hook.
	 * Separated method to avoid anonymous functions.
	 */
	public function remove_cart_actions_init() {
		remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart', 30 );
		remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart', 10 );
		
		// Remove all add to cart forms entirely.
		remove_action( 'woocommerce_simple_add_to_cart', 'woocommerce_simple_add_to_cart', 30 );
		remove_action( 'woocommerce_grouped_add_to_cart', 'woocommerce_grouped_add_to_cart', 30 );
		remove_action( 'woocommerce_variable_add_to_cart', 'woocommerce_variable_add_to_cart', 30 );
		remove_action( 'woocommerce_external_add_to_cart', 'woocommerce_external_add_to_cart', 30 );
	}

	/**
	 * Redirect cart and checkout pages to shop.
	 */
	public function redirect_cart_checkout() {
		if ( is_cart() || is_checkout() ) {
			wp_safe_redirect( wc_get_page_permalink( 'shop' ) );
			exit;
		}
	}

	/**
	 * Disable cart functionality in REST API.
	 *
	 * @param bool   $permission Permission status.
	 * @param string $context Context.
	 * @param int    $object_id Object ID.
	 * @param string $object_type Object type.
	 * @return bool
	 */
	public function disable_cart_rest_api( $permission, $context, $object_id, $object_type ) {
		if ( 'cart' === $object_type ) {
			return false;
		}
		return $permission;
	}

	/**
	 * Hide add to cart buttons with CSS (nuclear backup method).
	 */
	public function hide_cart_buttons_css() {
		?>
		<style type="text/css">
		/* DRWS: Force hide all add to cart elements */
		.single_add_to_cart_button,
		button[name="add-to-cart"],
		.add_to_cart_button,
		.add-to-cart,
		.ajax_add_to_cart,
		form.cart button[type="submit"],
		form.cart .single_add_to_cart_button,
		.woocommerce-variation-add-to-cart,
		.product-type-simple .add_to_cart_button,
		.product-type-variable .add_to_cart_button {
			display: none !important;
			visibility: hidden !important;
			opacity: 0 !important;
			pointer-events: none !important;
		}
		</style>
		<?php
	}

	/**
	 * Add WhatsApp button.
	 */
	private function add_whatsapp_button() {
		// Method 1: Try standard WooCommerce hooks (works 80% of themes).
		add_action( 'woocommerce_single_product_summary', array( $this, 'render_whatsapp_button_single' ), 30 );
		add_action( 'woocommerce_after_shop_loop_item', array( $this, 'render_whatsapp_button_loop' ), 10 );
		
		// Method 2: Try alternative product hooks (works 15% of themes).
		add_action( 'woocommerce_before_add_to_cart_form', array( $this, 'render_whatsapp_button_single' ), 10 );
		add_action( 'woocommerce_product_meta_end', array( $this, 'render_whatsapp_button_single' ), 10 );
		
		// Method 3: FORCE with JavaScript injection (works 100% - nuclear option).
		add_action( 'wp_footer', array( $this, 'inject_button_via_javascript' ), 9999 );
		
		// Method 4: Handle blocks.
		add_filter( 'woocommerce_blocks_product_grid_item_html', array( $this, 'add_whatsapp_button_to_blocks' ), 10, 3 );
	}

	/**
	 * Whether button has been rendered.
	 *
	 * @var bool
	 */
	private $button_rendered = false;

	/**
	 * Render WhatsApp button for single product page.
	 */
	public function render_whatsapp_button_single() {
		// Prevent duplicate buttons from multiple hooks.
		if ( $this->button_rendered ) {
			return;
		}
		
		global $product;

		if ( ! $product ) {
			return;
		}

		$whatsapp_url = $this->get_whatsapp_url( $product );

		if ( empty( $whatsapp_url ) ) {
			// Debug: Phone number not configured.
			if ( current_user_can( 'manage_options' ) ) {
				echo '<!-- Digital Rise WhatsApp: Phone number not configured in WooCommerce > WhatsApp Settings -->';
			}
			return;
		}

		// Mark as rendered.
		$this->button_rendered = true;

		echo '<div class="drws-whatsapp-button-wrapper">';
		printf(
			'<a href="%s" class="drws-whatsapp-button drws-whatsapp-button-single button alt" target="_blank" rel="noopener noreferrer">%s</a>',
			esc_url( $whatsapp_url ),
			$this->get_whatsapp_button_html()
		);
		echo '</div>';
	}

	/**
	 * Render WhatsApp button for shop loop.
	 */
	public function render_whatsapp_button_loop() {
		global $product;

		if ( ! $product ) {
			return;
		}

		$whatsapp_url = $this->get_whatsapp_url( $product );

		if ( empty( $whatsapp_url ) ) {
			return;
		}

		printf(
			'<a href="%s" class="drws-whatsapp-button drws-whatsapp-button-loop button" target="_blank" rel="noopener noreferrer">%s</a>',
			esc_url( $whatsapp_url ),
			$this->get_whatsapp_button_html()
		);
	}

	/**
	 * Add WhatsApp button to WooCommerce blocks.
	 *
	 * @param string     $html Product grid item HTML.
	 * @param array      $data Product data.
	 * @param WC_Product $product Product object.
	 * @return string
	 */
	public function add_whatsapp_button_to_blocks( $html, $data, $product ) {
		if ( ! $product ) {
			return $html;
		}

		$whatsapp_url = $this->get_whatsapp_url( $product );

		if ( empty( $whatsapp_url ) ) {
			return $html;
		}

		$button = sprintf(
			'<a href="%s" class="drws-whatsapp-button drws-whatsapp-button-block button" target="_blank" rel="noopener noreferrer">%s</a>',
			esc_url( $whatsapp_url ),
			$this->get_whatsapp_button_html()
		);

		// Add button before closing li tag.
		$html = str_replace( '</li>', $button . '</li>', $html );

		return $html;
	}

	/**
	 * Get WhatsApp URL for product.
	 *
	 * @param WC_Product $product Product object.
	 * @return string
	 */
	private function get_whatsapp_url( $product ) {
		$country_code = get_option( 'drws_country_code', '+216' );
		$phone_number = get_option( 'drws_whatsapp_number', '' );

		if ( empty( $phone_number ) ) {
			return '';
		}

		$full_number = $country_code . $phone_number;
		$message = get_option( 'drws_whatsapp_message', 'Je suis intéressé par ce bien immobilier :' );
		$product_url = get_permalink( $product->get_id() );

		// Append product URL to message.
		$full_message = $message . ' ' . $product_url;

		// URL encode the message.
		$encoded_message = rawurlencode( $full_message );

		// Construct WhatsApp URL.
		$whatsapp_url = 'https://wa.me/' . $full_number . '?text=' . $encoded_message;

		return $whatsapp_url;
	}

	/**
	 * Get WhatsApp button HTML.
	 *
	 * @return string
	 */
	private function get_whatsapp_button_html() {
		$svg_icon = '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="currentColor" class="drws-whatsapp-icon"><path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z"/></svg>';
		$button_text = esc_html__( 'Contact via WhatsApp', 'digital-rise-whatsapp-solutions' );

		return $svg_icon . ' ' . $button_text;
	}

	/**
	 * Inject WhatsApp button via JavaScript (nuclear option - works on ANY theme).
	 * This method uses JavaScript to find where the Add to Cart button should be
	 * and injects the WhatsApp button there.
	 */
	public function inject_button_via_javascript() {
		// Only on product pages.
		if ( ! is_product() ) {
			return;
		}

		// Check if already rendered via PHP.
		if ( $this->button_rendered ) {
			return;
		}

		global $product;
		if ( ! $product ) {
			return;
		}

		$whatsapp_url = $this->get_whatsapp_url( $product );
		if ( empty( $whatsapp_url ) ) {
			return;
		}

		// Prepare button HTML for JavaScript.
		$button_html = sprintf(
			'<div class="drws-whatsapp-button-wrapper drws-js-injected"><a href="%s" class="drws-whatsapp-button drws-whatsapp-button-single button alt" target="_blank" rel="noopener noreferrer">%s</a></div>',
			esc_url( $whatsapp_url ),
			$this->get_whatsapp_button_html()
		);

		// Output JavaScript that injects the button.
		?>
		<script type="text/javascript">
		(function($) {
			'use strict';
			
			// Wait for page to be ready.
			$(document).ready(function() {
				// Check if button already exists (from PHP hooks).
				if ($('.drws-whatsapp-button').length > 0) {
					return; // Button already rendered, exit.
				}
				
				// Button HTML to inject.
				var buttonHtml = <?php echo wp_json_encode( $button_html ); ?>;
				
				// Try multiple selectors to find the right place to inject button.
				// These are common WooCommerce product page elements.
				var selectors = [
					'.single_add_to_cart_button',           // Standard add to cart button
					'form.cart',                             // Cart form
					'.product_meta',                         // Product meta (SKU, categories)
					'.woocommerce-product-details__short-description', // Short description
					'.product_title',                        // Product title
					'.summary.entry-summary',                // Product summary wrapper
					'.product-info',                         // Common theme class
					'.product-details',                      // Common theme class
					'.single-product-summary'                // Common theme class
				];
				
				var injected = false;
				
				// Try each selector until we find one.
				for (var i = 0; i < selectors.length; i++) {
					var $target = $(selectors[i]).first();
					
					if ($target.length > 0) {
						// Found a target! Inject after it.
						$target.after(buttonHtml);
						injected = true;
						<?php if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) : ?>
						<?php if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) : ?>
						console.log('DRWS: WhatsApp button injected after ' + selectors[i]);
						<?php endif; ?>
						<?php endif; ?>
						break;
					}
				}
				
				// If still not injected, try prepending to .summary.
				if (!injected) {
					var $summary = $('.summary.entry-summary, .product-summary, .product-info-main').first();
					if ($summary.length > 0) {
						$summary.prepend(buttonHtml);
						injected = true;
						<?php if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) : ?>
						console.log('DRWS: WhatsApp button prepended to summary');
						<?php endif; ?>
					}
				}
				
				// Last resort: append to body (better than nothing).
				if (!injected) {
					$('body.single-product .site-main, body.single-product main, body.single-product article').first().prepend(buttonHtml);
					<?php if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) : ?>
					console.log('DRWS: WhatsApp button injected as last resort');
					<?php endif; ?>
				}
			});
		})(jQuery);
		</script>
		<?php
	}

	/**
	 * Enqueue frontend assets.
	 */
	public function enqueue_frontend_assets() {
		$css = "
		/* WhatsApp Button - positioned like standard Add to Cart */
		.drws-whatsapp-button-wrapper {
			margin-bottom: 1em;
			clear: both;
		}
		
		.drws-whatsapp-button {
			display: inline-flex;
			align-items: center;
			justify-content: center;
			gap: 8px;
			padding: 12px 24px;
			background-color: #25D366 !important;
			color: #ffffff !important;
			text-decoration: none !important;
			border-radius: 4px;
			font-weight: 600;
			font-size: 16px;
			transition: all 0.3s ease;
			border: none;
			cursor: pointer;
			text-align: center;
			line-height: 1.5;
			min-height: 48px;
			width: 100%;
			box-sizing: border-box;
		}
		
		.drws-whatsapp-button:hover,
		.drws-whatsapp-button:focus {
			background-color: #128C7E !important;
			color: #ffffff !important;
			text-decoration: none !important;
			transform: translateY(-2px);
			box-shadow: 0 4px 8px rgba(37, 211, 102, 0.3);
		}
		
		.drws-whatsapp-button:active {
			transform: translateY(0);
		}
		
		.drws-whatsapp-icon {
			width: 20px;
			height: 20px;
			flex-shrink: 0;
		}
		
		/* Loop/Shop page buttons */
		.drws-whatsapp-button-loop {
			width: 100%;
			margin: 10px 0 0 0;
		}
		
		/* Block editor buttons */
		.drws-whatsapp-button-block {
			width: auto;
			margin: 10px auto 0;
			display: inline-flex;
		}
		
		/* Ensure proper display in product grids */
		.woocommerce ul.products li.product .drws-whatsapp-button {
			margin-top: 10px;
			width: 100%;
		}
		
		/* Mobile responsiveness */
		@media (max-width: 768px) {
			.drws-whatsapp-button {
				font-size: 14px;
				padding: 10px 20px;
				min-height: 44px;
			}
			
			.drws-whatsapp-icon {
				width: 18px;
				height: 18px;
			}
		}
		
		/* Ensure button shows in correct position - override theme styles if needed */
		.single-product .drws-whatsapp-button-wrapper {
			display: block !important;
			visibility: visible !important;
			opacity: 1 !important;
		}
		
		/* Hide cart/checkout pages */
		.woocommerce-cart .woocommerce,
		.woocommerce-checkout .woocommerce {
			display: none;
		}
		";

		// Register inline style - use wp_head to ensure it always loads.
		wp_register_style( 'drws-frontend-styles', false );
		wp_enqueue_style( 'drws-frontend-styles' );
		wp_add_inline_style( 'drws-frontend-styles', $css );
	}
}
