<?php
/**
 * Admin Settings Class
 *
 * @package Digital_Rise_WhatsApp_Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Settings Class
 */
class DRWS_Admin_Settings {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ), 99 );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );
		add_filter( 'plugin_action_links_' . DRWS_PLUGIN_BASENAME, array( $this, 'add_settings_link' ) );
	}

	/**
	 * Add admin menu under WooCommerce.
	 */
	public function add_admin_menu() {
		add_submenu_page(
			'woocommerce',
			esc_html__( 'WhatsApp Settings', 'digital-rise-whatsapp-solutions' ),
			esc_html__( 'WhatsApp Settings', 'digital-rise-whatsapp-solutions' ),
			'manage_woocommerce',
			'drws-settings',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Add settings link on plugins page.
	 *
	 * @param array $links Plugin action links.
	 * @return array
	 */
	public function add_settings_link( $links ) {
		$settings_link = '<a href="' . esc_url( admin_url( 'admin.php?page=drws-settings' ) ) . '">' . esc_html__( 'Settings', 'digital-rise-whatsapp-solutions' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		register_setting(
			'drws_settings_group',
			'drws_country_code',
			array(
				'type'              => 'string',
				'sanitize_callback' => array( $this, 'sanitize_country_code' ),
				'default'           => '+216',
			)
		);

		register_setting(
			'drws_settings_group',
			'drws_whatsapp_number',
			array(
				'type'              => 'string',
				'sanitize_callback' => array( $this, 'sanitize_phone_number' ),
				'default'           => '',
			)
		);

		register_setting(
			'drws_settings_group',
			'drws_whatsapp_message',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_textarea_field',
				'default'           => 'Je suis intéressé par ce bien immobilier :',
			)
		);

		add_settings_section(
			'drws_main_section',
			esc_html__( 'WhatsApp Configuration', 'digital-rise-whatsapp-solutions' ),
			array( $this, 'render_section_description' ),
			'drws-settings'
		);

		add_settings_field(
			'drws_country_code',
			esc_html__( 'Country Code', 'digital-rise-whatsapp-solutions' ),
			array( $this, 'render_country_code_field' ),
			'drws-settings',
			'drws_main_section'
		);

		add_settings_field(
			'drws_whatsapp_number',
			esc_html__( 'WhatsApp Number', 'digital-rise-whatsapp-solutions' ),
			array( $this, 'render_phone_number_field' ),
			'drws-settings',
			'drws_main_section'
		);

		add_settings_field(
			'drws_whatsapp_message',
			esc_html__( 'Default Message', 'digital-rise-whatsapp-solutions' ),
			array( $this, 'render_message_field' ),
			'drws-settings',
			'drws_main_section'
		);
	}

	/**
	 * Render settings section description.
	 */
	public function render_section_description() {
		echo '<p>' . esc_html__( 'Configure your WhatsApp contact settings. The product URL will be automatically appended to the message when users click the WhatsApp button.', 'digital-rise-whatsapp-solutions' ) . '</p>';
	}

	/**
	 * Render country code field.
	 */
	public function render_country_code_field() {
		$country_code = get_option( 'drws_country_code', '+216' );
		$country_codes = $this->get_country_codes();
		?>
		<select name="drws_country_code" id="drws_country_code" class="regular-text">
			<?php foreach ( $country_codes as $code => $label ) : ?>
				<option value="<?php echo esc_attr( $code ); ?>" <?php selected( $country_code, $code ); ?>>
					<?php echo esc_html( $label ); ?>
				</option>
			<?php endforeach; ?>
		</select>
		<p class="description">
			<?php esc_html_e( 'Select your country code.', 'digital-rise-whatsapp-solutions' ); ?>
		</p>
		<?php
	}

	/**
	 * Render phone number field.
	 */
	public function render_phone_number_field() {
		$phone_number = get_option( 'drws_whatsapp_number', '' );
		?>
		<input type="text" 
			   name="drws_whatsapp_number" 
			   id="drws_whatsapp_number" 
			   value="<?php echo esc_attr( $phone_number ); ?>" 
			   class="regular-text" 
			   placeholder="<?php esc_attr_e( 'e.g., 98765432', 'digital-rise-whatsapp-solutions' ); ?>"
			   pattern="[0-9]*"
			   inputmode="numeric" />
		<p class="description">
			<?php esc_html_e( 'Enter your WhatsApp number without country code (numbers only).', 'digital-rise-whatsapp-solutions' ); ?>
		</p>
		<?php
	}

	/**
	 * Render message field.
	 */
	public function render_message_field() {
		$message = get_option( 'drws_whatsapp_message', 'Je suis intéressé par ce bien immobilier :' );
		?>
		<textarea name="drws_whatsapp_message" 
				  id="drws_whatsapp_message" 
				  rows="4" 
				  class="large-text"><?php echo esc_textarea( $message ); ?></textarea>
		<p class="description">
			<?php esc_html_e( 'This message will be sent via WhatsApp. The product URL will be automatically added after your message.', 'digital-rise-whatsapp-solutions' ); ?>
		</p>
		<?php
	}

	/**
	 * Get list of country codes.
	 *
	 * @return array
	 */
	private function get_country_codes() {
		return array(
			'+1'    => '+1 (United States/Canada)',
			'+20'   => '+20 (Egypt)',
			'+27'   => '+27 (South Africa)',
			'+30'   => '+30 (Greece)',
			'+31'   => '+31 (Netherlands)',
			'+32'   => '+32 (Belgium)',
			'+33'   => '+33 (France)',
			'+34'   => '+34 (Spain)',
			'+39'   => '+39 (Italy)',
			'+40'   => '+40 (Romania)',
			'+41'   => '+41 (Switzerland)',
			'+43'   => '+43 (Austria)',
			'+44'   => '+44 (United Kingdom)',
			'+45'   => '+45 (Denmark)',
			'+46'   => '+46 (Sweden)',
			'+47'   => '+47 (Norway)',
			'+48'   => '+48 (Poland)',
			'+49'   => '+49 (Germany)',
			'+51'   => '+51 (Peru)',
			'+52'   => '+52 (Mexico)',
			'+53'   => '+53 (Cuba)',
			'+54'   => '+54 (Argentina)',
			'+55'   => '+55 (Brazil)',
			'+56'   => '+56 (Chile)',
			'+57'   => '+57 (Colombia)',
			'+58'   => '+58 (Venezuela)',
			'+60'   => '+60 (Malaysia)',
			'+61'   => '+61 (Australia)',
			'+62'   => '+62 (Indonesia)',
			'+63'   => '+63 (Philippines)',
			'+64'   => '+64 (New Zealand)',
			'+65'   => '+65 (Singapore)',
			'+66'   => '+66 (Thailand)',
			'+81'   => '+81 (Japan)',
			'+82'   => '+82 (South Korea)',
			'+84'   => '+84 (Vietnam)',
			'+86'   => '+86 (China)',
			'+90'   => '+90 (Turkey)',
			'+91'   => '+91 (India)',
			'+92'   => '+92 (Pakistan)',
			'+93'   => '+93 (Afghanistan)',
			'+94'   => '+94 (Sri Lanka)',
			'+95'   => '+95 (Myanmar)',
			'+98'   => '+98 (Iran)',
			'+212'  => '+212 (Morocco)',
			'+213'  => '+213 (Algeria)',
			'+216'  => '+216 (Tunisia)',
			'+218'  => '+218 (Libya)',
			'+220'  => '+220 (Gambia)',
			'+221'  => '+221 (Senegal)',
			'+222'  => '+222 (Mauritania)',
			'+223'  => '+223 (Mali)',
			'+224'  => '+224 (Guinea)',
			'+225'  => '+225 (Ivory Coast)',
			'+226'  => '+226 (Burkina Faso)',
			'+227'  => '+227 (Niger)',
			'+228'  => '+228 (Togo)',
			'+229'  => '+229 (Benin)',
			'+230'  => '+230 (Mauritius)',
			'+231'  => '+231 (Liberia)',
			'+232'  => '+232 (Sierra Leone)',
			'+233'  => '+233 (Ghana)',
			'+234'  => '+234 (Nigeria)',
			'+235'  => '+235 (Chad)',
			'+236'  => '+236 (Central African Republic)',
			'+237'  => '+237 (Cameroon)',
			'+238'  => '+238 (Cape Verde)',
			'+239'  => '+239 (Sao Tome and Principe)',
			'+240'  => '+240 (Equatorial Guinea)',
			'+241'  => '+241 (Gabon)',
			'+242'  => '+242 (Republic of the Congo)',
			'+243'  => '+243 (Democratic Republic of the Congo)',
			'+244'  => '+244 (Angola)',
			'+245'  => '+245 (Guinea-Bissau)',
			'+246'  => '+246 (Diego Garcia)',
			'+248'  => '+248 (Seychelles)',
			'+249'  => '+249 (Sudan)',
			'+250'  => '+250 (Rwanda)',
			'+251'  => '+251 (Ethiopia)',
			'+252'  => '+252 (Somalia)',
			'+253'  => '+253 (Djibouti)',
			'+254'  => '+254 (Kenya)',
			'+255'  => '+255 (Tanzania)',
			'+256'  => '+256 (Uganda)',
			'+257'  => '+257 (Burundi)',
			'+258'  => '+258 (Mozambique)',
			'+260'  => '+260 (Zambia)',
			'+261'  => '+261 (Madagascar)',
			'+262'  => '+262 (Reunion)',
			'+263'  => '+263 (Zimbabwe)',
			'+264'  => '+264 (Namibia)',
			'+265'  => '+265 (Malawi)',
			'+266'  => '+266 (Lesotho)',
			'+267'  => '+267 (Botswana)',
			'+268'  => '+268 (Eswatini)',
			'+269'  => '+269 (Comoros)',
			'+290'  => '+290 (Saint Helena)',
			'+291'  => '+291 (Eritrea)',
			'+297'  => '+297 (Aruba)',
			'+298'  => '+298 (Faroe Islands)',
			'+299'  => '+299 (Greenland)',
			'+350'  => '+350 (Gibraltar)',
			'+351'  => '+351 (Portugal)',
			'+352'  => '+352 (Luxembourg)',
			'+353'  => '+353 (Ireland)',
			'+354'  => '+354 (Iceland)',
			'+355'  => '+355 (Albania)',
			'+356'  => '+356 (Malta)',
			'+357'  => '+357 (Cyprus)',
			'+358'  => '+358 (Finland)',
			'+359'  => '+359 (Bulgaria)',
			'+370'  => '+370 (Lithuania)',
			'+371'  => '+371 (Latvia)',
			'+372'  => '+372 (Estonia)',
			'+373'  => '+373 (Moldova)',
			'+374'  => '+374 (Armenia)',
			'+375'  => '+375 (Belarus)',
			'+376'  => '+376 (Andorra)',
			'+377'  => '+377 (Monaco)',
			'+378'  => '+378 (San Marino)',
			'+380'  => '+380 (Ukraine)',
			'+381'  => '+381 (Serbia)',
			'+382'  => '+382 (Montenegro)',
			'+383'  => '+383 (Kosovo)',
			'+385'  => '+385 (Croatia)',
			'+386'  => '+386 (Slovenia)',
			'+387'  => '+387 (Bosnia and Herzegovina)',
			'+389'  => '+389 (North Macedonia)',
			'+420'  => '+420 (Czech Republic)',
			'+421'  => '+421 (Slovakia)',
			'+423'  => '+423 (Liechtenstein)',
			'+500'  => '+500 (Falkland Islands)',
			'+501'  => '+501 (Belize)',
			'+502'  => '+502 (Guatemala)',
			'+503'  => '+503 (El Salvador)',
			'+504'  => '+504 (Honduras)',
			'+505'  => '+505 (Nicaragua)',
			'+506'  => '+506 (Costa Rica)',
			'+507'  => '+507 (Panama)',
			'+508'  => '+508 (Saint Pierre and Miquelon)',
			'+509'  => '+509 (Haiti)',
			'+590'  => '+590 (Guadeloupe)',
			'+591'  => '+591 (Bolivia)',
			'+592'  => '+592 (Guyana)',
			'+593'  => '+593 (Ecuador)',
			'+594'  => '+594 (French Guiana)',
			'+595'  => '+595 (Paraguay)',
			'+596'  => '+596 (Martinique)',
			'+597'  => '+597 (Suriname)',
			'+598'  => '+598 (Uruguay)',
			'+599'  => '+599 (Curacao)',
			'+670'  => '+670 (East Timor)',
			'+672'  => '+672 (Antarctica)',
			'+673'  => '+673 (Brunei)',
			'+674'  => '+674 (Nauru)',
			'+675'  => '+675 (Papua New Guinea)',
			'+676'  => '+676 (Tonga)',
			'+677'  => '+677 (Solomon Islands)',
			'+678'  => '+678 (Vanuatu)',
			'+679'  => '+679 (Fiji)',
			'+680'  => '+680 (Palau)',
			'+681'  => '+681 (Wallis and Futuna)',
			'+682'  => '+682 (Cook Islands)',
			'+683'  => '+683 (Niue)',
			'+685'  => '+685 (Samoa)',
			'+686'  => '+686 (Kiribati)',
			'+687'  => '+687 (New Caledonia)',
			'+688'  => '+688 (Tuvalu)',
			'+689'  => '+689 (French Polynesia)',
			'+690'  => '+690 (Tokelau)',
			'+691'  => '+691 (Micronesia)',
			'+692'  => '+692 (Marshall Islands)',
			'+850'  => '+850 (North Korea)',
			'+852'  => '+852 (Hong Kong)',
			'+853'  => '+853 (Macau)',
			'+855'  => '+855 (Cambodia)',
			'+856'  => '+856 (Laos)',
			'+880'  => '+880 (Bangladesh)',
			'+886'  => '+886 (Taiwan)',
			'+960'  => '+960 (Maldives)',
			'+961'  => '+961 (Lebanon)',
			'+962'  => '+962 (Jordan)',
			'+963'  => '+963 (Syria)',
			'+964'  => '+964 (Iraq)',
			'+965'  => '+965 (Kuwait)',
			'+966'  => '+966 (Saudi Arabia)',
			'+967'  => '+967 (Yemen)',
			'+968'  => '+968 (Oman)',
			'+970'  => '+970 (Palestine)',
			'+971'  => '+971 (United Arab Emirates)',
			'+972'  => '+972 (Israel)',
			'+973'  => '+973 (Bahrain)',
			'+974'  => '+974 (Qatar)',
			'+975'  => '+975 (Bhutan)',
			'+976'  => '+976 (Mongolia)',
			'+977'  => '+977 (Nepal)',
			'+992'  => '+992 (Tajikistan)',
			'+993'  => '+993 (Turkmenistan)',
			'+994'  => '+994 (Azerbaijan)',
			'+995'  => '+995 (Georgia)',
			'+996'  => '+996 (Kyrgyzstan)',
			'+998'  => '+998 (Uzbekistan)',
		);
	}

	/**
	 * Sanitize country code.
	 *
	 * @param string $value Country code value.
	 * @return string
	 */
	public function sanitize_country_code( $value ) {
		$country_codes = array_keys( $this->get_country_codes() );
		
		if ( in_array( $value, $country_codes, true ) ) {
			return $value;
		}

		return '+216';
	}

	/**
	 * Sanitize phone number.
	 *
	 * @param string $value Phone number value.
	 * @return string
	 */
	public function sanitize_phone_number( $value ) {
		// Remove all non-numeric characters.
		return preg_replace( '/[^0-9]/', '', $value );
	}

	/**
	 * Render settings page.
	 */
	public function render_settings_page() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'digital-rise-whatsapp-solutions' ) );
		}

		// Get current settings.
		$country_code = get_option( 'drws_country_code', '+216' );
		$phone_number = get_option( 'drws_whatsapp_number', '' );
		$full_number = $country_code . $phone_number;
		?>
		<div class="wrap drws-settings-wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<?php settings_errors(); ?>

			<div class="drws-settings-container">
				<div class="drws-settings-main">
					<form method="post" action="options.php">
						<?php
						settings_fields( 'drws_settings_group' );
						do_settings_sections( 'drws-settings' );
						submit_button( esc_html__( 'Save Settings', 'digital-rise-whatsapp-solutions' ) );
						?>
					</form>

					<?php if ( ! empty( $phone_number ) ) : ?>
					<div class="drws-preview-section">
						<h2><?php esc_html_e( 'Preview', 'digital-rise-whatsapp-solutions' ); ?></h2>
						<p>
							<strong><?php esc_html_e( 'Full WhatsApp Number:', 'digital-rise-whatsapp-solutions' ); ?></strong><br>
							<code><?php echo esc_html( $full_number ); ?></code>
						</p>
						<p>
							<strong><?php esc_html_e( 'WhatsApp Link Format:', 'digital-rise-whatsapp-solutions' ); ?></strong><br>
							<code>https://wa.me/<?php echo esc_html( $full_number ); ?></code>
						</p>
					</div>
					<?php endif; ?>
				</div>

				<div class="drws-settings-sidebar">
					<div class="drws-sidebar-box">
						<h3><?php esc_html_e( 'About This Plugin', 'digital-rise-whatsapp-solutions' ); ?></h3>
						<p><?php esc_html_e( 'Digital Rise WhatsApp Solutions replaces WooCommerce checkout with WhatsApp contact buttons, perfect for inquiry-based businesses.', 'digital-rise-whatsapp-solutions' ); ?></p>
						<hr>
						<p>
							<strong><?php esc_html_e( 'Version:', 'digital-rise-whatsapp-solutions' ); ?></strong> <?php echo esc_html( DRWS_VERSION ); ?><br>
							<strong><?php esc_html_e( 'Author:', 'digital-rise-whatsapp-solutions' ); ?></strong> <a href="https://digital-rise-solutions.com/oussama-ben-rejab-seo-expert-seo-consultant/" target="_blank" rel="noopener noreferrer">Oussama Ben Rejab</a>
						</p>
					</div>

					<div class="drws-sidebar-box">
						<h3><?php esc_html_e( 'Need Help?', 'digital-rise-whatsapp-solutions' ); ?></h3>
						<p><?php esc_html_e( 'For support and updates, visit:', 'digital-rise-whatsapp-solutions' ); ?></p>
						<p>
							<a href="https://digital-rise-solutions.com/" target="_blank" rel="noopener noreferrer" class="button button-secondary">
								<?php esc_html_e( 'Visit Digital Rise Solutions', 'digital-rise-whatsapp-solutions' ); ?>
							</a>
						</p>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Enqueue admin styles.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_styles( $hook ) {
		if ( 'woocommerce_page_drws-settings' !== $hook ) {
			return;
		}

		$css = "
		.drws-settings-wrap {
			max-width: 1200px;
		}
		
		.drws-settings-container {
			display: flex;
			gap: 30px;
			margin-top: 20px;
		}
		
		.drws-settings-main {
			flex: 1;
			background: #fff;
			padding: 20px;
			border: 1px solid #ccd0d4;
			box-shadow: 0 1px 1px rgba(0,0,0,.04);
		}
		
		.drws-settings-sidebar {
			width: 300px;
			flex-shrink: 0;
		}
		
		.drws-sidebar-box {
			background: #fff;
			padding: 20px;
			border: 1px solid #ccd0d4;
			box-shadow: 0 1px 1px rgba(0,0,0,.04);
			margin-bottom: 20px;
		}
		
		.drws-sidebar-box h3 {
			margin-top: 0;
			font-size: 14px;
			font-weight: 600;
			text-transform: uppercase;
			color: #23282d;
		}
		
		.drws-sidebar-box hr {
			margin: 15px 0;
			border: 0;
			border-top: 1px solid #ddd;
		}
		
		.drws-sidebar-box p {
			font-size: 13px;
			line-height: 1.6;
		}
		
		.drws-preview-section {
			margin-top: 30px;
			padding: 20px;
			background: #f8f9fa;
			border-left: 4px solid #25D366;
		}
		
		.drws-preview-section h2 {
			margin-top: 0;
			font-size: 18px;
		}
		
		.drws-preview-section code {
			background: #fff;
			padding: 4px 8px;
			border: 1px solid #ddd;
			border-radius: 3px;
			font-size: 13px;
		}
		
		@media (max-width: 782px) {
			.drws-settings-container {
				flex-direction: column;
			}
			
			.drws-settings-sidebar {
				width: 100%;
			}
		}
		";

		wp_add_inline_style( 'wp-admin', $css );
	}
}
