<?php
/**
 * Plugin Name: Digital Rise WhatsApp Solutions
 * Plugin URI: https://digital-rise-solutions.com/
 * Description: Replace WooCommerce checkout with a WhatsApp contact button for product inquiries.
 * Version: 1.1.0
 * Author: Oussama Ben Rejab
 * Author URI: https://digital-rise-solutions.com/oussama-ben-rejab-seo-expert-seo-consultant/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: digital-rise-whatsapp-solutions
 * Domain Path: /languages
 * Requires at least: 5.8
 * Tested up to: 6.7
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 9.5
 *
 * @package Digital_Rise_WhatsApp_Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants.
define( 'DRWS_VERSION', '1.1.0' );
define( 'DRWS_PLUGIN_FILE', __FILE__ );
define( 'DRWS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'DRWS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'DRWS_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main Plugin Class
 */
final class Digital_Rise_WhatsApp_Solutions {

	/**
	 * Single instance of the class.
	 *
	 * @var Digital_Rise_WhatsApp_Solutions
	 */
	private static $instance = null;

	/**
	 * Get single instance of the class.
	 *
	 * @return Digital_Rise_WhatsApp_Solutions
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'plugins_loaded', array( $this, 'init' ) );
		add_action( 'before_woocommerce_init', array( $this, 'declare_hpos_compatibility' ) );
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
	}

	/**
	 * Initialize plugin.
	 */
	public function init() {
		// Check if WooCommerce is active.
		if ( ! $this->is_woocommerce_active() ) {
			add_action( 'admin_notices', array( $this, 'woocommerce_missing_notice' ) );
			return;
		}

		// Load text domain for translations.
		load_plugin_textdomain( 'digital-rise-whatsapp-solutions', false, dirname( DRWS_PLUGIN_BASENAME ) . '/languages' );

		// Load required files.
		$this->load_files();

		// Initialize admin settings.
		if ( is_admin() ) {
			new DRWS_Admin_Settings();
		}

		// Initialize frontend functionality (always load, not just on !is_admin).
		new DRWS_Frontend();
	}

	/**
	 * Load required files.
	 */
	private function load_files() {
		require_once DRWS_PLUGIN_DIR . 'includes/class-drws-admin-settings.php';
		require_once DRWS_PLUGIN_DIR . 'includes/class-drws-frontend.php';
		require_once DRWS_PLUGIN_DIR . 'includes/drws-functions.php';
	}

	/**
	 * Check if WooCommerce is active.
	 *
	 * @return bool
	 */
	private function is_woocommerce_active() {
		return class_exists( 'WooCommerce' );
	}

	/**
	 * Display WooCommerce missing notice.
	 */
	public function woocommerce_missing_notice() {
		$message = sprintf(
			/* translators: %s: WooCommerce plugin link */
			esc_html__( 'Digital Rise WhatsApp Solutions requires WooCommerce to be installed and active. Please install %s.', 'digital-rise-whatsapp-solutions' ),
			'<a href="' . esc_url( admin_url( 'plugin-install.php?s=woocommerce&tab=search&type=term' ) ) . '">WooCommerce</a>'
		);

		printf(
			'<div class="notice notice-error"><p>%s</p></div>',
			wp_kses_post( $message )
		);
	}

	/**
	 * Declare HPOS compatibility.
	 */
	public function declare_hpos_compatibility() {
		if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
		}
	}

	/**
	 * Plugin activation hook.
	 */
	public function activate() {
		// Set default options if not already set.
		if ( false === get_option( 'drws_whatsapp_number' ) ) {
			update_option( 'drws_whatsapp_number', '' );
		}

		if ( false === get_option( 'drws_country_code' ) ) {
			// Use empty string - admin must select their country.
			update_option( 'drws_country_code', '' );
		}

		if ( false === get_option( 'drws_whatsapp_message' ) ) {
			// Use English default message (translation-ready).
			update_option( 'drws_whatsapp_message', __( 'I am interested in this product:', 'digital-rise-whatsapp-solutions' ) );
		}
	}

	/**
	 * Plugin deactivation hook.
	 */
	public function deactivate() {
		// Clear any transients or temporary data.
		// Note: We don't delete options here - user might reactivate.
		// Options are only deleted via uninstall.php.
		delete_transient( 'drws_activation_redirect' );
		wp_cache_flush();
	}
}

/**
 * Initialize the plugin.
 *
 * @return Digital_Rise_WhatsApp_Solutions
 */
function drws() {
	return Digital_Rise_WhatsApp_Solutions::get_instance();
}

// Start the plugin.
drws();
