<?php
/**
 * Sitemap.xml Generator
 * PHASE 5: Automatic sitemap generation and updates
 *
 * @package Digital_Rise_SEO_Solutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRSS_Sitemap {

    private $plugin;
    private $max_urls = 50000; // Google's limit

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        $this->init();
    }

    /**
     * Initialize sitemap functionality
     */
    public function init() {
        // Generate sitemap on activation
        add_action( 'drss_activated', array( $this, 'generate_sitemap' ) );
        
        // Regenerate sitemap when content changes
        add_action( 'save_post', array( $this, 'schedule_regeneration' ), 10, 2 );
        add_action( 'delete_post', array( $this, 'schedule_regeneration' ) );
        add_action( 'created_term', array( $this, 'schedule_regeneration' ) );
        add_action( 'edited_term', array( $this, 'schedule_regeneration' ) );
        add_action( 'delete_term', array( $this, 'schedule_regeneration' ) );
        
        // ISSUE #4 FIX: Handle sitemap requests using template_redirect (earlier than init)
        add_action( 'template_redirect', array( $this, 'handle_sitemap_request' ), 1 );
        
        // Admin actions
        add_action( 'wp_ajax_drss_regenerate_sitemap', array( $this, 'ajax_regenerate_sitemap' ) );
    }

    /**
     * Handle sitemap.xml requests
     * TASK #2: Sitemap index architecture with sub-sitemaps
     */
    public function handle_sitemap_request() {
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        
        // Main sitemap index
        if ( preg_match( '/\/sitemap\.xml$/i', $request_uri ) ) {
            $this->serve_sitemap_index();
            return;
        }
        
        // Sub-sitemaps: /sitemap/posts.xml, /sitemap/pages.xml, etc.
        if ( preg_match( '/\/sitemap\/([a-z_-]+)\.xml$/i', $request_uri, $matches ) ) {
            $type = $matches[1];
            $this->serve_sub_sitemap( $type );
            return;
        }
    }
    
    /**
     * Serve main sitemap index
     * Lists all sub-sitemaps
     */
    private function serve_sitemap_index() {
        $sitemap_index = get_transient( 'drss_sitemap_index_cache' );
        
        if ( ! $sitemap_index ) {
            $sitemap_index = $this->generate_sitemap_index();
            set_transient( 'drss_sitemap_index_cache', $sitemap_index, DAY_IN_SECONDS );
        }
        
        status_header( 200 );
        header( 'Content-Type: application/xml; charset=UTF-8' );
        header( 'X-Robots-Tag: noindex, follow', true );
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML content, already escaped
        echo $sitemap_index;
        exit;
    }
    
    /**
     * Generate sitemap index XML
     */
    private function generate_sitemap_index() {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<?xml-stylesheet type="text/xsl" href="' . esc_url( DRSS_PLUGIN_URL . 'assets/sitemap-index.xsl' ) . '"?>' . "\n";
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        $base_url = home_url( '/sitemap/' );
        $lastmod = gmdate( 'c' );
        
        // Define all sub-sitemaps
        $sub_sitemaps = array(
            'pages' => __( 'Pages', 'digital-rise-seo-solutions' ),
            'posts' => __( 'Posts', 'digital-rise-seo-solutions' ),
            'categories' => __( 'Categories', 'digital-rise-seo-solutions' ),
            'tags' => __( 'Tags', 'digital-rise-seo-solutions' ),
        );
        
        // Add authors sitemap
        $sub_sitemaps['authors'] = __( 'Authors', 'digital-rise-seo-solutions' );
        
        // Add WooCommerce sitemaps if available
        if ( class_exists( 'WooCommerce' ) ) {
            $sub_sitemaps['products'] = __( 'Products', 'digital-rise-seo-solutions' );
            $sub_sitemaps['product-categories'] = __( 'Product Categories', 'digital-rise-seo-solutions' );
            $sub_sitemaps['product-tags'] = __( 'Product Tags', 'digital-rise-seo-solutions' );
        }
        
        // Add Dokan sitemaps if available
        if ( class_exists( 'WeDevs_Dokan' ) ) {
            $sub_sitemaps['vendors'] = __( 'Vendors', 'digital-rise-seo-solutions' );
            $sub_sitemaps['vendor-products'] = __( 'Vendor Products', 'digital-rise-seo-solutions' );
        }
        
        $sub_sitemaps['images'] = __( 'Images', 'digital-rise-seo-solutions' );
        
        foreach ( $sub_sitemaps as $slug => $label ) {
            $xml .= '  <sitemap>' . "\n";
            $xml .= '    <loc>' . esc_url( $base_url . $slug . '.xml' ) . '</loc>' . "\n";
            $xml .= '    <lastmod>' . $lastmod . '</lastmod>' . "\n";
            $xml .= '  </sitemap>' . "\n";
        }
        
        $xml .= '</sitemapindex>';
        
        return $xml;
    }
    
    /**
     * Serve individual sub-sitemap
     */
    private function serve_sub_sitemap( $type ) {
        $cache_key = 'drss_sitemap_' . $type . '_cache';
        $sitemap = get_transient( $cache_key );
        
        if ( ! $sitemap ) {
            $sitemap = $this->generate_sub_sitemap( $type );
            set_transient( $cache_key, $sitemap, DAY_IN_SECONDS );
        }
        
        status_header( 200 );
        header( 'Content-Type: application/xml; charset=UTF-8' );
        header( 'X-Robots-Tag: noindex, follow', true );
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML content, already escaped
        echo $sitemap;
        exit;
    }
    
    /**
     * Generate individual sub-sitemap content
     */
    private function generate_sub_sitemap( $type ) {
        $urls = array();
        
        switch ( $type ) {
            case 'pages':
                $urls = $this->get_pages_urls();
                break;
            case 'posts':
                $urls = $this->get_posts_urls();
                break;
            case 'categories':
                $urls = $this->get_categories_urls();
                break;
            case 'tags':
                $urls = $this->get_tags_urls();
                break;
            case 'products':
                $urls = $this->get_products_urls();
                break;
            case 'product-categories':
                $urls = $this->get_product_categories_urls();
                break;
            case 'product-tags':
                $urls = $this->get_product_tags_urls();
                break;
            case 'authors':
                $urls = $this->get_authors_urls();
                break;
            case 'vendors':
                $urls = $this->get_vendors_urls();
                break;
            case 'vendor-products':
                $urls = $this->get_vendor_products_urls();
                break;
            case 'images':
                $urls = $this->get_images_urls();
                break;
        }
        
        return $this->build_sitemap_xml( $urls, $type );
    }

    /**
     * Serve the sitemap
     * ISSUE #4 FIX: Added error handling and debugging
     */
    private function serve_sitemap() {
        // Get cached sitemap or generate new one
        $sitemap = get_transient( 'drss_sitemap_cache' );
        
        if ( ! $sitemap ) {
            $sitemap = $this->generate_sitemap_content();
            
            // Check if sitemap generation failed
            if ( empty( $sitemap ) ) {
                // Generate a minimal valid sitemap as fallback
                $sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
                $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
                $sitemap .= '  <url>' . "\n";
                $sitemap .= '    <loc>' . esc_url( home_url( '/' ) ) . '</loc>' . "\n";
                $sitemap .= '    <lastmod>' . gmdate( 'c' ) . '</lastmod>' . "\n";
                $sitemap .= '    <changefreq>daily</changefreq>' . "\n";
                $sitemap .= '    <priority>1.0</priority>' . "\n";
                $sitemap .= '  </url>' . "\n";
                $sitemap .= '</urlset>';
            }
            
            set_transient( 'drss_sitemap_cache', $sitemap, DAY_IN_SECONDS );
        }

        // Output sitemap
        status_header( 200 );
        header( 'Content-Type: application/xml; charset=UTF-8' );
        header( 'X-Robots-Tag: noindex, follow', true );
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML content, already escaped
        echo $sitemap;
        exit;
    }


    /**
     * Get pages URLs
     */
    private function get_pages_urls() {
        $pages = get_posts( array(
            'post_type' => 'page',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'modified',
            'order' => 'DESC',
        ) );
        
        $urls = array();
        // Homepage
        $urls[] = array(
            'loc' => home_url( '/' ),
            'lastmod' => gmdate( 'c' ),
            'changefreq' => 'daily',
            'priority' => '1.0',
        );
        
        foreach ( $pages as $page ) {
            $urls[] = array(
                'loc' => get_permalink( $page->ID ),
                'lastmod' => get_post_modified_time( 'c', false, $page->ID ),
                'changefreq' => 'monthly',
                'priority' => '0.6',
            );
        }
        
        return $urls;
    }
    
    /**
     * Get posts URLs with images
     */
    private function get_posts_urls() {
        $posts = get_posts( array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'modified',
            'order' => 'DESC',
        ) );
        
        $urls = array();
        foreach ( $posts as $post ) {
            $urls[] = array(
                'loc' => get_permalink( $post->ID ),
                'lastmod' => get_post_modified_time( 'c', false, $post->ID ),
                'changefreq' => 'weekly',
                'priority' => '0.8',
                'images' => $this->get_post_images( $post->ID ),
            );
        }
        
        return $urls;
    }
    
    /**
     * Get categories URLs
     */
    private function get_categories_urls() {
        $categories = get_terms( array(
            'taxonomy' => 'category',
            'hide_empty' => true,
        ) );
        
        $urls = array();
        if ( ! is_wp_error( $categories ) && ! empty( $categories ) ) {
            foreach ( $categories as $category ) {
                $urls[] = array(
                    'loc' => get_term_link( $category ),
                    'lastmod' => gmdate( 'c' ),
                    'changefreq' => 'weekly',
                    'priority' => '0.5',
                );
            }
        }
        
        return $urls;
    }
    
    /**
     * Get tags URLs
     */
    private function get_tags_urls() {
        $tags = get_terms( array(
            'taxonomy' => 'post_tag',
            'hide_empty' => true,
        ) );
        
        $urls = array();
        if ( ! is_wp_error( $tags ) && ! empty( $tags ) ) {
            foreach ( $tags as $tag ) {
                $urls[] = array(
                    'loc' => get_term_link( $tag ),
                    'lastmod' => gmdate( 'c' ),
                    'changefreq' => 'weekly',
                    'priority' => '0.4',
                );
            }
        }
        
        return $urls;
    }
    
    /**
     * Get products URLs with images (WooCommerce)
     */
    private function get_products_urls() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return array();
        }
        
        $products = get_posts( array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'modified',
            'order' => 'DESC',
        ) );
        
        $urls = array();
        foreach ( $products as $product ) {
            $urls[] = array(
                'loc' => get_permalink( $product->ID ),
                'lastmod' => get_post_modified_time( 'c', false, $product->ID ),
                'changefreq' => 'weekly',
                'priority' => '0.9',
                'images' => $this->get_post_images( $product->ID ),
            );
        }
        
        return $urls;
    }
    
    /**
     * Get product categories URLs
     */
    private function get_product_categories_urls() {
        if ( ! taxonomy_exists( 'product_cat' ) ) {
            return array();
        }
        
        $categories = get_terms( array(
            'taxonomy' => 'product_cat',
            'hide_empty' => true,
        ) );
        
        $urls = array();
        if ( ! is_wp_error( $categories ) && ! empty( $categories ) ) {
            foreach ( $categories as $category ) {
                $urls[] = array(
                    'loc' => get_term_link( $category ),
                    'lastmod' => gmdate( 'c' ),
                    'changefreq' => 'weekly',
                    'priority' => '0.7',
                );
            }
        }
        
        return $urls;
    }
    
    /**
     * Get product tags URLs
     */
    private function get_product_tags_urls() {
        if ( ! taxonomy_exists( 'product_tag' ) ) {
            return array();
        }
        
        $tags = get_terms( array(
            'taxonomy' => 'product_tag',
            'hide_empty' => true,
        ) );
        
        $urls = array();
        if ( ! is_wp_error( $tags ) && ! empty( $tags ) ) {
            foreach ( $tags as $tag ) {
                $urls[] = array(
                    'loc' => get_term_link( $tag ),
                    'lastmod' => gmdate( 'c' ),
                    'changefreq' => 'weekly',
                    'priority' => '0.6',
                );
            }
        }
        
        return $urls;
    }
    
    /**
     * Get images URLs - TASK #3: Include ALL images
     */
    private function get_images_urls() {
        $images = get_posts( array(
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'post_mime_type' => 'image',
            'numberposts' => 5000, // Increased to get more images
            'orderby' => 'modified',
            'order' => 'DESC',
        ) );
        
        $urls = array();
        foreach ( $images as $image ) {
            $image_url = wp_get_attachment_url( $image->ID );
            if ( $image_url ) {
                $urls[] = array(
                    'loc' => $image_url,
                    'lastmod' => get_post_modified_time( 'c', false, $image->ID ),
                    'changefreq' => 'monthly',
                    'priority' => '0.3',
                );
            }
        }
        
        return $urls;
    }
    
    /**
     * Get authors URLs
     */
    private function get_authors_urls() {
        $authors = get_users( array(
            'who' => 'authors',
            'has_published_posts' => true,
            'orderby' => 'post_count',
            'order' => 'DESC',
        ) );
        
        $urls = array();
        foreach ( $authors as $author ) {
            $urls[] = array(
                'loc' => get_author_posts_url( $author->ID ),
                'lastmod' => gmdate( 'c' ),
                'changefreq' => 'weekly',
                'priority' => '0.6',
            );
        }
        
        return $urls;
    }
    
    /**
     * Get Dokan vendors URLs
     */
    private function get_vendors_urls() {
        if ( ! class_exists( 'WeDevs_Dokan' ) ) {
            return array();
        }
        
        $vendors = dokan_get_sellers( array( 'number' => -1 ) );
        $urls = array();
        
        if ( ! empty( $vendors['users'] ) ) {
            foreach ( $vendors['users'] as $vendor ) {
                $store_url = dokan_get_store_url( $vendor->ID );
                if ( $store_url ) {
                    $urls[] = array(
                        'loc' => $store_url,
                        'lastmod' => gmdate( 'c' ),
                        'changefreq' => 'daily',
                        'priority' => '0.8',
                    );
                }
            }
        }
        
        return $urls;
    }
    
    /**
     * Get Dokan vendor products URLs
     */
    private function get_vendor_products_urls() {
        if ( ! class_exists( 'WeDevs_Dokan' ) ) {
            return array();
        }
        
        $vendors = dokan_get_sellers( array( 'number' => -1 ) );
        $urls = array();
        
        if ( ! empty( $vendors['users'] ) ) {
            foreach ( $vendors['users'] as $vendor ) {
                // Get vendor products
                $products = get_posts( array(
                    'post_type' => 'product',
                    'post_status' => 'publish',
                    'author' => $vendor->ID,
                    'numberposts' => -1,
                ) );
                
                foreach ( $products as $product ) {
                    $urls[] = array(
                        'loc' => get_permalink( $product->ID ),
                        'lastmod' => get_post_modified_time( 'c', false, $product->ID ),
                        'changefreq' => 'weekly',
                        'priority' => '0.9',
                        'images' => $this->get_post_images( $product->ID ),
                    );
                }
            }
        }
        
        return $urls;
    }
    
    /**
     * Get images for a specific post - Google Image Sitemap
     */
    private function get_post_images( $post_id ) {
        $images = array();
        
        // Featured image
        $thumbnail_id = get_post_thumbnail_id( $post_id );
        if ( $thumbnail_id ) {
            $image_url = wp_get_attachment_url( $thumbnail_id );
            if ( $image_url ) {
                $images[] = array(
                    'loc' => $image_url,
                    'title' => get_the_title( $thumbnail_id ),
                    'caption' => wp_get_attachment_caption( $thumbnail_id ),
                );
            }
        }
        
        // Gallery images
        $attachments = get_attached_media( 'image', $post_id );
        foreach ( $attachments as $attachment ) {
            if ( $attachment->ID == $thumbnail_id ) {
                continue;
            }
            
            $image_url = wp_get_attachment_url( $attachment->ID );
            if ( $image_url ) {
                $images[] = array(
                    'loc' => $image_url,
                    'title' => get_the_title( $attachment->ID ),
                    'caption' => wp_get_attachment_caption( $attachment->ID ),
                );
            }
        }
        
        return $images;
    }

    /**
     * Generate sitemap content
     * ENHANCED: Includes ALL content types
     */
    public function generate_sitemap_content() {
        $urls = array();

        // Add homepage
        $urls[] = array(
            'loc' => home_url( '/' ),
            'lastmod' => get_lastpostmodified( 'GMT' ),
            'changefreq' => 'daily',
            'priority' => '1.0',
        );

        // Add posts
        $posts = get_posts( array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'numberposts' => $this->max_urls,
            'orderby' => 'modified',
            'order' => 'DESC',
        ) );

        foreach ( $posts as $post ) {
            $urls[] = array(
                'loc' => get_permalink( $post->ID ),
                'lastmod' => get_post_modified_time( 'c', false, $post->ID ),
                'changefreq' => 'weekly',
                'priority' => '0.8',
            );
        }

        // Add pages
        $pages = get_posts( array(
            'post_type' => 'page',
            'post_status' => 'publish',
            'numberposts' => $this->max_urls,
        ) );

        foreach ( $pages as $page ) {
            $urls[] = array(
                'loc' => get_permalink( $page->ID ),
                'lastmod' => get_post_modified_time( 'c', false, $page->ID ),
                'changefreq' => 'monthly',
                'priority' => '0.6',
            );
        }

        // Add products (WooCommerce)
        if ( class_exists( 'WooCommerce' ) ) {
            $products = get_posts( array(
                'post_type' => 'product',
                'post_status' => 'publish',
                'numberposts' => $this->max_urls,
            ) );

            foreach ( $products as $product ) {
                $urls[] = array(
                    'loc' => get_permalink( $product->ID ),
                    'lastmod' => get_post_modified_time( 'c', false, $product->ID ),
                    'changefreq' => 'weekly',
                    'priority' => '0.9',
                );
            }
        }

        // ENHANCED: Add post categories
        $post_categories = get_categories( array(
            'taxonomy' => 'category',
            'hide_empty' => true,
        ) );

        foreach ( $post_categories as $category ) {
            $urls[] = array(
                'loc' => get_category_link( $category->term_id ),
                'lastmod' => gmdate( 'c' ),
                'changefreq' => 'weekly',
                'priority' => '0.5',
            );
        }

        // ENHANCED: Add post tags
        $post_tags = get_terms( array(
            'taxonomy' => 'post_tag',
            'hide_empty' => true,
        ) );

        if ( ! is_wp_error( $post_tags ) && ! empty( $post_tags ) ) {
            foreach ( $post_tags as $tag ) {
                $urls[] = array(
                    'loc' => get_tag_link( $tag->term_id ),
                    'lastmod' => gmdate( 'c' ),
                    'changefreq' => 'weekly',
                    'priority' => '0.4',
                );
            }
        }

        // ENHANCED: Add product categories (WooCommerce)
        if ( taxonomy_exists( 'product_cat' ) ) {
            $product_cats = get_terms( array(
                'taxonomy' => 'product_cat',
                'hide_empty' => true,
            ) );

            if ( ! is_wp_error( $product_cats ) && ! empty( $product_cats ) ) {
                foreach ( $product_cats as $cat ) {
                    $urls[] = array(
                        'loc' => get_term_link( $cat ),
                        'lastmod' => gmdate( 'c' ),
                        'changefreq' => 'weekly',
                        'priority' => '0.7',
                    );
                }
            }
        }

        // ENHANCED: Add product tags (WooCommerce)
        if ( taxonomy_exists( 'product_tag' ) ) {
            $product_tags = get_terms( array(
                'taxonomy' => 'product_tag',
                'hide_empty' => true,
            ) );

            if ( ! is_wp_error( $product_tags ) && ! empty( $product_tags ) ) {
                foreach ( $product_tags as $tag ) {
                    $urls[] = array(
                        'loc' => get_term_link( $tag ),
                        'lastmod' => gmdate( 'c' ),
                        'changefreq' => 'weekly',
                        'priority' => '0.6',
                    );
                }
            }
        }

        // ENHANCED: Add images (attachments)
        $images = get_posts( array(
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'post_mime_type' => 'image',
            'numberposts' => 1000, // Limit to 1000 images
            'orderby' => 'modified',
            'order' => 'DESC',
        ) );

        foreach ( $images as $image ) {
            $image_url = wp_get_attachment_url( $image->ID );
            if ( $image_url ) {
                $urls[] = array(
                    'loc' => $image_url,
                    'lastmod' => get_post_modified_time( 'c', false, $image->ID ),
                    'changefreq' => 'monthly',
                    'priority' => '0.3',
                );
            }
        }

        // Generate XML
        return $this->build_sitemap_xml( $urls );
    }

    /**
     * Build sitemap XML from URLs array
     * ENHANCED: Supports Google Image Sitemap format + type parameter
     */
    private function build_sitemap_xml( $urls, $type = 'general' ) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<?xml-stylesheet type="text/xsl" href="' . esc_url( DRSS_PLUGIN_URL . 'assets/sitemap.xsl' ) . '"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"';
        $xml .= ' xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";

        foreach ( $urls as $url ) {
            $xml .= "  <url>\n";
            $xml .= '    <loc>' . esc_url( $url['loc'] ) . "</loc>\n";
            
            if ( ! empty( $url['lastmod'] ) ) {
                $xml .= '    <lastmod>' . esc_html( $url['lastmod'] ) . "</lastmod>\n";
            }
            
            if ( ! empty( $url['changefreq'] ) ) {
                $xml .= '    <changefreq>' . esc_html( $url['changefreq'] ) . "</changefreq>\n";
            }
            
            if ( ! empty( $url['priority'] ) ) {
                $xml .= '    <priority>' . esc_html( $url['priority'] ) . "</priority>\n";
            }
            
            // TASK #3: Add image data if this URL has images
            if ( ! empty( $url['images'] ) && is_array( $url['images'] ) ) {
                foreach ( $url['images'] as $image ) {
                    $xml .= "    <image:image>\n";
                    $xml .= '      <image:loc>' . esc_url( $image['loc'] ) . "</image:loc>\n";
                    if ( ! empty( $image['title'] ) ) {
                        $xml .= '      <image:title>' . esc_html( $image['title'] ) . "</image:title>\n";
                    }
                    if ( ! empty( $image['caption'] ) ) {
                        $xml .= '      <image:caption>' . esc_html( $image['caption'] ) . "</image:caption>\n";
                    }
                    $xml .= "    </image:image>\n";
                }
            }
            
            $xml .= "  </url>\n";
        }

        $xml .= '</urlset>';

        return $xml;
    }

    /**
     * Generate sitemap file
     */
    public function generate_sitemap() {
        $content = $this->generate_sitemap_content();
        
        // Save to transient cache
        set_transient( 'drss_sitemap_cache', $content, DAY_IN_SECONDS );
        
        // Update last generation time
        update_option( 'drss_sitemap_last_generated', current_time( 'mysql' ) );
        
        return true;
    }

    /**
     * Schedule sitemap regeneration (rate-limited)
     */
    public function schedule_regeneration( $post_id = 0, $post = null ) {
        // Ignore autosaves and revisions
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( $post && $post->post_status !== 'publish' ) {
            return;
        }

        // Rate limit: only regenerate once every 5 minutes
        $last_regeneration = get_transient( 'drss_sitemap_regenerating' );
        if ( $last_regeneration ) {
            return;
        }

        // Mark as regenerating
        set_transient( 'drss_sitemap_regenerating', 1, 5 * MINUTE_IN_SECONDS );
        
        // Clear cache to force regeneration on next request
        delete_transient( 'drss_sitemap_cache' );
    }

    /**
     * AJAX: Regenerate sitemap
     */
    public function ajax_regenerate_sitemap() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Unauthorized' ) );
        }

        $result = $this->generate_sitemap();

        if ( $result ) {
            $last_generated = get_option( 'drss_sitemap_last_generated' );
            wp_send_json_success( array(
                'message' => __( 'Sitemap regenerated successfully!', 'digital-rise-seo-solutions' ),
                'last_generated' => $last_generated,
                'url' => home_url( '/sitemap.xml' ),
            ) );
        } else {
            wp_send_json_error( array( 'message' => __( 'Failed to generate sitemap', 'digital-rise-seo-solutions' ) ) );
        }
    }

    /**
     * Get sitemap stats
     * ENHANCED: Shows counts for all content types
     */
    public function get_stats() {
        $last_generated = get_option( 'drss_sitemap_last_generated' );
        $cached = get_transient( 'drss_sitemap_cache' ) ? true : false;
        
        // Count URLs by type
        $url_count = 1; // Homepage
        
        // Posts
        $posts_count = wp_count_posts( 'post' )->publish;
        $url_count += $posts_count;
        
        // Pages
        $pages_count = wp_count_posts( 'page' )->publish;
        $url_count += $pages_count;
        
        // Products (WooCommerce)
        $products_count = 0;
        if ( class_exists( 'WooCommerce' ) ) {
            $products_count = wp_count_posts( 'product' )->publish;
            $url_count += $products_count;
        }
        
        // Post categories
        $post_cats_count = wp_count_terms( array( 'taxonomy' => 'category', 'hide_empty' => true ) );
        $url_count += $post_cats_count;
        
        // Post tags
        $post_tags_count = wp_count_terms( array( 'taxonomy' => 'post_tag', 'hide_empty' => true ) );
        $url_count += $post_tags_count;
        
        // Product categories
        $product_cats_count = 0;
        if ( taxonomy_exists( 'product_cat' ) ) {
            $product_cats_count = wp_count_terms( array( 'taxonomy' => 'product_cat', 'hide_empty' => true ) );
            $url_count += $product_cats_count;
        }
        
        // Product tags
        $product_tags_count = 0;
        if ( taxonomy_exists( 'product_tag' ) ) {
            $product_tags_count = wp_count_terms( array( 'taxonomy' => 'product_tag', 'hide_empty' => true ) );
            $url_count += $product_tags_count;
        }
        
        // Images (limited to 1000)
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Count query for stats
        $images_count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
             WHERE post_type = 'attachment' 
             AND post_status = 'inherit' 
             AND post_mime_type LIKE 'image/%'
             LIMIT 1000"
        );
        $url_count += (int) $images_count;

        return array(
            'exists' => $cached || $last_generated,
            'last_generated' => $last_generated,
            'cached' => $cached,
            'url_count' => $url_count,
            'breakdown' => array(
                'homepage' => 1,
                'posts' => $posts_count,
                'pages' => $pages_count,
                'products' => $products_count,
                'post_categories' => $post_cats_count,
                'post_tags' => $post_tags_count,
                'product_categories' => $product_cats_count,
                'product_tags' => $product_tags_count,
                'images' => (int) $images_count,
            ),
            'url' => home_url( '/sitemap.xml' ),
        );
    }
}
