<?php
/**
 * Image SEO Automation
 * MAX 7 WORDS for all fields (Google SEO best practice)
 * Uses customizable pattern from settings
 *
 * @package Digital_Rise_SEO_Solutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRSS_SEO {

    private $plugin;
    private $max_words = 7;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Auto-set image meta on NEW uploads
     * Uses pattern and limits to MAX 7 WORDS
     * FIXED: Ensures unique titles per attachment
     */
    public function auto_set_image_meta( $metadata, $attachment_id ) {
        // Get attachment
        $attachment = get_post( $attachment_id );
        
        if ( ! $attachment || $attachment->post_type !== 'attachment' ) {
            return $metadata;
        }
        
        // Check if it's an image
        if ( strpos( get_post_mime_type( $attachment_id ), 'image/' ) !== 0 ) {
            return $metadata;
        }

        // Generate SEO text using pattern (max 7 words)
        $seo_text = $this->generate_seo_text_from_pattern( $attachment_id );
        
        // FIXED: Make title unique
        $seo_text = $this->make_unique_seo_text( $seo_text, $attachment_id );

        // Set Title, ALT, Caption, Description
        $this->set_all_image_meta( $attachment_id, $seo_text );

        // Mark as processed
        update_post_meta( $attachment_id, '_drss_seo_processed', '1' );
        update_post_meta( $attachment_id, '_drss_seo_text', $seo_text );

        return $metadata;
    }

    /**
     * Apply SEO to EXISTING image (bulk optimization)
     * Uses pattern and limits to MAX 7 WORDS
     * FIXED: Appends unique suffix to prevent duplicate titles
     */
    public function apply_seo_to_image( $attachment_id ) {
        // Generate SEO text using pattern (max 7 words)
        $seo_text = $this->generate_seo_text_from_pattern( $attachment_id );
        
        // FIXED: Make title unique by checking for duplicates
        $seo_text = $this->make_unique_seo_text( $seo_text, $attachment_id );

        // Set Title, ALT, Caption, Description
        $this->set_all_image_meta( $attachment_id, $seo_text );

        // Mark as processed
        update_post_meta( $attachment_id, '_drss_seo_processed', '1' );
        update_post_meta( $attachment_id, '_drss_seo_text', $seo_text );

        return true;
    }
    
    /**
     * Make SEO text unique across attachments.
     * If another attachment already has this exact title, append -2, -3, etc.
     * This prevents "clicking image opens wrong image" caused by duplicate post_titles.
     */
    private function make_unique_seo_text( $seo_text, $attachment_id ) {
        global $wpdb;
        
        // Check if any OTHER attachment already has this exact title
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking for duplicate titles
        $existing_count = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
             WHERE post_type = 'attachment' 
             AND post_title = %s 
             AND ID != %d",
            $seo_text,
            $attachment_id
        ) );
        
        if ( $existing_count === 0 ) {
            return $seo_text;
        }
        
        // Find the next available suffix
        $counter = 2;
        $candidate = $seo_text . ' ' . $counter;
        
        while ( true ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Finding unique title
            $exists = (int) $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->posts} 
                 WHERE post_type = 'attachment' 
                 AND post_title = %s 
                 AND ID != %d",
                $candidate,
                $attachment_id
            ) );
            
            if ( $exists === 0 ) {
                return $candidate;
            }
            
            $counter++;
            $candidate = $seo_text . ' ' . $counter;
            
            // Safety valve
            if ( $counter > 10000 ) {
                return $seo_text . ' ' . $attachment_id;
            }
        }
    }

    /**
     * Generate SEO text from pattern with MAX 7 WORDS
     * PHASE 3 FIX #1: Enhanced with all variables + content-type detection
     * Pattern example: {site_name} {keyword} {content_title}
     */
    private function generate_seo_text_from_pattern( $attachment_id = 0 ) {
        // Detect content type for pattern selection
        $content_type = $this->detect_content_type( $attachment_id );
        
        // Get pattern from settings (content-type-specific or default)
        $pattern = $this->get_pattern_for_content_type( $content_type );
        
        if ( empty( $pattern ) ) {
            $pattern = '{site_name} {keyword}';
        }
        
        // Get all variables
        $site_name = get_bloginfo( 'name' );
        
        // Get keyword
        $keywords = new DRSS_Keywords( $this->plugin );
        $keyword = $keywords->get_random_keyword();
        
        // Get content title and author (from parent post if attached)
        $content_title = '';
        $author = '';
        if ( $attachment_id ) {
            $parent_id = wp_get_post_parent_id( $attachment_id );
            if ( $parent_id ) {
                $content_title = get_the_title( $parent_id );
                $author_obj = get_user_by( 'id', get_post_field( 'post_author', $parent_id ) );
                if ( $author_obj ) {
                    $author = $author_obj->display_name;
                }
            }
        }
        
        // Get category
        $category = '';
        if ( $attachment_id ) {
            $parent_id = wp_get_post_parent_id( $attachment_id );
            if ( $parent_id ) {
                $cats = get_the_category( $parent_id );
                if ( ! empty( $cats ) ) {
                    $category = $cats[0]->name;
                }
            }
        }
        
        // Replace variables - PHASE 3: All variables including {author}
        $replacements = array(
            '{site_name}' => $site_name,
            '{keyword}' => $keyword,
            '{content_title}' => $content_title,
            '{category}' => $category,
            '{date}' => wp_date( 'Y' ),
            '{year}' => wp_date( 'Y' ),
            '{month}' => wp_date( 'F' ),
            '{author}' => $author, // PHASE 3: Added author variable
        );
        
        $result = str_replace( array_keys( $replacements ), array_values( $replacements ), $pattern );
        
        // Clean up empty variables and extra spaces
        $result = preg_replace( '/\{[^}]+\}/', '', $result );
        $result = preg_replace( '/\s+/', ' ', $result );
        $result = trim( $result );
        
        // Limit to 7 words
        $words = preg_split( '/\s+/', $result );
        $words = array_slice( $words, 0, $this->max_words );
        $result = implode( ' ', $words );
        
        // Fallback
        if ( empty( $result ) ) {
            $result = ! empty( $site_name ) ? $site_name : 'Image';
        }
        
        return ucwords( $result );
    }
    
    /**
     * PHASE 3 FIX #6: Detect content type for pattern selection
     */
    private function detect_content_type( $attachment_id ) {
        if ( ! $attachment_id ) {
            return 'default';
        }
        
        $parent_id = wp_get_post_parent_id( $attachment_id );
        if ( ! $parent_id ) {
            return 'default';
        }
        
        $post_type = get_post_type( $parent_id );
        
        // Map post types to our pattern types
        $type_map = array(
            'post' => 'post',
            'page' => 'page',
            'product' => 'product',
        );
        
        return isset( $type_map[ $post_type ] ) ? $type_map[ $post_type ] : 'default';
    }
    
    /**
     * PHASE 3 FIX #6: Get pattern for specific content type
     */
    private function get_pattern_for_content_type( $content_type ) {
        // Check for content-type-specific pattern
        $option_key = 'image_seo_pattern_' . $content_type;
        $pattern = $this->plugin->get_option( $option_key, '' );
        
        // Fall back to default pattern if not set
        if ( empty( $pattern ) ) {
            $pattern = $this->plugin->get_option( 'image_seo_pattern', '{site_name} {keyword}' );
        }
        
        return $pattern;
    }

    /**
     * Set ALL image meta fields
     */
    private function set_all_image_meta( $attachment_id, $seo_text ) {
        // 1. Set Title, Caption, Description
        wp_update_post( array(
            'ID' => $attachment_id,
            'post_title' => $seo_text,
            'post_excerpt' => $seo_text, // Caption
            'post_content' => $seo_text, // Description
        ) );

        // 2. Set ALT text
        delete_post_meta( $attachment_id, '_wp_attachment_image_alt' );
        add_post_meta( $attachment_id, '_wp_attachment_image_alt', $seo_text, true );
    }

    /**
     * Get current SEO text for an image
     */
    public function get_image_seo_text( $attachment_id ) {
        return get_post_meta( $attachment_id, '_drss_seo_text', true );
    }

    /**
     * Check if image has been SEO processed
     */
    public function is_processed( $attachment_id ) {
        return get_post_meta( $attachment_id, '_drss_seo_processed', true ) === '1';
    }

    /**
     * Get SEO stats
     */
    public function get_stats() {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Statistics query
        $total = $wpdb->get_var( "
            SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_type = 'attachment' 
            AND post_mime_type LIKE 'image/%'
        " );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Statistics query
        $processed = $wpdb->get_var( "
            SELECT COUNT(*) FROM {$wpdb->postmeta} 
            WHERE meta_key = '_drss_seo_processed' 
            AND meta_value = '1'
        " );
        
        return array(
            'total' => (int) $total,
            'processed' => (int) $processed,
            'pending' => max( 0, (int) $total - (int) $processed ),
        );
    }
    
    /**
     * Get available variables for image SEO pattern
     * PHASE 3 FIX #1: Added {author} variable
     */
    public function get_available_variables() {
        return array(
            '{site_name}' => __( 'Your website name', 'digital-rise-seo-solutions' ),
            '{keyword}' => __( 'Random keyword from your list', 'digital-rise-seo-solutions' ),
            '{content_title}' => __( 'Title of the parent post/page/product', 'digital-rise-seo-solutions' ),
            '{category}' => __( 'Primary category of the parent content', 'digital-rise-seo-solutions' ),
            '{author}' => __( 'Author name of the parent content', 'digital-rise-seo-solutions' ),
            '{year}' => __( 'Current year', 'digital-rise-seo-solutions' ),
            '{month}' => __( 'Current month name', 'digital-rise-seo-solutions' ),
        );
    }
}
