<?php
/**
 * SEO Checklist
 * Based on Digital Rise Solutions SEO Checklist PDF
 * Provides a comprehensive checklist with tracking
 *
 * @package Digital_Rise_SEO_Solutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRSS_SEO_Checklist {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Get the full SEO checklist organized by category
     */
    public function get_checklist() {
        return array(
            'keyword_research' => array(
                'title' => __( 'Keyword Research & Strategy', 'digital-rise-seo-solutions' ),
                'icon'  => 'dashicons-search',
                'items' => array(
                    array( 'id' => 'kw_01', 'text' => __( 'Extract keywords: Find your top-ranking keywords', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'kw_02', 'text' => __( 'Choose the most profitable keywords', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'kw_03', 'text' => __( 'Conduct competitor research: Analyze your competitors', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'kw_04', 'text' => __( 'Identify your organic competitors', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'kw_05', 'text' => __( 'Steal keywords from your competitors', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'kw_06', 'text' => __( 'Determine the potential traffic each keyword can bring', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'kw_07', 'text' => __( 'Assign one main keyword and two secondary keywords to each page', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'kw_08', 'text' => __( 'Avoid keyword cannibalization: Don\'t use the same keyword on multiple pages', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                ),
            ),
            'on_page_seo' => array(
                'title' => __( 'On-Page SEO', 'digital-rise-seo-solutions' ),
                'icon'  => 'dashicons-edit-page',
                'items' => array(
                    array( 'id' => 'op_01', 'text' => __( 'Prepare content using the identified keywords', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'op_02', 'text' => __( 'Check keyword placement: Title tag, H1, first paragraph, anchor text, image alt, file/folder names', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'op_03', 'text' => __( 'Ensure every page has an H1 tag: Fix missing or duplicate H1 tags', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'op_04', 'text' => __( 'Review duplicate or missing description tags', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'op_05', 'text' => __( 'Identify pages missing title tags or having duplicate/long/short tags', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'op_06', 'text' => __( 'Check keyword density', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'op_07', 'text' => __( 'Consider buying a domain with relevant keywords', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'op_08', 'text' => __( 'Separate content into different pages based on topics', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'op_09', 'text' => __( 'Create PDF files using keywords', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'op_10', 'text' => __( 'Ensure a canonical page and avoid duplicate content', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'op_11', 'text' => __( 'Implement rich snippets and Schema Markup', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'op_12', 'text' => __( 'Optimize your website for voice search', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                ),
            ),
            'site_structure' => array(
                'title' => __( 'Site Structure & Links', 'digital-rise-seo-solutions' ),
                'icon'  => 'dashicons-networking',
                'items' => array(
                    array( 'id' => 'ss_01', 'text' => __( 'Verify max two levels deep page structure', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ss_02', 'text' => __( 'Implement onsite linking strategies', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ss_03', 'text' => __( 'Consolidate outgoing links on one page with nofollow tags', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ss_04', 'text' => __( 'Check for broken internal and external links', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ss_05', 'text' => __( 'Check for 404 errors and set up redirects', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ss_06', 'text' => __( 'Create an email signature with a backlink', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'ss_07', 'text' => __( 'Submit your site to relevant directories', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ss_08', 'text' => __( 'Exchange backlinks with other websites', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                ),
            ),
            'images_media' => array(
                'title' => __( 'Images & Media SEO', 'digital-rise-seo-solutions' ),
                'icon'  => 'dashicons-format-image',
                'items' => array(
                    array( 'id' => 'im_01', 'text' => __( 'Fix broken images and convert all to WEBP format', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'im_02', 'text' => __( 'Add ALT text to all images', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'im_03', 'text' => __( 'Detect and compress large images', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'im_04', 'text' => __( 'Check image-to-text ratio', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'im_05', 'text' => __( 'Create a 16x16px favicon with relevant keyword name', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                ),
            ),
            'technical_seo' => array(
                'title' => __( 'Technical SEO', 'digital-rise-seo-solutions' ),
                'icon'  => 'dashicons-admin-tools',
                'items' => array(
                    array( 'id' => 'ts_01', 'text' => __( 'Make your site accessible online', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ts_02', 'text' => __( 'Ensure HTTP redirects to HTTPS', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ts_03', 'text' => __( 'Create subdomains and redirect to main domain', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ts_04', 'text' => __( 'Publish a cookie policy', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ts_05', 'text' => __( 'Test website speed', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ts_06', 'text' => __( 'Check mobile-friendliness', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ts_07', 'text' => __( 'Configure robots.txt to disallow JS pages from crawling', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ts_08', 'text' => __( 'Generate sitemap, upload, and submit to search console', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'ts_09', 'text' => __( 'Review coding issues and make improvements', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ts_10', 'text' => __( 'WWW vs non-WWW consistency check', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ts_11', 'text' => __( 'Trailing slash consistency check', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'ts_12', 'text' => __( 'Pagination SEO check', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'ts_13', 'text' => __( 'Detect orphan pages (no internal links pointing to them)', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ts_14', 'text' => __( 'Check meta charset declaration', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'ts_15', 'text' => __( 'Check viewport meta tag', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                ),
            ),
            'search_console' => array(
                'title' => __( 'Search Console & Analytics', 'digital-rise-seo-solutions' ),
                'icon'  => 'dashicons-chart-bar',
                'items' => array(
                    array( 'id' => 'sc_01', 'text' => __( 'Install and set up Google Search Console', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'sc_02', 'text' => __( 'Use Search Console to inspect URLs', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'sc_03', 'text' => __( 'Implement a tag manager', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'sc_04', 'text' => __( 'Set up website analytics', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'sc_05', 'text' => __( 'Review pages with bounce rate over 45%', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'sc_06', 'text' => __( 'Create Google Business Profile and submit sitemap to Bing', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'sc_07', 'text' => __( 'Ensure site is indexed in Google, Bing, Yahoo', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'sc_08', 'text' => __( 'Install SEOquake to analyze performance', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'sc_09', 'text' => __( 'Use spider simulation tools', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'sc_10', 'text' => __( 'Set up Google Alerts for your business name', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'sc_11', 'text' => __( 'Consider running Google Ads PPC campaigns', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                ),
            ),
            'social_media' => array(
                'title' => __( 'Social Media SEO', 'digital-rise-seo-solutions' ),
                'icon'  => 'dashicons-share',
                'items' => array(
                    array( 'id' => 'sm_01', 'text' => __( 'Add trusted info: phone, address, hours, images, awards', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'sm_02', 'text' => __( 'Create social media profiles, start posting and sharing URLs', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'sm_03', 'text' => __( 'Generate and install Open Graph (OG) tags', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'sm_04', 'text' => __( 'Optimize social media pages using keywords', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'sm_05', 'text' => __( 'Integrate all social media channels together', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'sm_06', 'text' => __( 'Add "follow us" and "like" buttons on your website', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                ),
            ),
            'daily_routine' => array(
                'title' => __( 'Optimized Daily SEO Routine', 'digital-rise-seo-solutions' ),
                'icon'  => 'dashicons-calendar-alt',
                'items' => array(
                    array( 'id' => 'dr_01', 'text' => __( 'Check and track rankings and visibility', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'dr_02', 'text' => __( 'Identify and remove broken links', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'dr_03', 'text' => __( 'Review and address 404 error pages', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'dr_04', 'text' => __( 'Analyze website performance using Google Analytics', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'dr_05', 'text' => __( 'Monitor keyword positions and adjust', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'dr_06', 'text' => __( 'Create new content with backlinks', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'dr_07', 'text' => __( 'Search for link exchange partners', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'dr_08', 'text' => __( 'Find new business directories to list your site', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'dr_09', 'text' => __( 'Stay updated on SEO trends', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                    array( 'id' => 'dr_10', 'text' => __( 'Check for duplicate heading tags', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'dr_11', 'text' => __( 'Optimize content titles, H1s, and paragraphs', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'dr_12', 'text' => __( 'Analyze competitors\' backlinks', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'dr_13', 'text' => __( 'Acquire backlinks from competitors\' referring domains', 'digital-rise-seo-solutions' ), 'priority' => 'medium' ),
                    array( 'id' => 'dr_14', 'text' => __( 'Verify site is indexed in major search engines', 'digital-rise-seo-solutions' ), 'priority' => 'high' ),
                    array( 'id' => 'dr_15', 'text' => __( 'Share knowledge in the SEO community', 'digital-rise-seo-solutions' ), 'priority' => 'low' ),
                ),
            ),
        );
    }

    /**
     * Get saved checklist progress
     */
    public function get_progress() {
        return get_option( 'drss_checklist_progress', array() );
    }

    /**
     * Save checklist item status
     */
    public function save_item( $item_id, $completed ) {
        $progress = $this->get_progress();
        
        if ( $completed ) {
            $progress[ $item_id ] = array(
                'completed' => true,
                'date'      => current_time( 'mysql' ),
            );
        } else {
            unset( $progress[ $item_id ] );
        }
        
        update_option( 'drss_checklist_progress', $progress );
        return $progress;
    }

    /**
     * Get completion stats
     */
    public function get_stats() {
        $checklist = $this->get_checklist();
        $progress = $this->get_progress();
        
        $total = 0;
        $completed = 0;
        
        foreach ( $checklist as $category ) {
            foreach ( $category['items'] as $item ) {
                $total++;
                if ( isset( $progress[ $item['id'] ] ) && $progress[ $item['id'] ]['completed'] ) {
                    $completed++;
                }
            }
        }
        
        return array(
            'total'      => $total,
            'completed'  => $completed,
            'remaining'  => $total - $completed,
            'percentage' => $total > 0 ? round( ( $completed / $total ) * 100 ) : 0,
        );
    }
}
