<?php
/**
 * SEO Audit Engine
 * Crawls entire site and analyzes SEO health per-URL
 * NO external APIs - pure WordPress/PHP
 *
 * @package Digital_Rise_SEO_Solutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRSS_SEO_Audit {

    private $plugin;
    
    // Scoring weights (total = 100)
    const WEIGHTS = array(
        'title_tag'           => 10,
        'meta_description'    => 8,
        'h1_tag'              => 8,
        'h2_structure'        => 4,
        'url_length'          => 3,
        'image_alt'           => 7,
        'internal_links'      => 6,
        'external_links'      => 3,
        'canonical'           => 5,
        'meta_robots'         => 3,
        'og_tags'             => 4,
        'word_count'          => 7,
        'https'               => 5,
        'schema'              => 4,
        'viewport'            => 3,
        'charset'             => 2,
        'favicon'             => 2,
        'content_quality'     => 8,
        'broken_links'        => 5,
        'heading_hierarchy'   => 3,
    );

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Get all crawlable URLs from WordPress
     */
    public function collect_urls() {
        $urls = array();
        
        // Homepage
        $urls[] = array(
            'url'  => home_url( '/' ),
            'type' => 'homepage',
            'id'   => 0,
        );
        
        // Pages
        $pages = get_posts( array(
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ) );
        foreach ( $pages as $page_id ) {
            $urls[] = array(
                'url'  => get_permalink( $page_id ),
                'type' => 'page',
                'id'   => $page_id,
            );
        }
        
        // Posts
        $posts = get_posts( array(
            'post_type'      => 'post',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ) );
        foreach ( $posts as $post_id ) {
            $urls[] = array(
                'url'  => get_permalink( $post_id ),
                'type' => 'post',
                'id'   => $post_id,
            );
        }
        
        // Custom post types
        $cpts = get_post_types( array( 'public' => true, '_builtin' => false ), 'names' );
        foreach ( $cpts as $cpt ) {
            $cpt_posts = get_posts( array(
                'post_type'      => $cpt,
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'fields'         => 'ids',
            ) );
            foreach ( $cpt_posts as $cpt_id ) {
                $urls[] = array(
                    'url'  => get_permalink( $cpt_id ),
                    'type' => $cpt,
                    'id'   => $cpt_id,
                );
            }
        }
        
        return $urls;
    }

    /**
     * Analyze a single URL
     */
    public function analyze_url( $url_data ) {
        $url = $url_data['url'];
        $post_id = $url_data['id'];
        
        $result = array(
            'url'          => $url,
            'type'         => $url_data['type'],
            'post_id'      => $post_id,
            'score'        => 0,
            'checks'       => array(),
            'errors'       => array(),
            'warnings'     => array(),
            'passed'       => array(),
            'recommendations' => array(),
        );
        
        // Get content via WordPress internal methods (no HTTP request needed)
        $html = $this->get_page_html( $url_data );
        
        if ( empty( $html ) ) {
            $result['errors'][] = __( 'Could not retrieve page content', 'digital-rise-seo-solutions' );
            return $result;
        }
        
        // Run all checks
        $checks = array();
        $checks['title_tag']         = $this->check_title( $html, $post_id );
        $checks['meta_description']  = $this->check_meta_description( $html, $post_id );
        $checks['h1_tag']            = $this->check_h1( $html );
        $checks['h2_structure']      = $this->check_h2( $html );
        $checks['url_length']        = $this->check_url_length( $url );
        $checks['image_alt']         = $this->check_image_alts( $html );
        $checks['internal_links']    = $this->check_internal_links( $html );
        $checks['external_links']    = $this->check_external_links( $html );
        $checks['canonical']         = $this->check_canonical( $html );
        $checks['meta_robots']       = $this->check_meta_robots( $html );
        $checks['og_tags']           = $this->check_og_tags( $html );
        $checks['word_count']        = $this->check_word_count( $html );
        $checks['https']             = $this->check_https( $url );
        $checks['schema']            = $this->check_schema( $html );
        $checks['viewport']          = $this->check_viewport( $html );
        $checks['charset']           = $this->check_charset( $html );
        $checks['favicon']           = $this->check_favicon( $html );
        $checks['content_quality']   = $this->check_content_quality( $html, $post_id );
        $checks['broken_links']      = $this->check_broken_internal_links( $html );
        $checks['heading_hierarchy'] = $this->check_heading_hierarchy( $html );
        
        // Calculate score
        $total_score = 0;
        foreach ( $checks as $key => $check ) {
            $weight = isset( self::WEIGHTS[ $key ] ) ? self::WEIGHTS[ $key ] : 0;
            $check_score = ( $check['score'] / 100 ) * $weight;
            $total_score += $check_score;
            
            if ( $check['status'] === 'error' ) {
                $result['errors'][] = $check['message'];
            } elseif ( $check['status'] === 'warning' ) {
                $result['warnings'][] = $check['message'];
            } else {
                $result['passed'][] = $check['message'];
            }
            
            if ( ! empty( $check['recommendation'] ) ) {
                $result['recommendations'][] = $check['recommendation'];
            }
        }
        
        $result['score'] = round( $total_score );
        $result['checks'] = $checks;
        
        return $result;
    }

    /**
     * Get HTML content using WP internals (no HTTP request)
     */
    private function get_page_html( $url_data ) {
        $post_id = $url_data['id'];
        $type = $url_data['type'];
        
        // For homepage or posts, use output buffering with template
        if ( $post_id > 0 ) {
            $post = get_post( $post_id );
            if ( ! $post ) {
                return '';
            }
            
            // Build a synthetic HTML for analysis
            $html = $this->build_synthetic_html( $post );
            return $html;
        }
        
        // Homepage - try to get front page
        $front_page_id = get_option( 'page_on_front' );
        if ( $front_page_id ) {
            $post = get_post( $front_page_id );
            if ( $post ) {
                return $this->build_synthetic_html( $post );
            }
        }
        
        // Fallback: use wp_remote_get on local URL
        $response = wp_remote_get( $url_data['url'], array(
            'timeout'     => 15,
            'sslverify'   => false,
            'httpversion' => '1.1',
            'user-agent'  => 'DRSS-SEO-Audit/2.0',
        ) );
        
        if ( is_wp_error( $response ) ) {
            return '';
        }
        
        return wp_remote_retrieve_body( $response );
    }

    /**
     * Build HTML from post data for analysis
     */
    private function build_synthetic_html( $post ) {
        // Get the actual rendered title
        $seo_title = get_post_meta( $post->ID, '_drss_seo_title', true );
        $title = ! empty( $seo_title ) ? $seo_title : $post->post_title;
        
        // Get meta description
        $seo_desc = get_post_meta( $post->ID, '_drss_seo_description', true );
        $meta_desc = ! empty( $seo_desc ) ? $seo_desc : wp_trim_words( $post->post_content, 30, '' );
        
        // Get canonical
        $canonical = get_permalink( $post->ID );
        
        // Get OG tags
        $og_title = get_post_meta( $post->ID, '_drss_og_title', true );
        $og_desc = get_post_meta( $post->ID, '_drss_og_description', true );
        
        // Build HTML
        $content = apply_filters( 'the_content', $post->post_content );
        
        $html = '<!DOCTYPE html><html><head>';
        $html .= '<meta charset="' . get_bloginfo( 'charset' ) . '">';
        $html .= '<meta name="viewport" content="width=device-width, initial-scale=1">';
        $html .= '<title>' . esc_html( $title . ' | ' . get_bloginfo( 'name' ) ) . '</title>';
        
        if ( ! empty( $meta_desc ) ) {
            $html .= '<meta name="description" content="' . esc_attr( $meta_desc ) . '">';
        }
        
        $html .= '<link rel="canonical" href="' . esc_url( $canonical ) . '">';
        
        if ( ! empty( $og_title ) ) {
            $html .= '<meta property="og:title" content="' . esc_attr( $og_title ) . '">';
        }
        if ( ! empty( $og_desc ) ) {
            $html .= '<meta property="og:description" content="' . esc_attr( $og_desc ) . '">';
        }
        
        // Check favicon
        $favicon = get_site_icon_url();
        if ( $favicon ) {
            $html .= '<link rel="icon" href="' . esc_url( $favicon ) . '">';
        }
        
        $html .= '</head><body>';
        $html .= '<h1>' . esc_html( $post->post_title ) . '</h1>';
        $html .= $content;
        $html .= '</body></html>';
        
        return $html;
    }

    // ========== INDIVIDUAL CHECK METHODS ==========

    private function check_title( $html, $post_id ) {
        preg_match( '/<title[^>]*>(.*?)<\/title>/is', $html, $matches );
        $title = isset( $matches[1] ) ? trim( strip_tags( $matches[1] ) ) : '';
        $len = mb_strlen( $title );
        
        if ( empty( $title ) ) {
            return array( 'score' => 0, 'status' => 'error', 'message' => __( 'Missing title tag', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add a unique, descriptive title tag (50-60 characters)', 'digital-rise-seo-solutions' ), 'value' => '' );
        }
        if ( $len < 30 ) {
            return array( 'score' => 40, 'status' => 'warning', 'message' => sprintf( __( 'Title too short (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => __( 'Extend title to 50-60 characters for best results', 'digital-rise-seo-solutions' ), 'value' => $title );
        }
        if ( $len > 70 ) {
            return array( 'score' => 60, 'status' => 'warning', 'message' => sprintf( __( 'Title too long (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => __( 'Shorten title to under 60 characters to avoid truncation', 'digital-rise-seo-solutions' ), 'value' => $title );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( 'Title tag OK (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => '', 'value' => $title );
    }

    private function check_meta_description( $html, $post_id ) {
        preg_match( '/<meta\s+name=["\']description["\']\s+content=["\']([^"\']*?)["\']/is', $html, $matches );
        $desc = isset( $matches[1] ) ? trim( $matches[1] ) : '';
        $len = mb_strlen( $desc );
        
        if ( empty( $desc ) ) {
            return array( 'score' => 0, 'status' => 'error', 'message' => __( 'Missing meta description', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add a meta description (120-160 characters) with target keywords', 'digital-rise-seo-solutions' ), 'value' => '' );
        }
        if ( $len < 70 ) {
            return array( 'score' => 40, 'status' => 'warning', 'message' => sprintf( __( 'Meta description too short (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => __( 'Expand to 120-160 characters', 'digital-rise-seo-solutions' ), 'value' => $desc );
        }
        if ( $len > 170 ) {
            return array( 'score' => 60, 'status' => 'warning', 'message' => sprintf( __( 'Meta description too long (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => __( 'Shorten to under 160 characters', 'digital-rise-seo-solutions' ), 'value' => $desc );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( 'Meta description OK (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => '', 'value' => $desc );
    }

    private function check_h1( $html ) {
        preg_match_all( '/<h1[^>]*>(.*?)<\/h1>/is', $html, $matches );
        $count = count( $matches[1] );
        
        if ( $count === 0 ) {
            return array( 'score' => 0, 'status' => 'error', 'message' => __( 'Missing H1 tag', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add exactly one H1 tag per page with your main keyword', 'digital-rise-seo-solutions' ), 'value' => 0 );
        }
        if ( $count > 1 ) {
            return array( 'score' => 50, 'status' => 'warning', 'message' => sprintf( __( 'Multiple H1 tags found (%d)', 'digital-rise-seo-solutions' ), $count ), 'recommendation' => __( 'Use only one H1 tag per page', 'digital-rise-seo-solutions' ), 'value' => $count );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Single H1 tag present', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => 1 );
    }

    private function check_h2( $html ) {
        preg_match_all( '/<h2[^>]*>(.*?)<\/h2>/is', $html, $matches );
        $count = count( $matches[1] );
        
        if ( $count === 0 ) {
            return array( 'score' => 30, 'status' => 'warning', 'message' => __( 'No H2 tags found', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add H2 subheadings to structure your content', 'digital-rise-seo-solutions' ), 'value' => 0 );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( '%d H2 tags found', 'digital-rise-seo-solutions' ), $count ), 'recommendation' => '', 'value' => $count );
    }

    private function check_url_length( $url ) {
        $path = wp_parse_url( $url, PHP_URL_PATH );
        $len = mb_strlen( $path );
        
        if ( $len > 115 ) {
            return array( 'score' => 30, 'status' => 'warning', 'message' => sprintf( __( 'URL too long (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => __( 'Keep URLs under 75 characters for better SEO', 'digital-rise-seo-solutions' ), 'value' => $len );
        }
        if ( $len > 75 ) {
            return array( 'score' => 70, 'status' => 'warning', 'message' => sprintf( __( 'URL moderately long (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => __( 'Consider shortening the URL slug', 'digital-rise-seo-solutions' ), 'value' => $len );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( 'URL length OK (%d chars)', 'digital-rise-seo-solutions' ), $len ), 'recommendation' => '', 'value' => $len );
    }

    private function check_image_alts( $html ) {
        preg_match_all( '/<img[^>]*>/is', $html, $img_matches );
        $total_images = count( $img_matches[0] );
        
        if ( $total_images === 0 ) {
            return array( 'score' => 60, 'status' => 'warning', 'message' => __( 'No images found on page', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add relevant images with descriptive ALT text', 'digital-rise-seo-solutions' ), 'value' => '0/0' );
        }
        
        $missing_alt = 0;
        foreach ( $img_matches[0] as $img ) {
            if ( ! preg_match( '/alt=["\'][^"\']+["\']/i', $img ) ) {
                $missing_alt++;
            }
        }
        
        $with_alt = $total_images - $missing_alt;
        $ratio = $total_images > 0 ? ( $with_alt / $total_images ) * 100 : 0;
        
        if ( $missing_alt > 0 ) {
            $status = $ratio < 50 ? 'error' : 'warning';
            return array(
                'score'          => round( $ratio ),
                'status'         => $status,
                'message'        => sprintf( __( '%d of %d images missing ALT text', 'digital-rise-seo-solutions' ), $missing_alt, $total_images ),
                'recommendation' => __( 'Add descriptive ALT text to all images for accessibility and SEO', 'digital-rise-seo-solutions' ),
                'value'          => $with_alt . '/' . $total_images,
            );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( 'All %d images have ALT text', 'digital-rise-seo-solutions' ), $total_images ), 'recommendation' => '', 'value' => $total_images . '/' . $total_images );
    }

    private function check_internal_links( $html ) {
        $site_url = home_url();
        preg_match_all( '/<a\s[^>]*href=["\']([^"\']*)["\'][^>]*>/is', $html, $matches );
        
        $internal = 0;
        foreach ( $matches[1] as $href ) {
            if ( strpos( $href, $site_url ) === 0 || ( strpos( $href, '/' ) === 0 && strpos( $href, '//' ) !== 0 ) ) {
                $internal++;
            }
        }
        
        if ( $internal === 0 ) {
            return array( 'score' => 0, 'status' => 'error', 'message' => __( 'No internal links found', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add internal links to connect your content and help search engines crawl your site', 'digital-rise-seo-solutions' ), 'value' => 0 );
        }
        if ( $internal < 3 ) {
            return array( 'score' => 50, 'status' => 'warning', 'message' => sprintf( __( 'Only %d internal links', 'digital-rise-seo-solutions' ), $internal ), 'recommendation' => __( 'Add more internal links (3-5 minimum recommended)', 'digital-rise-seo-solutions' ), 'value' => $internal );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( '%d internal links found', 'digital-rise-seo-solutions' ), $internal ), 'recommendation' => '', 'value' => $internal );
    }

    private function check_external_links( $html ) {
        $site_url = home_url();
        preg_match_all( '/<a\s[^>]*href=["\']([^"\']*)["\'][^>]*>/is', $html, $matches );
        
        $external = 0;
        foreach ( $matches[1] as $href ) {
            if ( preg_match( '/^https?:\/\//i', $href ) && strpos( $href, $site_url ) !== 0 ) {
                $external++;
            }
        }
        
        if ( $external > 50 ) {
            return array( 'score' => 30, 'status' => 'warning', 'message' => sprintf( __( 'Too many external links (%d)', 'digital-rise-seo-solutions' ), $external ), 'recommendation' => __( 'Reduce external links and add nofollow to untrusted links', 'digital-rise-seo-solutions' ), 'value' => $external );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( '%d external links', 'digital-rise-seo-solutions' ), $external ), 'recommendation' => '', 'value' => $external );
    }

    private function check_canonical( $html ) {
        if ( preg_match( '/<link[^>]*rel=["\']canonical["\'][^>]*>/is', $html ) ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Canonical tag present', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => true );
        }
        return array( 'score' => 0, 'status' => 'error', 'message' => __( 'Missing canonical tag', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add a canonical tag to prevent duplicate content issues', 'digital-rise-seo-solutions' ), 'value' => false );
    }

    private function check_meta_robots( $html ) {
        if ( preg_match( '/<meta\s+name=["\']robots["\']\s+content=["\']([^"\']*?)["\']/is', $html, $matches ) ) {
            $content = strtolower( $matches[1] );
            if ( strpos( $content, 'noindex' ) !== false ) {
                return array( 'score' => 50, 'status' => 'warning', 'message' => __( 'Page set to noindex', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Remove noindex if you want this page indexed by search engines', 'digital-rise-seo-solutions' ), 'value' => $content );
            }
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Meta robots tag OK', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => $content );
        }
        return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'No restrictive robots meta (default: index,follow)', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => 'default' );
    }

    private function check_og_tags( $html ) {
        $has_title = preg_match( '/<meta\s+property=["\']og:title["\']/is', $html );
        $has_desc = preg_match( '/<meta\s+property=["\']og:description["\']/is', $html );
        $has_image = preg_match( '/<meta\s+property=["\']og:image["\']/is', $html );
        
        $found = (int) $has_title + (int) $has_desc + (int) $has_image;
        
        if ( $found === 0 ) {
            return array( 'score' => 0, 'status' => 'error', 'message' => __( 'No Open Graph tags found', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add og:title, og:description, and og:image tags for better social sharing', 'digital-rise-seo-solutions' ), 'value' => 0 );
        }
        if ( $found < 3 ) {
            return array( 'score' => 50, 'status' => 'warning', 'message' => sprintf( __( 'Incomplete OG tags (%d/3)', 'digital-rise-seo-solutions' ), $found ), 'recommendation' => __( 'Add missing og:title, og:description, or og:image tags', 'digital-rise-seo-solutions' ), 'value' => $found );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Open Graph tags present', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => 3 );
    }

    private function check_word_count( $html ) {
        // Strip tags and get text content from body
        $body = '';
        if ( preg_match( '/<body[^>]*>(.*?)<\/body>/is', $html, $m ) ) {
            $body = $m[1];
        } else {
            $body = $html;
        }
        
        $text = wp_strip_all_tags( $body );
        $text = preg_replace( '/\s+/', ' ', $text );
        $words = str_word_count( $text );
        
        if ( $words < 100 ) {
            return array( 'score' => 10, 'status' => 'error', 'message' => sprintf( __( 'Very thin content (%d words)', 'digital-rise-seo-solutions' ), $words ), 'recommendation' => __( 'Add at least 300 words of quality content for better rankings', 'digital-rise-seo-solutions' ), 'value' => $words );
        }
        if ( $words < 300 ) {
            return array( 'score' => 40, 'status' => 'warning', 'message' => sprintf( __( 'Thin content (%d words)', 'digital-rise-seo-solutions' ), $words ), 'recommendation' => __( 'Expand content to 600+ words for better search visibility', 'digital-rise-seo-solutions' ), 'value' => $words );
        }
        if ( $words < 600 ) {
            return array( 'score' => 70, 'status' => 'pass', 'message' => sprintf( __( 'Moderate content (%d words)', 'digital-rise-seo-solutions' ), $words ), 'recommendation' => __( 'Consider expanding to 1000+ words for competitive keywords', 'digital-rise-seo-solutions' ), 'value' => $words );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( 'Good content length (%d words)', 'digital-rise-seo-solutions' ), $words ), 'recommendation' => '', 'value' => $words );
    }

    private function check_https( $url ) {
        if ( strpos( $url, 'https://' ) === 0 ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'HTTPS enabled', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => true );
        }
        return array( 'score' => 0, 'status' => 'error', 'message' => __( 'Not using HTTPS', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Install an SSL certificate and redirect HTTP to HTTPS', 'digital-rise-seo-solutions' ), 'value' => false );
    }

    private function check_schema( $html ) {
        $has_jsonld = preg_match( '/<script\s+type=["\']application\/ld\+json["\']/is', $html );
        $has_microdata = preg_match( '/itemscope|itemtype/is', $html );
        
        if ( $has_jsonld || $has_microdata ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Schema markup detected', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => true );
        }
        return array( 'score' => 0, 'status' => 'warning', 'message' => __( 'No schema markup found', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add JSON-LD structured data for rich search results', 'digital-rise-seo-solutions' ), 'value' => false );
    }

    private function check_viewport( $html ) {
        if ( preg_match( '/<meta\s+name=["\']viewport["\']/is', $html ) ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Viewport meta tag present', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => true );
        }
        return array( 'score' => 0, 'status' => 'error', 'message' => __( 'Missing viewport meta tag', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add viewport meta for mobile-friendliness', 'digital-rise-seo-solutions' ), 'value' => false );
    }

    private function check_charset( $html ) {
        if ( preg_match( '/<meta\s+charset/is', $html ) || preg_match( '/content-type.*charset/is', $html ) ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Character encoding declared', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => true );
        }
        return array( 'score' => 0, 'status' => 'error', 'message' => __( 'Missing charset declaration', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add <meta charset="UTF-8"> to your page', 'digital-rise-seo-solutions' ), 'value' => false );
    }

    private function check_favicon( $html ) {
        $has_favicon = preg_match( '/<link[^>]*rel=["\'](?:shortcut )?icon["\']/is', $html );
        $has_site_icon = get_site_icon_url();
        
        if ( $has_favicon || $has_site_icon ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Favicon detected', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => true );
        }
        return array( 'score' => 0, 'status' => 'warning', 'message' => __( 'No favicon found', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Add a favicon (16x16px) for better brand recognition in tabs and bookmarks', 'digital-rise-seo-solutions' ), 'value' => false );
    }

    private function check_content_quality( $html, $post_id ) {
        $score = 100;
        $issues = array();
        
        if ( $post_id > 0 ) {
            $post = get_post( $post_id );
            if ( $post ) {
                // Title-H1 mismatch check
                $title_tag = '';
                preg_match( '/<title[^>]*>(.*?)<\/title>/is', $html, $tm );
                if ( isset( $tm[1] ) ) {
                    $title_tag = strtolower( trim( strip_tags( $tm[1] ) ) );
                }
                
                $h1 = '';
                preg_match( '/<h1[^>]*>(.*?)<\/h1>/is', $html, $hm );
                if ( isset( $hm[1] ) ) {
                    $h1 = strtolower( trim( strip_tags( $hm[1] ) ) );
                }
                
                if ( ! empty( $title_tag ) && ! empty( $h1 ) ) {
                    similar_text( $title_tag, $h1, $similarity );
                    if ( $similarity < 30 ) {
                        $score -= 30;
                        $issues[] = __( 'Title tag and H1 are very different', 'digital-rise-seo-solutions' );
                    }
                }
                
                // Keyword stuffing detection (basic)
                $content = strtolower( wp_strip_all_tags( $post->post_content ) );
                $words = str_word_count( $content, 1 );
                $word_freq = array_count_values( $words );
                $total_words = count( $words );
                
                if ( $total_words > 50 ) {
                    foreach ( $word_freq as $word => $freq ) {
                        if ( mb_strlen( $word ) > 3 ) {
                            $density = ( $freq / $total_words ) * 100;
                            if ( $density > 5 ) {
                                $score -= 20;
                                $issues[] = sprintf( __( 'Possible keyword stuffing: "%s" appears %.1f%% of the time', 'digital-rise-seo-solutions' ), $word, $density );
                                break;
                            }
                        }
                    }
                }
            }
        }
        
        $score = max( 0, $score );
        $msg = empty( $issues ) ? __( 'Content quality checks passed', 'digital-rise-seo-solutions' ) : implode( '; ', $issues );
        $status = $score >= 80 ? 'pass' : ( $score >= 50 ? 'warning' : 'error' );
        $rec = empty( $issues ) ? '' : __( 'Review content for quality issues', 'digital-rise-seo-solutions' );
        
        return array( 'score' => $score, 'status' => $status, 'message' => $msg, 'recommendation' => $rec, 'value' => $score );
    }

    private function check_broken_internal_links( $html ) {
        $site_url = home_url();
        preg_match_all( '/<a\s[^>]*href=["\']([^"\']*)["\'][^>]*>/is', $html, $matches );
        
        $broken = 0;
        $checked = 0;
        
        foreach ( $matches[1] as $href ) {
            if ( strpos( $href, $site_url ) === 0 || ( strpos( $href, '/' ) === 0 && strpos( $href, '//' ) !== 0 ) ) {
                $checked++;
                if ( $checked > 20 ) break; // Limit checks for performance
                
                $full_url = strpos( $href, '/' ) === 0 ? $site_url . $href : $href;
                $post_id = url_to_postid( $full_url );
                
                if ( $post_id === 0 && $full_url !== $site_url . '/' && $full_url !== $site_url ) {
                    // Check if it's a valid page/archive
                    if ( ! preg_match( '/\.(css|js|jpg|jpeg|png|gif|webp|svg|pdf|zip|mp3|mp4)$/i', $full_url ) ) {
                        // Skip anchor links and common non-post URLs
                        if ( strpos( $href, '#' ) !== 0 && strpos( $href, 'mailto:' ) !== 0 && strpos( $href, 'tel:' ) !== 0 ) {
                            $broken++;
                        }
                    }
                }
            }
        }
        
        if ( $broken > 0 ) {
            return array( 'score' => max( 0, 100 - ( $broken * 20 ) ), 'status' => 'error', 'message' => sprintf( __( '%d potentially broken internal links', 'digital-rise-seo-solutions' ), $broken ), 'recommendation' => __( 'Fix or remove broken internal links', 'digital-rise-seo-solutions' ), 'value' => $broken );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'No broken internal links detected', 'digital-rise-seo-solutions' ), 'recommendation' => '', 'value' => 0 );
    }

    private function check_heading_hierarchy( $html ) {
        preg_match_all( '/<h([1-6])[^>]*>/is', $html, $matches );
        $headings = array_map( 'intval', $matches[1] );
        
        if ( empty( $headings ) ) {
            return array( 'score' => 30, 'status' => 'warning', 'message' => __( 'No heading tags found', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Use heading tags (H1-H6) to structure your content', 'digital-rise-seo-solutions' ), 'value' => 0 );
        }
        
        // Check if hierarchy is proper (no skipping levels)
        $skip_found = false;
        for ( $i = 1; $i < count( $headings ); $i++ ) {
            if ( $headings[ $i ] > $headings[ $i - 1 ] + 1 ) {
                $skip_found = true;
                break;
            }
        }
        
        if ( $skip_found ) {
            return array( 'score' => 60, 'status' => 'warning', 'message' => __( 'Heading hierarchy has gaps (e.g., H1 → H3)', 'digital-rise-seo-solutions' ), 'recommendation' => __( 'Maintain proper heading hierarchy (H1 → H2 → H3)', 'digital-rise-seo-solutions' ), 'value' => count( $headings ) );
        }
        
        return array( 'score' => 100, 'status' => 'pass', 'message' => sprintf( __( 'Proper heading hierarchy (%d headings)', 'digital-rise-seo-solutions' ), count( $headings ) ), 'recommendation' => '', 'value' => count( $headings ) );
    }

    // ========== SITE-WIDE CHECKS ==========

    /**
     * Run site-wide technical SEO checks
     */
    public function run_site_checks() {
        $checks = array();
        
        // Sitemap check
        $checks['sitemap'] = $this->check_sitemap_exists();
        
        // Robots.txt check
        $checks['robots_txt'] = $this->check_robots_txt();
        
        // HTTPS check
        $checks['https'] = $this->check_site_https();
        
        // WWW consistency
        $checks['www_consistency'] = $this->check_www_consistency();
        
        // Duplicate titles (site-wide)
        $checks['duplicate_titles'] = $this->check_duplicate_titles();
        
        // Duplicate descriptions
        $checks['duplicate_descriptions'] = $this->check_duplicate_descriptions();
        
        return $checks;
    }

    private function check_sitemap_exists() {
        $sitemap_url = home_url( '/sitemap.xml' );
        $response = wp_remote_head( $sitemap_url, array( 'timeout' => 5, 'sslverify' => false ) );
        
        if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Sitemap.xml found', 'digital-rise-seo-solutions' ) );
        }
        
        // Check wp-sitemap.xml (WordPress default)
        $wp_sitemap = home_url( '/wp-sitemap.xml' );
        $response2 = wp_remote_head( $wp_sitemap, array( 'timeout' => 5, 'sslverify' => false ) );
        
        if ( ! is_wp_error( $response2 ) && wp_remote_retrieve_response_code( $response2 ) === 200 ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'WordPress sitemap found', 'digital-rise-seo-solutions' ) );
        }
        
        return array( 'score' => 0, 'status' => 'error', 'message' => __( 'No sitemap found - generate and submit to search engines', 'digital-rise-seo-solutions' ) );
    }

    private function check_robots_txt() {
        $robots_url = home_url( '/robots.txt' );
        $response = wp_remote_get( $robots_url, array( 'timeout' => 5, 'sslverify' => false ) );
        
        if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
            $body = wp_remote_retrieve_body( $response );
            if ( strpos( $body, 'Disallow: /' ) !== false && strpos( $body, 'Disallow: / ' ) === false ) {
                // Might be blocking everything
                if ( preg_match( '/Disallow:\s*\/\s*$/m', $body ) ) {
                    return array( 'score' => 30, 'status' => 'warning', 'message' => __( 'Robots.txt may be blocking entire site', 'digital-rise-seo-solutions' ) );
                }
            }
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Robots.txt found and configured', 'digital-rise-seo-solutions' ) );
        }
        
        return array( 'score' => 50, 'status' => 'warning', 'message' => __( 'No robots.txt found', 'digital-rise-seo-solutions' ) );
    }

    private function check_site_https() {
        if ( is_ssl() || strpos( home_url(), 'https://' ) === 0 ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'Site uses HTTPS', 'digital-rise-seo-solutions' ) );
        }
        return array( 'score' => 0, 'status' => 'error', 'message' => __( 'Site not using HTTPS - install SSL certificate', 'digital-rise-seo-solutions' ) );
    }

    private function check_www_consistency() {
        $site_url = home_url();
        $home_url = get_option( 'home' );
        $wp_url = get_option( 'siteurl' );
        
        $site_www = strpos( $site_url, '://www.' ) !== false;
        $home_www = strpos( $home_url, '://www.' ) !== false;
        $wp_www = strpos( $wp_url, '://www.' ) !== false;
        
        if ( ( $site_www === $home_www ) && ( $home_www === $wp_www ) ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'WWW/non-WWW URLs are consistent', 'digital-rise-seo-solutions' ) );
        }
        
        return array( 'score' => 50, 'status' => 'warning', 'message' => __( 'WWW/non-WWW inconsistency detected in site URLs', 'digital-rise-seo-solutions' ) );
    }

    private function check_duplicate_titles() {
        global $wpdb;
        
        $results = $wpdb->get_results(
            "SELECT pm.meta_value, COUNT(*) as cnt 
             FROM {$wpdb->postmeta} pm 
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
             WHERE pm.meta_key = '_drss_seo_title' 
             AND pm.meta_value != '' 
             AND p.post_status = 'publish' 
             GROUP BY pm.meta_value 
             HAVING cnt > 1 
             LIMIT 10"
        );
        
        if ( empty( $results ) ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'No duplicate SEO titles detected', 'digital-rise-seo-solutions' ) );
        }
        
        return array( 'score' => 50, 'status' => 'warning', 'message' => sprintf( __( '%d duplicate SEO titles found', 'digital-rise-seo-solutions' ), count( $results ) ) );
    }

    private function check_duplicate_descriptions() {
        global $wpdb;
        
        $results = $wpdb->get_results(
            "SELECT pm.meta_value, COUNT(*) as cnt 
             FROM {$wpdb->postmeta} pm 
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
             WHERE pm.meta_key = '_drss_seo_description' 
             AND pm.meta_value != '' 
             AND p.post_status = 'publish' 
             GROUP BY pm.meta_value 
             HAVING cnt > 1 
             LIMIT 10"
        );
        
        if ( empty( $results ) ) {
            return array( 'score' => 100, 'status' => 'pass', 'message' => __( 'No duplicate meta descriptions detected', 'digital-rise-seo-solutions' ) );
        }
        
        return array( 'score' => 50, 'status' => 'warning', 'message' => sprintf( __( '%d duplicate meta descriptions found', 'digital-rise-seo-solutions' ), count( $results ) ) );
    }

    /**
     * Calculate global website SEO score from all URL results
     */
    public function calculate_global_score( $url_results, $site_checks ) {
        if ( empty( $url_results ) ) {
            return 0;
        }
        
        // 70% from page scores, 30% from site-wide checks
        $page_total = 0;
        foreach ( $url_results as $r ) {
            $page_total += $r['score'];
        }
        $page_avg = $page_total / count( $url_results );
        
        $site_total = 0;
        $site_count = 0;
        foreach ( $site_checks as $check ) {
            $site_total += $check['score'];
            $site_count++;
        }
        $site_avg = $site_count > 0 ? $site_total / $site_count : 0;
        
        return round( ( $page_avg * 0.7 ) + ( $site_avg * 0.3 ) );
    }
}
