<?php
/**
 * Remote Connection for Owner Control
 *
 * @package Digital_Rise_SEO_Solutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRSS_Remote {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Schedule check-in
     */
    public function schedule_checkin() {
        if ( ! wp_next_scheduled( 'drss_remote_checkin' ) ) {
            wp_schedule_event( time(), 'daily', 'drss_remote_checkin' );
        }
    }

    /**
     * Perform check-in with owner server
     */
    public function do_checkin() {
        if ( ! $this->plugin->get_option( 'remote_connection', true ) ) {
            return;
        }

        $site_data = $this->get_site_data();
        
        $response = wp_remote_post( DRSS_OWNER_API . 'checkin', array(
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-DRIOP-Site-Key' => $this->plugin->get_option( 'remote_api_key' ),
            ),
            'body' => wp_json_encode( $site_data ),
        ) );

        if ( is_wp_error( $response ) ) {
            return;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( $data && isset( $data['success'] ) && $data['success'] ) {
            // Mark first checkin done
            if ( ! $this->plugin->get_option( 'first_checkin_done', false ) ) {
                $this->plugin->update_option( 'first_checkin_done', true );
            }
            
            // Update plan if provided
            if ( isset( $data['plan'] ) && ! empty( $this->plugin->get_option( 'license_key' ) ) ) {
                $this->plugin->update_option( 'current_plan', $data['plan'] );
            }

            // Process notices
            if ( isset( $data['notices'] ) && is_array( $data['notices'] ) ) {
                $this->process_notices( $data['notices'] );
            }

            // Store last check-in
            $this->plugin->update_option( 'last_checkin', current_time( 'mysql' ) );
        }
    }

    /**
     * Get site data for check-in
     */
    private function get_site_data() {
        global $wp_version;

        return array(
            'site_url' => home_url(),
            'site_name' => get_bloginfo( 'name' ),
            'admin_email' => get_option( 'admin_email' ),
            'user_email' => $this->plugin->get_option( 'user_email', '' ),
            'user_whatsapp' => $this->plugin->get_option( 'user_whatsapp', '' ),
            'wp_version' => $wp_version,
            'php_version' => phpversion(),
            'plugin_version' => DRSS_VERSION,
            'current_plan' => $this->plugin->current_plan,
            'license_key' => $this->plugin->get_option( 'license_key' ),
            'license_status' => $this->plugin->get_option( 'license_status' ),
            'installation_id' => $this->plugin->get_option( 'installation_id' ),
            'installation_date' => $this->plugin->get_option( 'installation_date' ),
            'woocommerce_active' => class_exists( 'WooCommerce' ),
            'dokan_active' => class_exists( 'WeDevs_Dokan' ),
            'multisite' => is_multisite(),
            'locale' => get_locale(),
            'total_images_optimized' => $this->get_total_optimized(),
            'is_first_checkin' => ! $this->plugin->get_option( 'first_checkin_done', false ),
        );
    }

    /**
     * Get total optimized images
     */
    private function get_total_optimized() {
        global $wpdb;
        $table = $wpdb->prefix . 'drss_optimization_log';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            return 0;
        }
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table count
        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}drss_optimization_log" );
    }

    /**
     * Process notices from owner - FIXED field names
     */
    private function process_notices( $notices ) {
        global $wpdb;
        $table = $wpdb->prefix . 'drss_remote_notices';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            return;
        }

        foreach ( $notices as $notice ) {
            // Accept both field naming conventions
            $notice_id = isset( $notice['notice_id'] ) ? $notice['notice_id'] : ( isset( $notice['id'] ) ? $notice['id'] : '' );
            $notice_type = isset( $notice['notice_type'] ) ? $notice['notice_type'] : ( isset( $notice['type'] ) ? $notice['type'] : 'info' );
            $notice_title = isset( $notice['notice_title'] ) ? $notice['notice_title'] : ( isset( $notice['title'] ) ? $notice['title'] : '' );
            $notice_content = isset( $notice['notice_content'] ) ? $notice['notice_content'] : ( isset( $notice['content'] ) ? $notice['content'] : '' );
            $notice_link = isset( $notice['notice_link'] ) ? $notice['notice_link'] : ( isset( $notice['link'] ) ? $notice['link'] : '' );
            
            if ( empty( $notice_id ) || empty( $notice_content ) ) {
                continue;
            }

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
            $existing = $wpdb->get_var( $wpdb->prepare(
                "SELECT id FROM {$wpdb->prefix}drss_remote_notices WHERE notice_id = %s",
                $notice_id
            ) );

            $data = array(
                'notice_type' => sanitize_text_field( $notice_type ),
                'notice_title' => sanitize_text_field( $notice_title ),
                'notice_content' => wp_kses_post( $notice_content ),
                'notice_link' => esc_url_raw( $notice_link ),
            );

            if ( $existing ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update
                $wpdb->update( $table, $data, array( 'notice_id' => $notice_id ) );
            } else {
                $data['notice_id'] = sanitize_text_field( $notice_id );
                $data['is_dismissed'] = 0;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert
                $wpdb->insert( $table, $data );
            }
        }
    }

    /**
     * Get active notices - from local DB or fetch from server
     */
    public function get_active_notices() {
        global $wpdb;
        $table = $wpdb->prefix . 'drss_remote_notices';
        
        // Check if table exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            // Try to fetch from server directly
            return $this->fetch_notices_from_server();
        }
        
        // Get from local database
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
        $results = $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}drss_remote_notices WHERE is_dismissed = 0 ORDER BY created_at DESC", ARRAY_A );
        
        // If no local notices and we haven't checked recently, fetch from server
        if ( empty( $results ) ) {
            $last_fetch = get_transient( 'drss_notices_last_fetch' );
            if ( ! $last_fetch ) {
                $server_notices = $this->fetch_notices_from_server();
                if ( ! empty( $server_notices ) ) {
                    // Store in local DB
                    foreach ( $server_notices as $notice ) {
                        $this->store_notice_locally( $notice );
                    }
                    set_transient( 'drss_notices_last_fetch', time(), HOUR_IN_SECONDS );
                    return $server_notices;
                }
            }
        }
        
        return is_array( $results ) ? $results : array();
    }
    
    /**
     * Fetch notices directly from owner server
     */
    private function fetch_notices_from_server() {
        if ( ! $this->plugin->get_option( 'remote_connection', true ) ) {
            return array();
        }
        
        $response = wp_remote_post( DRSS_OWNER_API . 'checkin', array(
            'timeout' => 10,
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body' => wp_json_encode( array(
                'site_url' => home_url(),
                'installation_id' => $this->plugin->get_option( 'installation_id' ),
                'plugin_version' => DRSS_VERSION,
                'fetch_notices_only' => true,
            )),
        ));
        
        if ( is_wp_error( $response ) ) {
            return array();
        }
        
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        if ( $data && isset( $data['notices'] ) && is_array( $data['notices'] ) ) {
            return $data['notices'];
        }
        
        return array();
    }
    
    /**
     * Store notice in local database
     */
    private function store_notice_locally( $notice ) {
        global $wpdb;
        $table = $wpdb->prefix . 'drss_remote_notices';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            return;
        }
        
        $notice_id = isset( $notice['notice_id'] ) ? $notice['notice_id'] : '';
        if ( empty( $notice_id ) ) {
            return;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
        $existing = $wpdb->get_var( $wpdb->prepare(
            "SELECT id FROM {$wpdb->prefix}drss_remote_notices WHERE notice_id = %s",
            $notice_id
        ));
        
        if ( ! $existing ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert
            $wpdb->insert( $table, array(
                'notice_id' => sanitize_text_field( $notice_id ),
                'notice_type' => sanitize_text_field( isset( $notice['notice_type'] ) ? $notice['notice_type'] : 'info' ),
                'notice_title' => sanitize_text_field( isset( $notice['notice_title'] ) ? $notice['notice_title'] : '' ),
                'notice_content' => wp_kses_post( isset( $notice['notice_content'] ) ? $notice['notice_content'] : '' ),
                'notice_link' => esc_url_raw( isset( $notice['notice_link'] ) ? $notice['notice_link'] : '' ),
                'is_dismissed' => 0,
            ));
        }
    }

    /**
     * Dismiss notice
     */
    public function dismiss_notice( $notice_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'drss_remote_notices';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            return false;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update
        return $wpdb->update( 
            $table, 
            array( 'is_dismissed' => 1 ), 
            array( 'notice_id' => sanitize_text_field( $notice_id ) )
        );
    }

    /**
     * Check if remote connection is enabled
     */
    public function is_enabled() {
        return (bool) $this->plugin->get_option( 'remote_connection', true );
    }
    
    /**
     * Force fetch notices now
     */
    public function fetch_notices_now() {
        $this->do_checkin();
    }
}
