<?php
/**
 * Plans Management
 *
 * @package Digital_Rise_SEO_Solutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRSS_Plans {

    private $plugin;
    private $features = array();

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        $this->define_features();
    }

    private function define_features() {
        $this->features = array(
            'free' => array(
                'image_compression' => true,
                'auto_resize' => true,
                'webp_conversion' => true,
                'basic_rename' => true,
                'auto_alt' => true,
                'wc_basic' => true,
                'bulk_optimize' => true,
                'css_js_minify' => true,
                // Restricted
                'custom_keywords' => false,
                'per_user_limits' => false,
                'per_product_limits' => false,
                'per_vendor_limits' => false,
                'vendor_dashboard' => false,
                'vendor_stats' => false,
                'advanced_seo' => false,
                'priority_support' => false,
                'unlimited_sites' => false,
                'white_label' => false,
                'disable_attribution' => false,
            ),
            'pro' => array(
                'image_compression' => true,
                'auto_resize' => true,
                'webp_conversion' => true,
                'basic_rename' => true,
                'auto_alt' => true,
                'wc_basic' => true,
                'bulk_optimize' => true,
                'css_js_minify' => true,
                'custom_keywords' => true,
                'per_user_limits' => true,
                'per_product_limits' => true,
                'advanced_seo' => true,
                'content_type_patterns' => true, // PHASE 3 FIX #6
                'social_preview' => true, // Social Media Preview
                'priority_support' => true,
                'disable_attribution' => true,
                // Restricted
                'per_vendor_limits' => false,
                'vendor_dashboard' => false,
                'vendor_stats' => false,
                'unlimited_sites' => false,
                'white_label' => false,
            ),
            'agency' => array(
                'image_compression' => true,
                'auto_resize' => true,
                'webp_conversion' => true,
                'basic_rename' => true,
                'auto_alt' => true,
                'wc_basic' => true,
                'bulk_optimize' => true,
                'css_js_minify' => true,
                'custom_keywords' => true,
                'per_user_limits' => true,
                'per_product_limits' => true,
                'per_vendor_limits' => true,
                'vendor_dashboard' => true,
                'vendor_stats' => true,
                'advanced_seo' => true,
                'content_type_patterns' => true, // PHASE 3 FIX #6
                'social_preview' => true, // Social Media Preview
                'priority_support' => true,
                'unlimited_sites' => true,
                'white_label' => true,
                'disable_attribution' => true,
            ),
        );
    }

    public function has_feature( $feature ) {
        $plan = $this->plugin->current_plan;
        if ( ! isset( $this->features[ $plan ] ) ) {
            $plan = 'free';
        }
        return isset( $this->features[ $plan ][ $feature ] ) && $this->features[ $plan ][ $feature ];
    }

    /**
     * Fetch prices from owner panel (cached for 5 minutes)
     */
    private function get_remote_prices() {
        // Check if force refresh requested
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Simple cache refresh, requires admin capability
        if ( isset( $_GET['refresh_prices'] ) && current_user_can( 'manage_options' ) ) {
            delete_transient( 'drss_plan_prices' );
        }

        $cached = get_transient( 'drss_plan_prices' );
        if ( $cached !== false ) {
            return $cached;
        }

        $response = wp_remote_get( DRSS_OWNER_API . 'prices', array(
            'timeout' => 10,
            'sslverify' => false,
        ));

        if ( is_wp_error( $response ) ) {
            return null;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( isset( $data['success'] ) && $data['success'] && isset( $data['prices'] ) ) {
            // Cache for 5 minutes only (300 seconds)
            set_transient( 'drss_plan_prices', $data['prices'], 5 * MINUTE_IN_SECONDS );
            return $data['prices'];
        }

        return null;
    }

    /**
     * Clear price cache (call this when needed)
     */
    public function clear_price_cache() {
        delete_transient( 'drss_plan_prices' );
    }

    /**
     * Format price for display
     */
    private function format_price_display( $plan_data ) {
        if ( ! $plan_data || ! isset( $plan_data['price'] ) ) {
            return array( 'price' => '$0', 'note' => '' );
        }

        $price = $plan_data['price'];
        $currency = $plan_data['currency'] ?? 'USD';
        $period = $plan_data['period'] ?? '';

        if ( $price === '0' || strtolower( $price ) === 'free' ) {
            return array( 'price' => __( 'Free', 'digital-rise-seo-solutions' ), 'note' => __( 'Forever Free', 'digital-rise-seo-solutions' ) );
        }

        $symbols = array( 'USD' => '$', 'EUR' => '€', 'TND' => 'DT ', 'GBP' => '£' );
        $symbol = $symbols[ $currency ] ?? $currency . ' ';

        $formatted_price = $symbol . $price;
        
        $note = '';
        if ( $period === 'month' ) {
            $note = __( 'per month', 'digital-rise-seo-solutions' );
        } elseif ( $period === 'year' ) {
            $note = __( 'per year', 'digital-rise-seo-solutions' );
        } else {
            $note = __( 'Lifetime', 'digital-rise-seo-solutions' );
        }

        return array( 'price' => $formatted_price, 'note' => $note );
    }

    /**
     * Get purchase URL from server
     */
    public function get_purchase_url() {
        $prices = $this->get_remote_prices();
        if ( $prices && isset( $prices['purchase_url'] ) && ! empty( $prices['purchase_url'] ) ) {
            return $prices['purchase_url'];
        }
        return DRSS_OWNER_URL . 'pricing';
    }

    public function get_plans() {
        // Fetch remote prices
        $remote_prices = $this->get_remote_prices();
        
        // Format prices (use remote if available, otherwise defaults)
        $free_price = $this->format_price_display( $remote_prices['free'] ?? null );
        $pro_price = $this->format_price_display( $remote_prices['pro'] ?? array( 'price' => '29', 'currency' => 'USD', 'period' => 'year' ) );
        $agency_price = $this->format_price_display( $remote_prices['developer'] ?? array( 'price' => '99', 'currency' => 'USD', 'period' => 'year' ) );

        return array(
            'free' => array(
                'name' => __( 'Free', 'digital-rise-seo-solutions' ),
                'price' => $free_price['price'],
                'price_note' => $free_price['note'],
                'description' => __( 'Perfect for bloggers and small websites', 'digital-rise-seo-solutions' ),
                'features' => array(
                    __( 'WebP conversion', 'digital-rise-seo-solutions' ),
                    __( 'Auto SEO rename', 'digital-rise-seo-solutions' ),
                    __( 'Auto ALT text & title', 'digital-rise-seo-solutions' ),
                    __( 'Bulk optimization', 'digital-rise-seo-solutions' ),
                    __( 'CSS & JS minification', 'digital-rise-seo-solutions' ),
                    __( 'WooCommerce integration', 'digital-rise-seo-solutions' ),
                    __( 'Max 6 words SEO', 'digital-rise-seo-solutions' ),
                ),
                'limitations' => array(
                    __( 'Attribution link required', 'digital-rise-seo-solutions' ),
                    __( 'No custom keywords', 'digital-rise-seo-solutions' ),
                    __( 'No vendor limits (Dokan)', 'digital-rise-seo-solutions' ),
                ),
            ),
            'pro' => array(
                'name' => __( 'Pro', 'digital-rise-seo-solutions' ),
                'price' => $pro_price['price'],
                'price_note' => $pro_price['note'],
                'description' => __( 'For businesses and online stores', 'digital-rise-seo-solutions' ),
                'features' => array(
                    __( 'Everything in Free', 'digital-rise-seo-solutions' ),
                    __( 'Custom keywords system', 'digital-rise-seo-solutions' ),
                    __( 'Per-user image limits', 'digital-rise-seo-solutions' ),
                    __( 'Per-product image limits', 'digital-rise-seo-solutions' ),
                    __( 'Page/Post SEO titles & meta', 'digital-rise-seo-solutions' ),
                    __( 'Priority email support', 'digital-rise-seo-solutions' ),
                    __( 'Remove attribution link', 'digital-rise-seo-solutions' ),
                ),
            ),
            'agency' => array(
                'name' => __( 'Agency', 'digital-rise-seo-solutions' ),
                'price' => $agency_price['price'],
                'price_note' => $agency_price['note'],
                'description' => __( 'For marketplaces and agencies (Dokan)', 'digital-rise-seo-solutions' ),
                'features' => array(
                    __( 'Everything in Pro', 'digital-rise-seo-solutions' ),
                    __( 'Unlimited sites license', 'digital-rise-seo-solutions' ),
                    __( 'Per-vendor image limits', 'digital-rise-seo-solutions' ),
                    __( 'Vendor dashboard widget', 'digital-rise-seo-solutions' ),
                    __( 'Vendor statistics', 'digital-rise-seo-solutions' ),
                    __( 'Vendor plan management', 'digital-rise-seo-solutions' ),
                    __( 'White-label option', 'digital-rise-seo-solutions' ),
                    __( 'Priority 24/7 support', 'digital-rise-seo-solutions' ),
                ),
            ),
        );
    }

    public function get_current_plan_name() {
        $plans = $this->get_plans();
        $current = $this->plugin->current_plan;
        return isset( $plans[ $current ]['name'] ) ? $plans[ $current ]['name'] : __( 'Free', 'digital-rise-seo-solutions' );
    }
}
