<?php
/**
 * Plugin Name: Digital Rise SEO Solutions
 * Plugin URI: https://digital-rise-solutions.com/digital-rise-seo-solutions/
 * Description: Professional SEO audit, image optimization, and SEO automation for WordPress, WooCommerce, and Dokan. Full on-site SEO analyzer with scoring system.
 * Version: 2.0.0
 * Author: Digital Rise Solutions
 * Author URI: https://digital-rise-solutions.com/
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: digital-rise-seo-solutions
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 9.5
 *
 * @package Digital_Rise_SEO_Solutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'DRSS_VERSION', '2.0.0' );
define( 'DRSS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'DRSS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'DRSS_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
define( 'DRSS_PLUGIN_FILE', __FILE__ );
define( 'DRSS_TEXT_DOMAIN', 'digital-rise-seo-solutions' );
define( 'DRSS_OWNER_URL', 'https://digital-rise-solutions.com/' );
define( 'DRSS_OWNER_API', 'https://digital-rise-solutions.com/wp-json/drss-license/v1/' );
define( 'DRSS_SUPPORT_EMAIL', 'seo.oussama@gmail.com' );
define( 'DRSS_LOGO_URL', 'https://digital-rise-solutions.com/wp-content/uploads/2023/06/digital-rise-solutions-digital-marketing-tunisia.webp' );
define( 'DRSS_MAX_FILE_SIZE', 50 );

/**
 * WooCommerce HPOS Compatibility
 */
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', __FILE__, true );
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'product_block_editor', __FILE__, true );
    }
});

/**
 * Main plugin class
 */
final class Digital_Rise_SEO_Solutions {

    private static $instance = null;
    public $options = array();
    public $current_plan = 'free';

    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->load_options();
        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    private function load_options() {
        $defaults = array(
            'current_plan'              => 'free',
            'license_key'               => '',
            'license_status'            => 'inactive',
            'license_expires'           => '',
            'max_width'                 => 1920,
            'max_height'                => 1080,
            'target_file_size'          => 100,
            'compression_quality'       => 82,
            'auto_compress'             => true,
            'confirm_before_optimize'   => false,
            'convert_to_webp'           => true,
            'max_images_per_user'       => 0,
            'max_images_per_product'    => 10,
            'max_images_per_vendor'     => 50,
            'css_minify_enabled'        => false,
            'js_minify_enabled'         => false,
            'css_combine_enabled'       => false,
            'js_combine_enabled'        => false,
            'minify_exclude'            => '',
            'keywords'                  => '',
            'seo_pattern'               => '{site_name} {content_title} {keyword}',
            'auto_rename'               => true,
            'auto_alt'                  => true,
            'auto_description'          => true,
            'page_seo_enabled'          => true,
            'default_title_pattern'     => '{title} | {site_name}',
            'default_meta_pattern'      => '{excerpt}',
            'remote_connection'         => true,
            'remote_api_key'            => '',
            'user_email'                => '',
            'user_whatsapp'             => '',
            'show_attribution'          => true,
            'wc_integration'            => true,
            'dokan_integration'         => true,
            'installation_id'           => '',
            'installation_date'         => '',
        );

        $saved_options = get_option( 'drss_options', array() );
        
        // Migration from old plugin
        if ( empty( $saved_options ) ) {
            $old_options = get_option( 'driop_options', array() );
            if ( ! empty( $old_options ) ) {
                $saved_options = $old_options;
                update_option( 'drss_options', $saved_options );
                delete_option( 'driop_options' );
            }
        }
        
        $this->options = wp_parse_args( $saved_options, $defaults );
        $this->current_plan = $this->options['current_plan'];
        
        if ( empty( $this->options['installation_id'] ) ) {
            $this->options['installation_id'] = wp_generate_uuid4();
            $this->options['installation_date'] = current_time( 'mysql' );
            update_option( 'drss_options', $this->options );
        }
    }

    private function load_dependencies() {
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-activator.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-deactivator.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-image-optimizer.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-seo.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-page-seo.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-keywords.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-license.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-remote.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-minifier.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-woocommerce.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-dokan.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-plans.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-bulk-optimizer.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-sitemap.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-robots.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-seo-audit.php';
        require_once DRSS_PLUGIN_DIR . 'includes/class-drss-seo-checklist.php';

        if ( is_admin() ) {
            require_once DRSS_PLUGIN_DIR . 'admin/class-drss-admin.php';
        }

        require_once DRSS_PLUGIN_DIR . 'public/class-drss-public.php';
    }

    private function set_locale() {
        // WordPress.org handles translations automatically since WP 4.6
    }

    private function define_admin_hooks() {
        if ( ! is_admin() ) {
            return;
        }

        $admin = new DRSS_Admin( $this );
        
        add_action( 'admin_enqueue_scripts', array( $admin, 'enqueue_styles' ) );
        add_action( 'admin_enqueue_scripts', array( $admin, 'enqueue_scripts' ) );
        add_action( 'admin_menu', array( $admin, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $admin, 'register_settings' ) );
        add_action( 'admin_notices', array( $admin, 'display_admin_notices' ) );
        
        // AJAX handlers
        add_action( 'wp_ajax_drss_dismiss_notice', array( $admin, 'ajax_dismiss_notice' ) );
        add_action( 'wp_ajax_drss_optimize_image', array( $admin, 'ajax_optimize_image' ) );
        add_action( 'wp_ajax_drss_bulk_optimize', array( $admin, 'ajax_bulk_optimize' ) );
        add_action( 'wp_ajax_drss_reset_optimization', array( $admin, 'ajax_reset_optimization' ) );
        add_action( 'wp_ajax_drss_activate_license', array( $admin, 'ajax_activate_license' ) );
        add_action( 'wp_ajax_drss_deactivate_license', array( $admin, 'ajax_deactivate_license' ) );
        add_action( 'wp_ajax_drss_get_optimization_progress', array( $admin, 'ajax_get_optimization_progress' ) );
        add_action( 'wp_ajax_drss_sync_notices', array( $admin, 'ajax_sync_notices' ) );
        add_action( 'wp_ajax_drss_submit_review', array( $admin, 'handle_review_submission' ) );
        
        // SEO Audit AJAX
        add_action( 'wp_ajax_drss_run_seo_audit', array( $admin, 'ajax_run_seo_audit' ) );
        add_action( 'wp_ajax_drss_get_audit_status', array( $admin, 'ajax_get_audit_status' ) );
        add_action( 'wp_ajax_drss_get_url_details', array( $admin, 'ajax_get_url_details' ) );
        add_action( 'wp_ajax_drss_toggle_checklist_item', array( $admin, 'ajax_toggle_checklist_item' ) );
        
        // Page SEO meta boxes
        $page_seo = new DRSS_Page_SEO( $this );
        add_action( 'add_meta_boxes', array( $page_seo, 'add_meta_boxes' ) );
        add_action( 'admin_enqueue_scripts', array( $page_seo, 'enqueue_metabox_assets' ) );
        add_action( 'save_post', array( $page_seo, 'save_meta_box' ) );
        add_action( 'edited_term', array( $page_seo, 'save_term_meta' ), 10, 3 );
        add_action( 'created_term', array( $page_seo, 'save_term_meta' ), 10, 3 );
    }

    private function define_public_hooks() {
        $public = new DRSS_Public( $this );
        $image_optimizer = new DRSS_Image_Optimizer( $this );
        $seo = new DRSS_SEO( $this );
        $page_seo = new DRSS_Page_SEO( $this );
        $remote = new DRSS_Remote( $this );
        $license = new DRSS_License( $this );
        $minifier = new DRSS_Minifier( $this );
        $sitemap = new DRSS_Sitemap( $this );
        $robots = new DRSS_Robots( $this );
        
        add_action( 'wp_enqueue_scripts', array( $public, 'enqueue_styles' ) );
        add_action( 'wp_loaded', array( $minifier, 'init' ) );
        add_filter( 'plugin_action_links_' . DRSS_PLUGIN_BASENAME, array( $this, 'add_plugin_action_links' ) );
        add_filter( 'big_image_size_threshold', '__return_false' );
        add_filter( 'wp_handle_upload', array( $image_optimizer, 'handle_upload' ), 10, 2 );
        add_filter( 'wp_generate_attachment_metadata', array( $image_optimizer, 'process_attachment' ), 10, 2 );
        add_filter( 'wp_generate_attachment_metadata', array( $seo, 'auto_set_image_meta' ), 30, 2 );
        
        if ( $this->options['page_seo_enabled'] ) {
            add_action( 'wp_head', array( $page_seo, 'output_meta_tags' ), 1 );
            add_filter( 'pre_get_document_title', array( $page_seo, 'filter_title' ), 999 );
            add_filter( 'document_title_parts', array( $page_seo, 'filter_title_parts' ), 999 );
        }
        
        // Taxonomy SEO fields
        $taxonomies = array( 'category', 'post_tag', 'product_cat', 'product_tag' );
        foreach ( $taxonomies as $tax ) {
            add_action( $tax . '_add_form_fields', array( $page_seo, 'add_term_fields' ) );
            add_action( $tax . '_edit_form_fields', array( $page_seo, 'edit_term_fields' ) );
        }
        
        if ( $this->options['remote_connection'] ) {
            add_action( 'admin_init', array( $remote, 'schedule_checkin' ) );
            add_action( 'drss_remote_checkin', array( $remote, 'do_checkin' ) );
            
            if ( ! get_transient( 'drss_initial_checkin_done' ) ) {
                add_action( 'admin_init', array( $remote, 'do_checkin' ) );
                set_transient( 'drss_initial_checkin_done', 1, DAY_IN_SECONDS );
            }
        }
        
        add_action( 'admin_init', array( $license, 'maybe_validate_license' ) );
        
        if ( $this->options['show_attribution'] && $this->current_plan === 'free' ) {
            add_action( 'wp_footer', array( $public, 'output_attribution' ) );
        }
        
        if ( $this->options['wc_integration'] && class_exists( 'WooCommerce' ) ) {
            $wc = new DRSS_WooCommerce( $this );
            add_action( 'add_meta_boxes', array( $wc, 'add_product_seo_meta_box' ) );
        }
        
        if ( $this->options['dokan_integration'] && class_exists( 'WeDevs_Dokan' ) ) {
            $dokan = new DRSS_Dokan( $this );
            add_filter( 'dokan_product_image_upload', array( $dokan, 'check_vendor_image_limit' ), 10, 2 );
        }
    }
    
    public function add_plugin_action_links( $links ) {
        $settings_link = '<a href="' . admin_url( 'admin.php?page=drss-dashboard' ) . '">' . __( 'Settings', 'digital-rise-seo-solutions' ) . '</a>';
        $audit_link = '<a href="' . admin_url( 'admin.php?page=drss-seo-audit' ) . '" style="color: #2ecc71; font-weight: bold;">' . __( 'SEO Audit', 'digital-rise-seo-solutions' ) . '</a>';
        
        array_unshift( $links, $settings_link );
        $links[] = $audit_link;
        
        return $links;
    }

    public function get_option( $key, $default = null ) {
        $value = isset( $this->options[ $key ] ) ? $this->options[ $key ] : $default;
        
        if ( $key === 'keywords' && is_string( $value ) && ! empty( $value ) ) {
            if ( strpos( $value, 'b64:' ) === 0 ) {
                $value = base64_decode( substr( $value, 4 ) );
            }
            $value = str_replace( array( "\r\n", "\r" ), "\n", $value );
        }
        
        return $value;
    }

    public function update_option( $key, $value ) {
        if ( $key === 'keywords' && is_string( $value ) ) {
            $value = str_replace( array( "\r\n", "\r" ), "\n", $value );
            $value = 'b64:' . base64_encode( $value );
        }
        
        $this->options[ $key ] = $value;
        update_option( 'drss_options', $this->options );
    }

    public function save_options( $options ) {
        $this->options = wp_parse_args( $options, $this->options );
        update_option( 'drss_options', $this->options );
    }

    public function has_feature( $feature ) {
        $plans = new DRSS_Plans( $this );
        return $plans->has_feature( $feature );
    }

    public function is_plan( $plan ) {
        return $this->current_plan === $plan;
    }

    public function is_plan_at_least( $plan ) {
        $levels = array( 'free' => 0, 'pro' => 1, 'agency' => 2 );
        $current_level = isset( $levels[ $this->current_plan ] ) ? $levels[ $this->current_plan ] : 0;
        $required_level = isset( $levels[ $plan ] ) ? $levels[ $plan ] : 0;
        return $current_level >= $required_level;
    }
}

function drss_activate() {
    require_once DRSS_PLUGIN_DIR . 'includes/class-drss-activator.php';
    DRSS_Activator::activate();
}
register_activation_hook( __FILE__, 'drss_activate' );

function drss_deactivate() {
    require_once DRSS_PLUGIN_DIR . 'includes/class-drss-deactivator.php';
    DRSS_Deactivator::deactivate();
}
register_deactivation_hook( __FILE__, 'drss_deactivate' );

function drss() {
    return Digital_Rise_SEO_Solutions::get_instance();
}

add_action( 'plugins_loaded', 'drss', 0 );
