<?php
/**
 * Admin Dashboard
 *
 * @package Digital_Rise_SEO_Solutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRSS_Admin {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        
        // Add AJAX handler for review submission
        add_action( 'wp_ajax_drss_submit_review', array( $this, 'handle_review_submission' ) );
    }

    public function enqueue_styles( $hook ) {
        if ( strpos( $hook, 'drss' ) === false ) return;
        wp_enqueue_style( 'drss-admin', DRSS_PLUGIN_URL . 'admin/css/drss-admin.css', array(), DRSS_VERSION );
    }

    public function enqueue_scripts( $hook ) {
        if ( strpos( $hook, 'drss' ) === false ) return;
        wp_enqueue_script( 'drss-admin', DRSS_PLUGIN_URL . 'admin/js/drss-admin.js', array( 'jquery' ), DRSS_VERSION, true );
        wp_localize_script( 'drss-admin', 'drss_admin', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'drss_admin_nonce' ),
            'strings' => array(
                'optimizing' => __( 'Optimizing...', 'digital-rise-seo-solutions' ),
                'complete' => __( 'Complete!', 'digital-rise-seo-solutions' ),
                'error' => __( 'Error occurred', 'digital-rise-seo-solutions' ),
            ),
        ) );
    }

    public function add_admin_menu() {
        add_menu_page( __( 'Digital Rise SEO', 'digital-rise-seo-solutions' ), __( 'Digital Rise SEO', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-dashboard', array( $this, 'render_dashboard' ), 'dashicons-chart-area', 65 );
        add_submenu_page( 'drss-dashboard', __( 'Dashboard', 'digital-rise-seo-solutions' ), __( 'Dashboard', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-dashboard', array( $this, 'render_dashboard' ) );
        add_submenu_page( 'drss-dashboard', __( 'SEO Audit', 'digital-rise-seo-solutions' ), __( '🔍 SEO Audit', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-seo-audit', array( $this, 'render_seo_audit' ) );
        add_submenu_page( 'drss-dashboard', __( 'SEO Checklist', 'digital-rise-seo-solutions' ), __( '📋 SEO Checklist', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-seo-checklist', array( $this, 'render_seo_checklist' ) );
        add_submenu_page( 'drss-dashboard', __( 'Image Optimization', 'digital-rise-seo-solutions' ), __( 'Optimization', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-optimization', array( $this, 'render_optimization' ) );
        add_submenu_page( 'drss-dashboard', __( 'Bulk Optimize', 'digital-rise-seo-solutions' ), __( 'Bulk Optimize', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-bulk', array( $this, 'render_bulk_optimize' ) );
        add_submenu_page( 'drss-dashboard', __( 'Keywords', 'digital-rise-seo-solutions' ), __( 'Keywords', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-keywords', array( $this, 'render_keywords' ) );
        add_submenu_page( 'drss-dashboard', __( 'SEO Titles & Meta', 'digital-rise-seo-solutions' ), __( 'SEO Titles', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-page-seo', array( $this, 'render_page_seo' ) );
        add_submenu_page( 'drss-dashboard', __( 'Plans', 'digital-rise-seo-solutions' ), __( 'Plans', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-plans', array( $this, 'render_plans' ) );
        add_submenu_page( 'drss-dashboard', __( 'License', 'digital-rise-seo-solutions' ), __( 'License', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-license', array( $this, 'render_license' ) );
        add_submenu_page( 'drss-dashboard', __( 'Settings', 'digital-rise-seo-solutions' ), __( 'Settings', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-settings', array( $this, 'render_settings' ) );
        // Hidden page for viewing pending reviews (if emails fail)
        add_submenu_page( null, __( 'Pending Reviews', 'digital-rise-seo-solutions' ), __( 'Pending Reviews', 'digital-rise-seo-solutions' ), 'manage_options', 'drss-reviews', array( $this, 'render_reviews' ) );
    }

    public function register_settings() {
        register_setting( 'drss_options_group', 'drss_options', array(
            'sanitize_callback' => array( $this, 'sanitize_options' ),
        ) );
    }

    public function sanitize_options( $options ) {
        if ( ! is_array( $options ) ) {
            return array();
        }
        
        $sanitized = array();
        foreach ( $options as $key => $value ) {
            if ( is_bool( $value ) ) {
                $sanitized[ $key ] = (bool) $value;
            } elseif ( is_numeric( $value ) ) {
                $sanitized[ $key ] = absint( $value );
            } elseif ( is_array( $value ) ) {
                $sanitized[ $key ] = array_map( 'sanitize_text_field', $value );
            } else {
                $sanitized[ $key ] = sanitize_text_field( $value );
            }
        }
        return $sanitized;
    }

    public function display_admin_notices() {
        if ( get_transient( 'drss_activation_redirect' ) ) {
            delete_transient( 'drss_activation_redirect' );
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Redirect check only, no data processing
            if ( ! isset( $_GET['activate-multi'] ) ) {
                wp_safe_redirect( admin_url( 'admin.php?page=drss-dashboard' ) );
                exit;
            }
        }

        // FIX #2: Show YOUR branded notices on ALL admin pages
        // WordPress notices stay normal, YOUR notices get branded box
        $remote = new DRSS_Remote( $this->plugin );
        $notices = $remote->get_active_notices();
        
        if ( ! empty( $notices ) && is_array( $notices ) ) {
            foreach ( $notices as $notice ) {
                if ( ! is_array( $notice ) || empty( $notice['notice_content'] ) ) {
                    continue;
                }
                
                $type = isset( $notice['notice_type'] ) && in_array( $notice['notice_type'], array( 'info', 'warning', 'error', 'success', 'promo' ) ) ? $notice['notice_type'] : 'info';
                $notice_id = isset( $notice['notice_id'] ) ? $notice['notice_id'] : 'notice-' . md5( $notice['notice_content'] );
                
                // Border color based on type
                $border_colors = array(
                    'promo' => '#ffd700',
                    'error' => '#dc3545',
                    'warning' => '#ffc107',
                    'success' => '#4CAF50',
                    'info' => '#2196F3',
                );
                $border_color = isset( $border_colors[ $type ] ) ? $border_colors[ $type ] : '#2196F3';
                
                // YOUR BRANDED NOTICE BOX - Only for YOUR notices from owner panel
                ?>
                <div class="notice is-dismissible drss-remote-notice drss-branded-notice" data-notice-id="<?php echo esc_attr( $notice_id ); ?>" style="background: linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%); border: none; border-left: 4px solid <?php echo esc_attr( $border_color ); ?>; color: white; padding: 15px 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); margin: 5px 15px 15px 0;">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <?php
                        // Show logo
                        $logo_url = DRSS_LOGO_URL;
                        if ( ! empty( $logo_url ) && @getimagesize( $logo_url ) ) : ?>
                            <img src="<?php echo esc_url( $logo_url ); ?>" alt="Digital Rise" style="height: 40px; border-radius: 6px; background: white; padding: 8px;">
                        <?php else : ?>
                            <div style="width: 40px; height: 40px; background: white; border-radius: 6px; display: flex; align-items: center; justify-content: center; font-size: 18px; font-weight: bold; color: #1e3a5f;">DR</div>
                        <?php endif; ?>
                        
                        <div style="flex: 1;">
                            <?php if ( ! empty( $notice['notice_title'] ) ) : ?>
                                <strong style="font-size: 16px; display: block; margin-bottom: 5px;"><?php echo esc_html( $notice['notice_title'] ); ?></strong>
                            <?php endif; ?>
                            <div style="font-size: 14px; opacity: 0.95;">
                                <?php echo wp_kses_post( $notice['notice_content'] ); ?>
                                <?php if ( ! empty( $notice['notice_link'] ) ) : ?>
                                    <a href="<?php echo esc_url( $notice['notice_link'] ); ?>" target="_blank" style="color: #ffd700; font-weight: 600; text-decoration: underline; margin-left: 10px;"><?php esc_html_e( 'Learn more →', 'digital-rise-seo-solutions' ); ?></a>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <?php if ( $type === 'promo' ) : ?>
                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-license' ) ); ?>" class="button" style="background: #ffd700; color: #1e3a5f; border: none; font-weight: bold; padding: 10px 20px; white-space: nowrap;">
                                ⭐ <?php esc_html_e( 'Upgrade Now', 'digital-rise-seo-solutions' ); ?>
                            </a>
                        <?php endif; ?>
                        
                        <span class="drss-plan-badge drss-plan-<?php echo esc_attr( $this->plugin->current_plan ); ?>" style="<?php echo $this->plugin->current_plan === 'free' ? 'background: #ffd700; color: #333;' : ''; ?> padding: 6px 12px; border-radius: 20px; font-size: 12px; font-weight: 600; text-transform: uppercase;">
                            <?php echo esc_html( ucfirst( $this->plugin->current_plan ) ); ?>
                        </span>
                    </div>
                </div>
                <?php
            }
        }
    }

    private function render_header( $title ) {
        $logo_url = DRSS_LOGO_URL;
        ?>
        <div class="drss-header">
            <div class="drss-header-logo">
                <?php if ( ! empty( $logo_url ) && @getimagesize( $logo_url ) ) : ?>
                    <img src="<?php echo esc_url( $logo_url ); ?>" alt="Digital Rise" height="50">
                <?php else : ?>
                    <span class="drss-logo-text">Digital Rise SEO Solutions</span>
                <?php endif; ?>
            </div>
            <div class="drss-header-info" style="flex: 1;">
                <h1><?php echo esc_html( $title ); ?></h1>
                <span class="drss-plan-badge drss-plan-<?php echo esc_attr( $this->plugin->current_plan ); ?>">
                    <?php echo esc_html( ucfirst( $this->plugin->current_plan ) ); ?>
                </span>
            </div>
            <?php if ( $this->plugin->current_plan === 'free' ) : ?>
                <div class="drss-header-actions" style="display: flex; gap: 10px; align-items: center;">
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-license' ) ); ?>" class="button button-primary button-hero" style="background: #ffd700; border-color: #ffd700; color: #333; font-weight: 600; text-shadow: none;">
                        ⭐ <?php esc_html_e( 'Upgrade to Pro', 'digital-rise-seo-solutions' ); ?>
                    </a>
                    <button type="button" id="drss-review-btn" class="button button-secondary" style="background: #0073aa; color: white; border-color: #0073aa; font-weight: 600; padding: 8px 15px; height: auto;">
                        <span class="dashicons dashicons-star-filled" style="color: #ffd700; margin-top: 3px;"></span> <?php esc_html_e( 'Review Plugin', 'digital-rise-seo-solutions' ); ?>
                    </button>
                </div>
            <?php else : ?>
                <div class="drss-header-actions" style="display: flex; gap: 10px; align-items: center;">
                    <button type="button" id="drss-review-btn" class="button button-primary" style="background: #0073aa; color: white; border-color: #0073aa; font-weight: 600; padding: 8px 15px; height: auto;">
                        <span class="dashicons dashicons-star-filled" style="color: #ffd700; margin-top: 3px;"></span> <?php esc_html_e( 'Review Plugin', 'digital-rise-seo-solutions' ); ?>
                    </button>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Review Modal -->
        <div id="drss-review-modal" style="display: none; position: fixed; z-index: 999999; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.6);">
            <div style="background-color: #fefefe; margin: 5% auto; padding: 0; border-radius: 8px; width: 90%; max-width: 500px; box-shadow: 0 4px 20px rgba(0,0,0,0.3);">
                <!-- Header -->
                <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px 8px 0 0;">
                    <h2 style="margin: 0; color: white; display: flex; align-items: center;">
                        <span class="dashicons dashicons-star-filled" style="font-size: 24px; margin-right: 10px;"></span>
                        <?php esc_html_e( 'Review Our Plugin', 'digital-rise-seo-solutions' ); ?>
                    </h2>
                    <p style="margin: 10px 0 0 0; opacity: 0.9;"><?php esc_html_e( 'Your feedback helps us improve!', 'digital-rise-seo-solutions' ); ?></p>
                </div>
                
                <!-- Body -->
                <div style="padding: 30px;">
                    <!-- Email Configuration Notice -->
                    <div style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin-bottom: 25px; border-radius: 4px;">
                        <p style="margin: 0; color: #856404; font-size: 14px; line-height: 1.6;">
                            <strong>ℹ️ <?php esc_html_e( 'Note:', 'digital-rise-seo-solutions' ); ?></strong>
                            <?php esc_html_e( 'If email delivery is not configured on your server, please send your review directly to:', 'digital-rise-seo-solutions' ); ?>
                            <br>
                            <a href="mailto:seo.oussama@gmail.com" style="color: #667eea; font-weight: 600; text-decoration: none;">
                                📧 seo.oussama@gmail.com
                            </a>
                        </p>
                    </div>
                    
                    <form id="drss-review-form">
                        <!-- Star Rating -->
                        <div style="margin-bottom: 25px; text-align: center;">
                            <label style="display: block; margin-bottom: 15px; font-weight: 600; font-size: 16px; color: #333;">
                                <?php esc_html_e( 'Rate your experience:', 'digital-rise-seo-solutions' ); ?>
                            </label>
                            <div class="drss-star-rating" style="font-size: 40px; cursor: pointer;">
                                <span class="drss-star" data-rating="1">☆</span>
                                <span class="drss-star" data-rating="2">☆</span>
                                <span class="drss-star" data-rating="3">☆</span>
                                <span class="drss-star" data-rating="4">☆</span>
                                <span class="drss-star" data-rating="5">☆</span>
                            </div>
                            <input type="hidden" id="drss-rating" name="rating" value="0">
                        </div>
                        
                        <!-- Review Text -->
                        <div style="margin-bottom: 20px;">
                            <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #333;">
                                <?php esc_html_e( 'Your Review:', 'digital-rise-seo-solutions' ); ?>
                            </label>
                            <textarea name="review" id="drss-review-text" rows="5" style="width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;" placeholder="<?php esc_attr_e( 'Tell us about your experience...', 'digital-rise-seo-solutions' ); ?>" required></textarea>
                        </div>
                        
                        <!-- Contact Info (Optional) -->
                        <div style="margin-bottom: 20px;">
                            <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #333;">
                                <?php esc_html_e( 'Email (optional):', 'digital-rise-seo-solutions' ); ?>
                            </label>
                            <input type="email" name="email" id="drss-review-email" style="width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px;" placeholder="your@email.com">
                        </div>
                        
                        <div style="margin-bottom: 20px;">
                            <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #333;">
                                <?php esc_html_e( 'Phone (optional):', 'digital-rise-seo-solutions' ); ?>
                            </label>
                            <input type="tel" name="phone" id="drss-review-phone" style="width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px;" placeholder="+1234567890">
                        </div>
                        
                        <!-- Buttons -->
                        <div style="display: flex; gap: 10px; margin-top: 25px;">
                            <button type="submit" class="button button-primary" style="flex: 1; height: 40px; font-size: 15px;">
                                <span class="dashicons dashicons-yes" style="margin-top: 3px;"></span> <?php esc_html_e( 'Submit Review', 'digital-rise-seo-solutions' ); ?>
                            </button>
                            <button type="button" id="drss-copy-review" class="button" style="height: 40px;" title="<?php esc_attr_e( 'Copy review to send manually', 'digital-rise-seo-solutions' ); ?>">
                                <span class="dashicons dashicons-clipboard" style="margin-top: 3px;"></span>
                            </button>
                            <button type="button" class="button drss-close-modal" style="flex: 0 0 auto; height: 40px;">
                                <?php esc_html_e( 'Cancel', 'digital-rise-seo-solutions' ); ?>
                            </button>
                        </div>
                    </form>
                    
                    <!-- Success Message -->
                    <div id="drss-review-success" style="display: none; text-align: center; padding: 20px;">
                        <span class="dashicons dashicons-yes-alt" style="font-size: 60px; color: #46b450;"></span>
                        <h3 style="color: #46b450; margin: 15px 0;"><?php esc_html_e( 'Thank You!', 'digital-rise-seo-solutions' ); ?></h3>
                        <p><?php esc_html_e( 'Your review has been sent successfully.', 'digital-rise-seo-solutions' ); ?></p>
                        <button type="button" class="button button-primary drss-close-modal" style="margin-top: 15px;">
                            <?php esc_html_e( 'Close', 'digital-rise-seo-solutions' ); ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
            .drss-star { 
                color: #ddd; 
                transition: color 0.2s; 
                display: inline-block;
                margin: 0 3px;
            }
            .drss-star:hover,
            .drss-star.active { 
                color: #ffd700; 
            }
            .drss-star.hovered {
                color: #ffd700;
            }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            var selectedRating = 0;
            
            // Open modal
            $('#drss-review-btn').on('click', function() {
                $('#drss-review-modal').fadeIn(300);
            });
            
            // Close modal
            $('.drss-close-modal, #drss-review-modal').on('click', function(e) {
                if (e.target === this) {
                    $('#drss-review-modal').fadeOut(300);
                    setTimeout(function() {
                        $('#drss-review-form')[0].reset();
                        $('#drss-review-success').hide();
                        $('#drss-review-form').show();
                        $('.drss-star').removeClass('active');
                        selectedRating = 0;
                    }, 300);
                }
            });
            
            // Star rating
            $('.drss-star').on('mouseenter', function() {
                var rating = $(this).data('rating');
                $('.drss-star').each(function() {
                    if ($(this).data('rating') <= rating) {
                        $(this).addClass('hovered');
                    } else {
                        $(this).removeClass('hovered');
                    }
                });
            });
            
            $('.drss-star-rating').on('mouseleave', function() {
                $('.drss-star').removeClass('hovered');
            });
            
            $('.drss-star').on('click', function() {
                selectedRating = $(this).data('rating');
                $('#drss-rating').val(selectedRating);
                
                $('.drss-star').each(function() {
                    if ($(this).data('rating') <= selectedRating) {
                        $(this).addClass('active').text('★');
                    } else {
                        $(this).removeClass('active').text('☆');
                    }
                });
            });
            
            // Copy review to clipboard
            $('#drss-copy-review').on('click', function() {
                var stars = '';
                for (var i = 0; i < selectedRating; i++) stars += '★';
                for (var i = selectedRating; i < 5; i++) stars += '☆';
                
                var reviewText = 'PLUGIN REVIEW\n\n';
                reviewText += 'Rating: ' + stars + ' (' + selectedRating + '/5)\n\n';
                reviewText += 'Review:\n' + $('#drss-review-text').val() + '\n\n';
                reviewText += 'Website: <?php echo esc_js( home_url() ); ?>\n';
                
                var email = $('#drss-review-email').val();
                if (email) reviewText += 'Email: ' + email + '\n';
                
                var phone = $('#drss-review-phone').val();
                if (phone) reviewText += 'Phone: ' + phone + '\n';
                
                reviewText += '\nPlease send to: seo.oussama@gmail.com';
                
                // Copy to clipboard
                if (navigator.clipboard && navigator.clipboard.writeText) {
                    navigator.clipboard.writeText(reviewText).then(function() {
                        var $btn = $('#drss-copy-review');
                        var originalHtml = $btn.html();
                        $btn.html('<span class="dashicons dashicons-yes" style="margin-top: 3px; color: #46b450;"></span>');
                        setTimeout(function() {
                            $btn.html(originalHtml);
                        }, 2000);
                        alert('<?php esc_html_e( 'Review copied! You can paste it in an email to seo.oussama@gmail.com', 'digital-rise-seo-solutions' ); ?>');
                    }).catch(function() {
                        // Fallback for older browsers
                        copyToClipboardFallback(reviewText);
                    });
                } else {
                    // Fallback for older browsers
                    copyToClipboardFallback(reviewText);
                }
            });
            
            // Fallback copy method
            function copyToClipboardFallback(text) {
                var $temp = $('<textarea>');
                $('body').append($temp);
                $temp.val(text).select();
                try {
                    document.execCommand('copy');
                    alert('<?php esc_html_e( 'Review copied! You can paste it in an email to seo.oussama@gmail.com', 'digital-rise-seo-solutions' ); ?>');
                } catch (err) {
                    alert('<?php esc_html_e( 'Could not copy. Please copy manually.', 'digital-rise-seo-solutions' ); ?>');
                }
                $temp.remove();
            }
            
            // Submit review
            $('#drss-review-form').on('submit', function(e) {
                e.preventDefault();
                
                if (selectedRating === 0) {
                    alert('<?php esc_html_e( 'Please select a star rating!', 'digital-rise-seo-solutions' ); ?>');
                    return;
                }
                
                var $btn = $(this).find('button[type="submit"]');
                $btn.prop('disabled', true).text('<?php esc_html_e( 'Sending...', 'digital-rise-seo-solutions' ); ?>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'drss_submit_review',
                        nonce: '<?php echo esc_js( wp_create_nonce( 'drss_review_nonce' ) ); ?>',
                        rating: selectedRating,
                        review: $('#drss-review-text').val(),
                        email: $('#drss-review-email').val(),
                        phone: $('#drss-review-phone').val(),
                        site_url: '<?php echo esc_js( home_url() ); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#drss-review-form').fadeOut(200, function() {
                                $('#drss-review-success').fadeIn(200);
                            });
                        } else {
                            alert(response.data || '<?php esc_html_e( 'Error sending review. Please try again.', 'digital-rise-seo-solutions' ); ?>');
                            $btn.prop('disabled', false).html('<span class="dashicons dashicons-yes" style="margin-top: 3px;"></span> <?php esc_html_e( 'Submit Review', 'digital-rise-seo-solutions' ); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php esc_html_e( 'Error sending review. Please try again.', 'digital-rise-seo-solutions' ); ?>');
                        $btn.prop('disabled', false).html('<span class="dashicons dashicons-yes" style="margin-top: 3px;"></span> <?php esc_html_e( 'Submit Review', 'digital-rise-seo-solutions' ); ?>');
                    }
                });
            });
        });
        </script>
        <?php
    }

    public function render_dashboard() {
        // Handle sitemap cache clear
        if ( isset( $_POST['drss_clear_sitemap_cache'] ) && check_admin_referer( 'drss_clear_sitemap_nonce' ) ) {
            // Clear all sitemap caches
            delete_transient( 'drss_sitemap_index_cache' );
            delete_transient( 'drss_sitemap_pages_cache' );
            delete_transient( 'drss_sitemap_posts_cache' );
            delete_transient( 'drss_sitemap_categories_cache' );
            delete_transient( 'drss_sitemap_tags_cache' );
            delete_transient( 'drss_sitemap_products_cache' );
            delete_transient( 'drss_sitemap_product-categories_cache' );
            delete_transient( 'drss_sitemap_product-tags_cache' );
            delete_transient( 'drss_sitemap_authors_cache' );
            delete_transient( 'drss_sitemap_vendors_cache' );
            delete_transient( 'drss_sitemap_vendor-products_cache' );
            delete_transient( 'drss_sitemap_images_cache' );
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Sitemap cache cleared! All sitemaps will regenerate with updated image URLs.', 'digital-rise-seo-solutions' ) . '</p></div>';
        }
        
        // Handle contact form save
        if ( isset( $_POST['drss_save_contact'] ) && check_admin_referer( 'drss_contact_nonce' ) ) {
            $options = $this->plugin->options;
            $options['user_email'] = isset( $_POST['user_email'] ) ? sanitize_email( wp_unslash( $_POST['user_email'] ) ) : '';
            $options['user_whatsapp'] = isset( $_POST['user_whatsapp'] ) ? sanitize_text_field( wp_unslash( $_POST['user_whatsapp'] ) ) : '';
            $this->plugin->save_options( $options );
            
            // Trigger immediate sync
            $remote = new DRSS_Remote( $this->plugin );
            $remote->do_checkin();
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Contact info saved and synced!', 'digital-rise-seo-solutions' ) . '</p></div>';
        }
        
        // Auto-sync on page load (once per hour)
        $last_auto_sync = get_transient( 'drss_auto_sync' );
        if ( ! $last_auto_sync ) {
            $remote = new DRSS_Remote( $this->plugin );
            $remote->do_checkin();
            set_transient( 'drss_auto_sync', time(), HOUR_IN_SECONDS );
        }
        
        $optimizer = new DRSS_Image_Optimizer( $this->plugin );
        $stats = $optimizer->get_stats();
        $bulk = new DRSS_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        $wc = new DRSS_WooCommerce( $this->plugin );
        $dokan = new DRSS_Dokan( $this->plugin );
        $user_email = $this->plugin->get_option( 'user_email', '' );
        $user_whatsapp = $this->plugin->get_option( 'user_whatsapp', '' );
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'Dashboard', 'digital-rise-seo-solutions' ) ); ?>
            
            <div class="drss-dashboard-grid">
                <!-- Contact Info Card - FIRST -->
                <div class="drss-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #fff;">
                    <h2 style="color: #fff;">📧 <?php esc_html_e( 'Your Contact Info', 'digital-rise-seo-solutions' ); ?></h2>
                    <form method="post">
                        <?php wp_nonce_field( 'drss_contact_nonce' ); ?>
                        <p>
                            <label style="color: #fff;"><?php esc_html_e( 'Email:', 'digital-rise-seo-solutions' ); ?></label><br>
                            <input type="email" name="user_email" value="<?php echo esc_attr( $user_email ); ?>" style="width: 100%; padding: 8px;" placeholder="your@email.com">
                        </p>
                        <p>
                            <label style="color: #fff;"><?php esc_html_e( 'WhatsApp:', 'digital-rise-seo-solutions' ); ?></label><br>
                            <input type="text" name="user_whatsapp" value="<?php echo esc_attr( $user_whatsapp ); ?>" style="width: 100%; padding: 8px;" placeholder="+216 XX XXX XXX">
                        </p>
                        <p>
                            <button type="submit" name="drss_save_contact" class="button" style="background: #fff; color: #667eea; border: none; font-weight: bold;">
                                <?php esc_html_e( 'Save & Sync', 'digital-rise-seo-solutions' ); ?>
                            </button>
                        </p>
                    </form>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'Optimization Stats', 'digital-rise-seo-solutions' ); ?></h2>
                    <div class="drss-stats">
                        <div class="drss-stat">
                            <span class="drss-stat-num"><?php echo esc_html( $stats['total_images'] ); ?></span>
                            <span class="drss-stat-label"><?php esc_html_e( 'Total Images', 'digital-rise-seo-solutions' ); ?></span>
                        </div>
                        <div class="drss-stat">
                            <span class="drss-stat-num"><?php echo esc_html( $stats['webp_count'] ); ?></span>
                            <span class="drss-stat-label"><?php esc_html_e( 'WebP', 'digital-rise-seo-solutions' ); ?></span>
                        </div>
                        <div class="drss-stat">
                            <span class="drss-stat-num"><?php echo esc_html( $stats['optimized'] ); ?></span>
                            <span class="drss-stat-label"><?php esc_html_e( 'Optimized', 'digital-rise-seo-solutions' ); ?></span>
                        </div>
                    </div>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'Image Status', 'digital-rise-seo-solutions' ); ?></h2>
                    <div class="drss-progress-wrap">
                        <div class="drss-progress-bar">
                            <div class="drss-progress-fill" style="width: <?php echo esc_attr( $progress['percentage'] ); ?>%"></div>
                        </div>
                        <?php
                        // translators: %1$d is the number of optimized images, %2$d is the total number of images
                        ?>
                        <p><?php printf( esc_html__( '%1$d of %2$d images optimized', 'digital-rise-seo-solutions' ), (int) $progress['optimized'], (int) $progress['total'] ); ?></p>
                    </div>
                    <?php if ( $progress['remaining'] > 0 ) : ?>
                        <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-bulk' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Optimize All', 'digital-rise-seo-solutions' ); ?></a></p>
                    <?php endif; ?>
                </div>
                
                <div class="drss-card">
                    <h2>🗺️ <?php esc_html_e( 'Sitemap', 'digital-rise-seo-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Your sitemap is automatically updated when images are optimized.', 'digital-rise-seo-solutions' ); ?></p>
                    <p style="margin-bottom: 15px;">
                        <a href="<?php echo esc_url( home_url( '/sitemap.xml' ) ); ?>" target="_blank" class="button">
                            <?php esc_html_e( 'View Sitemap', 'digital-rise-seo-solutions' ); ?>
                        </a>
                    </p>
                    <form method="post" style="display: inline;">
                        <?php wp_nonce_field( 'drss_clear_sitemap_nonce' ); ?>
                        <button type="submit" name="drss_clear_sitemap_cache" class="button">
                            🔄 <?php esc_html_e( 'Refresh Sitemap Now', 'digital-rise-seo-solutions' ); ?>
                        </button>
                    </form>
                    <p class="description" style="margin-top: 10px;">
                        <?php esc_html_e( 'Clear cache and regenerate sitemap with latest image URLs.', 'digital-rise-seo-solutions' ); ?>
                    </p>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'WooCommerce', 'digital-rise-seo-solutions' ); ?></h2>
                    <?php $wc_stats = $wc->get_stats(); ?>
                    <?php if ( $wc_stats['active'] ) : ?>
                        <p class="drss-status-active"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e( 'Active', 'digital-rise-seo-solutions' ); ?></p>
                        <?php
                        // translators: %d is the number of products
                        ?>
                        <p><?php printf( esc_html__( 'Products: %d', 'digital-rise-seo-solutions' ), (int) $wc_stats['total_products'] ); ?></p>
                    <?php else : ?>
                        <p class="drss-status-inactive"><?php esc_html_e( 'Not installed', 'digital-rise-seo-solutions' ); ?></p>
                    <?php endif; ?>
                </div>

                <div class="drss-card <?php echo ! $dokan_stats['active'] ? 'drss-locked-feature' : ''; ?>">
                    <h2><?php esc_html_e( 'Dokan', 'digital-rise-seo-solutions' ); ?></h2>
                    <?php $dokan_stats = $dokan->get_stats(); ?>
                    <?php if ( $dokan_stats['active'] ) : ?>
                        <p class="drss-status-active"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e( 'Active', 'digital-rise-seo-solutions' ); ?></p>
                        <?php
                        // translators: %d is the number of vendors
                        ?>
                        <p><?php printf( esc_html__( 'Vendors: %d', 'digital-rise-seo-solutions' ), (int) $dokan_stats['total_vendors'] ); ?></p>
                    <?php else : ?>
                        <p class="drss-status-inactive"><span class="dashicons dashicons-lock"></span> <?php esc_html_e( 'Dokan not installed', 'digital-rise-seo-solutions' ); ?></p>
                        <p><?php esc_html_e( 'Install Dokan to enable multi-vendor image management and per-vendor limits.', 'digital-rise-seo-solutions' ); ?></p>
                        <p>
                            <a href="<?php echo esc_url( admin_url( 'plugin-install.php?s=dokan&tab=search&type=term' ) ); ?>" class="button"><?php esc_html_e( 'Install Dokan', 'digital-rise-seo-solutions' ); ?></a>
                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-plans' ) ); ?>" class="button button-primary"><?php esc_html_e( 'View Pro Features', 'digital-rise-seo-solutions' ); ?></a>
                        </p>
                    <?php endif; ?>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'Quick Actions', 'digital-rise-seo-solutions' ); ?></h2>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-bulk' ) ); ?>" class="button"><?php esc_html_e( 'Bulk Optimize', 'digital-rise-seo-solutions' ); ?></a></p>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-keywords' ) ); ?>" class="button"><?php esc_html_e( 'Manage Keywords', 'digital-rise-seo-solutions' ); ?></a></p>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-settings' ) ); ?>" class="button"><?php esc_html_e( 'Settings', 'digital-rise-seo-solutions' ); ?></a></p>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'Connection Status', 'digital-rise-seo-solutions' ); ?></h2>
                    <?php 
                    $last_checkin = $this->plugin->get_option( 'last_checkin' );
                    $remote = new DRSS_Remote( $this->plugin );
                    $notices = $remote->get_active_notices();
                    ?>
                    <p><strong><?php esc_html_e( 'Last Sync:', 'digital-rise-seo-solutions' ); ?></strong> 
                        <?php echo $last_checkin ? esc_html( human_time_diff( strtotime( $last_checkin ) ) ) . ' ago' : esc_html__( 'Never', 'digital-rise-seo-solutions' ); ?>
                    </p>
                    <p><strong><?php esc_html_e( 'Active Notices:', 'digital-rise-seo-solutions' ); ?></strong> <?php echo count( $notices ); ?></p>
                    <p><strong><?php esc_html_e( 'Status:', 'digital-rise-seo-solutions' ); ?></strong> 
                        <span style="color: green;">✅ <?php esc_html_e( 'Auto-sync enabled', 'digital-rise-seo-solutions' ); ?></span>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }

    public function render_optimization() {
        if ( isset( $_POST['drss_save_optimization'] ) && check_admin_referer( 'drss_optimization_nonce' ) ) {
            $options = $this->plugin->options;
            $options['max_width'] = isset( $_POST['max_width'] ) ? absint( $_POST['max_width'] ) : 2048;
            $options['max_height'] = isset( $_POST['max_height'] ) ? absint( $_POST['max_height'] ) : 2048;
            $options['target_file_size'] = isset( $_POST['target_file_size'] ) ? absint( $_POST['target_file_size'] ) : 100;
            $options['compression_quality'] = isset( $_POST['compression_quality'] ) ? min( 100, max( 1, absint( $_POST['compression_quality'] ) ) ) : 82;
            $options['auto_compress'] = isset( $_POST['auto_compress'] );
            $options['max_images_per_user'] = isset( $_POST['max_images_per_user'] ) ? absint( $_POST['max_images_per_user'] ) : 0;
            $options['max_images_per_product'] = isset( $_POST['max_images_per_product'] ) ? absint( $_POST['max_images_per_product'] ) : 10;
            $options['max_images_per_vendor'] = isset( $_POST['max_images_per_vendor'] ) ? absint( $_POST['max_images_per_vendor'] ) : 50;
            $this->plugin->save_options( $options );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-seo-solutions' ) . '</p></div>';
        }
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'Image Optimization', 'digital-rise-seo-solutions' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'drss_optimization_nonce' ); ?>
                
                <div class="drss-card">
                    <h2><?php esc_html_e( 'Size Settings', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Max Width (px)', 'digital-rise-seo-solutions' ); ?></th>
                            <td><input type="number" name="max_width" value="<?php echo esc_attr( $this->plugin->get_option( 'max_width', 1920 ) ); ?>" min="100" max="10000"></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Max Height (px)', 'digital-rise-seo-solutions' ); ?></th>
                            <td><input type="number" name="max_height" value="<?php echo esc_attr( $this->plugin->get_option( 'max_height', 1080 ) ); ?>" min="100" max="10000"></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Target File Size (KB)', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="number" name="target_file_size" value="<?php echo esc_attr( $this->plugin->get_option( 'target_file_size', 100 ) ); ?>" min="10" max="500">
                                <p class="description"><?php esc_html_e( 'Images will be compressed to this target size (default: 100KB)', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Compression Quality', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="range" name="compression_quality" id="compression_quality" value="<?php echo esc_attr( $this->plugin->get_option( 'compression_quality', 82 ) ); ?>" min="1" max="100">
                                <span id="quality_value"><?php echo esc_html( $this->plugin->get_option( 'compression_quality', 82 ) ); ?>%</span>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Auto Compress', 'digital-rise-seo-solutions' ); ?></th>
                            <td><label><input type="checkbox" name="auto_compress" <?php checked( $this->plugin->get_option( 'auto_compress', true ) ); ?>> <?php esc_html_e( 'Automatically compress on upload', 'digital-rise-seo-solutions' ); ?></label></td>
                        </tr>
                    </table>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'Upload Limits', 'digital-rise-seo-solutions' ); ?></h2>
                    <?php if ( ! $this->plugin->has_feature( 'per_user_limits' ) ) : ?>
                        <div class="drss-pro-notice"><span class="dashicons dashicons-lock"></span> <?php esc_html_e( 'Upload limits require Pro plan.', 'digital-rise-seo-solutions' ); ?></div>
                    <?php endif; ?>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per User', 'digital-rise-seo-solutions' ); ?></th>
                            <td><input type="number" name="max_images_per_user" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_user', 0 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_user_limits' ) ); ?>><p class="description"><?php esc_html_e( '0 = unlimited', 'digital-rise-seo-solutions' ); ?></p></td>
                        </tr>
                        <?php if ( class_exists( 'WooCommerce' ) ) : ?>
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per Product', 'digital-rise-seo-solutions' ); ?></th>
                            <td><input type="number" name="max_images_per_product" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_product', 10 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_product_limits' ) ); ?>></td>
                        </tr>
                        <?php endif; ?>
                        <?php if ( class_exists( 'WeDevs_Dokan' ) || class_exists( 'Dokan_Pro' ) ) : ?>
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per Vendor', 'digital-rise-seo-solutions' ); ?></th>
                            <td><input type="number" name="max_images_per_vendor" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_vendor', 50 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_vendor_limits' ) ); ?>></td>
                        </tr>
                        <?php endif; ?>
                    </table>
                </div>

                <p class="submit"><input type="submit" name="drss_save_optimization" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-seo-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    public function render_bulk_optimize() {
        $bulk = new DRSS_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'Bulk Optimize', 'digital-rise-seo-solutions' ) ); ?>
            
            <div class="drss-card">
                <h2><?php esc_html_e( 'Optimize All Images', 'digital-rise-seo-solutions' ); ?></h2>
                <p><?php esc_html_e( 'This will convert images to WebP and apply SEO to all your images.', 'digital-rise-seo-solutions' ); ?></p>
                
                <div class="drss-bulk-progress">
                    <div class="drss-progress-bar">
                        <div class="drss-progress-fill" id="bulk-progress-fill" style="width: <?php echo esc_attr( $progress['percentage'] ); ?>%"></div>
                    </div>
                    <p id="bulk-progress-text">
                        <?php
                        // translators: %1$d is optimized images, %2$d is total images, %3$d is remaining images
                        printf( esc_html__( '%1$d of %2$d images optimized (%3$d remaining)', 'digital-rise-seo-solutions' ), (int) $progress['optimized'], (int) $progress['total'], (int) $progress['remaining'] );
                        ?>
                    </p>
                </div>

                <table class="form-table">
                    <tr>
                        <th><?php esc_html_e( 'Options', 'digital-rise-seo-solutions' ); ?></th>
                        <td>
                            <p><label><input type="checkbox" id="apply_seo" checked> <?php esc_html_e( 'Apply SEO (ALT text, title, description)', 'digital-rise-seo-solutions' ); ?></label></p>
                            <p><label><input type="checkbox" id="reoptimize_all"> <?php esc_html_e( 'Re-optimize ALL images (including previously optimized)', 'digital-rise-seo-solutions' ); ?></label></p>
                        </td>
                    </tr>
                </table>

                <p>
                    <button type="button" id="start-bulk-optimize" class="button button-primary button-hero">
                        <?php esc_html_e( 'Start Bulk Optimization', 'digital-rise-seo-solutions' ); ?>
                    </button>
                    <button type="button" id="stop-bulk-optimize" class="button button-secondary" style="display:none;">
                        <?php esc_html_e( 'Stop', 'digital-rise-seo-solutions' ); ?>
                    </button>
                    <button type="button" id="reset-optimization" class="button">
                        <?php esc_html_e( 'Reset All (Mark as Unoptimized)', 'digital-rise-seo-solutions' ); ?>
                    </button>
                </p>

                <div id="bulk-log" class="drss-bulk-log" style="display:none;"></div>
            </div>
        </div>
        <?php
    }

    public function render_keywords() {
        $keywords = new DRSS_Keywords( $this->plugin );
        
        if ( isset( $_POST['drss_save_keywords'] ) && check_admin_referer( 'drss_keywords_nonce' ) ) {
            // FIX #4: Use sanitize_textarea_field (preserves line breaks) like old version
            $keywords_input = isset( $_POST['keywords'] ) ? sanitize_textarea_field( wp_unslash( $_POST['keywords'] ) ) : '';
            $keywords->save_keywords( $keywords_input );
            
            // Save default pattern
            $this->plugin->update_option( 'image_seo_pattern', isset( $_POST['image_seo_pattern'] ) ? sanitize_text_field( wp_unslash( $_POST['image_seo_pattern'] ) ) : '{site_name} {keyword}' );
            
            // PHASE 3 FIX #6: Save content-type-specific patterns
            $content_types = array( 'post', 'page', 'product' );
            foreach ( $content_types as $type ) {
                $key = 'image_seo_pattern_' . $type;
                if ( isset( $_POST[ $key ] ) ) {
                    $this->plugin->update_option( $key, sanitize_text_field( wp_unslash( $_POST[ $key ] ) ) );
                }
            }
            
            $this->plugin->update_option( 'auto_rename', isset( $_POST['auto_rename'] ) );
            $this->plugin->update_option( 'auto_alt', isset( $_POST['auto_alt'] ) );
            $this->plugin->update_option( 'auto_title', isset( $_POST['auto_title'] ) );
            $this->plugin->update_option( 'auto_description', isset( $_POST['auto_description'] ) );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-seo-solutions' ) . '</p></div>';
        }
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'Keywords Management', 'digital-rise-seo-solutions' ) ); ?>
            
            <?php if ( ! $this->plugin->has_feature( 'custom_keywords' ) ) : ?>
                <div class="drss-card">
                    <div class="drss-pro-notice">
                        <span class="dashicons dashicons-lock"></span> 
                        <?php esc_html_e( 'Custom keywords require Pro plan. Free plan uses basic patterns.', 'digital-rise-seo-solutions' ); ?>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-license' ) ); ?>" class="button"><?php esc_html_e( 'Upgrade', 'digital-rise-seo-solutions' ); ?></a>
                    </div>
                </div>
            <?php endif; ?>
            
            <form method="post">
                <?php wp_nonce_field( 'drss_keywords_nonce' ); ?>
                
                <div class="drss-card">
                    <h2><?php esc_html_e( 'Keywords', 'digital-rise-seo-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Enter keywords for image SEO. One keyword phrase per line.', 'digital-rise-seo-solutions' ); ?></p>
                    <textarea name="keywords" rows="10" class="large-text code" <?php disabled( ! $this->plugin->has_feature( 'custom_keywords' ) ); ?> placeholder="<?php esc_attr_e( "Example:\nreal estate tunisia\nbuy apartment kairouan\nluxury villa sousse", 'digital-rise-seo-solutions' ); ?>"><?php echo esc_textarea( $this->plugin->get_option( 'keywords', '' ) ); ?></textarea>
                    <?php
                    // translators: %d is the number of keywords
                    ?>
                    <p class="description"><?php printf( esc_html__( 'Total keywords: %d (one per line recommended)', 'digital-rise-seo-solutions' ), (int) $keywords->get_count() ); ?></p>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'Image SEO Pattern', 'digital-rise-seo-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Define how image names, titles, ALT text, and descriptions are generated. Maximum 7 words will be used.', 'digital-rise-seo-solutions' ); ?></p>
                    
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="image_seo_pattern" value="<?php echo esc_attr( $this->plugin->get_option( 'image_seo_pattern', '{site_name} {keyword}' ) ); ?>" class="large-text" id="image-seo-pattern">
                                <p class="description"><?php esc_html_e( 'This pattern will be applied to: Image filename, Title, ALT text, Caption, Description', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Available Variables', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <div class="drss-variable-buttons">
                                    <?php 
                                    $seo = new DRSS_SEO( $this->plugin );
                                    $vars = $seo->get_available_variables();
                                    foreach ( $vars as $var => $desc ) : 
                                    ?>
                                        <button type="button" class="button drss-insert-var" data-var="<?php echo esc_attr( $var ); ?>" title="<?php echo esc_attr( $desc ); ?>">
                                            <?php echo esc_html( $var ); ?>
                                        </button>
                                    <?php endforeach; ?>
                                </div>
                                <p class="description" style="margin-top: 10px;">
                                    <strong><?php esc_html_e( 'Click a variable to insert it. You can also add static text.', 'digital-rise-seo-solutions' ); ?></strong><br>
                                    <?php esc_html_e( 'Example: "{site_name} {keyword} best quality" → "My Store Real Estate Tunisia Best Quality"', 'digital-rise-seo-solutions' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Word Limit', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <strong><?php esc_html_e( '7 words maximum', 'digital-rise-seo-solutions' ); ?></strong>
                                <p class="description"><?php esc_html_e( 'Google recommends short, descriptive image names and ALT text. The pattern result will be trimmed to 7 words.', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <?php if ( $this->plugin->has_feature( 'content_type_patterns' ) ) : ?>
                <div class="drss-card">
                    <h2>🎯 <?php esc_html_e( 'Content-Type-Specific Patterns', 'digital-rise-seo-solutions' ); ?> 
                        <span class="drss-pro-badge">PRO</span>
                    </h2>
                    <p><?php esc_html_e( 'Define different SEO patterns for different content types. If not set, the default pattern above will be used.', 'digital-rise-seo-solutions' ); ?></p>
                    
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Blog Posts', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="image_seo_pattern_post" value="<?php echo esc_attr( $this->plugin->get_option( 'image_seo_pattern_post', '' ) ); ?>" class="large-text" placeholder="<?php esc_attr_e( 'e.g., {site_name} {content_title} {category}', 'digital-rise-seo-solutions' ); ?>">
                                <p class="description"><?php esc_html_e( 'Pattern for images in blog posts', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Pages', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="image_seo_pattern_page" value="<?php echo esc_attr( $this->plugin->get_option( 'image_seo_pattern_page', '' ) ); ?>" class="large-text" placeholder="<?php esc_attr_e( 'e.g., {site_name} {content_title}', 'digital-rise-seo-solutions' ); ?>">
                                <p class="description"><?php esc_html_e( 'Pattern for images in pages', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Products (WooCommerce)', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="image_seo_pattern_product" value="<?php echo esc_attr( $this->plugin->get_option( 'image_seo_pattern_product', '' ) ); ?>" class="large-text" placeholder="<?php esc_attr_e( 'e.g., {content_title} {keyword} {site_name}', 'digital-rise-seo-solutions' ); ?>">
                                <p class="description"><?php esc_html_e( 'Pattern for product images', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
                <?php else : ?>
                <div class="drss-card drss-locked-feature">
                    <h2>🎯 <?php esc_html_e( 'Content-Type-Specific Patterns', 'digital-rise-seo-solutions' ); ?> 
                        <span class="drss-pro-badge">PRO</span>
                    </h2>
                    <div class="drss-pro-notice">
                        <span class="dashicons dashicons-lock"></span> 
                        <?php esc_html_e( 'Define different SEO patterns for posts, pages, and products. Upgrade to Pro or Agency to unlock this feature.', 'digital-rise-seo-solutions' ); ?>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-license' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Upgrade Now', 'digital-rise-seo-solutions' ); ?></a>
                    </div>
                </div>
                <?php endif; ?>
                
                <div class="drss-card">
                    <h2><?php esc_html_e( 'Auto Features', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'On Upload', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="auto_rename" <?php checked( $this->plugin->get_option( 'auto_rename', true ) ); ?>> <?php esc_html_e( 'Auto rename images (SEO-friendly filenames)', 'digital-rise-seo-solutions' ); ?></label><br>
                                <label><input type="checkbox" name="auto_alt" <?php checked( $this->plugin->get_option( 'auto_alt', true ) ); ?>> <?php esc_html_e( 'Auto generate ALT text', 'digital-rise-seo-solutions' ); ?></label><br>
                                <label><input type="checkbox" name="auto_title" <?php checked( $this->plugin->get_option( 'auto_title', true ) ); ?>> <?php esc_html_e( 'Auto generate Title', 'digital-rise-seo-solutions' ); ?></label><br>
                                <label><input type="checkbox" name="auto_description" <?php checked( $this->plugin->get_option( 'auto_description', true ) ); ?>> <?php esc_html_e( 'Auto generate Caption & Description', 'digital-rise-seo-solutions' ); ?></label>
                            </td>
                        </tr>
                    </table>
                </div>

                <script>
                jQuery(document).ready(function($) {
                    $('.drss-insert-var').on('click', function() {
                        var varText = $(this).data('var');
                        var $input = $('#image-seo-pattern');
                        var currentVal = $input.val();
                        $input.val(currentVal + (currentVal ? ' ' : '') + varText);
                        $input.focus();
                    });
                });
                </script>
                <style>
                .drss-variable-buttons { display: flex; flex-wrap: wrap; gap: 5px; }
                .drss-variable-buttons .button { font-family: monospace; }
                </style>

                <p class="submit"><input type="submit" name="drss_save_keywords" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-seo-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    public function render_plans() {
        $plans = new DRSS_Plans( $this->plugin );
        
        // Handle refresh prices
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Simple cache clear, no data modification
        if ( isset( $_GET['refresh_prices'] ) ) {
            $plans->clear_price_cache();
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Prices refreshed from server!', 'digital-rise-seo-solutions' ) . '</p></div>';
        }
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'Plans & Features', 'digital-rise-seo-solutions' ) ); ?>
            
            <?php if ( $this->plugin->current_plan === 'free' ) : ?>
            <!-- CUSTOM BRANDED UPGRADE NOTICE - ONLY FOR YOUR PLUGIN -->
            <div class="drss-card" style="background: linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%); color: white; padding: 25px; margin-bottom: 25px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; align-items: center; gap: 20px;">
                    <?php
                    $logo_url = DRSS_LOGO_URL;
                    if ( ! empty( $logo_url ) && @getimagesize( $logo_url ) ) : ?>
                        <img src="<?php echo esc_url( $logo_url ); ?>" alt="Digital Rise" style="height: 60px; border-radius: 8px; background: white; padding: 10px;">
                    <?php else : ?>
                        <div style="width: 60px; height: 60px; background: white; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 24px; font-weight: bold; color: #1e3a5f;">DR</div>
                    <?php endif; ?>
                    
                    <div style="flex: 1;">
                        <h2 style="color: white; margin: 0 0 8px 0; font-size: 24px;">
                            🚀 <?php esc_html_e( 'Upgrade to Pro!', 'digital-rise-seo-solutions' ); ?>
                        </h2>
                        <p style="margin: 0; font-size: 16px; opacity: 0.95;">
                            <?php esc_html_e( 'Unlock custom keywords, advanced SEO features, content-type patterns, and more.', 'digital-rise-seo-solutions' ); ?>
                        </p>
                    </div>
                    
                    <div>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-license' ) ); ?>" class="button button-hero" style="background: #ffd700; color: #1e3a5f; border: none; font-weight: bold; font-size: 16px; padding: 15px 30px; box-shadow: 0 4px 10px rgba(255, 215, 0, 0.3); transition: all 0.3s;" onmouseover="this.style.background='#ffed4e'; this.style.transform='translateY(-2px)'; this.style.boxShadow='0 6px 15px rgba(255, 215, 0, 0.5)';" onmouseout="this.style.background='#ffd700'; this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 10px rgba(255, 215, 0, 0.3)';">
                            ⭐ <?php esc_html_e( 'Upgrade Now', 'digital-rise-seo-solutions' ); ?>
                        </a>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <p style="margin-bottom: 20px;">
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-plans&refresh_prices=1' ) ); ?>" class="button">
                    <span class="dashicons dashicons-update" style="margin-top: 3px;"></span> <?php esc_html_e( 'Refresh Prices', 'digital-rise-seo-solutions' ); ?>
                </a>
            </p>
            
            <div class="drss-plans-grid">
                <?php foreach ( $plans->get_plans() as $plan_key => $plan ) : ?>
                    <div class="drss-plan-card <?php echo $this->plugin->current_plan === $plan_key ? 'drss-plan-current' : ''; ?>">
                        <h3><?php echo esc_html( $plan['name'] ); ?></h3>
                        <div class="drss-plan-price"><?php echo esc_html( $plan['price'] ); ?></div>
                        <?php if ( ! empty( $plan['price_note'] ) ) : ?>
                            <div class="drss-plan-price-note" style="color: #666; font-size: 14px; margin-top: -10px; margin-bottom: 15px;"><?php echo esc_html( $plan['price_note'] ); ?></div>
                        <?php endif; ?>
                        <p><?php echo esc_html( $plan['description'] ); ?></p>
                        <ul>
                            <?php foreach ( $plan['features'] as $feature ) : ?>
                                <li><span class="dashicons dashicons-yes"></span> <?php echo esc_html( $feature ); ?></li>
                            <?php endforeach; ?>
                            <?php if ( isset( $plan['limitations'] ) ) : ?>
                                <?php foreach ( $plan['limitations'] as $limitation ) : ?>
                                    <li class="limitation"><span class="dashicons dashicons-no"></span> <?php echo esc_html( $limitation ); ?></li>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </ul>
                        <?php if ( $this->plugin->current_plan === $plan_key ) : ?>
                            <span class="button disabled"><?php esc_html_e( 'Current Plan', 'digital-rise-seo-solutions' ); ?></span>
                        <?php elseif ( $plan_key !== 'free' ) : ?>
                            <a href="<?php echo esc_url( $plans->get_purchase_url() ); ?>" target="_blank" class="button button-primary"><?php esc_html_e( 'Upgrade', 'digital-rise-seo-solutions' ); ?></a>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }

    public function render_license() {
        $license = new DRSS_License( $this->plugin );
        $status = $license->get_status();
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'License & Upgrade', 'digital-rise-seo-solutions' ) ); ?>
            
            <div class="drss-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
                <h2 style="color: white;"><?php esc_html_e( 'Upgrade Your Plan', 'digital-rise-seo-solutions' ); ?></h2>
                <p style="font-size: 16px;"><?php esc_html_e( 'Unlock all features with Pro or Developer plan. Easy payment via USDT/Crypto.', 'digital-rise-seo-solutions' ); ?></p>
                <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" rel="noopener" class="button button-hero" style="background: #fff; color: #667eea; border: none; font-weight: bold; font-size: 16px; padding: 12px 30px;">
                    <span class="dashicons dashicons-cart" style="margin-top: 3px;"></span> <?php esc_html_e( 'Buy License Now', 'digital-rise-seo-solutions' ); ?>
                </a>
                <p style="margin-top: 15px; opacity: 0.9;"><?php esc_html_e( 'After payment, contact us to receive your license key:', 'digital-rise-seo-solutions' ); ?> <a href="mailto:<?php echo esc_attr( DRSS_SUPPORT_EMAIL ); ?>" style="color: white; font-weight: bold;"><?php echo esc_html( DRSS_SUPPORT_EMAIL ); ?></a></p>
            </div>

            <div class="drss-card">
                <h2><?php esc_html_e( 'License Activation', 'digital-rise-seo-solutions' ); ?></h2>
                
                <div class="drss-license-status">
                    <p><strong><?php esc_html_e( 'Status:', 'digital-rise-seo-solutions' ); ?></strong> 
                        <span class="drss-status-<?php echo esc_attr( $status['status'] ); ?>">
                            <?php echo esc_html( ucfirst( $status['status'] ) ); ?>
                        </span>
                    </p>
                    <p><strong><?php esc_html_e( 'Plan:', 'digital-rise-seo-solutions' ); ?></strong> <?php echo esc_html( ucfirst( $status['plan'] ) ); ?></p>
                    <?php if ( $status['key'] ) : ?>
                        <p><strong><?php esc_html_e( 'Key:', 'digital-rise-seo-solutions' ); ?></strong> <?php echo esc_html( $license->get_masked_key() ); ?></p>
                    <?php endif; ?>
                    <?php if ( $status['expires'] ) : ?>
                        <p><strong><?php esc_html_e( 'Expires:', 'digital-rise-seo-solutions' ); ?></strong> <?php echo esc_html( $status['expires'] ); ?></p>
                    <?php endif; ?>
                </div>

                <form id="license-form">
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'License Key', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="license_key" id="license_key" value="" class="regular-text" placeholder="<?php esc_attr_e( 'Enter your license key', 'digital-rise-seo-solutions' ); ?>">
                            </td>
                        </tr>
                    </table>
                    <p>
                        <button type="button" id="activate-license" class="button button-primary"><?php esc_html_e( 'Activate License', 'digital-rise-seo-solutions' ); ?></button>
                        <?php if ( $status['key'] ) : ?>
                            <button type="button" id="deactivate-license" class="button"><?php esc_html_e( 'Deactivate', 'digital-rise-seo-solutions' ); ?></button>
                        <?php endif; ?>
                    </p>
                </form>
                <div id="license-message"></div>
            </div>

            <div class="drss-card drss-affiliate-card">
                <h2><?php esc_html_e( 'Support Development', 'digital-rise-seo-solutions' ); ?></h2>
                <p><?php esc_html_e( 'If you find this plugin useful, please consider supporting its development.', 'digital-rise-seo-solutions' ); ?></p>
                <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" rel="noopener" class="button button-primary button-hero" style="background: #2ecc71; border-color: #27ae60;"><?php esc_html_e( 'Donate with USDT', 'digital-rise-seo-solutions' ); ?></a>
                <p style="margin-top: 15px;"><strong><?php esc_html_e( 'Need RedotPay for easy crypto payments?', 'digital-rise-seo-solutions' ); ?></strong></p>
                <a href="https://url.hk/i/en/4bq59" target="_blank" rel="noopener" class="button"><?php esc_html_e( 'Get RedotPay', 'digital-rise-seo-solutions' ); ?></a>
            </div>
        </div>
        <?php
    }

    public function render_settings() {
        if ( isset( $_POST['drss_save_settings'] ) && check_admin_referer( 'drss_settings_nonce' ) ) {
            $options = $this->plugin->options;
            $options['remote_connection'] = isset( $_POST['remote_connection'] );
            $options['show_attribution'] = isset( $_POST['show_attribution'] );
            $options['wc_integration'] = isset( $_POST['wc_integration'] );
            $options['dokan_integration'] = isset( $_POST['dokan_integration'] );
            $options['confirm_before_optimize'] = isset( $_POST['confirm_before_optimize'] );
            $options['convert_to_webp'] = isset( $_POST['convert_to_webp'] );
            $options['css_minify_enabled'] = isset( $_POST['css_minify_enabled'] );
            $options['js_minify_enabled'] = isset( $_POST['js_minify_enabled'] );
            $options['minify_exclude'] = isset( $_POST['minify_exclude'] ) ? sanitize_textarea_field( wp_unslash( $_POST['minify_exclude'] ) ) : '';
            $options['user_email'] = isset( $_POST['user_email'] ) ? sanitize_email( wp_unslash( $_POST['user_email'] ) ) : '';
            $options['user_whatsapp'] = isset( $_POST['user_whatsapp'] ) ? sanitize_text_field( wp_unslash( $_POST['user_whatsapp'] ) ) : '';
            $this->plugin->save_options( $options );
            
            // Sync contact info with server (like dashboard button)
            $remote = new DRSS_Remote( $this->plugin );
            $remote->do_checkin();
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved and synced!', 'digital-rise-seo-solutions' ) . '</p></div>';
        }
        
        // Handle cache clear
        if ( isset( $_POST['drss_clear_cache'] ) && check_admin_referer( 'drss_settings_nonce' ) ) {
            $minifier = new DRSS_Minifier( $this->plugin );
            $minifier->clear_cache();
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Cache cleared.', 'digital-rise-seo-solutions' ) . '</p></div>';
        }
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'Settings', 'digital-rise-seo-solutions' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'drss_settings_nonce' ); ?>
                
                <div class="drss-card">
                    <h2><?php esc_html_e( 'Image Optimization Behavior', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Confirm Before Optimizing', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="confirm_before_optimize" <?php checked( $this->plugin->get_option( 'confirm_before_optimize', false ) ); ?>> <?php esc_html_e( 'Ask before automatically optimizing images', 'digital-rise-seo-solutions' ); ?></label>
                                <p class="description"><?php esc_html_e( 'When enabled, images will NOT be auto-optimized on upload. Use Bulk Optimize to process images manually.', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Convert to WebP', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <?php 
                                $optimizer = new DRSS_Image_Optimizer( $this->plugin );
                                $webp_supported = $optimizer->supports_webp();
                                ?>
                                <label>
                                    <input type="checkbox" name="convert_to_webp" <?php checked( $this->plugin->get_option( 'convert_to_webp', true ) ); ?> <?php disabled( ! $webp_supported ); ?>>
                                    <?php esc_html_e( 'Convert all uploaded images to WebP format', 'digital-rise-seo-solutions' ); ?>
                                </label>
                                <?php if ( $webp_supported ) : ?>
                                    <p class="description" style="color: green;">✓ <?php esc_html_e( 'WebP is supported on your server.', 'digital-rise-seo-solutions' ); ?></p>
                                <?php else : ?>
                                    <p class="description" style="color: red;">✗ <?php esc_html_e( 'WebP is NOT supported. Please update GD library with WebP support.', 'digital-rise-seo-solutions' ); ?></p>
                                <?php endif; ?>
                                <p class="description"><?php esc_html_e( 'WebP provides 25-34% smaller file sizes compared to JPEG. Your image URLs will look like: yoursite.com/image-keyword.webp', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'CSS & JS Optimization', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Minify CSS', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="css_minify_enabled" <?php checked( $this->plugin->get_option( 'css_minify_enabled', false ) ); ?>> <?php esc_html_e( 'Enable CSS minification', 'digital-rise-seo-solutions' ); ?></label>
                                <p class="description"><?php esc_html_e( 'Removes whitespace and comments from CSS files to reduce file size.', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Minify JavaScript', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="js_minify_enabled" <?php checked( $this->plugin->get_option( 'js_minify_enabled', false ) ); ?>> <?php esc_html_e( 'Enable JavaScript minification', 'digital-rise-seo-solutions' ); ?></label>
                                <p class="description"><?php esc_html_e( 'Removes whitespace and comments from JS files. Use with caution.', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Exclude Handles', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="minify_exclude" rows="4" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'minify_exclude', '' ) ); ?></textarea>
                                <p class="description"><?php esc_html_e( 'Enter script/style handles to exclude from minification (one per line). jQuery core is always excluded.', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Cache', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <?php 
                                $minifier = new DRSS_Minifier( $this->plugin );
                                $cache_size = $minifier->get_cache_size();
                                // translators: %s is the cache size (e.g., "5 MB")
                                ?>
                                <p><?php printf( esc_html__( 'Cache size: %s', 'digital-rise-seo-solutions' ), esc_html( size_format( $cache_size ) ) ); ?></p>
                                <button type="submit" name="drss_clear_cache" class="button"><?php esc_html_e( 'Clear Cache', 'digital-rise-seo-solutions' ); ?></button>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="drss-card">
                    <h2><?php esc_html_e( 'Your Contact Information', 'digital-rise-seo-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Optional: Provide your contact information to receive updates and support.', 'digital-rise-seo-solutions' ); ?></p>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Email Address', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="email" name="user_email" value="<?php echo esc_attr( $this->plugin->get_option( 'user_email', '' ) ); ?>" class="regular-text" placeholder="your@email.com">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'WhatsApp Number', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="user_whatsapp" value="<?php echo esc_attr( $this->plugin->get_option( 'user_whatsapp', '' ) ); ?>" class="regular-text" placeholder="+1234567890">
                                <p class="description"><?php esc_html_e( 'Include country code (e.g., +1 for USA)', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'General', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Remote Connection', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="remote_connection" <?php checked( $this->plugin->get_option( 'remote_connection', true ) ); ?>> <?php esc_html_e( 'Enable connection to Digital Rise Solutions for license validation', 'digital-rise-seo-solutions' ); ?></label>
                                <p class="description"><?php esc_html_e( 'This is optional. Some features require this connection.', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Attribution', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="show_attribution" <?php checked( $this->plugin->get_option( 'show_attribution', true ) ); ?> <?php disabled( ! $this->plugin->has_feature( 'disable_attribution' ) && $this->plugin->current_plan === 'free' ); ?>> <?php esc_html_e( 'Show attribution link', 'digital-rise-seo-solutions' ); ?></label>
                                <?php if ( ! $this->plugin->has_feature( 'disable_attribution' ) ) : ?>
                                    <p class="description"><?php esc_html_e( 'Upgrade to Pro to disable attribution.', 'digital-rise-seo-solutions' ); ?></p>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'Integrations', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'WooCommerce', 'digital-rise-seo-solutions' ); ?></th>
                            <td><label><input type="checkbox" name="wc_integration" <?php checked( $this->plugin->get_option( 'wc_integration', true ) ); ?>> <?php esc_html_e( 'Enable WooCommerce integration', 'digital-rise-seo-solutions' ); ?></label></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Dokan', 'digital-rise-seo-solutions' ); ?></th>
                            <td><label><input type="checkbox" name="dokan_integration" <?php checked( $this->plugin->get_option( 'dokan_integration', true ) ); ?>> <?php esc_html_e( 'Enable Dokan integration', 'digital-rise-seo-solutions' ); ?></label></td>
                        </tr>
                    </table>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'System Info', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr><th><?php esc_html_e( 'Plugin Version', 'digital-rise-seo-solutions' ); ?></th><td><?php echo esc_html( DRSS_VERSION ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'WordPress', 'digital-rise-seo-solutions' ); ?></th><td><?php echo esc_html( get_bloginfo( 'version' ) ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'PHP', 'digital-rise-seo-solutions' ); ?></th><td><?php echo esc_html( phpversion() ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'GD Library', 'digital-rise-seo-solutions' ); ?></th><td><?php echo function_exists( 'imagecreatetruecolor' ) ? '✓' : '✗'; ?></td></tr>
                        <tr><th><?php esc_html_e( 'Installation ID', 'digital-rise-seo-solutions' ); ?></th><td><code><?php echo esc_html( $this->plugin->get_option( 'installation_id' ) ); ?></code></td></tr>
                    </table>
                </div>
                
                <div class="drss-card" style="background: linear-gradient(135deg, #2ecc71, #27ae60); color: #fff;">
                    <h2 style="color: #fff;"><?php esc_html_e( 'Support This Plugin', 'digital-rise-seo-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'If you find this plugin useful, please consider making a donation to support continued development.', 'digital-rise-seo-solutions' ); ?></p>
                    <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" rel="noopener" class="button button-hero" style="background: #fff; color: #27ae60; border: none;"><?php esc_html_e( 'Donate with USDT', 'digital-rise-seo-solutions' ); ?></a>
                </div>

                <p class="submit"><input type="submit" name="drss_save_settings" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-seo-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    // AJAX Handlers
    public function ajax_dismiss_notice() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        $notice = isset( $_POST['notice'] ) ? sanitize_text_field( wp_unslash( $_POST['notice'] ) ) : '';
        if ( $notice === 'upgrade_banner' ) {
            update_option( 'drss_upgrade_dismissed', true );
        } else {
            $remote = new DRSS_Remote( $this->plugin );
            $remote->dismiss_notice( $notice );
        }
        wp_send_json_success();
    }

    public function ajax_optimize_image() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        $attachment_id = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;
        
        $optimizer = new DRSS_Image_Optimizer( $this->plugin );
        $result = $optimizer->optimize_existing( $attachment_id );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result['message'] );
        }
    }

    public function ajax_bulk_optimize() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        $apply_seo = isset( $_POST['apply_seo'] ) && $_POST['apply_seo'] === 'true';
        $reoptimize = isset( $_POST['reoptimize'] ) && $_POST['reoptimize'] === 'true';
        
        $bulk = new DRSS_Bulk_Optimizer( $this->plugin );
        $result = $bulk->optimize_batch( 5, $apply_seo, $reoptimize );
        
        wp_send_json_success( $result );
    }

    public function ajax_reset_optimization() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        
        $bulk = new DRSS_Bulk_Optimizer( $this->plugin );
        $bulk->reset_optimization_status();
        
        wp_send_json_success( array( 'message' => __( 'All images marked as unoptimized.', 'digital-rise-seo-solutions' ) ) );
    }

    public function ajax_activate_license() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        $license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';
        
        $license = new DRSS_License( $this->plugin );
        $result = $license->activate_license( $license_key );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result['message'] );
        }
    }

    public function ajax_deactivate_license() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        
        $license = new DRSS_License( $this->plugin );
        $result = $license->deactivate_license();
        
        wp_send_json_success( $result );
    }

    public function ajax_get_optimization_progress() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        
        $bulk = new DRSS_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        
        wp_send_json_success( $progress );
    }

    public function ajax_sync_notices() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        
        $remote = new DRSS_Remote( $this->plugin );
        $remote->do_checkin();
        
        $notices = $remote->get_active_notices();
        
        wp_send_json_success( array( 
            'message' => __( 'Notices synced successfully.', 'digital-rise-seo-solutions' ),
            'count' => count( $notices ),
        ));
    }

    public function render_page_seo() {
        // Handle save
        if ( isset( $_POST['drss_save_page_seo'] ) && check_admin_referer( 'drss_page_seo_nonce' ) ) {
            $this->plugin->update_option( 'page_seo_enabled', isset( $_POST['page_seo_enabled'] ) );
            
            // Posts patterns
            $this->plugin->update_option( 'seo_title_post', isset( $_POST['seo_title_post'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_post'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_post', isset( $_POST['seo_meta_post'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_post'] ) ) : '' );
            
            // Pages patterns
            $this->plugin->update_option( 'seo_title_page', isset( $_POST['seo_title_page'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_page'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_page', isset( $_POST['seo_meta_page'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_page'] ) ) : '' );
            
            // Products patterns (WooCommerce)
            $this->plugin->update_option( 'seo_title_product', isset( $_POST['seo_title_product'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_product'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_product', isset( $_POST['seo_meta_product'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_product'] ) ) : '' );
            
            // Categories patterns
            $this->plugin->update_option( 'seo_title_category', isset( $_POST['seo_title_category'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_category'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_category', isset( $_POST['seo_meta_category'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_category'] ) ) : '' );
            
            // Tags patterns
            $this->plugin->update_option( 'seo_title_tag', isset( $_POST['seo_title_tag'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_tag'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_tag', isset( $_POST['seo_meta_tag'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_tag'] ) ) : '' );
            
            // Product categories patterns (WooCommerce)
            $this->plugin->update_option( 'seo_title_product_cat', isset( $_POST['seo_title_product_cat'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_product_cat'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_product_cat', isset( $_POST['seo_meta_product_cat'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_product_cat'] ) ) : '' );
            
            // Homepage patterns
            $this->plugin->update_option( 'seo_title_home', isset( $_POST['seo_title_home'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_home'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_home', isset( $_POST['seo_meta_home'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_home'] ) ) : '' );
            
            // Archive patterns
            $this->plugin->update_option( 'seo_title_archive', isset( $_POST['seo_title_archive'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_archive'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_archive', isset( $_POST['seo_meta_archive'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_archive'] ) ) : '' );
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-seo-solutions' ) . '</p></div>';
        }
        
        $page_seo = new DRSS_Page_SEO( $this->plugin );
        $variables = $page_seo->get_available_variables();
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'SEO Titles & Meta Descriptions', 'digital-rise-seo-solutions' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'drss_page_seo_nonce' ); ?>
                
                <div class="drss-card">
                    <h2><?php esc_html_e( 'General Settings', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Enable Page SEO', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="page_seo_enabled" <?php checked( $this->plugin->get_option( 'page_seo_enabled', true ) ); ?>>
                                    <?php esc_html_e( 'Automatically generate SEO title and meta description', 'digital-rise-seo-solutions' ); ?>
                                </label>
                                <p class="description"><?php esc_html_e( 'When enabled, the patterns below will be used unless overridden on individual content.', 'digital-rise-seo-solutions' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <style>
                    .drss-quick-vars-settings { 
                        margin-top: 8px; 
                        padding: 10px; 
                        background: #f9f9f9; 
                        border-left: 3px solid #2271b1; 
                        border-radius: 3px;
                    }
                    .drss-quick-vars-settings small { 
                        color: #646970; 
                        font-weight: 600; 
                        display: block; 
                        margin-bottom: 5px;
                    }
                    .drss-quick-vars-settings code { 
                        cursor: pointer; 
                        padding: 4px 8px; 
                        background: #fff; 
                        border-radius: 3px; 
                        margin: 3px; 
                        display: inline-block; 
                        color: #1d2327; 
                        border: 1px solid #ddd; 
                        transition: all 0.2s;
                        font-size: 12px;
                    }
                    .drss-quick-vars-settings code:hover { 
                        background: #2271b1; 
                        color: #fff; 
                        border-color: #2271b1; 
                        transform: translateY(-1px);
                        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                    }
                    .drss-quick-vars-settings code.spacer-sep { 
                        background: #e3f2fd; 
                        border-color: #2196f3;
                    }
                    .drss-quick-vars-settings code.spacer-sep:hover { 
                        background: #2196f3; 
                        color: #fff;
                    }
                </style>
                
                <script>
                jQuery(document).ready(function($) {
                    // Handle variable insertion in settings page
                    var lastFocusedField = null;
                    
                    // Track which field is focused
                    $('input[type="text"], textarea').on('focus', function() {
                        lastFocusedField = this;
                    });
                    
                    // Handle variable click
                    $(document).on('click', '.drss-var-insert', function(e) {
                        e.preventDefault();
                        var varText = $(this).data('var');
                        
                        if (lastFocusedField) {
                            var $field = $(lastFocusedField);
                            var field = lastFocusedField;
                            var currentVal = $field.val() || '';
                            var pos = field.selectionStart !== undefined ? field.selectionStart : currentVal.length;
                            
                            var newVal = currentVal.substring(0, pos) + varText + currentVal.substring(pos);
                            $field.val(newVal);
                            
                            // Update cursor position
                            $field.focus();
                            if (field.setSelectionRange) {
                                field.setSelectionRange(pos + varText.length, pos + varText.length);
                            }
                        }
                    });
                });
                </script>
                
                <!-- Homepage -->
                <div class="drss-card">
                    <h2><span class="dashicons dashicons-admin-home"></span> <?php esc_html_e( 'Homepage', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_home" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_home', '{site_name} - {site_tagline}' ) ); ?>" class="large-text">
                                <div class="drss-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-seo-solutions' ); ?></small>
                                    <code class="drss-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="drss-var-insert" data-var="{site_tagline}">{site_tagline}</code>
                                    <code class="drss-var-insert" data-var="{year}">{year}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="seo_meta_home" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_home', '{site_tagline}' ) ); ?></textarea>
                                <div class="drss-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-seo-solutions' ); ?></small>
                                    <code class="drss-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="drss-var-insert" data-var="{site_tagline}">{site_tagline}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Posts -->
                <div class="drss-card">
                    <h2><span class="dashicons dashicons-admin-post"></span> <?php esc_html_e( 'Posts', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_post" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_post', '{title} | {site_name}' ) ); ?>" class="large-text">
                                <div class="drss-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-seo-solutions' ); ?></small>
                                    <code class="drss-var-insert" data-var="{title}">{title}</code>
                                    <code class="drss-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="drss-var-insert" data-var="{category}">{category}</code>
                                    <code class="drss-var-insert" data-var="{author}">{author}</code>
                                    <code class="drss-var-insert" data-var="{date}">{date}</code>
                                    <code class="drss-var-insert" data-var="{year}">{year}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="seo_meta_post" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_post', '{excerpt}' ) ); ?></textarea>
                                <div class="drss-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-seo-solutions' ); ?></small>
                                    <code class="drss-var-insert" data-var="{excerpt}">{excerpt}</code>
                                    <code class="drss-var-insert" data-var="{title}">{title}</code>
                                    <code class="drss-var-insert" data-var="{category}">{category}</code>
                                    <code class="drss-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Pages -->
                <div class="drss-card">
                    <h2><span class="dashicons dashicons-admin-page"></span> <?php esc_html_e( 'Pages', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_page" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_page', '{title} | {site_name}' ) ); ?>" class="large-text">
                                <div class="drss-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-seo-solutions' ); ?></small>
                                    <code class="drss-var-insert" data-var="{title}">{title}</code>
                                    <code class="drss-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="drss-var-insert" data-var="{parent_title}">{parent_title}</code>
                                    <code class="drss-var-insert" data-var="{year}">{year}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="seo_meta_page" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_page', '{excerpt}' ) ); ?></textarea>
                                <div class="drss-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-seo-solutions' ); ?></small>
                                    <code class="drss-var-insert" data-var="{excerpt}">{excerpt}</code>
                                    <code class="drss-var-insert" data-var="{title}">{title}</code>
                                    <code class="drss-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>

                <?php if ( class_exists( 'WooCommerce' ) ) : ?>
                <!-- Products -->
                <div class="drss-card">
                    <h2><span class="dashicons dashicons-cart"></span> <?php esc_html_e( 'Products (WooCommerce)', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_product" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_product', '{title} - Buy Online | {site_name}' ) ); ?>" class="large-text">
                                <div class="drss-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-seo-solutions' ); ?></small>
                                    <code class="drss-var-insert" data-var="{title}">{title}</code>
                                    <code class="drss-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="drss-var-insert" data-var="{product_price}">{product_price}</code>
                                    <code class="drss-var-insert" data-var="{product_sku}">{product_sku}</code>
                                    <code class="drss-var-insert" data-var="{category}">{category}</code>
                                    <code class="drss-var-insert" data-var="{year}">{year}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="seo_meta_product" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_product', 'Buy {title} at best price. {excerpt}' ) ); ?></textarea>
                                <div class="drss-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-seo-solutions' ); ?></small>
                                    <code class="drss-var-insert" data-var="{excerpt}">{excerpt}</code>
                                    <code class="drss-var-insert" data-var="{title}">{title}</code>
                                    <code class="drss-var-insert" data-var="{product_price}">{product_price}</code>
                                    <code class="drss-var-insert" data-var="{category}">{category}</code>
                                    <code class="drss-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="drss-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <!-- Product Categories -->
                <div class="drss-card">
                    <h2><span class="dashicons dashicons-category"></span> <?php esc_html_e( 'Product Categories', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_product_cat" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_product_cat', '{title} - Shop {count} Products | {site_name}' ) ); ?>" class="large-text">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="seo_meta_product_cat" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_product_cat', 'Browse our {title} collection. {count} products available at {site_name}.' ) ); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>
                <?php endif; ?>

                <!-- Categories -->
                <div class="drss-card">
                    <h2><span class="dashicons dashicons-category"></span> <?php esc_html_e( 'Categories', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_category" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_category', '{title} Archives | {site_name}' ) ); ?>" class="large-text">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="seo_meta_category" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_category', 'Browse all posts in {title}. {count} articles available.' ) ); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Tags -->
                <div class="drss-card">
                    <h2><span class="dashicons dashicons-tag"></span> <?php esc_html_e( 'Tags', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_tag" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_tag', 'Posts Tagged "{title}" | {site_name}' ) ); ?>" class="large-text">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="seo_meta_tag" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_tag', 'All posts tagged with {title}. {count} articles available.' ) ); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Archives -->
                <div class="drss-card">
                    <h2><span class="dashicons dashicons-archive"></span> <?php esc_html_e( 'Date Archives', 'digital-rise-seo-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_archive" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_archive', 'Archives: {date} | {site_name}' ) ); ?>" class="large-text">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-seo-solutions' ); ?></th>
                            <td>
                                <textarea name="seo_meta_archive" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_archive', 'Archive of posts from {date} on {site_name}.' ) ); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'Available Variables', 'digital-rise-seo-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Use these variables in your patterns. They will be replaced with actual values.', 'digital-rise-seo-solutions' ); ?></p>
                    
                    <table class="widefat striped" style="max-width: 800px;">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Variable', 'digital-rise-seo-solutions' ); ?></th>
                                <th><?php esc_html_e( 'Description', 'digital-rise-seo-solutions' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $variables as $var => $desc ) : ?>
                                <tr>
                                    <td><code><?php echo esc_html( $var ); ?></code></td>
                                    <td><?php echo esc_html( $desc ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="drss-card">
                    <h2><?php esc_html_e( 'How It Works', 'digital-rise-seo-solutions' ); ?></h2>
                    <ol>
                        <li><strong><?php esc_html_e( 'Default patterns:', 'digital-rise-seo-solutions' ); ?></strong> <?php esc_html_e( 'Set above for each content type, applied automatically.', 'digital-rise-seo-solutions' ); ?></li>
                        <li><strong><?php esc_html_e( 'Custom override:', 'digital-rise-seo-solutions' ); ?></strong> <?php esc_html_e( 'Edit any post, page, or product and scroll down to find the "SEO Settings" meta box to override.', 'digital-rise-seo-solutions' ); ?></li>
                        <li><strong><?php esc_html_e( 'Priority:', 'digital-rise-seo-solutions' ); ?></strong> <?php esc_html_e( 'Custom values always take precedence over default patterns.', 'digital-rise-seo-solutions' ); ?></li>
                    </ol>
                </div>

                <p class="submit"><input type="submit" name="drss_save_page_seo" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-seo-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }
    
    
    /**
     * Render pending reviews page (hidden, accessed via direct URL)
     */
    public function render_reviews() {
        $reviews = get_option( 'drss_pending_reviews', array() );
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'Pending Reviews', 'digital-rise-seo-solutions' ) ); ?>
            
            <div class="drss-card">
                <h2><?php esc_html_e( 'Reviews Pending Email Delivery', 'digital-rise-seo-solutions' ); ?></h2>
                <p><?php esc_html_e( 'These reviews were submitted but email delivery failed. You can manually send them to seo.oussama@gmail.com', 'digital-rise-seo-solutions' ); ?></p>
                
                <?php if ( empty( $reviews ) ) : ?>
                    <p style="background: #d4edda; padding: 15px; border-left: 4px solid #28a745; border-radius: 4px;">
                        ✅ <?php esc_html_e( 'No pending reviews. All reviews were delivered successfully!', 'digital-rise-seo-solutions' ); ?>
                    </p>
                <?php else : ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Date', 'digital-rise-seo-solutions' ); ?></th>
                                <th><?php esc_html_e( 'Rating', 'digital-rise-seo-solutions' ); ?></th>
                                <th><?php esc_html_e( 'Review', 'digital-rise-seo-solutions' ); ?></th>
                                <th><?php esc_html_e( 'Website', 'digital-rise-seo-solutions' ); ?></th>
                                <th><?php esc_html_e( 'Contact', 'digital-rise-seo-solutions' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $reviews as $review ) : ?>
                                <tr>
                                    <td><?php echo esc_html( $review['date'] ); ?></td>
                                    <td><?php echo esc_html( str_repeat( '★', $review['rating'] ) . str_repeat( '☆', 5 - $review['rating'] ) ); ?></td>
                                    <td><?php echo esc_html( wp_trim_words( $review['review'], 20 ) ); ?></td>
                                    <td><a href="<?php echo esc_url( $review['site_url'] ); ?>" target="_blank"><?php echo esc_html( $review['site_url'] ); ?></a></td>
                                    <td>
                                        <?php if ( ! empty( $review['email'] ) ) : ?>
                                            📧 <a href="mailto:<?php echo esc_attr( $review['email'] ); ?>"><?php echo esc_html( $review['email'] ); ?></a><br>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $review['phone'] ) ) : ?>
                                            📞 <?php echo esc_html( $review['phone'] ); ?>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <p style="margin-top: 20px;">
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=drss-reviews&action=clear' ) ); ?>" class="button button-secondary" onclick="return confirm('<?php esc_attr_e( 'Clear all pending reviews?', 'digital-rise-seo-solutions' ); ?>');">
                            <?php esc_html_e( 'Clear All', 'digital-rise-seo-solutions' ); ?>
                        </a>
                    </p>
                <?php endif; ?>
            </div>
        </div>
        <?php
        
        // Handle clear action
        if ( isset( $_GET['action'] ) && $_GET['action'] === 'clear' ) {
            delete_option( 'drss_pending_reviews' );
            wp_safe_redirect( admin_url( 'admin.php?page=drss-reviews' ) );
            exit;
        }
    }
    
    /**
     * Handle review submission via AJAX
     */
    public function handle_review_submission() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'drss_review_nonce' ) ) {
            wp_send_json_error( __( 'Security check failed', 'digital-rise-seo-solutions' ) );
        }
        
        // Get review data
        $rating = isset( $_POST['rating'] ) ? intval( $_POST['rating'] ) : 0;
        $review = isset( $_POST['review'] ) ? sanitize_textarea_field( wp_unslash( $_POST['review'] ) ) : '';
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        $phone = isset( $_POST['phone'] ) ? sanitize_text_field( wp_unslash( $_POST['phone'] ) ) : '';
        $site_url = isset( $_POST['site_url'] ) ? esc_url_raw( wp_unslash( $_POST['site_url'] ) ) : '';
        
        // Validate
        if ( $rating < 1 || $rating > 5 || empty( $review ) ) {
            wp_send_json_error( __( 'Please provide a rating and review', 'digital-rise-seo-solutions' ) );
        }
        
        // Prepare email
        $to = 'seo.oussama@gmail.com';
        $subject = sprintf( 
            '[%s] Plugin Review - %d Stars', 
            wp_parse_url( $site_url, PHP_URL_HOST ), 
            $rating 
        );
        
        // Create star display
        $stars = str_repeat( '★', $rating ) . str_repeat( '☆', 5 - $rating );
        
        // Build HTML email
        $message = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; border-radius: 8px 8px 0 0; text-align: center; }
        .stars { font-size: 32px; color: #ffd700; margin: 15px 0; }
        .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 8px 8px; }
        .review-box { background: white; padding: 20px; border-left: 4px solid #667eea; margin: 20px 0; border-radius: 4px; }
        .info-row { padding: 10px 0; border-bottom: 1px solid #eee; }
        .label { font-weight: bold; color: #667eea; display: inline-block; width: 120px; }
        .footer { text-align: center; color: #999; margin-top: 30px; font-size: 12px; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1 style="margin: 0; color: white;">📧 New Plugin Review</h1>
            <div class="stars">' . esc_html( $stars ) . '</div>
            <p style="margin: 0; opacity: 0.9;">' . esc_html( $rating ) . ' out of 5 stars</p>
        </div>
        
        <div class="content">
            <h2 style="color: #667eea; margin-top: 0;">Review Details</h2>
            
            <div class="review-box">
                <p style="margin: 0; font-size: 16px; line-height: 1.8;">' . nl2br( esc_html( $review ) ) . '</p>
            </div>
            
            <h3 style="color: #667eea;">Contact Information</h3>
            <div class="info-row">
                <span class="label">🌐 Website:</span>
                <a href="' . esc_url( $site_url ) . '" style="color: #667eea; text-decoration: none;">' . esc_html( $site_url ) . '</a>
            </div>';
            
        if ( ! empty( $email ) ) {
            $message .= '
            <div class="info-row">
                <span class="label">📧 Email:</span>
                <a href="mailto:' . esc_attr( $email ) . '" style="color: #667eea; text-decoration: none;">' . esc_html( $email ) . '</a>
            </div>';
        }
        
        if ( ! empty( $phone ) ) {
            $message .= '
            <div class="info-row">
                <span class="label">📞 Phone:</span>
                <span>' . esc_html( $phone ) . '</span>
            </div>';
        }
        
        $message .= '
            <div class="footer">
                <p>Sent from Digital Rise Digital Rise SEO Solutions Plugin</p>
                <p>Received: ' . esc_html( current_time( 'F j, Y \a\t g:i a' ) ) . '</p>
            </div>
        </div>
    </div>
</body>
</html>';
        
        // Email headers
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_bloginfo( 'name' ) . ' <' . get_bloginfo( 'admin_email' ) . '>',
        );
        
        if ( ! empty( $email ) ) {
            $headers[] = 'Reply-To: ' . $email;
        }
        
        // Try to send email with wp_mail first
        $sent = wp_mail( $to, $subject, $message, $headers );
        
        // If wp_mail fails, try alternative method using PHP mail()
        if ( ! $sent ) {
            // Create simple text version as fallback
            $text_message = "New Plugin Review\n\n";
            $text_message .= "Rating: " . $stars . " (" . $rating . "/5)\n\n";
            $text_message .= "Review:\n" . $review . "\n\n";
            $text_message .= "Website: " . $site_url . "\n";
            if ( ! empty( $email ) ) {
                $text_message .= "Email: " . $email . "\n";
            }
            if ( ! empty( $phone ) ) {
                $text_message .= "Phone: " . $phone . "\n";
            }
            $text_message .= "\nReceived: " . current_time( 'F j, Y \a\t g:i a' );
            
            // Build headers for PHP mail
            $simple_headers = "From: " . get_bloginfo( 'name' ) . " <" . get_bloginfo( 'admin_email' ) . ">\r\n";
            $simple_headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
            if ( ! empty( $email ) ) {
                $simple_headers .= "Reply-To: " . $email . "\r\n";
            }
            
            // Try PHP mail as fallback
            $sent = mail( $to, $subject, $text_message, $simple_headers );
        }
        
        // If still failed, save to database as backup
        if ( ! $sent ) {
            $review_data = array(
                'rating' => $rating,
                'review' => $review,
                'email' => $email,
                'phone' => $phone,
                'site_url' => $site_url,
                'date' => current_time( 'mysql' ),
            );
            
            // Save to options table
            $reviews = get_option( 'drss_pending_reviews', array() );
            $reviews[] = $review_data;
            update_option( 'drss_pending_reviews', $reviews );
            
            // Still return success to user but log the issue
            error_log( 'DRIOP Review: Email failed but saved to database. Review from: ' . $site_url );
            wp_send_json_success( __( 'Thank you! Your review has been recorded and will be sent shortly.', 'digital-rise-seo-solutions' ) );
        }
        
        if ( $sent ) {
            wp_send_json_success( __( 'Thank you for your review!', 'digital-rise-seo-solutions' ) );
        } else {
            wp_send_json_error( __( 'Failed to send review. Please try again.', 'digital-rise-seo-solutions' ) );
        }
    }

    // ==================== SEO AUDIT ====================

    public function render_seo_audit() {
        $audit = new DRSS_SEO_Audit( $this->plugin );
        $last_results = get_option( 'drss_audit_results', array() );
        $last_site_checks = get_option( 'drss_audit_site_checks', array() );
        $last_scan_time = get_option( 'drss_audit_last_scan', '' );
        $global_score = 0;
        if ( ! empty( $last_results ) && ! empty( $last_site_checks ) ) {
            $global_score = $audit->calculate_global_score( $last_results, $last_site_checks );
        }
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'SEO Audit', 'digital-rise-seo-solutions' ) ); ?>
            <div class="drss-audit-score-card" style="display:flex;align-items:center;gap:30px;background:#fff;padding:30px;border-radius:12px;box-shadow:0 2px 12px rgba(0,0,0,.08);margin-bottom:25px;">
                <div style="width:120px;height:120px;border-radius:50%;display:flex;flex-direction:column;align-items:center;justify-content:center;font-weight:700;border:6px solid <?php echo $global_score >= 80 ? '#4CAF50' : ($global_score >= 50 ? '#ff9800' : '#f44336'); ?>;">
                    <span style="font-size:36px;line-height:1;" id="global-score"><?php echo esc_html( $global_score ); ?></span>
                    <span style="font-size:12px;color:#666;">/100</span>
                </div>
                <div style="flex:1;">
                    <h2 style="margin:0 0 8px;"><?php esc_html_e( 'Website SEO Score', 'digital-rise-seo-solutions' ); ?></h2>
                    <p style="color:#666;margin:0 0 15px;">
                        <?php if ( $last_scan_time ) : ?>
                            <?php printf( esc_html__( 'Last scan: %s — %d URLs analyzed', 'digital-rise-seo-solutions' ), esc_html( human_time_diff( strtotime( $last_scan_time ) ) . ' ago' ), count( $last_results ) ); ?>
                        <?php else : ?>
                            <?php esc_html_e( 'No scan yet. Run your first SEO audit now!', 'digital-rise-seo-solutions' ); ?>
                        <?php endif; ?>
                    </p>
                    <button type="button" id="drss-run-audit" class="button button-primary button-hero">🔍 <?php esc_html_e( 'Run SEO Scan', 'digital-rise-seo-solutions' ); ?></button>
                    <div id="drss-scan-progress" style="display:none;margin-top:15px;">
                        <div style="background:#e0e0e0;height:20px;border-radius:10px;overflow:hidden;"><div id="scan-progress-fill" style="background:linear-gradient(90deg,#4CAF50,#8BC34A);height:100%;width:0%;transition:width .3s;"></div></div>
                        <p id="scan-progress-text" style="margin:8px 0 0;"><?php esc_html_e( 'Starting scan...', 'digital-rise-seo-solutions' ); ?></p>
                    </div>
                </div>
            </div>

            <?php if ( ! empty( $last_site_checks ) ) : ?>
            <div class="drss-card"><h2>🌐 <?php esc_html_e( 'Site-Wide Technical Checks', 'digital-rise-seo-solutions' ); ?></h2>
            <table class="wp-list-table widefat fixed striped"><thead><tr><th><?php esc_html_e( 'Check', 'digital-rise-seo-solutions' ); ?></th><th style="width:100px"><?php esc_html_e( 'Status', 'digital-rise-seo-solutions' ); ?></th></tr></thead><tbody>
            <?php foreach ( $last_site_checks as $check ) : ?>
                <tr><td><?php echo esc_html( $check['message'] ); ?></td>
                <td><span class="drss-status-badge" style="background:<?php echo $check['status']==='pass'?'#4CAF50':($check['status']==='warning'?'#ff9800':'#f44336'); ?>;color:#fff;padding:4px 10px;border-radius:12px;font-size:12px;"><?php echo esc_html( ucfirst( $check['status'] ) ); ?></span></td></tr>
            <?php endforeach; ?>
            </tbody></table></div>
            <?php endif; ?>

            <?php if ( ! empty( $last_results ) ) : ?>
            <div class="drss-card"><h2>📊 <?php esc_html_e( 'Per-Page SEO Results', 'digital-rise-seo-solutions' ); ?></h2>
            <table class="wp-list-table widefat fixed striped" id="drss-audit-results"><thead><tr>
                <th style="width:70px"><?php esc_html_e( 'Score', 'digital-rise-seo-solutions' ); ?></th>
                <th><?php esc_html_e( 'URL', 'digital-rise-seo-solutions' ); ?></th>
                <th style="width:80px"><?php esc_html_e( 'Type', 'digital-rise-seo-solutions' ); ?></th>
                <th style="width:80px"><?php esc_html_e( 'Errors', 'digital-rise-seo-solutions' ); ?></th>
                <th style="width:90px"><?php esc_html_e( 'Warnings', 'digital-rise-seo-solutions' ); ?></th>
                <th style="width:80px"><?php esc_html_e( 'Details', 'digital-rise-seo-solutions' ); ?></th>
            </tr></thead><tbody>
            <?php foreach ( $last_results as $i => $r ) : 
                $color = $r['score'] >= 80 ? '#4CAF50' : ($r['score'] >= 50 ? '#ff9800' : '#f44336');
            ?>
                <tr class="drss-url-row" data-index="<?php echo esc_attr( $i ); ?>">
                    <td><span style="display:inline-block;width:42px;height:42px;border-radius:50%;background:<?php echo esc_attr( $color ); ?>;color:#fff;text-align:center;line-height:42px;font-weight:700;font-size:14px;"><?php echo esc_html( $r['score'] ); ?></span></td>
                    <td style="word-break:break-all;"><a href="<?php echo esc_url( $r['url'] ); ?>" target="_blank"><?php echo esc_html( str_replace( home_url(), '', $r['url'] ) ?: '/' ); ?></a></td>
                    <td><span style="background:#f0f0f0;padding:3px 8px;border-radius:10px;font-size:11px;"><?php echo esc_html( $r['type'] ); ?></span></td>
                    <td><span style="color:#f44336;font-weight:600;"><?php echo count( $r['errors'] ); ?></span></td>
                    <td><span style="color:#ff9800;font-weight:600;"><?php echo count( $r['warnings'] ); ?></span></td>
                    <td><button type="button" class="button button-small drss-expand-url" data-index="<?php echo esc_attr( $i ); ?>">▶</button></td>
                </tr>
                <tr class="drss-url-details" id="drss-details-<?php echo esc_attr( $i ); ?>" style="display:none;">
                    <td colspan="6" style="padding:20px;background:#f9f9f9;">
                        <?php if ( ! empty( $r['errors'] ) ) : ?>
                            <div style="margin-bottom:12px;"><strong style="color:#f44336;">❌ <?php esc_html_e( 'Errors:', 'digital-rise-seo-solutions' ); ?></strong>
                            <ul style="margin:5px 0 0 20px;"><?php foreach ( $r['errors'] as $e ) : ?><li><?php echo esc_html( $e ); ?></li><?php endforeach; ?></ul></div>
                        <?php endif; ?>
                        <?php if ( ! empty( $r['warnings'] ) ) : ?>
                            <div style="margin-bottom:12px;"><strong style="color:#ff9800;">⚠️ <?php esc_html_e( 'Warnings:', 'digital-rise-seo-solutions' ); ?></strong>
                            <ul style="margin:5px 0 0 20px;"><?php foreach ( $r['warnings'] as $w ) : ?><li><?php echo esc_html( $w ); ?></li><?php endforeach; ?></ul></div>
                        <?php endif; ?>
                        <?php if ( ! empty( $r['passed'] ) ) : ?>
                            <div style="margin-bottom:12px;"><strong style="color:#4CAF50;">✅ <?php esc_html_e( 'Passed:', 'digital-rise-seo-solutions' ); ?></strong>
                            <ul style="margin:5px 0 0 20px;"><?php foreach ( $r['passed'] as $p ) : ?><li><?php echo esc_html( $p ); ?></li><?php endforeach; ?></ul></div>
                        <?php endif; ?>
                        <?php if ( ! empty( $r['recommendations'] ) ) : ?>
                            <div><strong style="color:#2196F3;">💡 <?php esc_html_e( 'Recommendations:', 'digital-rise-seo-solutions' ); ?></strong>
                            <ul style="margin:5px 0 0 20px;"><?php foreach ( $r['recommendations'] as $rec ) : ?><li><?php echo esc_html( $rec ); ?></li><?php endforeach; ?></ul></div>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
            </tbody></table></div>
            <?php endif; ?>

            <div class="drss-card">
                <h2>📐 <?php esc_html_e( 'Scoring Logic', 'digital-rise-seo-solutions' ); ?></h2>
                <p><?php esc_html_e( 'Each URL is scored 0-100 based on weighted checks. The global score is 70% page average + 30% site-wide checks.', 'digital-rise-seo-solutions' ); ?></p>
                <table class="widefat striped" style="max-width:600px;"><thead><tr><th><?php esc_html_e( 'Check', 'digital-rise-seo-solutions' ); ?></th><th style="width:80px"><?php esc_html_e( 'Weight', 'digital-rise-seo-solutions' ); ?></th></tr></thead><tbody>
                <?php 
                $weights = DRSS_SEO_Audit::WEIGHTS;
                arsort( $weights );
                foreach ( $weights as $k => $w ) : ?>
                    <tr><td><?php echo esc_html( ucwords( str_replace( '_', ' ', $k ) ) ); ?></td><td><strong><?php echo esc_html( $w ); ?>%</strong></td></tr>
                <?php endforeach; ?>
                </tbody></table>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            // Expand/collapse URL details
            $(document).on('click', '.drss-expand-url', function() {
                var idx = $(this).data('index');
                var $details = $('#drss-details-' + idx);
                $details.toggle();
                $(this).text($details.is(':visible') ? '▼' : '▶');
            });

            // Run SEO Scan
            $('#drss-run-audit').on('click', function() {
                var $btn = $(this);
                $btn.prop('disabled', true).text('<?php esc_html_e( 'Scanning...', 'digital-rise-seo-solutions' ); ?>');
                $('#drss-scan-progress').show();

                $.ajax({
                    url: drss_admin.ajax_url,
                    type: 'POST',
                    data: { action: 'drss_run_seo_audit', nonce: drss_admin.nonce },
                    timeout: 300000,
                    success: function(response) {
                        if (response.success) {
                            $('#scan-progress-fill').css('width', '100%');
                            $('#scan-progress-text').text('<?php esc_html_e( 'Complete! Reloading...', 'digital-rise-seo-solutions' ); ?>');
                            setTimeout(function() { location.reload(); }, 1000);
                        } else {
                            alert(response.data || 'Error');
                            $btn.prop('disabled', false).text('🔍 <?php esc_html_e( 'Run SEO Scan', 'digital-rise-seo-solutions' ); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php esc_html_e( 'Scan failed. Try again.', 'digital-rise-seo-solutions' ); ?>');
                        $btn.prop('disabled', false).text('🔍 <?php esc_html_e( 'Run SEO Scan', 'digital-rise-seo-solutions' ); ?>');
                    }
                });
            });
        });
        </script>
        <?php
    }

    public function ajax_run_seo_audit() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        set_time_limit( 300 );
        
        $audit = new DRSS_SEO_Audit( $this->plugin );
        $urls = $audit->collect_urls();
        $results = array();
        
        foreach ( $urls as $url_data ) {
            $results[] = $audit->analyze_url( $url_data );
        }
        
        $site_checks = $audit->run_site_checks();
        $global_score = $audit->calculate_global_score( $results, $site_checks );
        
        update_option( 'drss_audit_results', $results );
        update_option( 'drss_audit_site_checks', $site_checks );
        update_option( 'drss_audit_last_scan', current_time( 'mysql' ) );
        
        wp_send_json_success( array(
            'global_score' => $global_score,
            'total_urls'   => count( $results ),
            'message'      => __( 'Scan complete!', 'digital-rise-seo-solutions' ),
        ));
    }

    public function ajax_get_audit_status() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        $results = get_option( 'drss_audit_results', array() );
        wp_send_json_success( array( 'total' => count( $results ) ) );
    }

    public function ajax_get_url_details() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        $index = isset( $_POST['index'] ) ? absint( $_POST['index'] ) : 0;
        $results = get_option( 'drss_audit_results', array() );
        if ( isset( $results[ $index ] ) ) {
            wp_send_json_success( $results[ $index ] );
        }
        wp_send_json_error( 'Not found' );
    }

    // ==================== SEO CHECKLIST ====================

    public function render_seo_checklist() {
        $checklist_obj = new DRSS_SEO_Checklist( $this->plugin );
        $checklist = $checklist_obj->get_checklist();
        $progress = $checklist_obj->get_progress();
        $stats = $checklist_obj->get_stats();
        ?>
        <div class="wrap drss-wrap">
            <?php $this->render_header( __( 'SEO Checklist', 'digital-rise-seo-solutions' ) ); ?>
            
            <div style="display:flex;gap:20px;align-items:center;background:#fff;padding:25px;border-radius:12px;box-shadow:0 2px 12px rgba(0,0,0,.08);margin-bottom:25px;">
                <div style="width:100px;height:100px;border-radius:50%;background:conic-gradient(#4CAF50 0% <?php echo esc_attr( $stats['percentage'] ); ?>%, #e0e0e0 <?php echo esc_attr( $stats['percentage'] ); ?>% 100%);display:flex;align-items:center;justify-content:center;">
                    <div style="width:80px;height:80px;border-radius:50%;background:#fff;display:flex;align-items:center;justify-content:center;font-size:22px;font-weight:700;"><?php echo esc_html( $stats['percentage'] ); ?>%</div>
                </div>
                <div>
                    <h2 style="margin:0 0 5px;"><?php esc_html_e( 'SEO Checklist Progress', 'digital-rise-seo-solutions' ); ?></h2>
                    <p style="color:#666;margin:0;"><?php printf( esc_html__( '%d of %d tasks completed (%d remaining)', 'digital-rise-seo-solutions' ), $stats['completed'], $stats['total'], $stats['remaining'] ); ?></p>
                </div>
            </div>

            <?php foreach ( $checklist as $cat_key => $category ) : ?>
            <div class="drss-card" style="margin-bottom:15px;">
                <h2><span class="dashicons <?php echo esc_attr( $category['icon'] ); ?>" style="margin-right:8px;"></span> <?php echo esc_html( $category['title'] ); ?></h2>
                <div class="drss-checklist-items">
                <?php foreach ( $category['items'] as $item ) : 
                    $is_done = isset( $progress[ $item['id'] ] ) && $progress[ $item['id'] ]['completed'];
                    $priority_colors = array( 'high' => '#f44336', 'medium' => '#ff9800', 'low' => '#9e9e9e' );
                    $pcolor = isset( $priority_colors[ $item['priority'] ] ) ? $priority_colors[ $item['priority'] ] : '#9e9e9e';
                ?>
                    <label class="drss-checklist-item" style="display:flex;align-items:center;gap:12px;padding:10px 15px;border-bottom:1px solid #f0f0f0;cursor:pointer;<?php echo $is_done ? 'opacity:.6;text-decoration:line-through;' : ''; ?>">
                        <input type="checkbox" class="drss-checklist-toggle" data-id="<?php echo esc_attr( $item['id'] ); ?>" <?php checked( $is_done ); ?> style="width:18px;height:18px;">
                        <span style="flex:1;"><?php echo esc_html( $item['text'] ); ?></span>
                        <span style="font-size:10px;padding:2px 8px;border-radius:10px;background:<?php echo esc_attr( $pcolor ); ?>;color:#fff;text-transform:uppercase;"><?php echo esc_html( $item['priority'] ); ?></span>
                        <?php if ( $is_done && isset( $progress[ $item['id'] ]['date'] ) ) : ?>
                            <span style="font-size:11px;color:#999;"><?php echo esc_html( human_time_diff( strtotime( $progress[ $item['id'] ]['date'] ) ) . ' ago' ); ?></span>
                        <?php endif; ?>
                    </label>
                <?php endforeach; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>

        <script>
        jQuery(document).ready(function($) {
            $('.drss-checklist-toggle').on('change', function() {
                var $this = $(this);
                var itemId = $this.data('id');
                var completed = $this.is(':checked');
                var $label = $this.closest('.drss-checklist-item');
                
                $label.css({ opacity: completed ? '.6' : '1', textDecoration: completed ? 'line-through' : 'none' });
                
                $.post(drss_admin.ajax_url, {
                    action: 'drss_toggle_checklist_item',
                    nonce: drss_admin.nonce,
                    item_id: itemId,
                    completed: completed ? 1 : 0
                });
            });
        });
        </script>
        <?php
    }

    public function ajax_toggle_checklist_item() {
        check_ajax_referer( 'drss_admin_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }
        $item_id = isset( $_POST['item_id'] ) ? sanitize_text_field( wp_unslash( $_POST['item_id'] ) ) : '';
        $completed = isset( $_POST['completed'] ) && $_POST['completed'] === '1';
        
        $checklist = new DRSS_SEO_Checklist( $this->plugin );
        $progress = $checklist->save_item( $item_id, $completed );
        
        wp_send_json_success( array( 'progress' => $checklist->get_stats() ) );
    }

    private function get_score_class( $score ) {
        if ( $score >= 80 ) return 'drss-score-good';
        if ( $score >= 50 ) return 'drss-score-ok';
        return 'drss-score-bad';
    }
}
