<?php
/**
 * Class DRPS_Shortcode
 * Registers the [drps_posts] shortcode.
 *
 * Resolution order (highest priority first):
 *   1. Per-post meta box settings (if non-empty)
 *   2. Global admin settings (always present after activation)
 *
 * The shortcode itself takes no user-editable attributes — everything
 * is controlled visually via admin settings or the per-post meta box.
 *
 * @package DigitalRisePostsSolutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRPS_Shortcode {

    public function __construct() {
        add_shortcode( 'drps_posts', array( $this, 'shortcode_handler' ) );
    }

    /**
     * Shortcode handler — thin wrapper that calls the static render method.
     */
    public function shortcode_handler( $atts, $content = '', $shortcode_tag = '' ) {
        return self::render( $atts, $content, get_the_ID() );
    }

    // ── Static render (also called by AJAX live preview) ─────────────────
    /**
     * Main render function.
     * @param array  $atts     Shortcode attributes (unused — kept for WP compat).
     * @param string $content  Shortcode inner content (unused).
     * @param int    $post_id  The current post/page ID (0 for admin preview).
     * @return string          Rendered HTML.
     */
    public static function render( $atts = array(), $content = '', $post_id = 0 ) {
        // 1. Resolve effective settings
        $s = self::resolve_settings( $post_id );

        // 2. Build WP_Query args
        $query_args = array(
            'post_type'      => 'post',
            'post_status'    => 'publish',
            'posts_per_page' => 20, // fetch enough for carousel/grid
            'orderby'        => 'date',
            'order'          => 'DESC',
        );

        // Category filters
        if ( ! empty( $s['include_cats'] ) ) {
            $query_args['category__in'] = self::parse_ids( $s['include_cats'] );
        }
        if ( ! empty( $s['exclude_cats'] ) ) {
            $query_args['category__not_in'] = self::parse_ids( $s['exclude_cats'] );
        }
        // Tag filters
        if ( ! empty( $s['include_tags'] ) ) {
            $query_args['tag__in'] = self::parse_ids( $s['include_tags'] );
        }
        if ( ! empty( $s['exclude_tags'] ) ) {
            $query_args['tag__not_in'] = self::parse_ids( $s['exclude_tags'] );
        }

        $query = new WP_Query( $query_args );

        if ( ! $query->have_posts() ) {
            return '<div class="drps-no-posts"><p>' . esc_html__( 'No posts found.', DRPS_TEXT_DOMAIN ) . '</p></div>';
        }

        // 3. Build HTML
        $layout = $s['layout'];
        $html   = '';

        if ( 'carousel' === $layout ) {
            $html = self::render_carousel( $query, $s );
        } else {
            $html = self::render_grid( $query, $s );
        }

        $query->rewind_posts();

        return $html;
    }

    // ── Resolve settings: meta overrides global ──────────────────────────
    /**
     * For each setting, use the per-post meta value if it is non-empty,
     * otherwise fall back to the global option.
     *
     * @param int $post_id
     * @return array Resolved settings.
     */
    public static function resolve_settings( $post_id = 0 ) {
        $fields = array(
            'layout',
            'loop',
            'autoplay_speed',
            'posts_per_view',
            'include_cats',
            'exclude_cats',
            'include_tags',
            'exclude_tags',
        );

        $resolved = array();
        foreach ( $fields as $field ) {
            $meta_value  = ( $post_id > 0 ) ? get_post_meta( $post_id, 'drps_meta_' . $field, true ) : '';
            $global_value = get_option( 'drps_' . $field, '' );

            // Use meta if non-empty, else global
            $resolved[ $field ] = ( '' !== $meta_value ) ? $meta_value : $global_value;
        }

        // Ensure loop is boolean-ish
        $resolved['loop'] = ( '1' === $resolved['loop'] );

        return $resolved;
    }

    // ── Carousel HTML ────────────────────────────────────────────────────
    private static function render_carousel( $query, $s ) {
        $ppv = absint( $s['posts_per_view'] );
        if ( $ppv < 1 ) { $ppv = 3; }

        $html = '<div class="drps-carousel-wrapper" data-loop="' . ( $s['loop'] ? '1' : '0' ) . '" data-speed="' . absint( $s['autoplay_speed'] ) . '" data-ppv="' . $ppv . '">';
        $html .= '<div class="drps-carousel-track">';

        while ( $query->have_posts() ) {
            $query->the_post();
            $html .= self::render_post_card( get_the_ID() );
        }
        wp_reset_postdata();

        $html .= '</div>'; // track

        // Navigation arrows
        $html .= '<button class="drps-carousel-prev" aria-label="' . esc_attr__( 'Previous', DRPS_TEXT_DOMAIN ) . '">&#10094;</button>';
        $html .= '<button class="drps-carousel-next" aria-label="' . esc_attr__( 'Next', DRPS_TEXT_DOMAIN ) . '">&#10095;</button>';

        $html .= '</div>'; // wrapper
        return $html;
    }

    // ── Grid HTML ────────────────────────────────────────────────────────
    private static function render_grid( $query, $s ) {
        $html = '<div class="drps-grid">';

        while ( $query->have_posts() ) {
            $query->the_post();
            $html .= self::render_post_card( get_the_ID() );
        }
        wp_reset_postdata();

        $html .= '</div>';
        return $html;
    }

    // ── Single post card (shared by carousel & grid) ────────────────────
    /**
     * Renders one post card: thumbnail + title + 45-word excerpt.
     * All output is properly escaped.
     */
    private static function render_post_card( $post_id ) {
        $post  = get_post( $post_id );
        $title = get_the_title( $post_id );
        $url   = get_permalink( $post_id );

        // Thumbnail
        $thumb_html = '';
        if ( has_post_thumbnail( $post_id ) ) {
            $thumb_html = get_the_post_thumbnail( $post_id, 'medium', array( 'class' => 'drps-post-thumb' ) );
        } else {
            // Placeholder if no thumbnail
            $thumb_html = '<img src="' . esc_url( DRPS_PLUGIN_URL . 'assets/images/placeholder.svg' ) . '" alt="" class="drps-post-thumb drps-placeholder" />';
        }

        // Excerpt — limited to 45 words, no tags
        $excerpt = self::get_excerpt( $post, 45 );

        $html  = '<div class="drps-post-card">';
        $html .= '<a href="' . esc_url( $url ) . '" class="drps-post-link" aria-label="' . esc_attr( $title ) . '">';
        $html .= '<div class="drps-post-thumb-wrap">' . $thumb_html . '</div>';
        $html .= '</a>';
        $html .= '<div class="drps-post-content">';
        $html .= '<a href="' . esc_url( $url ) . '" class="drps-post-title-link"><h3 class="drps-post-title">' . esc_html( $title ) . '</h3></a>';
        $html .= '<p class="drps-post-excerpt">' . esc_html( $excerpt ) . '</p>';
        $html .= '</div>';
        $html .= '</div>';

        return $html;
    }

    // ── Helper: 45-word excerpt ──────────────────────────────────────────
    /**
     * Generates a clean, tag-free excerpt limited to $max_words words.
     * Appends "…" if truncated.
     */
    private static function get_excerpt( $post, $max_words = 45 ) {
        // Use manual excerpt if set, otherwise strip tags from content
        $text = ( '' !== $post->post_excerpt )
            ? $post->post_excerpt
            : strip_tags( $post->post_content );

        // Collapse whitespace
        $text = preg_replace( '/\s+/', ' ', trim( $text ) );

        $words = explode( ' ', $text );
        if ( count( $words ) <= $max_words ) {
            return implode( ' ', $words );
        }
        return implode( ' ', array_slice( $words, 0, $max_words ) ) . ' \u{2026}'; // unicode ellipsis
    }

    // ── Helper: parse comma-separated IDs ───────────────────────────────
    private static function parse_ids( $value ) {
        return array_filter(
            array_map( 'absint', explode( ',', $value ) )
        );
    }
}
