<?php
/**
 * Class DRPS_Metabox
 * Adds a "DR Posts Settings" meta box to the Page and Post editors.
 * Each field overrides the corresponding global setting when the shortcode
 * [drps_posts] is rendered on that page/post.
 *
 * @package DigitalRisePostsSolutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRPS_Metabox {

    /** Meta key prefix used for all per-post overrides */
    const PREFIX = 'drps_meta_';

    public function __construct() {
        add_action( 'add_meta_boxes',      array( $this, 'add_meta_box' ) );
        add_action( 'save_post',          array( $this, 'save_meta' ), 10, 3 );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_metabox_assets' ) );
    }

    // ── Register meta box on pages & posts ───────────────────────────────
    public function add_meta_box() {
        $screens = array( 'page', 'post' );
        foreach ( $screens as $screen ) {
            add_meta_box(
                'drps_meta_box',                                         // ID
                esc_html__( 'DR Posts Settings', DRPS_TEXT_DOMAIN ),    // Title
                array( $this, 'render_meta_box' ),                      // Callback
                $screen,                                                 // Screen
                'sidebar',                                               // Context
                'high'                                                   // Priority
            );
        }
    }

    // ── Enqueue metabox-specific JS (autocomplete) on post editor screens ─
    public function enqueue_metabox_assets( $hook ) {
        // Only on new/edit post screens
        if ( ! in_array( $hook, array( 'post.php', 'post-new.php' ), true ) ) {
            return;
        }
        wp_enqueue_script(
            'drps-metabox-js',
            DRPS_PLUGIN_URL . 'assets/js/metabox.js',
            array( 'jquery' ),
            DRPS_VERSION,
            true
        );
        wp_enqueue_style(
            'drps-metabox-css',
            DRPS_PLUGIN_URL . 'assets/css/metabox.css',
            array(),
            DRPS_VERSION
        );
        wp_localize_script(
            'drps-metabox-js',
            'drps_metabox',
            array(
                'ajax_url' => esc_url( admin_url( 'admin-ajax.php' ) ),
                'nonce'    => wp_create_nonce( 'drps_autocomplete_nonce' ),
            )
        );
    }

    // ── Render the meta box HTML ─────────────────────────────────────────
    public function render_meta_box( $post ) {
        // Retrieve saved meta values (fall back to empty / global defaults)
        $layout         = get_post_meta( $post->ID, self::PREFIX . 'layout',         true );
        $loop           = get_post_meta( $post->ID, self::PREFIX . 'loop',           true );
        $autoplay_speed = get_post_meta( $post->ID, self::PREFIX . 'autoplay_speed', true );
        $posts_per_view = get_post_meta( $post->ID, self::PREFIX . 'posts_per_view', true );
        $inc_cats       = get_post_meta( $post->ID, self::PREFIX . 'include_cats',   true );
        $exc_cats       = get_post_meta( $post->ID, self::PREFIX . 'exclude_cats',   true );
        $inc_tags       = get_post_meta( $post->ID, self::PREFIX . 'include_tags',   true );
        $exc_tags       = get_post_meta( $post->ID, self::PREFIX . 'exclude_tags',   true );

        // Nonce field for save verification
        wp_nonce_field( 'drps_metabox_save_action', 'drps_metabox_nonce' );

        ?>
        <div class="drps-metabox">
            <p class="drps-metabox-note">
                <?php esc_html_e( 'These settings override the global defaults for this page/post only. Leave blank to use global settings.', DRPS_TEXT_DOMAIN ); ?>
            </p>

            <!-- Layout -->
            <div class="drps-mb-field">
                <label for="drps_meta_layout"><strong><?php esc_html_e( 'Layout', DRPS_TEXT_DOMAIN ); ?></strong></label>
                <select id="drps_meta_layout" name="drps_meta_layout">
                    <option value=""        <?php selected( $layout, '' ); ?>><?php esc_html_e( '— Global default —', DRPS_TEXT_DOMAIN ); ?></option>
                    <option value="carousel" <?php selected( $layout, 'carousel' ); ?>><?php esc_html_e( 'Carousel', DRPS_TEXT_DOMAIN ); ?></option>
                    <option value="grid"     <?php selected( $layout, 'grid' ); ?>><?php esc_html_e( 'Grid', DRPS_TEXT_DOMAIN ); ?></option>
                </select>
            </div>

            <!-- Loop -->
            <div class="drps-mb-field">
                <label><strong><?php esc_html_e( 'Infinite Loop', DRPS_TEXT_DOMAIN ); ?></strong></label><br/>
                <select id="drps_meta_loop" name="drps_meta_loop">
                    <option value=""  <?php selected( $loop, '' ); ?>><?php esc_html_e( '— Global default —', DRPS_TEXT_DOMAIN ); ?></option>
                    <option value="1" <?php selected( $loop, '1' ); ?>><?php esc_html_e( 'On', DRPS_TEXT_DOMAIN ); ?></option>
                    <option value="0" <?php selected( $loop, '0' ); ?>><?php esc_html_e( 'Off', DRPS_TEXT_DOMAIN ); ?></option>
                </select>
            </div>

            <!-- Autoplay Speed -->
            <div class="drps-mb-field">
                <label for="drps_meta_autoplay_speed"><strong><?php esc_html_e( 'Autoplay Speed (ms)', DRPS_TEXT_DOMAIN ); ?></strong></label>
                <input type="number" id="drps_meta_autoplay_speed" name="drps_meta_autoplay_speed"
                       value="<?php echo esc_attr( $autoplay_speed ); ?>" min="500" step="100" placeholder="<?php esc_attr_e( 'e.g. 2000', DRPS_TEXT_DOMAIN ); ?>" />
            </div>

            <!-- Posts Per View -->
            <div class="drps-mb-field">
                <label for="drps_meta_posts_per_view"><strong><?php esc_html_e( 'Posts Per View', DRPS_TEXT_DOMAIN ); ?></strong></label>
                <input type="number" id="drps_meta_posts_per_view" name="drps_meta_posts_per_view"
                       value="<?php echo esc_attr( $posts_per_view ); ?>" min="1" max="6" step="1" placeholder="<?php esc_attr_e( 'e.g. 3', DRPS_TEXT_DOMAIN ); ?>" />
            </div>

            <!-- Include Categories (autocomplete) -->
            <div class="drps-mb-field">
                <label><strong><?php esc_html_e( 'Include Categories', DRPS_TEXT_DOMAIN ); ?></strong></label>
                <div class="drps-autocomplete-wrap">
                    <input type="text" class="drps-autocomplete" data-field="drps_meta_include_cats" data-type="category" placeholder="<?php esc_attr_e( 'Type to search…', DRPS_TEXT_DOMAIN ); ?>" autocomplete="off" />
                    <div class="drps-autocomplete-results"></div>
                </div>
                <input type="hidden" id="drps_meta_include_cats" name="drps_meta_include_cats" value="<?php echo esc_attr( $inc_cats ); ?>" />
                <div class="drps-tags-container" data-field="drps_meta_include_cats"></div>
            </div>

            <!-- Exclude Categories -->
            <div class="drps-mb-field">
                <label><strong><?php esc_html_e( 'Exclude Categories', DRPS_TEXT_DOMAIN ); ?></strong></label>
                <div class="drps-autocomplete-wrap">
                    <input type="text" class="drps-autocomplete" data-field="drps_meta_exclude_cats" data-type="category" placeholder="<?php esc_attr_e( 'Type to search…', DRPS_TEXT_DOMAIN ); ?>" autocomplete="off" />
                    <div class="drps-autocomplete-results"></div>
                </div>
                <input type="hidden" id="drps_meta_exclude_cats" name="drps_meta_exclude_cats" value="<?php echo esc_attr( $exc_cats ); ?>" />
                <div class="drps-tags-container" data-field="drps_meta_exclude_cats"></div>
            </div>

            <!-- Include Tags -->
            <div class="drps-mb-field">
                <label><strong><?php esc_html_e( 'Include Tags', DRPS_TEXT_DOMAIN ); ?></strong></label>
                <div class="drps-autocomplete-wrap">
                    <input type="text" class="drps-autocomplete" data-field="drps_meta_include_tags" data-type="tag" placeholder="<?php esc_attr_e( 'Type to search…', DRPS_TEXT_DOMAIN ); ?>" autocomplete="off" />
                    <div class="drps-autocomplete-results"></div>
                </div>
                <input type="hidden" id="drps_meta_include_tags" name="drps_meta_include_tags" value="<?php echo esc_attr( $inc_tags ); ?>" />
                <div class="drps-tags-container" data-field="drps_meta_include_tags"></div>
            </div>

            <!-- Exclude Tags -->
            <div class="drps-mb-field">
                <label><strong><?php esc_html_e( 'Exclude Tags', DRPS_TEXT_DOMAIN ); ?></strong></label>
                <div class="drps-autocomplete-wrap">
                    <input type="text" class="drps-autocomplete" data-field="drps_meta_exclude_tags" data-type="tag" placeholder="<?php esc_attr_e( 'Type to search…', DRPS_TEXT_DOMAIN ); ?>" autocomplete="off" />
                    <div class="drps-autocomplete-results"></div>
                </div>
                <input type="hidden" id="drps_meta_exclude_tags" name="drps_meta_exclude_tags" value="<?php echo esc_attr( $exc_tags ); ?>" />
                <div class="drps-tags-container" data-field="drps_meta_exclude_tags"></div>
            </div>
        </div>
        <?php
    }

    // ── Save meta box fields ─────────────────────────────────────────────
    public function save_meta( $post_id, $post, $update ) {
        // Skip on autosave
        if ( wp_is_post_autosave( $post_id ) ) {
            return;
        }
        // Skip on revision
        if ( 'revision' === $post->post_type ) {
            return;
        }
        // Verify nonce
        if ( ! isset( $_POST['drps_metabox_nonce'] ) ||
             ! wp_verify_nonce( $_POST['drps_metabox_nonce'], 'drps_metabox_save_action' ) ) {
            return;
        }
        // Capability check
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        // Sanitize and save each meta field
        $fields = array(
            'layout',
            'loop',
            'autoplay_speed',
            'posts_per_view',
            'include_cats',
            'exclude_cats',
            'include_tags',
            'exclude_tags',
        );

        foreach ( $fields as $field ) {
            $key   = self::PREFIX . $field;
            $value = isset( $_POST[ $key ] ) ? sanitize_text_field( $_POST[ $key ] ) : '';
            // For numeric fields, ensure integer sanitization
            if ( in_array( $field, array( 'autoplay_speed', 'posts_per_view' ), true ) && '' !== $value ) {
                $value = absint( $value );
            }
            update_post_meta( $post_id, $key, $value );
        }
    }
}
