<?php
/**
 * Class DRPS_Ajax
 * Handles AJAX requests for:
 *   1. Autocomplete search (categories & tags) — used in admin + metabox.
 *   2. Live preview rendering — returns the rendered shortcode HTML for the admin preview panel.
 *
 * @package DigitalRisePostsSolutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRPS_Ajax {

    public function __construct() {
        // Autocomplete (logged-in users only)
        add_action( 'wp_ajax_drps_autocomplete', array( $this, 'autocomplete' ) );

        // Live preview (logged-in users only)
        add_action( 'wp_ajax_drps_live_preview', array( $this, 'live_preview' ) );
    }

    // ── Autocomplete handler ─────────────────────────────────────────────
    /**
     * Returns matching categories or tags as JSON.
     * Expected POST params: type (category|tag), search (string).
     */
    public function autocomplete() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'drps_autocomplete_nonce' ) ) {
            wp_send_json_fail( esc_html__( 'Security check failed.', DRPS_TEXT_DOMAIN ) );
        }

        $type   = sanitize_key( $_POST['type'] ?? '' );
        $search = sanitize_text_field( $_POST['search'] ?? '' );

        if ( empty( $search ) ) {
            wp_send_json_success( array() );
        }

        $results = array();

        if ( 'category' === $type ) {
            // Query categories whose name matches the search string
            $cats = get_categories( array(
                'search'   => $search,
                'number'   => 10,
                'hide_empty' => false,
            ) );
            foreach ( $cats as $cat ) {
                $results[] = array(
                    'id'   => $cat->term_id,
                    'name' => $cat->name,
                );
            }
        } elseif ( 'tag' === $type ) {
            // Query tags whose name matches the search string
            $tags = get_tags( array(
                'search'   => $search,
                'number'   => 10,
                'hide_empty' => false,
            ) );
            foreach ( $tags as $tag ) {
                $results[] = array(
                    'id'   => $tag->term_id,
                    'name' => $tag->name,
                );
            }
        }

        wp_send_json_success( $results );
    }

    // ── Live preview handler ─────────────────────────────────────────────
    /**
     * Accepts the current form values via POST, temporarily overrides
     * global options, renders the shortcode, then restores original values.
     * Returns the rendered HTML for the admin live-preview panel.
     */
    public function live_preview() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'drps_live_preview_nonce' ) ) {
            wp_send_json_fail( esc_html__( 'Security check failed.', DRPS_TEXT_DOMAIN ) );
        }

        // Temporarily override global options with the form values
        $overrides = array(
            'drps_layout'         => sanitize_key( $_POST['layout'] ?? 'carousel' ),
            'drps_loop'           => $_POST['loop'] === '1' ? '1' : '0',
            'drps_autoplay_speed' => absint( $_POST['autoplay_speed'] ?? 2000 ),
            'drps_posts_per_view' => absint( $_POST['posts_per_view'] ?? 3 ),
            'drps_include_cats'   => sanitize_text_field( $_POST['include_cats'] ?? '' ),
            'drps_exclude_cats'   => sanitize_text_field( $_POST['exclude_cats'] ?? '' ),
            'drps_include_tags'   => sanitize_text_field( $_POST['include_tags'] ?? '' ),
            'drps_exclude_tags'   => sanitize_text_field( $_POST['exclude_tags'] ?? '' ),
        );

        // Save originals so we can restore after rendering
        $originals = array();
        foreach ( $overrides as $key => $value ) {
            $originals[ $key ] = get_option( $key );
            update_option( $key, $value );
        }

        // Render the shortcode with the temporary settings
        // Pass post_id = 0 so resolve_settings uses global only
        $html = DRPS_Shortcode::render( array(), '', 0 );

        // Restore original values
        foreach ( $originals as $key => $value ) {
            update_option( $key, $value );
        }

        wp_send_json_success( array( 'html' => $html ) );
    }
}
