<?php
/**
 * Class DRPS_Admin
 * Handles the WordPress admin settings page, form submission,
 * live preview, logo, donate button, and post-activation instructions.
 *
 * @package DigitalRisePostsSolutions
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRPS_Admin {

    /**
     * Constructor — register hooks.
     */
    public function __construct() {
        add_action( 'admin_menu',    array( $this, 'add_settings_page' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_action( 'admin_init',    array( $this, 'save_settings' ) );
        // Admin notice shown once after activation
        add_action( 'admin_notices', array( $this, 'activation_notice' ) );
    }

    // ── Register admin page ──────────────────────────────────────────────
    public function add_settings_page() {
        add_menu_page(
            esc_html__( 'DR Posts Settings', DRPS_TEXT_DOMAIN ),
            esc_html__( 'DR Posts', DRPS_TEXT_DOMAIN ),
            'manage_options',
            'drps-settings',
            array( $this, 'render_settings_page' ),
            'dashicons-grid-view',
            500
        );
    }

    // ── Enqueue admin CSS + JS ───────────────────────────────────────────
    public function enqueue_admin_assets( $hook ) {
        if ( 'toplevel_page_drps-settings' !== $hook ) {
            return;
        }
        wp_enqueue_style(
            'drps-admin-css',
            DRPS_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            DRPS_VERSION
        );
        wp_enqueue_script(
            'drps-admin-js',
            DRPS_PLUGIN_URL . 'assets/js/admin.js',
            array( 'jquery' ),
            DRPS_VERSION,
            true // load in footer
        );
        // Pass AJAX URL + nonce to JS via wp_localize_script
        wp_localize_script(
            'drps-admin-js',
            'drps_admin',
            array(
                'ajax_url' => esc_url( admin_url( 'admin-ajax.php' ) ),
                'nonce'    => wp_create_nonce( 'drps_live_preview_nonce' ),
            )
        );
    }

    // ── Save settings (nonce-checked) ────────────────────────────────────
    public function save_settings() {
        // Only run on our settings page POST
        if ( ! isset( $_POST['drps_save_settings'] ) ) {
            return;
        }
        // Verify nonce
        if ( ! wp_verify_nonce( $_POST['drps_settings_nonce'], 'drps_save_settings_action' ) ) {
            wp_die( esc_html__( 'Security check failed.', DRPS_TEXT_DOMAIN ) );
        }
        // Capability check
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Unauthorized.', DRPS_TEXT_DOMAIN ) );
        }

        // Sanitize & save each option
        update_option( 'drps_layout',         sanitize_key( $_POST['drps_layout'] ?? 'carousel' ) );
        update_option( 'drps_loop',           isset( $_POST['drps_loop'] ) ? '1' : '0' );
        update_option( 'drps_autoplay_speed', absint( $_POST['drps_autoplay_speed'] ?? 2000 ) );
        update_option( 'drps_posts_per_view', absint( $_POST['drps_posts_per_view'] ?? 3 ) );

        // Categories / Tags — stored as comma-separated IDs (sanitized)
        update_option( 'drps_include_cats', sanitize_text_field( $_POST['drps_include_cats'] ?? '' ) );
        update_option( 'drps_exclude_cats', sanitize_text_field( $_POST['drps_exclude_cats'] ?? '' ) );
        update_option( 'drps_include_tags', sanitize_text_field( $_POST['drps_include_tags'] ?? '' ) );
        update_option( 'drps_exclude_tags', sanitize_text_field( $_POST['drps_exclude_tags'] ?? '' ) );

        // Add success notice via transient
        set_transient( 'drps_settings_saved', true, 30 );

        // Redirect to avoid resubmission
        wp_redirect( admin_url( 'admin.php?page=drps-settings' ) );
        exit;
    }

    // ── Post-activation instruction notice ──────────────────────────────
    public function activation_notice() {
        if ( ! get_option( 'drps_activated' ) ) {
            return;
        }
        // Show only once — then delete the flag
        ?>
        <div class="notice notice-success is-dismissible drps-activation-notice">
            <h3 style="margin-top:10px;">
                <?php esc_html_e( '✅ Digital Rise Posts Solutions — Activated!', DRPS_TEXT_DOMAIN ); ?>
            </h3>
            <p><strong><?php esc_html_e( 'How to use:', DRPS_TEXT_DOMAIN ); ?></strong></p>
            <ol style="margin:8px 0 8px 20px; line-height:1.8;">
                <li><?php esc_html_e( 'Go to <strong>DR Posts</strong> in your left menu to configure global layout, autoplay, categories, and more.', DRPS_TEXT_DOMAIN ); ?></li>
                <li><?php esc_html_e( 'Open any Page or Post editor — a <strong>"DR Posts Settings"</strong> meta box will appear on the right. Use it to override global settings for that specific page/post.', DRPS_TEXT_DOMAIN ); ?></li>
                <li><?php esc_html_e( 'Place the shortcode <code>[drps_posts]</code> anywhere inside a page or post where you want posts to display. It automatically uses the meta box settings (or global defaults).', DRPS_TEXT_DOMAIN ); ?></li>
                <li><?php esc_html_e( 'Save and preview — your posts carousel or grid is live!', DRPS_TEXT_DOMAIN ); ?></li>
            </ol>
            <p style="margin-top:4px; color:#888; font-size:13px;">
                <?php esc_html_e( 'This notice will disappear once dismissed.', DRPS_TEXT_DOMAIN ); ?>
            </p>
        </div>
        <?php
        // Delete flag so it only shows once
        delete_option( 'drps_activated' );
    }

    // ── Render the full settings page ────────────────────────────────────
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', DRPS_TEXT_DOMAIN ) );
        }

        // Show saved transient notice
        $saved = get_transient( 'drps_settings_saved' );
        if ( $saved ) {
            delete_transient( 'drps_settings_saved' );
        }

        // Pull current option values
        $layout         = get_option( 'drps_layout', 'carousel' );
        $loop           = get_option( 'drps_loop', '1' );
        $autoplay_speed = get_option( 'drps_autoplay_speed', '2000' );
        $posts_per_view = get_option( 'drps_posts_per_view', '3' );
        $inc_cats       = get_option( 'drps_include_cats', '' );
        $exc_cats       = get_option( 'drps_exclude_cats', '' );
        $inc_tags       = get_option( 'drps_include_tags', '' );
        $exc_tags       = get_option( 'drps_exclude_tags', '' );

        // Logo path
        $logo_path = DRPS_PLUGIN_URL . 'assets/images/digital-rise-posts-solutions-logo.png';

        ?>
        <div class="wrap drps-admin-wrap">

            <!-- ── Logo ── -->
            <div class="drps-logo-wrap">
                <img src="<?php echo esc_url( $logo_path ); ?>" alt="<?php esc_attr_e( 'Digital Rise Posts Solutions', DRPS_TEXT_DOMAIN ); ?>" class="drps-logo" style="max-width:300px; max-height:80px; margin-bottom:12px;" />
            </div>

            <h1 class="drps-page-title">
                <?php esc_html_e( 'DR Posts Settings', DRPS_TEXT_DOMAIN ); ?>
            </h1>

            <!-- ── Saved notice ── -->
            <?php if ( $saved ): ?>
                <div class="notice notice-success"><p><?php esc_html_e( '✅ Settings saved successfully!', DRPS_TEXT_DOMAIN ); ?></p></div>
            <?php endif; ?>

            <!-- ── Donate button ── -->
            <div class="drps-donate-wrap">
                <a href="<?php echo esc_url( DRPS_DONATE_URL ); ?>" target="_blank" class="drps-donate-btn">
                    <?php esc_html_e( '💛 Support This Plugin — Donate', DRPS_TEXT_DOMAIN ); ?>
                </a>
            </div>

            <!-- ── Settings form ── -->
            <form method="post" action="">
                <?php wp_nonce_field( 'drps_save_settings_action', 'drps_settings_nonce' ); ?>
                <input type="hidden" name="drps_save_settings" value="1" />

                <div class="drps-settings-grid">
                    <!-- LEFT: Form fields -->
                    <div class="drps-settings-fields">

                        <!-- Layout -->
                        <div class="drps-field-group">
                            <label for="drps_layout"><strong><?php esc_html_e( 'Layout', DRPS_TEXT_DOMAIN ); ?></strong></label>
                            <select id="drps_layout" name="drps_layout">
                                <option value="carousel" <?php selected( $layout, 'carousel' ); ?>><?php esc_html_e( 'Carousel', DRPS_TEXT_DOMAIN ); ?></option>
                                <option value="grid"     <?php selected( $layout, 'grid' );      ?>><?php esc_html_e( 'Grid', DRPS_TEXT_DOMAIN ); ?></option>
                            </select>
                        </div>

                        <!-- Loop -->
                        <div class="drps-field-group">
                            <label><strong><?php esc_html_e( 'Infinite Loop', DRPS_TEXT_DOMAIN ); ?></strong></label><br/>
                            <label class="drps-toggle">
                                <input type="checkbox" id="drps_loop" name="drps_loop" <?php checked( $loop, '1' ); ?> />
                                <span class="drps-toggle-slider"></span>
                                <span class="drps-toggle-label"><?php esc_html_e( 'Loop on/off', DRPS_TEXT_DOMAIN ); ?></span>
                            </label>
                        </div>

                        <!-- Autoplay Speed -->
                        <div class="drps-field-group">
                            <label for="drps_autoplay_speed"><strong><?php esc_html_e( 'Autoplay Speed (ms)', DRPS_TEXT_DOMAIN ); ?></strong></label>
                            <input type="number" id="drps_autoplay_speed" name="drps_autoplay_speed" value="<?php echo esc_attr( $autoplay_speed ); ?>" min="500" step="100" />
                            <span class="drps-hint"><?php esc_html_e( 'e.g. 2000 = 2 seconds', DRPS_TEXT_DOMAIN ); ?></span>
                        </div>

                        <!-- Posts Per View -->
                        <div class="drps-field-group">
                            <label for="drps_posts_per_view"><strong><?php esc_html_e( 'Posts Per View', DRPS_TEXT_DOMAIN ); ?></strong></label>
                            <input type="number" id="drps_posts_per_view" name="drps_posts_per_view" value="<?php echo esc_attr( $posts_per_view ); ?>" min="1" max="6" step="1" />
                        </div>

                        <!-- Include Categories (autocomplete) -->
                        <div class="drps-field-group">
                            <label for="drps_include_cats_input"><strong><?php esc_html_e( 'Include Categories', DRPS_TEXT_DOMAIN ); ?></strong></label>
                            <div class="drps-autocomplete-wrap">
                                <input type="text" id="drps_include_cats_input" class="drps-autocomplete" data-field="drps_include_cats" data-type="category" placeholder="<?php esc_attr_e( 'Type to search…', DRPS_TEXT_DOMAIN ); ?>" autocomplete="off" />
                                <div class="drps-autocomplete-results"></div>
                            </div>
                            <input type="hidden" id="drps_include_cats" name="drps_include_cats" value="<?php echo esc_attr( $inc_cats ); ?>" />
                            <div class="drps-tags-container" data-field="drps_include_cats"></div>
                        </div>

                        <!-- Exclude Categories -->
                        <div class="drps-field-group">
                            <label for="drps_exclude_cats_input"><strong><?php esc_html_e( 'Exclude Categories', DRPS_TEXT_DOMAIN ); ?></strong></label>
                            <div class="drps-autocomplete-wrap">
                                <input type="text" id="drps_exclude_cats_input" class="drps-autocomplete" data-field="drps_exclude_cats" data-type="category" placeholder="<?php esc_attr_e( 'Type to search…', DRPS_TEXT_DOMAIN ); ?>" autocomplete="off" />
                                <div class="drps-autocomplete-results"></div>
                            </div>
                            <input type="hidden" id="drps_exclude_cats" name="drps_exclude_cats" value="<?php echo esc_attr( $exc_cats ); ?>" />
                            <div class="drps-tags-container" data-field="drps_exclude_cats"></div>
                        </div>

                        <!-- Include Tags -->
                        <div class="drps-field-group">
                            <label for="drps_include_tags_input"><strong><?php esc_html_e( 'Include Tags', DRPS_TEXT_DOMAIN ); ?></strong></label>
                            <div class="drps-autocomplete-wrap">
                                <input type="text" id="drps_include_tags_input" class="drps-autocomplete" data-field="drps_include_tags" data-type="tag" placeholder="<?php esc_attr_e( 'Type to search…', DRPS_TEXT_DOMAIN ); ?>" autocomplete="off" />
                                <div class="drps-autocomplete-results"></div>
                            </div>
                            <input type="hidden" id="drps_include_tags" name="drps_include_tags" value="<?php echo esc_attr( $inc_tags ); ?>" />
                            <div class="drps-tags-container" data-field="drps_include_tags"></div>
                        </div>

                        <!-- Exclude Tags -->
                        <div class="drps-field-group">
                            <label for="drps_exclude_tags_input"><strong><?php esc_html_e( 'Exclude Tags', DRPS_TEXT_DOMAIN ); ?></strong></label>
                            <div class="drps-autocomplete-wrap">
                                <input type="text" id="drps_exclude_tags_input" class="drps-autocomplete" data-field="drps_exclude_tags" data-type="tag" placeholder="<?php esc_attr_e( 'Type to search…', DRPS_TEXT_DOMAIN ); ?>" autocomplete="off" />
                                <div class="drps-autocomplete-results"></div>
                            </div>
                            <input type="hidden" id="drps_exclude_tags" name="drps_exclude_tags" value="<?php echo esc_attr( $exc_tags ); ?>" />
                            <div class="drps-tags-container" data-field="drps_exclude_tags"></div>
                        </div>

                        <!-- Submit -->
                        <div class="drps-field-group">
                            <button type="submit" class="button button-primary drps-save-btn">
                                <?php esc_html_e( '💾 Save Settings', DRPS_TEXT_DOMAIN ); ?>
                            </button>
                        </div>
                    </div>

                    <!-- RIGHT: Live Preview -->
                    <div class="drps-preview-panel">
                        <h3 class="drps-preview-title"><?php esc_html_e( '👁 Live Preview', DRPS_TEXT_DOMAIN ); ?></h3>
                        <div id="drps-live-preview" class="drps-preview-box">
                            <p class="drps-preview-loading"><?php esc_html_e( 'Loading preview…', DRPS_TEXT_DOMAIN ); ?></p>
                        </div>
                    </div>
                </div>
            </form>
        </div>
        <?php
    }
}
