/**
 * Digital Rise Posts Solutions — Metabox JS
 * Initializes autocomplete + tag chips inside the post/page editor metabox.
 *
 * Shares the same UX pattern as admin.js but is a separate, lightweight file
 * so it only loads on post/page editor screens.
 *
 * Settings injected via wp_localize_script as window.drps_metabox.
 */

( function ( $ ) {
    'use strict';

    var settings = window.drps_metabox || {};
    var chipNameCache = {};

    // ── DOM Ready ─────────────────────────────────────────────────────────
    $( document ).ready( function () {

        // Init autocomplete on every .drps-autocomplete inside the metabox
        $( '.drps-metabox .drps-autocomplete' ).each( function () {
            initAutocomplete( $( this ) );
        } );

        // Hydrate chips from hidden field values (page load)
        $( '.drps-metabox .drps-tags-container' ).each( function () {
            var fieldId = $( this ).data( 'field' );
            var value   = $( '#' + fieldId ).val();
            if ( value ) {
                // We don't have names cached yet on load, so show IDs.
                // A production enhancement would bulk-fetch names here.
                renderChips( fieldId, value );
            }
        } );
    } );

    // ═══════════════════════════════════════════════════════════════════════
    // AUTOCOMPLETE
    // ═══════════════════════════════════════════════════════════════════════

    function initAutocomplete( $input ) {
        var $wrap    = $input.closest( '.drps-autocomplete-wrap' );
        var $results = $wrap.find( '.drps-autocomplete-results' );
        var fieldId  = $input.data( 'field' );
        var type     = $input.data( 'type' );
        var debounceTimer;

        $input.on( 'input', function () {
            var search = $.trim( $input.val() );
            clearTimeout( debounceTimer );

            if ( search.length < 1 ) {
                $results.hide().empty();
                return;
            }

            debounceTimer = setTimeout( function () {
                $.ajax( {
                    url:  settings.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'drps_autocomplete',
                        nonce:  settings.nonce,
                        type:   type,
                        search: search
                    },
                    success: function ( response ) {
                        $results.empty();
                        if ( response.success && response.data && response.data.length ) {
                            $.each( response.data, function ( i, item ) {
                                var $item = $( '<div class="drps-ac-item" data-id="' + item.id + '">'
                                    + $( '<span>' ).text( item.name ).html()
                                    + '</div>' );
                                $item.on( 'click', function () {
                                    selectTag( fieldId, item.id, item.name );
                                    $results.hide().empty();
                                    $input.val( '' );
                                } );
                                $results.append( $item );
                            } );
                            $results.show();
                        } else {
                            $results.hide();
                        }
                    },
                    error: function () {
                        $results.hide();
                    }
                } );
            }, 300 );
        } );

        // Close on outside click
        $( document ).on( 'click', function ( e ) {
            if ( ! $wrap.is( e.target ) && $wrap.find( e.target ).length === 0 ) {
                $results.hide();
            }
        } );
    }

    // ═══════════════════════════════════════════════════════════════════════
    // TAG SELECTION & CHIPS
    // ═══════════════════════════════════════════════════════════════════════

    function selectTag( fieldId, id, name ) {
        chipNameCache[ String( id ) ] = name;

        var $hidden = $( '#' + fieldId );
        var current = $hidden.val();
        var ids     = current ? current.split( ',' ) : [];

        if ( ids.indexOf( String( id ) ) === -1 ) {
            ids.push( String( id ) );
            $hidden.val( ids.join( ',' ) );
            renderChips( fieldId, $hidden.val() );
        }
    }

    function removeTag( fieldId, id ) {
        var $hidden = $( '#' + fieldId );
        var ids     = $hidden.val() ? $hidden.val().split( ',' ) : [];
        ids = ids.filter( function ( val ) { return val !== String( id ); } );
        $hidden.val( ids.join( ',' ) );
        renderChips( fieldId, $hidden.val() );
    }

    function renderChips( fieldId, idsStr ) {
        var $container = $( '.drps-metabox .drps-tags-container[data-field="' + fieldId + '"]' );
        var ids        = idsStr ? idsStr.split( ',' ) : [];
        $container.empty();

        ids.forEach( function ( id ) {
            if ( ! id ) return;
            var name  = chipNameCache[ id ] || 'ID: ' + id;
            var $chip = $( '<div class="drps-tag-chip">'
                + '<span class="drps-tag-name">' + $( '<span>' ).text( name ).html() + '</span>'
                + '<span class="drps-tag-remove" data-id="' + id + '" data-field="' + fieldId + '">&times;</span>'
                + '</div>' );
            $container.append( $chip );
        } );

        // Bind remove clicks
        $container.find( '.drps-tag-remove' ).on( 'click', function () {
            removeTag( $( this ).data( 'field' ), String( $( this ).data( 'id' ) ) );
        } );
    }

} )( jQuery );
