/**
 * Digital Rise Posts Solutions — Frontend Carousel JS
 * Vanilla JS (no jQuery, no external libs).
 *
 * Features:
 *   • Infinite loop via clone-based technique
 *   • Autoplay with configurable speed
 *   • Responsive posts-per-view (reads breakpoints)
 *   • Previous / Next arrow navigation
 *   • Smooth CSS-transition-based sliding
 *
 * Settings are injected via wp_localize_script as window.drps_settings.
 */

( function () {
    'use strict';

    // Wait for DOM ready
    document.addEventListener( 'DOMContentLoaded', function () {
        var wrappers = document.querySelectorAll( '.drps-carousel-wrapper' );
        wrappers.forEach( function ( wrapper ) {
            initCarousel( wrapper );
        } );
    } );

    /**
     * Initialize a single carousel instance.
     * @param {HTMLElement} wrapper  The .drps-carousel-wrapper element.
     */
    function initCarousel( wrapper ) {
        var track      = wrapper.querySelector( '.drps-carousel-track' );
        var cards      = track.querySelectorAll( '.drps-post-card' );
        var prevBtn    = wrapper.querySelector( '.drps-carousel-prev' );
        var nextBtn    = wrapper.querySelector( '.drps-carousel-next' );

        // Read data attributes set by PHP (per-instance)
        var loopEnabled = wrapper.getAttribute( 'data-loop' ) === '1';
        var speed       = parseInt( wrapper.getAttribute( 'data-speed' ), 10 ) || 2000;
        var ppvAttr     = parseInt( wrapper.getAttribute( 'data-ppv' ), 10 ) || 3;

        var totalCards  = cards.length;
        var currentIdx  = 0;       // logical index (before clones)
        var ppv         = ppvAttr; // posts per view (may change on resize)
        var isSliding   = false;
        var autoplayTimer = null;

        // ── Responsive PPV helper ──────────────────────────────────────
        function getResponsivePPV() {
            var w = window.innerWidth;
            if ( w <= 480 )  return 1;
            if ( w <= 768 )  return 2;
            return ppvAttr; // desktop = user setting
        }

        // ── Set CSS custom property --drps-ppv on the wrapper ──────────
        function setPPV( val ) {
            ppv = val;
            wrapper.style.setProperty( '--drps-ppv', String( ppv ) );
        }

        // ── Clone cards for infinite loop ──────────────────────────────
        function setupClones() {
            // Remove any previously added clones (on resize)
            var clones = track.querySelectorAll( '.drps-clone' );
            clones.forEach( function ( c ) { c.parentNode.removeChild( c ); } );

            if ( ! loopEnabled || totalCards === 0 ) return;

            // Clone last `ppv` cards → prepend; clone first `ppv` cards → append
            var cloneCount = Math.min( ppv, totalCards );

            // Prepend clones (end of original list, reversed)
            for ( var i = cloneCount - 1; i >= 0; i-- ) {
                var clone = cards[ totalCards - 1 - i ].cloneNode( true );
                clone.classList.add( 'drps-clone' );
                track.insertBefore( clone, track.firstChild );
            }
            // Append clones (beginning of original list)
            for ( var j = 0; j < cloneCount; j++ ) {
                var clone2 = cards[ j ].cloneNode( true );
                clone2.classList.add( 'drps-clone' );
                track.appendChild( clone2 );
            }
        }

        // ── Position track at the correct offset (no animation) ────────
        function setPosition( idx, animate ) {
            var cloneCount = loopEnabled ? Math.min( ppv, totalCards ) : 0;
            var offset     = ( idx + cloneCount ) * ( 100 / ppv );
            if ( animate ) {
                track.style.transition = 'transform 0.45s cubic-bezier(0.25,0.46,0.45,0.94)';
            } else {
                track.style.transition = 'none';
            }
            track.style.transform = 'translateX(-' + offset + '%)';
        }

        // ── Slide to next ──────────────────────────────────────────────
        function slideNext() {
            if ( isSliding ) return;
            isSliding = true;
            currentIdx++;
            setPosition( currentIdx, true );
        }

        // ── Slide to prev ──────────────────────────────────────────────
        function slidePrev() {
            if ( isSliding ) return;
            isSliding = true;
            currentIdx--;
            setPosition( currentIdx, true );
        }

        // ── Transition end: handle clone jumps for infinite loop ───────
        track.addEventListener( 'transitionend', function () {
            isSliding = false;

            if ( ! loopEnabled ) {
                // Clamp at boundaries
                if ( currentIdx < 0 )                       currentIdx = 0;
                if ( currentIdx > totalCards - ppv )        currentIdx = totalCards - ppv;
                setPosition( currentIdx, false );
                return;
            }

            // Jump back to real card if we've gone past the end
            if ( currentIdx >= totalCards ) {
                currentIdx = 0;
                setPosition( currentIdx, false );
            }
            // Jump forward if we've gone before the start
            if ( currentIdx < 0 ) {
                currentIdx = totalCards - 1;
                setPosition( currentIdx, false );
            }
        } );

        // ── Autoplay ───────────────────────────────────────────────────
        function startAutoplay() {
            if ( speed <= 0 ) return;
            autoplayTimer = setInterval( function () {
                slideNext();
            }, speed );
        }

        function stopAutoplay() {
            if ( autoplayTimer ) {
                clearInterval( autoplayTimer );
                autoplayTimer = null;
            }
        }

        // ── Arrow button handlers ──────────────────────────────────────
        if ( prevBtn ) {
            prevBtn.addEventListener( 'click', function () {
                stopAutoplay();
                slidePrev();
                startAutoplay();
            } );
        }
        if ( nextBtn ) {
            nextBtn.addEventListener( 'click', function () {
                stopAutoplay();
                slideNext();
                startAutoplay();
            } );
        }

        // ── Resize handler ─────────────────────────────────────────────
        var resizeTimeout;
        window.addEventListener( 'resize', function () {
            clearTimeout( resizeTimeout );
            resizeTimeout = setTimeout( function () {
                var newPPV = getResponsivePPV();
                if ( newPPV !== ppv ) {
                    setPPV( newPPV );
                    setupClones();
                    // Reset to start to avoid visual glitch
                    currentIdx = 0;
                    setPosition( currentIdx, false );
                }
            }, 150 );
        } );

        // ── Init sequence ──────────────────────────────────────────────
        setPPV( getResponsivePPV() );
        setupClones();
        setPosition( currentIdx, false );
        startAutoplay();
    }

})();
