/**
 * Digital Rise Posts Solutions — Admin JS
 * Handles:
 *   1. Autocomplete for category / tag inputs (AJAX-powered).
 *   2. Tag chip creation / removal (selected items shown as pills).
 *   3. Live preview: fetches rendered shortcode HTML whenever any setting changes.
 *
 * Requires jQuery (enqueued by WordPress core on admin pages).
 * Settings injected via wp_localize_script as window.drps_admin.
 */

( function ( $ ) {
    'use strict';

    var settings = window.drps_admin || {};

    // ── On DOM ready ──────────────────────────────────────────────────────
    $( document ).ready( function () {
        // Initialize all autocomplete inputs on this page
        $( '.drps-autocomplete' ).each( function () {
            initAutocomplete( $( this ) );
        } );

        // Populate tag chips from hidden field values on page load
        $( '.drps-tags-container' ).each( function () {
            var fieldId = $( this ).data( 'field' );
            var value   = $( '#' + fieldId ).val();
            if ( value ) {
                renderChips( fieldId, value );
            }
        } );

        // Live preview: trigger initial load + listen for changes
        loadPreview();
        bindPreviewTriggers();
    } );

    // ═══════════════════════════════════════════════════════════════════════
    // AUTOCOMPLETE
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Set up keystroke listener on an autocomplete input.
     * @param {jQuery} $input  The text input element.
     */
    function initAutocomplete( $input ) {
        var $wrap    = $input.closest( '.drps-autocomplete-wrap' );
        var $results = $wrap.find( '.drps-autocomplete-results' );
        var fieldId  = $input.data( 'field' );
        var type     = $input.data( 'type' ); // 'category' or 'tag'
        var debounceTimer;

        $input.on( 'input', function () {
            var search = $.trim( $input.val() );
            clearTimeout( debounceTimer );

            if ( search.length < 1 ) {
                $results.hide().empty();
                return;
            }

            // Debounce: wait 300ms after user stops typing
            debounceTimer = setTimeout( function () {
                $.ajax( {
                    url:      settings.ajax_url,
                    type:     'POST',
                    data:     {
                        action: 'drps_autocomplete',
                        nonce:  settings.nonce,
                        type:   type,
                        search: search
                    },
                    success:  function ( response ) {
                        $results.empty();
                        if ( response.success && response.data && response.data.length ) {
                            $.each( response.data, function ( i, item ) {
                                var $item = $( '<div class="drps-ac-item" data-id="' + item.id + '">' + $( '<span>' ).text( item.name ).html() + '</div>' );
                                $item.on( 'click', function () {
                                    selectTag( fieldId, item.id, item.name );
                                    $results.hide().empty();
                                    $input.val( '' );
                                } );
                                $results.append( $item );
                            } );
                            $results.show();
                        } else {
                            $results.hide();
                        }
                    },
                    error: function () {
                        $results.hide();
                    }
                } );
            }, 300 );
        } );

        // Close results when clicking outside
        $( document ).on( 'click', function ( e ) {
            if ( ! $wrap.is( e.target ) && $wrap.find( e.target ).length === 0 ) {
                $results.hide();
            }
        } );
    }

    /**
     * Add a selected tag/category to the hidden field and re-render chips.
     */
    function selectTag( fieldId, id, name ) {
        var $hidden = $( '#' + fieldId );
        var current = $hidden.val();
        var ids     = current ? current.split( ',' ) : [];

        // Avoid duplicates
        if ( ids.indexOf( String( id ) ) === -1 ) {
            ids.push( String( id ) );
            $hidden.val( ids.join( ',' ) );
            renderChips( fieldId, $hidden.val() );
            // Trigger preview update
            loadPreview();
        }
    }

    /**
     * Remove a tag by ID from the hidden field and re-render chips.
     */
    function removeTag( fieldId, id ) {
        var $hidden = $( '#' + fieldId );
        var ids     = $hidden.val() ? $hidden.val().split( ',' ) : [];
        ids = ids.filter( function ( val ) { return val !== String( id ); } );
        $hidden.val( ids.join( ',' ) );
        renderChips( fieldId, $hidden.val() );
        loadPreview();
    }

    // ═══════════════════════════════════════════════════════════════════════
    // TAG CHIPS — render pill badges for selected items
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Given a comma-separated ID string, render chip badges.
     * We need the name for display; we retrieve it via a quick AJAX call
     * or store it in a data cache. For simplicity we use the cached names
     * stored on each chip's creation. On page load we fetch names in bulk.
     */
    var chipNameCache = {};

    function renderChips( fieldId, idsStr ) {
        var $container = $( '.drps-tags-container[data-field="' + fieldId + '"]' );
        var ids        = idsStr ? idsStr.split( ',' ) : [];
        $container.empty();

        ids.forEach( function ( id ) {
            if ( ! id ) return;
            var name = chipNameCache[ id ] || 'ID: ' + id;
            var $chip = $( '<div class="drps-tag-chip">' +
                '<span class="drps-tag-name">' + $( '<span>' ).text( name ).html() + '</span>' +
                '<span class="drps-tag-remove" data-id="' + id + '" data-field="' + fieldId + '">&times;</span>' +
                '</div>' );
            $container.append( $chip );
        } );

        // Remove button handler
        $container.find( '.drps-tag-remove' ).on( 'click', function () {
            removeTag( $( this ).data( 'field' ), String( $( this ).data( 'id' ) ) );
        } );
    }

    /**
     * Cache a tag/category name when it's selected from autocomplete.
     * Also called on page load to hydrate chips with names.
     */
    ( function cacheFromAutocomplete() {
        // Patch selectTag to also cache the name
        var origSelect = selectTag;
        selectTag = function ( fieldId, id, name ) {
            chipNameCache[ String( id ) ] = name;
            origSelect( fieldId, id, name );
        };
    } )();

    // ═══════════════════════════════════════════════════════════════════════
    // LIVE PREVIEW
    // ═══════════════════════════════════════════════════════════════════════

    /**
     * Bind change/input events on all settings fields to trigger preview refresh.
     */
    function bindPreviewTriggers() {
        var triggers = [
            '#drps_layout',
            '#drps_loop',
            '#drps_autoplay_speed',
            '#drps_posts_per_view'
        ];
        triggers.forEach( function ( sel ) {
            $( sel ).on( 'change input', function () {
                loadPreview();
            } );
        } );
    }

    /**
     * Collect current form values and send AJAX request to render preview.
     */
    function loadPreview() {
        var $preview = $( '#drps-live-preview' );
        $preview.html( '<p class="drps-preview-loading">Loading…</p>' );

        $.ajax( {
            url:  settings.ajax_url,
            type: 'POST',
            data: {
                action:         'drps_live_preview',
                nonce:          settings.nonce,
                layout:         $( '#drps_layout' ).val(),
                loop:           $( '#drps_loop' ).is( ':checked' ) ? '1' : '0',
                autoplay_speed: $( '#drps_autoplay_speed' ).val(),
                posts_per_view: $( '#drps_posts_per_view' ).val(),
                include_cats:   $( '#drps_include_cats' ).val(),
                exclude_cats:   $( '#drps_exclude_cats' ).val(),
                include_tags:   $( '#drps_include_tags' ).val(),
                exclude_tags:   $( '#drps_exclude_tags' ).val()
            },
            success: function ( response ) {
                if ( response.success ) {
                    $preview.html( response.data.html );
                } else {
                    $preview.html( '<p style="color:#c00;">Preview error.</p>' );
                }
            },
            error: function () {
                $preview.html( '<p style="color:#c00;">Could not load preview.</p>' );
            }
        } );
    }

} )( jQuery );
