<?php
/**
 * Robots.txt Handler
 * PHASE 5: Automatic robots.txt generation and customization
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Robots {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        $this->init();
    }

    /**
     * Initialize robots.txt functionality
     */
    public function init() {
        // Filter WordPress robots.txt
        add_filter( 'robots_txt', array( $this, 'modify_robots_txt' ), 10, 2 );
        
        // Generate default robots.txt on activation
        add_action( 'driop_activated', array( $this, 'ensure_robots_txt' ) );
    }

    /**
     * Modify WordPress robots.txt output
     */
    public function modify_robots_txt( $output, $public ) {
        // Get custom robots.txt content
        $custom_content = $this->plugin->get_option( 'robots_txt_content', '' );
        
        if ( ! empty( $custom_content ) ) {
            return $custom_content;
        }

        // Generate default enhanced robots.txt
        return $this->generate_default_robots_txt( $public );
    }

    /**
     * Generate default robots.txt content
     */
    private function generate_default_robots_txt( $public = '1' ) {
        $site_url = parse_url( home_url(), PHP_URL_HOST );
        $sitemap_url = home_url( '/sitemap.xml' );

        $robots = '';

        if ( '0' === (string) $public ) {
            // Site is not public - disallow everything
            $robots .= "User-agent: *\n";
            $robots .= "Disallow: /\n\n";
        } else {
            // Site is public - allow with restrictions
            $robots .= "User-agent: *\n";
            $robots .= "Disallow: /wp-admin/\n";
            $robots .= "Allow: /wp-admin/admin-ajax.php\n";
            $robots .= "Disallow: /wp-includes/\n";
            $robots .= "Disallow: /wp-content/plugins/\n";
            $robots .= "Disallow: /wp-content/cache/\n";
            $robots .= "Disallow: /wp-content/themes/\n";
            $robots .= "Disallow: /trackback/\n";
            $robots .= "Disallow: /feed/\n";
            $robots .= "Disallow: /comments/\n";
            $robots .= "Disallow: /category/*/*\n";
            $robots .= "Disallow: */trackback\n";
            $robots .= "Disallow: */feed\n";
            $robots .= "Disallow: */comments\n";
            $robots .= "Disallow: /*?\n";
            $robots .= "Disallow: /*.php$\n";
            $robots .= "Disallow: /*.js$\n";
            $robots .= "Disallow: /*.css$\n";
            $robots .= "Disallow: /*.inc$\n";
            $robots .= "Disallow: /*.wmv$\n";
            $robots .= "Disallow: /*.cgi$\n";
            $robots .= "Disallow: /*.xhtml$\n";
            $robots .= "\n";

            // Allow images
            $robots .= "Allow: *.jpg$\n";
            $robots .= "Allow: *.jpeg$\n";
            $robots .= "Allow: *.gif$\n";
            $robots .= "Allow: *.png$\n";
            $robots .= "Allow: *.webp$\n";
            $robots .= "Allow: *.svg$\n";
            $robots .= "\n";

            // WooCommerce specific rules
            if ( class_exists( 'WooCommerce' ) ) {
                $robots .= "# WooCommerce\n";
                $robots .= "Disallow: /cart/\n";
                $robots .= "Disallow: /checkout/\n";
                $robots .= "Disallow: /my-account/\n";
                $robots .= "Disallow: /*add-to-cart=*\n";
                $robots .= "Disallow: /*add_to_wishlist=*\n";
                $robots .= "\n";
            }

            // Sitemap reference
            $robots .= "# Sitemap\n";
            $robots .= "Sitemap: " . esc_url( $sitemap_url ) . "\n\n";

            // Crawl rate (optional)
            $robots .= "# Crawl Rate\n";
            $robots .= "Crawl-delay: 10\n\n";

            // Specific bot rules
            $robots .= "# Specific Bots\n";
            $robots .= "User-agent: Googlebot\n";
            $robots .= "Allow: /\n\n";

            $robots .= "User-agent: Googlebot-Image\n";
            $robots .= "Allow: /\n\n";

            $robots .= "User-agent: Bingbot\n";
            $robots .= "Allow: /\n\n";
        }

        return $robots;
    }

    /**
     * Ensure robots.txt exists
     */
    public function ensure_robots_txt() {
        // WordPress handles robots.txt virtually, so we just need to ensure our filter is active
        // Save default content to options for customization
        $existing = $this->plugin->get_option( 'robots_txt_content', '' );
        
        if ( empty( $existing ) ) {
            // Don't save default - let WordPress handle it dynamically
            // User can customize via settings if needed
        }
    }

    /**
     * Get robots.txt content
     */
    public function get_robots_txt() {
        $custom_content = $this->plugin->get_option( 'robots_txt_content', '' );
        
        if ( ! empty( $custom_content ) ) {
            return $custom_content;
        }

        return $this->generate_default_robots_txt( get_option( 'blog_public' ) );
    }

    /**
     * Save custom robots.txt
     */
    public function save_robots_txt( $content ) {
        // Sanitize content
        $content = wp_strip_all_tags( $content, true ); // Keep line breaks
        
        // Save to options
        $this->plugin->update_option( 'robots_txt_content', $content );
        
        return true;
    }

    /**
     * Reset to default
     */
    public function reset_to_default() {
        $this->plugin->update_option( 'robots_txt_content', '' );
        return true;
    }

    /**
     * Get robots.txt stats
     */
    public function get_stats() {
        $custom_content = $this->plugin->get_option( 'robots_txt_content', '' );
        $is_custom = ! empty( $custom_content );
        $content = $this->get_robots_txt();
        
        return array(
            'exists' => true,
            'is_custom' => $is_custom,
            'url' => home_url( '/robots.txt' ),
            'content' => $content,
            'line_count' => substr_count( $content, "\n" ) + 1,
        );
    }

    /**
     * Test robots.txt accessibility
     */
    public function test_accessibility() {
        $url = home_url( '/robots.txt' );
        $response = wp_remote_get( $url );

        if ( is_wp_error( $response ) ) {
            return array(
                'accessible' => false,
                'error' => $response->get_error_message(),
            );
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        return array(
            'accessible' => $code === 200,
            'http_code' => $code,
            'has_sitemap' => strpos( $body, 'Sitemap:' ) !== false,
            'byte_size' => strlen( $body ),
        );
    }
}
