<?php
/**
 * License Management System
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_License {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Maybe validate license (scheduled check)
     */
    public function maybe_validate_license() {
        $license_key = $this->plugin->get_option( 'license_key' );
        
        if ( empty( $license_key ) ) {
            return;
        }

        // Check once per day
        $last_check = get_transient( 'driop_license_last_check' );
        if ( $last_check ) {
            return;
        }

        $this->validate_license( $license_key );
        set_transient( 'driop_license_last_check', time(), DAY_IN_SECONDS );
    }

    /**
     * Activate license
     */
    public function activate_license( $license_key ) {
        if ( empty( $license_key ) ) {
            return array(
                'success' => false,
                'message' => __( 'Please enter a license key.', 'digital-rise-image-optimizer-for-seo' ),
            );
        }

        $response = $this->api_request( 'activate', array(
            'license_key' => $license_key,
            'site_url' => home_url(),
            'site_name' => get_bloginfo( 'name' ),
            'admin_email' => get_option( 'admin_email' ),
            'installation_id' => $this->plugin->get_option( 'installation_id' ),
            'plugin_version' => DRIOP_VERSION,
        ) );

        if ( is_wp_error( $response ) ) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( ! $data ) {
            return array(
                'success' => false,
                'message' => __( 'Invalid response from license server.', 'digital-rise-image-optimizer-for-seo' ),
            );
        }

        if ( isset( $data['success'] ) && $data['success'] ) {
            // Save license info
            $options = $this->plugin->options;
            $options['license_key'] = $license_key;
            $options['license_status'] = 'active';
            $options['current_plan'] = isset( $data['plan'] ) ? $data['plan'] : 'pro';
            $options['license_expires'] = isset( $data['expires'] ) ? $data['expires'] : '';
            $this->plugin->save_options( $options );

            delete_transient( 'driop_license_last_check' );

            return array(
                'success' => true,
                'message' => isset( $data['message'] ) ? $data['message'] : __( 'License activated successfully!', 'digital-rise-image-optimizer-for-seo' ),
                'plan' => isset( $data['plan'] ) ? $data['plan'] : 'pro',
            );
        }

        return array(
            'success' => false,
            'message' => isset( $data['message'] ) ? $data['message'] : __( 'License activation failed.', 'digital-rise-image-optimizer-for-seo' ),
        );
    }

    /**
     * Deactivate license
     */
    public function deactivate_license() {
        $license_key = $this->plugin->get_option( 'license_key' );

        if ( empty( $license_key ) ) {
            return array(
                'success' => false,
                'message' => __( 'No license key found.', 'digital-rise-image-optimizer-for-seo' ),
            );
        }

        $response = $this->api_request( 'deactivate', array(
            'license_key' => $license_key,
            'site_url' => home_url(),
            'installation_id' => $this->plugin->get_option( 'installation_id' ),
        ) );

        // Reset license regardless of API response
        $options = $this->plugin->options;
        $options['license_key'] = '';
        $options['license_status'] = 'inactive';
        $options['current_plan'] = 'free';
        $options['license_expires'] = '';
        $this->plugin->save_options( $options );

        delete_transient( 'driop_license_last_check' );

        return array(
            'success' => true,
            'message' => __( 'License deactivated successfully.', 'digital-rise-image-optimizer-for-seo' ),
        );
    }

    /**
     * Validate license
     */
    public function validate_license( $license_key = null ) {
        if ( ! $license_key ) {
            $license_key = $this->plugin->get_option( 'license_key' );
        }

        if ( empty( $license_key ) ) {
            return false;
        }

        $response = $this->api_request( 'validate', array(
            'license_key' => $license_key,
            'site_url' => home_url(),
            'installation_id' => $this->plugin->get_option( 'installation_id' ),
        ) );

        if ( is_wp_error( $response ) ) {
            // Don't deactivate on network error, keep current status
            return $this->plugin->get_option( 'license_status' ) === 'active';
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( $data && isset( $data['valid'] ) && $data['valid'] ) {
            $options = $this->plugin->options;
            $options['license_status'] = 'active';
            if ( isset( $data['plan'] ) ) {
                $options['current_plan'] = $data['plan'];
            }
            if ( isset( $data['expires'] ) ) {
                $options['license_expires'] = $data['expires'];
            }
            $this->plugin->save_options( $options );
            return true;
        }

        // License invalid or expired
        if ( $data && isset( $data['valid'] ) && ! $data['valid'] ) {
            $options = $this->plugin->options;
            $options['license_status'] = isset( $data['status'] ) ? $data['status'] : 'invalid';
            if ( isset( $data['status'] ) && $data['status'] === 'expired' ) {
                // Keep plan but mark as expired
                $options['license_status'] = 'expired';
            } else {
                $options['current_plan'] = 'free';
            }
            $this->plugin->save_options( $options );
        }

        return false;
    }

    /**
     * Make API request
     */
    private function api_request( $endpoint, $data ) {
        $url = DRIOP_OWNER_API . $endpoint;

        $response = wp_remote_post( $url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-DRIOP-Plugin-Version' => DRIOP_VERSION,
            ),
            'body' => wp_json_encode( $data ),
        ) );

        return $response;
    }

    /**
     * Get license status
     */
    public function get_status() {
        return array(
            'key' => $this->plugin->get_option( 'license_key' ),
            'status' => $this->plugin->get_option( 'license_status', 'inactive' ),
            'plan' => $this->plugin->current_plan,
            'expires' => $this->plugin->get_option( 'license_expires' ),
        );
    }

    /**
     * Check if license is active
     */
    public function is_active() {
        return $this->plugin->get_option( 'license_status' ) === 'active';
    }

    /**
     * Get masked license key for display
     */
    public function get_masked_key() {
        $key = $this->plugin->get_option( 'license_key' );
        
        if ( empty( $key ) || strlen( $key ) < 8 ) {
            return '';
        }

        return substr( $key, 0, 4 ) . str_repeat( '*', strlen( $key ) - 8 ) . substr( $key, -4 );
    }
}
