<?php
/**
 * Keywords Management System
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Keywords {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Get all keywords as array
     */
    public function get_keywords() {
        $keywords_raw = $this->plugin->get_option( 'keywords', '' );
        
        if ( empty( $keywords_raw ) ) {
            return array();
        }

        // Parse keywords (supports newlines and commas)
        $keywords_raw = str_replace( array( "\r\n", "\r" ), "\n", $keywords_raw );
        $lines = explode( "\n", $keywords_raw );
        
        $keywords = array();
        foreach ( $lines as $line ) {
            $line = trim( $line );
            if ( empty( $line ) ) continue;
            
            // Check if line has commas
            if ( strpos( $line, ',' ) !== false ) {
                $parts = explode( ',', $line );
                foreach ( $parts as $part ) {
                    $part = trim( $part );
                    if ( ! empty( $part ) ) {
                        $keywords[] = $part;
                    }
                }
            } else {
                $keywords[] = $line;
            }
        }

        return array_unique( array_filter( $keywords ) );
    }

    /**
     * Get random keyword
     */
    public function get_random_keyword() {
        $keywords = $this->get_keywords();
        
        if ( empty( $keywords ) ) {
            return '';
        }

        $random_key = array_rand( $keywords );
        $keyword = $keywords[ $random_key ];
        
        // Track usage
        $this->track_usage( $keyword );
        
        return $keyword;
    }

    /**
     * Get keyword with least usage (for better distribution)
     */
    public function get_least_used_keyword() {
        global $wpdb;
        
        $keywords = $this->get_keywords();
        
        if ( empty( $keywords ) ) {
            return '';
        }

        $table = $wpdb->prefix . 'driop_keywords';
        
        // Get usage counts
        $placeholders = implode( ',', array_fill( 0, count( $keywords ), '%s' ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare -- Dynamic placeholders for IN clause
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT keyword, usage_count FROM {$wpdb->prefix}driop_keywords WHERE keyword IN ($placeholders) ORDER BY usage_count ASC",
                $keywords
            ),
            OBJECT_K
        );

        // Find keyword with least usage
        $min_usage = PHP_INT_MAX;
        $selected = $keywords[0];
        
        foreach ( $keywords as $keyword ) {
            $usage = isset( $results[ $keyword ] ) ? $results[ $keyword ]->usage_count : 0;
            if ( $usage < $min_usage ) {
                $min_usage = $usage;
                $selected = $keyword;
            }
        }

        $this->track_usage( $selected );
        
        return $selected;
    }

    /**
     * Track keyword usage
     */
    private function track_usage( $keyword ) {
        global $wpdb;
        
        $table = $wpdb->prefix . 'driop_keywords';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table insert/update
        $wpdb->query(
            $wpdb->prepare(
                "INSERT INTO {$wpdb->prefix}driop_keywords (keyword, usage_count) VALUES (%s, 1) 
                ON DUPLICATE KEY UPDATE usage_count = usage_count + 1",
                $keyword
            )
        );
    }

    /**
     * Save keywords - preserve line breaks
     * Uses sanitize_textarea_field which preserves \n
     */
    public function save_keywords( $keywords_text ) {
        // Already sanitized by caller using sanitize_textarea_field
        // This preserves line breaks properly
        $this->plugin->update_option( 'keywords', $keywords_text );
    }

    /**
     * Get keyword count
     */
    public function get_count() {
        return count( $this->get_keywords() );
    }

    /**
     * Get keywords with usage stats
     */
    public function get_keywords_with_stats() {
        global $wpdb;
        
        $keywords = $this->get_keywords();
        $table = $wpdb->prefix . 'driop_keywords';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
        $results = $wpdb->get_results(
            "SELECT keyword, usage_count FROM {$wpdb->prefix}driop_keywords ORDER BY usage_count DESC",
            OBJECT_K
        );

        $stats = array();
        foreach ( $keywords as $keyword ) {
            $stats[] = array(
                'keyword' => $keyword,
                'usage_count' => isset( $results[ $keyword ] ) ? $results[ $keyword ]->usage_count : 0,
            );
        }

        return $stats;
    }

    /**
     * Clear usage stats
     */
    public function clear_stats() {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table truncate
        $wpdb->query( "TRUNCATE TABLE {$wpdb->prefix}driop_keywords" );
    }

    /**
     * Import keywords from file content
     */
    public function import_keywords( $content ) {
        $existing = $this->plugin->get_option( 'keywords', '' );
        
        if ( ! empty( $existing ) ) {
            $existing .= "\n";
        }
        
        $existing .= $content;
        
        $this->save_keywords( $existing );
        
        return $this->get_count();
    }
}
