<?php
/**
 * Image Optimizer - SIMPLE VERSION
 * ONLY: WebP conversion + Rename
 * NO resize, NO compression
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Image_Optimizer {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Handle upload - Rename, Resize, Compress, Convert to WebP
     */
    public function handle_upload( $upload, $context = 'upload' ) {
        // Only process images
        if ( ! isset( $upload['type'] ) || strpos( $upload['type'], 'image/' ) !== 0 ) {
            return $upload;
        }

        // Skip GIFs (animated)
        if ( $upload['type'] === 'image/gif' ) {
            return $upload;
        }
        
        // Check if user wants to confirm before optimizing
        if ( $this->plugin->get_option( 'confirm_before_optimize', false ) ) {
            // User wants to manually optimize - skip auto-optimization
            return $upload;
        }

        $file_path = $upload['file'];
        
        // Step 1: Rename file if enabled
        if ( $this->plugin->get_option( 'auto_rename', true ) ) {
            $upload = $this->rename_file( $upload );
            $file_path = $upload['file'];
        }
        
        // Step 2: Resize if exceeds max dimensions
        $upload = $this->resize_image( $upload );
        $file_path = $upload['file'];
        
        // Step 3: Compress to target size if enabled
        $upload = $this->compress_image( $upload );
        $file_path = $upload['file'];

        // Step 4: Convert to WebP if enabled
        if ( $this->plugin->get_option( 'convert_to_webp', true ) && $this->supports_webp() ) {
            if ( $upload['type'] !== 'image/webp' ) {
                $upload = $this->convert_to_webp_simple( $upload );
            }
        }

        return $upload;
    }

    /**
     * Process attachment metadata - just update meta, no image changes
     */
    public function process_attachment( $metadata, $attachment_id ) {
        if ( empty( $metadata ) || ! is_array( $metadata ) ) {
            return $metadata;
        }

        // Store original info
        $file = get_attached_file( $attachment_id );
        if ( $file && file_exists( $file ) ) {
            update_post_meta( $attachment_id, '_driop_original_size', filesize( $file ) );
            // FIX #2: Set BOTH meta keys to ensure stats count properly
            update_post_meta( $attachment_id, '_driop_processed', 1 );
            update_post_meta( $attachment_id, '_driop_optimized', 1 );
            
            // Clear sitemap cache so new images appear immediately
            delete_transient( 'driop_sitemap_images_cache' );
            delete_transient( 'driop_sitemap_index_cache' );
        }

        // Increment user count
        $user_id = get_current_user_id();
        if ( $user_id ) {
            $this->increment_user_image_count( $user_id );
        }

        return $metadata;
    }

    /**
     * Simple rename - just change filename, no image manipulation
     * Uses simple sequential numbering: site-name-1, site-name-2, etc.
     * MAX 6 WORDS for SEO
     * CHECKS DATABASE to avoid conflicts with existing images
     */
    private function rename_file( $upload ) {
        $file_path = $upload['file'];
        $path_info = pathinfo( $file_path );
        
        // Build SEO-friendly base name (max 6 words)
        $site_name = $this->get_seo_name_max_words( 6 );
        
        if ( empty( $site_name ) ) {
            $site_name = 'image';
        }

        $extension = isset( $path_info['extension'] ) ? strtolower( $path_info['extension'] ) : 'jpg';
        $upload_dir = $path_info['dirname'];
        
        // Find the next available number by checking BOTH filesystem AND database
        $counter = $this->get_next_available_number( $site_name, $extension, $upload_dir );
        
        $new_filename = $site_name . '-' . $counter . '.' . $extension;
        $new_path = $upload_dir . '/' . $new_filename;
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Renaming uploaded file before WordPress processes it
        if ( @rename( $file_path, $new_path ) ) {
            $upload['file'] = $new_path;
            $upload['url'] = str_replace( basename( $upload['url'] ), $new_filename, $upload['url'] );
        }

        return $upload;
    }

    /**
     * Get next available number by checking filesystem AND database GLOBALLY.
     * FIXED: Checks across ALL upload year/month directories (not just current one)
     * and checks ALL extensions (jpg, png, webp) to prevent collisions after WebP conversion.
     * This prevents the case where site-name-keyword-1.jpg in 2025/01/ and
     * site-name-keyword-1.webp in 2025/02/ both exist and confuse WordPress.
     */
    private function get_next_available_number( $base_name, $extension, $upload_dir ) {
        global $wpdb;
        
        // Search for ALL existing files with this base name pattern across entire uploads,
        // regardless of extension (so .jpg, .png, .webp are all caught)
        $pattern = $base_name . '-';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Finding existing filenames globally
        $existing = $wpdb->get_col( $wpdb->prepare(
            "SELECT meta_value FROM {$wpdb->postmeta} 
             WHERE meta_key = '_wp_attached_file' 
             AND meta_value LIKE %s",
            '%' . $wpdb->esc_like( $pattern ) . '%'
        ) );
        
        // Also check GUIDs to catch any that may have been updated without _wp_attached_file
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Finding existing filenames in GUIDs
        $existing_guids = $wpdb->get_col( $wpdb->prepare(
            "SELECT guid FROM {$wpdb->posts} 
             WHERE post_type = 'attachment' 
             AND guid LIKE %s",
            '%' . $wpdb->esc_like( $pattern ) . '%'
        ) );
        
        $max_number = 0;
        
        // Find highest number in database (_wp_attached_file)
        foreach ( $existing as $file ) {
            $filename = basename( $file );
            // Match base-name-NUMBER with any extension
            if ( preg_match( '/' . preg_quote( $base_name, '/' ) . '-(\d+)\.[a-z]+$/i', $filename, $matches ) ) {
                $num = (int) $matches[1];
                if ( $num > $max_number ) {
                    $max_number = $num;
                }
            }
        }
        
        // Find highest number in GUIDs
        foreach ( $existing_guids as $guid ) {
            $filename = basename( $guid );
            if ( preg_match( '/' . preg_quote( $base_name, '/' ) . '-(\d+)\.[a-z]+$/i', $filename, $matches ) ) {
                $num = (int) $matches[1];
                if ( $num > $max_number ) {
                    $max_number = $num;
                }
            }
        }
        
        // Start from the next number after the highest found
        $counter = $max_number + 1;
        
        // Also verify against filesystem for the current directory (belt and suspenders)
        // Check ALL common image extensions to prevent cross-extension collisions
        $extensions_to_check = array( $extension, 'jpg', 'jpeg', 'png', 'webp' );
        $extensions_to_check = array_unique( $extensions_to_check );
        
        $collision = true;
        while ( $collision ) {
            $collision = false;
            foreach ( $extensions_to_check as $ext ) {
                $check_path = $upload_dir . '/' . $base_name . '-' . $counter . '.' . $ext;
                if ( file_exists( $check_path ) ) {
                    $collision = true;
                    $counter++;
                    break; // restart the while loop with new counter
                }
            }
        }
        
        return $counter;
    }

    /**
     * Get SEO name with max words limit
     * Combines site name + keyword, limited to max words
     */
    private function get_seo_name_max_words( $max_words = 6 ) {
        $site_name = sanitize_title( get_bloginfo( 'name' ) );
        
        // Get keyword if available
        $keywords = new DRIOP_Keywords( $this->plugin );
        $keyword = $keywords->get_random_keyword();
        
        // Combine site name and keyword
        $parts = array();
        if ( ! empty( $site_name ) ) {
            $parts[] = $site_name;
        }
        if ( ! empty( $keyword ) ) {
            $parts[] = sanitize_title( $keyword );
        }
        
        $full_name = implode( '-', $parts );
        
        // Split into words and limit
        $words = explode( '-', $full_name );
        $words = array_slice( $words, 0, $max_words );
        
        return implode( '-', $words );
    }

    /**
     * Convert to WebP - HIGH QUALITY, preserve dimensions
     */
    private function convert_to_webp_simple( $upload ) {
        $file_path = $upload['file'];
        
        if ( ! file_exists( $file_path ) ) {
            return $upload;
        }

        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return $upload;
        }

        $width = $image_info[0];
        $height = $image_info[1];
        $mime = $image_info['mime'];
        

        // Skip if already WebP or GIF
        if ( $mime === 'image/webp' || $mime === 'image/gif' ) {
            return $upload;
        }

        // Create image resource
        $source = null;
        switch ( $mime ) {
            case 'image/jpeg':
                $source = @imagecreatefromjpeg( $file_path );
                break;
            case 'image/png':
                $source = @imagecreatefrompng( $file_path );
                break;
        }

        if ( ! $source ) {
            return $upload;
        }

        // Verify source dimensions match original
        $source_width = imagesx( $source );
        $source_height = imagesy( $source );

        // Prepare output path
        $path_info = pathinfo( $file_path );
        $webp_path = $path_info['dirname'] . '/' . $path_info['filename'] . '.webp';

        // Handle PNG transparency
        if ( $mime === 'image/png' ) {
            imagepalettetotruecolor( $source );
            imagealphablending( $source, true );
            imagesavealpha( $source, true );
        }

        // Save as WebP with HIGH quality (100 = best quality)
        $quality = 100;
        $success = @imagewebp( $source, $webp_path, $quality );
        
        // Clean up
        imagedestroy( $source );

        if ( $success && file_exists( $webp_path ) ) {
            // Verify WebP dimensions
            $webp_info = @getimagesize( $webp_path );
            if ( $webp_info ) {
                
                // Only use WebP if dimensions match!
                if ( $webp_info[0] === $width && $webp_info[1] === $height ) {
                    // Delete original
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Deleting original file after WebP conversion
                    @unlink( $file_path );
                    
                    // Update upload info
                    $upload['file'] = $webp_path;
                    $upload['url'] = str_replace( $path_info['basename'], $path_info['filename'] . '.webp', $upload['url'] );
                    $upload['type'] = 'image/webp';
                } else {
                    // Dimensions don't match - something went wrong, keep original
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Cleaning up failed WebP conversion
                    @unlink( $webp_path );
                }
            }
        }

        return $upload;
    }
    
    /**
     * Resize image if it exceeds max dimensions
     */
    private function resize_image( $upload ) {
        $max_width = $this->plugin->get_option( 'max_width', 0 );
        $max_height = $this->plugin->get_option( 'max_height', 0 );
        
        // Skip if no limits set
        if ( ! $max_width && ! $max_height ) {
            return $upload;
        }
        
        $file_path = $upload['file'];
        $image_info = @getimagesize( $file_path );
        
        if ( ! $image_info ) {
            return $upload;
        }
        
        $current_width = $image_info[0];
        $current_height = $image_info[1];
        $mime_type = $image_info['mime'];
        
        // Check if resize needed
        $needs_resize = false;
        if ( $max_width > 0 && $current_width > $max_width ) {
            $needs_resize = true;
        }
        if ( $max_height > 0 && $current_height > $max_height ) {
            $needs_resize = true;
        }
        
        if ( ! $needs_resize ) {
            return $upload;
        }
        
        // Calculate new dimensions (maintain aspect ratio)
        $ratio = $current_width / $current_height;
        
        if ( $max_width > 0 && $max_height > 0 ) {
            // Both limits set - use the most restrictive
            if ( $current_width / $max_width > $current_height / $max_height ) {
                $new_width = $max_width;
                $new_height = (int) ( $max_width / $ratio );
            } else {
                $new_height = $max_height;
                $new_width = (int) ( $max_height * $ratio );
            }
        } elseif ( $max_width > 0 ) {
            $new_width = $max_width;
            $new_height = (int) ( $max_width / $ratio );
        } else {
            $new_height = $max_height;
            $new_width = (int) ( $max_height * $ratio );
        }
        
        // Create image resource
        $source = null;
        switch ( $mime_type ) {
            case 'image/jpeg':
                $source = @imagecreatefromjpeg( $file_path );
                break;
            case 'image/png':
                $source = @imagecreatefrompng( $file_path );
                break;
            case 'image/webp':
                $source = @imagecreatefromwebp( $file_path );
                break;
        }
        
        if ( ! $source ) {
            return $upload;
        }
        
        // Create resized image
        $resized = imagecreatetruecolor( $new_width, $new_height );
        
        // Preserve transparency for PNG/WebP
        if ( $mime_type === 'image/png' || $mime_type === 'image/webp' ) {
            imagealphablending( $resized, false );
            imagesavealpha( $resized, true );
            $transparent = imagecolorallocatealpha( $resized, 255, 255, 255, 127 );
            imagefilledrectangle( $resized, 0, 0, $new_width, $new_height, $transparent );
        }
        
        // Resize
        imagecopyresampled( $resized, $source, 0, 0, 0, 0, $new_width, $new_height, $current_width, $current_height );
        
        // Save back to file
        $quality = $this->plugin->get_option( 'compression_quality', 85 );
        $success = false;
        
        switch ( $mime_type ) {
            case 'image/jpeg':
                $success = @imagejpeg( $resized, $file_path, $quality );
                break;
            case 'image/png':
                $png_quality = (int) ( ( 100 - $quality ) / 11.111111 );
                $success = @imagepng( $resized, $file_path, $png_quality );
                break;
            case 'image/webp':
                $success = @imagewebp( $resized, $file_path, $quality );
                break;
        }
        
        imagedestroy( $source );
        imagedestroy( $resized );
        
        return $upload;
    }
    
    /**
     * Compress image to target file size
     */
    private function compress_image( $upload ) {
        if ( ! $this->plugin->get_option( 'auto_compress', false ) ) {
            return $upload;
        }
        
        $target_size = $this->plugin->get_option( 'target_size', 100 ) * 1024; // Convert KB to bytes
        $file_path = $upload['file'];
        
        if ( ! file_exists( $file_path ) ) {
            return $upload;
        }
        
        $current_size = filesize( $file_path );
        
        // Already under target size
        if ( $current_size <= $target_size ) {
            return $upload;
        }
        
        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return $upload;
        }
        
        $mime_type = $image_info['mime'];
        
        // Load image
        $source = null;
        switch ( $mime_type ) {
            case 'image/jpeg':
                $source = @imagecreatefromjpeg( $file_path );
                break;
            case 'image/png':
                $source = @imagecreatefrompng( $file_path );
                break;
            case 'image/webp':
                $source = @imagecreatefromwebp( $file_path );
                break;
        }
        
        if ( ! $source ) {
            return $upload;
        }
        
        // Try different quality levels
        $quality = $this->plugin->get_option( 'compression_quality', 85 );
        $min_quality = 50;
        
        while ( $quality >= $min_quality && $current_size > $target_size ) {
            // Save with current quality
            switch ( $mime_type ) {
                case 'image/jpeg':
                    @imagejpeg( $source, $file_path, $quality );
                    break;
                case 'image/png':
                    $png_quality = (int) ( ( 100 - $quality ) / 11.111111 );
                    @imagepng( $source, $file_path, $png_quality );
                    break;
                case 'image/webp':
                    @imagewebp( $source, $file_path, $quality );
                    break;
            }
            
            clearstatcache( true, $file_path );
            $current_size = filesize( $file_path );
            
            // Reduce quality for next attempt
            $quality -= 5;
        }
        
        imagedestroy( $source );
        
        return $upload;
    }

    /**
     * Check if server supports WebP
     */
    public function supports_webp() {
        if ( ! function_exists( 'imagewebp' ) ) {
            return false;
        }
        
        if ( function_exists( 'gd_info' ) ) {
            $gd_info = gd_info();
            return isset( $gd_info['WebP Support'] ) && $gd_info['WebP Support'];
        }
        
        return false;
    }

    /**
     * Get user image count
     */
    public function get_user_image_count( $user_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'driop_user_image_count';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking custom table
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            return 0;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
        $count = $wpdb->get_var( $wpdb->prepare( 
            "SELECT image_count FROM {$wpdb->prefix}driop_user_image_count WHERE user_id = %d", 
            $user_id 
        ) );
        
        return $count ? (int) $count : 0;
    }

    /**
     * Increment user image count
     */
    public function increment_user_image_count( $user_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'driop_user_image_count';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking custom table
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            return;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table insert/update
        $wpdb->query( $wpdb->prepare(
            "INSERT INTO {$wpdb->prefix}driop_user_image_count (user_id, image_count) VALUES (%d, 1) 
             ON DUPLICATE KEY UPDATE image_count = image_count + 1",
            $user_id
        ) );
    }

    /**
     * Get statistics
     */
    public function get_stats() {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Statistics query
        $total_images = $wpdb->get_var( "
            SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_type = 'attachment' 
            AND post_mime_type LIKE 'image/%'
        " );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Statistics query
        $webp_count = $wpdb->get_var( "
            SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_type = 'attachment' 
            AND post_mime_type = 'image/webp'
        " );
        
        // FIX #2: Count images with EITHER meta key for backward compatibility
        // - _driop_optimized: Set during both upload and bulk optimization (now)
        // - _driop_processed: Legacy key from earlier versions
        // Using OR ensures we count all optimized images regardless of when they were processed
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Statistics query
        $optimized = $wpdb->get_var( "
            SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta} 
            WHERE (meta_key = '_driop_optimized' OR meta_key = '_driop_processed') 
            AND meta_value = '1'
        " );
        
        return array(
            'total_images' => (int) $total_images,
            'webp_count' => (int) $webp_count,
            'optimized' => (int) $optimized,
            'pending' => max( 0, (int) $total_images - (int) $optimized ),
        );
    }

    /**
     * Optimize existing image (for bulk optimization)
     * Converts to WebP, renames file, and updates SEO
     * 
     * @param int  $attachment_id Image attachment ID
     * @param bool $reoptimize    Force re-optimization even if already done
     */
    public function optimize_existing( $attachment_id, $reoptimize = false ) {
        // Check if already optimized - skip UNLESS reoptimize is true
        if ( ! $reoptimize && get_post_meta( $attachment_id, '_driop_optimized', true ) ) {
            return array(
                'success' => true,
                'message' => 'Already optimized',
                'savings' => 0,
            );
        }
        
        $file_path = get_attached_file( $attachment_id );
        
        if ( ! $file_path || ! file_exists( $file_path ) ) {
            // Try to get from attachment metadata as fallback
            $metadata = wp_get_attachment_metadata( $attachment_id );
            if ( $metadata && isset( $metadata['file'] ) ) {
                $upload_dir = wp_upload_dir();
                $file_path = $upload_dir['basedir'] . '/' . $metadata['file'];
            }
            
            // Still not found - mark as optimized to skip in future
            if ( ! $file_path || ! file_exists( $file_path ) ) {
                update_post_meta( $attachment_id, '_driop_optimized', 1 );
                return array(
                    'success' => false,
                    'message' => 'File not found',
                    'savings' => 0,
                );
            }
        }

        $original_size = filesize( $file_path );
        $mime_type = get_post_mime_type( $attachment_id );
        
        // Skip GIFs
        if ( $mime_type === 'image/gif' ) {
            update_post_meta( $attachment_id, '_driop_optimized', 1 );
            return array(
                'success' => true,
                'message' => 'GIF skipped',
                'savings' => 0,
            );
        }

        // Step 1: Rename file with SEO name (max 6 words)
        $new_file_path = $this->rename_existing_file( $file_path, $attachment_id );
        if ( $new_file_path && $new_file_path !== $file_path ) {
            $file_path = $new_file_path;
        }

        // Step 2: Convert to WebP if not already
        $mime_type = get_post_mime_type( $attachment_id ); // Refresh mime type
        if ( $mime_type !== 'image/webp' && $this->supports_webp() ) {
            $webp_path = $this->convert_existing_to_webp( $file_path, $attachment_id );
            if ( $webp_path && $webp_path !== $file_path ) {
                $file_path = $webp_path;
            }
        }

        // Calculate savings
        clearstatcache( true, $file_path );
        $new_size = file_exists( $file_path ) ? filesize( $file_path ) : $original_size;
        $savings = $original_size - $new_size;

        // Mark as optimized
        update_post_meta( $attachment_id, '_driop_optimized', 1 );
        update_post_meta( $attachment_id, '_driop_original_size', $original_size );
        update_post_meta( $attachment_id, '_driop_optimized_size', $new_size );

        return array(
            'success' => true,
            'message' => 'Optimized',
            'savings' => $savings,
        );
    }

    /**
     * Rename existing file with SEO name (max 6 words)
     * FIXED: Checks BOTH filesystem AND database across ALL upload dirs to prevent duplicates.
     * Also renames thumbnail/intermediate sizes and updates post_name slug.
     */
    private function rename_existing_file( $file_path, $attachment_id ) {
        $path_info = pathinfo( $file_path );
        $upload_dir = $path_info['dirname'];
        $extension = isset( $path_info['extension'] ) ? strtolower( $path_info['extension'] ) : 'jpg';
        
        // Get SEO name (max 6 words)
        $seo_name = $this->get_seo_name_max_words( 6 );
        
        if ( empty( $seo_name ) ) {
            $seo_name = 'image';
        }
        
        // FIXED: Use the robust database+filesystem uniqueness checker
        $counter = $this->get_next_available_number( $seo_name, $extension, $upload_dir );
        
        $new_filename = $seo_name . '-' . $counter . '.' . $extension;
        $new_path = $upload_dir . '/' . $new_filename;
        
        // Skip if same path
        if ( $new_path === $file_path ) {
            return $file_path;
        }
        
        // FIXED: Also rename thumbnail/intermediate sizes BEFORE renaming the main file
        $metadata = wp_get_attachment_metadata( $attachment_id );
        $old_basename = $path_info['filename']; // e.g. "old-name" without extension
        $new_basename = $seo_name . '-' . $counter; // e.g. "new-name-5" without extension
        
        if ( $metadata && isset( $metadata['sizes'] ) && is_array( $metadata['sizes'] ) ) {
            foreach ( $metadata['sizes'] as $size_key => $size_data ) {
                if ( ! empty( $size_data['file'] ) ) {
                    $old_thumb_path = $upload_dir . '/' . $size_data['file'];
                    if ( file_exists( $old_thumb_path ) ) {
                        // Build new thumbnail filename: replace old base with new base, keep dimensions suffix
                        $new_thumb_file = str_replace( $old_basename, $new_basename, $size_data['file'] );
                        $new_thumb_path = $upload_dir . '/' . $new_thumb_file;
                        // phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Renaming thumbnail file
                        if ( @rename( $old_thumb_path, $new_thumb_path ) ) {
                            $metadata['sizes'][ $size_key ]['file'] = $new_thumb_file;
                        }
                    }
                }
            }
        }
        
        // Rename main file
        // phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Renaming attachment file
        if ( @rename( $file_path, $new_path ) ) {
            // Update attachment file path in database
            update_attached_file( $attachment_id, $new_path );
            
            // Update attachment metadata (including renamed thumbnails)
            if ( $metadata && isset( $metadata['file'] ) ) {
                $metadata['file'] = str_replace( $path_info['basename'], $new_filename, $metadata['file'] );
                wp_update_attachment_metadata( $attachment_id, $metadata );
            }
            
            // Update GUID
            global $wpdb;
            $old_url = wp_get_attachment_url( $attachment_id );
            $new_url = str_replace( $path_info['basename'], $new_filename, $old_url );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Updating attachment GUID and slug
            $wpdb->update( 
                $wpdb->posts, 
                array(
                    'guid' => $new_url,
                    'post_name' => sanitize_title( $new_basename ), // FIXED: Update slug too
                ),
                array( 'ID' => $attachment_id ) 
            );
            
            // FIXED: Update references in post_content across the site
            $this->update_content_references( $path_info['basename'], $new_filename );
            
            return $new_path;
        }
        
        return $file_path;
    }
    
    /**
     * Update image references across the ENTIRE database after rename/conversion.
     * Covers: post_content, postmeta (including serialized), options, and WooCommerce tables.
     * This is critical to prevent broken images after PNG→WebP or filename changes.
     */
    private function update_content_references( $old_filename, $new_filename ) {
        global $wpdb;
        
        // Only update if filenames actually differ
        if ( $old_filename === $new_filename ) {
            return;
        }
        
        $old_escaped = $wpdb->esc_like( $old_filename );
        
        // 1. Update post_content references (covers posts, pages, products, custom post types)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk update image references in content
        $wpdb->query( $wpdb->prepare(
            "UPDATE {$wpdb->posts} SET post_content = REPLACE(post_content, %s, %s) WHERE post_content LIKE %s",
            $old_filename,
            $new_filename,
            '%' . $old_escaped . '%'
        ) );
        
        // 2. Update postmeta values (covers page builders like Elementor, ACF fields, custom fields)
        // Only update non-serialized values to avoid corrupting serialized data
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk update image references in postmeta
        $wpdb->query( $wpdb->prepare(
            "UPDATE {$wpdb->postmeta} SET meta_value = REPLACE(meta_value, %s, %s) 
             WHERE meta_value LIKE %s 
             AND meta_value NOT LIKE 'a:%' 
             AND meta_value NOT LIKE 'O:%'
             AND meta_key != '_wp_attached_file'
             AND meta_key != '_wp_attachment_metadata'",
            $old_filename,
            $new_filename,
            '%' . $old_escaped . '%'
        ) );
        
        // 3. Handle serialized postmeta (Elementor, Divi, WPBakery, etc.)
        // We need to do safe serialized replacement with length correction
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Finding serialized meta with old references
        $serialized_rows = $wpdb->get_results( $wpdb->prepare(
            "SELECT meta_id, meta_value FROM {$wpdb->postmeta} 
             WHERE meta_value LIKE %s 
             AND (meta_value LIKE 'a:%%' OR meta_value LIKE 'O:%%')
             AND meta_key != '_wp_attachment_metadata'",
            '%' . $old_escaped . '%'
        ) );
        
        if ( $serialized_rows ) {
            foreach ( $serialized_rows as $row ) {
                $new_value = $this->safe_serialized_replace( $row->meta_value, $old_filename, $new_filename );
                if ( $new_value !== $row->meta_value ) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Updating serialized meta
                    $wpdb->update(
                        $wpdb->postmeta,
                        array( 'meta_value' => $new_value ),
                        array( 'meta_id' => $row->meta_id )
                    );
                }
            }
        }
        
        // 4. Update options table (widgets, customizer, theme mods)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk update image references in options
        $wpdb->query( $wpdb->prepare(
            "UPDATE {$wpdb->options} SET option_value = REPLACE(option_value, %s, %s) 
             WHERE option_value LIKE %s 
             AND option_value NOT LIKE 'a:%' 
             AND option_value NOT LIKE 'O:%'
             AND option_name NOT LIKE '_transient%'",
            $old_filename,
            $new_filename,
            '%' . $old_escaped . '%'
        ) );
        
        // 5. Handle serialized options (widget data, customizer, etc.)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Finding serialized options with old references
        $serialized_options = $wpdb->get_results( $wpdb->prepare(
            "SELECT option_id, option_value FROM {$wpdb->options} 
             WHERE option_value LIKE %s 
             AND (option_value LIKE 'a:%%' OR option_value LIKE 'O:%%')
             AND option_name NOT LIKE '_transient%%'",
            '%' . $old_escaped . '%'
        ) );
        
        if ( $serialized_options ) {
            foreach ( $serialized_options as $row ) {
                $new_value = $this->safe_serialized_replace( $row->option_value, $old_filename, $new_filename );
                if ( $new_value !== $row->option_value ) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Updating serialized option
                    $wpdb->update(
                        $wpdb->options,
                        array( 'option_value' => $new_value ),
                        array( 'option_id' => $row->option_id )
                    );
                }
            }
        }
    }
    
    /**
     * Safely replace strings inside serialized data.
     * Handles PHP serialized string length correction so unserialization doesn't break.
     * Example: s:15:"old-image-1.png" → s:16:"old-image-1.webp"
     */
    private function safe_serialized_replace( $serialized, $search, $replace ) {
        // Try unserialize → replace → reserialize approach first (safest)
        $data = @unserialize( $serialized );
        if ( $data !== false ) {
            $data = $this->recursive_replace( $data, $search, $replace );
            $new_serialized = serialize( $data );
            return $new_serialized;
        }
        
        // Fallback: regex-based serialized string length correction
        // This handles cases where unserialize fails (e.g., broken serialization)
        $search_len = strlen( $search );
        $replace_len = strlen( $replace );
        $diff = $replace_len - $search_len;
        
        // Find all serialized string entries containing the search string
        // Pattern: s:LENGTH:"...search_string..."
        $result = preg_replace_callback(
            '/s:(\d+):"([^"]*' . preg_quote( $search, '/' ) . '[^"]*)"/',
            function ( $matches ) use ( $search, $replace, $diff ) {
                $old_len = (int) $matches[1];
                $old_str = $matches[2];
                $new_str = str_replace( $search, $replace, $old_str );
                $count = substr_count( $old_str, $search );
                $new_len = $old_len + ( $diff * $count );
                return 's:' . $new_len . ':"' . $new_str . '"';
            },
            $serialized
        );
        
        return $result !== null ? $result : $serialized;
    }
    
    /**
     * Recursively replace a string in arrays/objects (for unserialized data).
     */
    private function recursive_replace( $data, $search, $replace ) {
        if ( is_string( $data ) ) {
            return str_replace( $search, $replace, $data );
        }
        
        if ( is_array( $data ) ) {
            foreach ( $data as $key => $value ) {
                $data[ $key ] = $this->recursive_replace( $value, $search, $replace );
            }
        }
        
        if ( is_object( $data ) ) {
            foreach ( get_object_vars( $data ) as $prop => $value ) {
                $data->$prop = $this->recursive_replace( $value, $search, $replace );
            }
        }
        
        return $data;
    }

    /**
     * Convert existing file to WebP
     * FIXED: Updates GUID, post_name, and thumbnail sizes after conversion.
     * FIXED: Keeps original file as fallback for browsers that don't support WebP
     *        and for backward compatibility with hardcoded URLs.
     *        Uses .htaccess rewrite to serve WebP when browser supports it.
     */
    private function convert_existing_to_webp( $file_path, $attachment_id ) {
        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return $file_path;
        }

        $mime = $image_info['mime'];
        $width = $image_info[0];
        $height = $image_info[1];

        // Skip if already WebP or GIF
        if ( $mime === 'image/webp' || $mime === 'image/gif' ) {
            return $file_path;
        }

        // Create image resource
        $source = null;
        switch ( $mime ) {
            case 'image/jpeg':
                $source = @imagecreatefromjpeg( $file_path );
                break;
            case 'image/png':
                $source = @imagecreatefrompng( $file_path );
                break;
        }

        if ( ! $source ) {
            return $file_path;
        }

        // Prepare output path
        $path_info = pathinfo( $file_path );
        $webp_filename = $path_info['filename'] . '.webp';
        $webp_path = $path_info['dirname'] . '/' . $webp_filename;
        
        // FIXED: Check if a .webp with the same name already exists on disk
        if ( file_exists( $webp_path ) && $webp_path !== $file_path ) {
            imagedestroy( $source );
            return $file_path; // Don't overwrite someone else's file
        }

        // Handle PNG transparency
        if ( $mime === 'image/png' ) {
            imagepalettetotruecolor( $source );
            imagealphablending( $source, true );
            imagesavealpha( $source, true );
        }

        // Save as WebP with high quality
        $success = @imagewebp( $source, $webp_path, 100 );
        imagedestroy( $source );

        if ( $success && file_exists( $webp_path ) ) {
            // Verify dimensions match
            $webp_info = @getimagesize( $webp_path );
            if ( $webp_info && $webp_info[0] === $width && $webp_info[1] === $height ) {
                
                // FIXED: KEEP the original file instead of deleting it!
                // Store original path so we can reference it / clean up later if desired
                update_post_meta( $attachment_id, '_driop_original_file', $file_path );
                update_post_meta( $attachment_id, '_driop_original_extension', $path_info['extension'] );
                
                // Update attachment to point to WebP
                update_attached_file( $attachment_id, $webp_path );
                
                // Update mime type
                wp_update_post( array(
                    'ID' => $attachment_id,
                    'post_mime_type' => 'image/webp',
                ) );

                // Update attachment metadata
                $metadata = wp_get_attachment_metadata( $attachment_id );
                if ( $metadata ) {
                    $old_basename = $path_info['basename'];
                    $metadata['file'] = str_replace( $old_basename, $webp_filename, $metadata['file'] );
                    
                    // FIXED: Convert thumbnail/intermediate sizes to WebP too (keep originals)
                    if ( isset( $metadata['sizes'] ) && is_array( $metadata['sizes'] ) ) {
                        foreach ( $metadata['sizes'] as $size_key => $size_data ) {
                            if ( ! empty( $size_data['file'] ) ) {
                                $thumb_path = $path_info['dirname'] . '/' . $size_data['file'];
                                if ( file_exists( $thumb_path ) ) {
                                    $thumb_info = pathinfo( $size_data['file'] );
                                    $thumb_webp_name = $thumb_info['filename'] . '.webp';
                                    $thumb_webp_path = $path_info['dirname'] . '/' . $thumb_webp_name;
                                    
                                    $thumb_source = null;
                                    switch ( $mime ) {
                                        case 'image/jpeg':
                                            $thumb_source = @imagecreatefromjpeg( $thumb_path );
                                            break;
                                        case 'image/png':
                                            $thumb_source = @imagecreatefrompng( $thumb_path );
                                            break;
                                    }
                                    
                                    if ( $thumb_source ) {
                                        if ( $mime === 'image/png' ) {
                                            imagepalettetotruecolor( $thumb_source );
                                            imagealphablending( $thumb_source, true );
                                            imagesavealpha( $thumb_source, true );
                                        }
                                        
                                        if ( @imagewebp( $thumb_source, $thumb_webp_path, 100 ) ) {
                                            // KEEP original thumbnail — don't unlink
                                            $metadata['sizes'][ $size_key ]['file'] = $thumb_webp_name;
                                            $metadata['sizes'][ $size_key ]['mime-type'] = 'image/webp';
                                        }
                                        imagedestroy( $thumb_source );
                                    }
                                }
                            }
                        }
                    }
                    
                    wp_update_attachment_metadata( $attachment_id, $metadata );
                }
                
                // FIXED: Update GUID after WebP conversion (was missing!)
                global $wpdb;
                $old_url = wp_get_attachment_url( $attachment_id );
                $new_url = preg_replace( '/\.' . preg_quote( $path_info['extension'], '/' ) . '$/', '.webp', $old_url );
                if ( $new_url !== $old_url ) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Updating GUID after WebP conversion
                    $wpdb->update(
                        $wpdb->posts,
                        array( 'guid' => $new_url ),
                        array( 'ID' => $attachment_id )
                    );
                }
                
                // FIXED: Update content references from old extension to .webp
                $this->update_content_references( $path_info['basename'], $webp_filename );
                
                // FIXED: Ensure .htaccess WebP rewrite rules are in place
                $this->ensure_webp_htaccess_rules();

                return $webp_path;
            } else {
                // Dimensions don't match, delete webp and keep original
                // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Cleaning up failed WebP conversion
                @unlink( $webp_path );
            }
        }

        return $file_path;
    }
    
    /**
     * Add .htaccess rewrite rules to the uploads directory.
     * 
     * Strategy for SEO:
     *   - When a bot/browser requests old .jpg/.png URL and .webp exists:
     *     → 301 redirect to .webp URL (proper SEO canonical signal)
     *   - When a browser that doesn't support WebP requests .webp:
     *     → Serve the original .jpg/.png as fallback (transparent rewrite, no redirect)
     *   - Vary: Accept header for CDN compatibility
     * 
     * This ensures:
     *   1. Google only indexes the .webp URL (one canonical per image)
     *   2. Old bookmarks/links get properly redirected
     *   3. Older browsers still see the image (fallback)
     *   4. No duplicate content in search results
     */
    private function ensure_webp_htaccess_rules() {
        // Only add once per request
        static $rules_checked = false;
        if ( $rules_checked ) {
            return;
        }
        $rules_checked = true;
        
        $upload_dir = wp_upload_dir();
        $htaccess_path = $upload_dir['basedir'] . '/.htaccess';
        
        $marker = 'DRIOP WebP Rewrite';
        
        // Check if our rules already exist
        if ( file_exists( $htaccess_path ) ) {
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Reading .htaccess
            $existing = file_get_contents( $htaccess_path );
            if ( strpos( $existing, $marker ) !== false ) {
                return; // Already added
            }
        }
        
        $rules = "\n# BEGIN " . $marker . "\n";
        $rules .= "<IfModule mod_rewrite.c>\n";
        $rules .= "RewriteEngine On\n\n";
        
        // Rule 1: 301 redirect old .jpg/.png URLs → .webp when WebP version exists
        // This tells Google "the canonical image is the .webp one" — prevents duplicate indexing
        // Pattern: image-name.jpg → image-name.webp (where extension is replaced)
        $rules .= "# 301 redirect old jpg/png to webp (SEO canonical)\n";
        $rules .= "RewriteCond %{HTTP_ACCEPT} image/webp\n";
        $rules .= "RewriteCond %{REQUEST_FILENAME} ^(.+)\\.(jpe?g|png)$\n";
        $rules .= "RewriteCond %1.webp -f\n";
        $rules .= "RewriteRule ^(.+)\\.(jpe?g|png)$ $1.webp [R=301,L]\n\n";
        
        // Rule 2: Fallback for browsers WITHOUT WebP support
        // If .webp is requested but browser doesn't support it, serve original silently
        $rules .= "# Fallback: serve original to non-WebP browsers\n";
        $rules .= "RewriteCond %{HTTP_ACCEPT} !image/webp\n";
        $rules .= "RewriteCond %{REQUEST_FILENAME} ^(.+)\\.webp$\n";
        $rules .= "RewriteCond %1.jpg -f\n";
        $rules .= "RewriteRule ^(.+)\\.webp$ $1.jpg [T=image/jpeg,L]\n\n";
        
        $rules .= "RewriteCond %{HTTP_ACCEPT} !image/webp\n";
        $rules .= "RewriteCond %{REQUEST_FILENAME} ^(.+)\\.webp$\n";
        $rules .= "RewriteCond %1.png -f\n";
        $rules .= "RewriteRule ^(.+)\\.webp$ $1.png [T=image/png,L]\n\n";

        // Rule 3: If .webp doesn't exist on disk, try original (safety net)
        $rules .= "# Safety: if webp file missing, try original\n";
        $rules .= "RewriteCond %{REQUEST_FILENAME} ^(.+)\\.webp$\n";
        $rules .= "RewriteCond %{REQUEST_FILENAME} !-f\n";
        $rules .= "RewriteCond %1.jpg -f\n";
        $rules .= "RewriteRule ^(.+)\\.webp$ $1.jpg [T=image/jpeg,L]\n\n";
        
        $rules .= "RewriteCond %{REQUEST_FILENAME} ^(.+)\\.webp$\n";
        $rules .= "RewriteCond %{REQUEST_FILENAME} !-f\n";
        $rules .= "RewriteCond %1.png -f\n";
        $rules .= "RewriteRule ^(.+)\\.webp$ $1.png [T=image/png,L]\n";
        
        $rules .= "</IfModule>\n\n";
        
        // Add Vary header so CDNs/proxies cache both versions correctly
        $rules .= "<IfModule mod_headers.c>\n";
        $rules .= "  <FilesMatch \"\\.(jpe?g|png|webp)$\">\n";
        $rules .= "    Header append Vary Accept\n";
        $rules .= "  </FilesMatch>\n";
        $rules .= "</IfModule>\n\n";
        
        // Serve WebP with correct MIME type
        $rules .= "<IfModule mod_mime.c>\n";
        $rules .= "  AddType image/webp .webp\n";
        $rules .= "</IfModule>\n";
        $rules .= "# END " . $marker . "\n";
        
        // Append to .htaccess
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Writing htaccess rules
        @file_put_contents( $htaccess_path, $rules, FILE_APPEND | LOCK_EX );
    }
}
