<?php
/**
 * Dokan Integration - Enhanced Vendor Management
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Dokan {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        
        if ( $this->is_active() ) {
            $this->init_hooks();
        }
    }
    
    /**
     * Initialize hooks for Dokan integration
     */
    private function init_hooks() {
        // Add vendor dashboard widget (Agency plan)
        if ( $this->plugin->has_feature( 'vendor_dashboard' ) ) {
            add_action( 'dokan_dashboard_content_inside_before', array( $this, 'render_vendor_dashboard_widget' ) );
        }
        
        // Add image optimization stats to vendor settings
        add_action( 'dokan_settings_after_store_phone', array( $this, 'render_vendor_image_stats' ) );
        
        // Track vendor image uploads
        add_action( 'add_attachment', array( $this, 'track_vendor_upload' ) );
    }

    public function is_active() {
        return class_exists( 'WeDevs_Dokan' ) || function_exists( 'dokan' );
    }

    /**
     * Check vendor image limit before upload
     */
    public function check_vendor_image_limit( $upload, $vendor_id = 0 ) {
        if ( ! $this->plugin->has_feature( 'per_vendor_limits' ) ) {
            return $upload;
        }

        if ( ! $vendor_id ) {
            $vendor_id = get_current_user_id();
        }

        if ( ! $this->is_vendor( $vendor_id ) ) {
            return $upload;
        }

        $vendor_plan = $this->get_vendor_plan( $vendor_id );
        $max_per_vendor = $this->get_vendor_image_limit( $vendor_id, $vendor_plan );
        
        if ( $max_per_vendor <= 0 ) {
            return $upload; // Unlimited
        }

        $current_count = $this->get_vendor_image_count( $vendor_id );

        if ( $current_count >= $max_per_vendor ) {
            return array(
                'error' => sprintf(
                    // translators: %d is the maximum number of images allowed
                    __( 'You have reached your image limit of %d images. Please upgrade your plan or delete unused images.', 'digital-rise-image-optimizer-for-seo' ),
                    $max_per_vendor
                ),
            );
        }

        return $upload;
    }

    /**
     * Get vendor plan
     */
    public function get_vendor_plan( $vendor_id ) {
        $plan = get_user_meta( $vendor_id, '_driop_vendor_plan', true );
        return $plan ? $plan : 'basic';
    }
    
    /**
     * Set vendor plan
     */
    public function set_vendor_plan( $vendor_id, $plan ) {
        update_user_meta( $vendor_id, '_driop_vendor_plan', $plan );
    }
    
    /**
     * Get vendor image limit based on plan
     */
    public function get_vendor_image_limit( $vendor_id, $plan = null ) {
        if ( ! $plan ) {
            $plan = $this->get_vendor_plan( $vendor_id );
        }
        
        $limits = $this->get_vendor_plan_limits();
        return isset( $limits[ $plan ]['images'] ) ? $limits[ $plan ]['images'] : 50;
    }
    
    /**
     * Get vendor plan limits configuration
     */
    public function get_vendor_plan_limits() {
        return array(
            'basic' => array(
                'name' => __( 'Basic Vendor', 'digital-rise-image-optimizer-for-seo' ),
                'images' => 50,
                'products' => 20,
            ),
            'standard' => array(
                'name' => __( 'Standard Vendor', 'digital-rise-image-optimizer-for-seo' ),
                'images' => 150,
                'products' => 50,
            ),
            'premium' => array(
                'name' => __( 'Premium Vendor', 'digital-rise-image-optimizer-for-seo' ),
                'images' => 500,
                'products' => 200,
            ),
            'unlimited' => array(
                'name' => __( 'Unlimited Vendor', 'digital-rise-image-optimizer-for-seo' ),
                'images' => 0, // 0 = unlimited
                'products' => 0,
            ),
        );
    }

    public function is_vendor( $user_id ) {
        if ( function_exists( 'dokan_is_user_seller' ) ) {
            return dokan_is_user_seller( $user_id );
        }
        $user = get_user_by( 'id', $user_id );
        if ( ! $user ) return false;
        return in_array( 'seller', (array) $user->roles, true ) || in_array( 'vendor', (array) $user->roles, true );
    }

    public function get_vendor_image_count( $vendor_id ) {
        global $wpdb;
        $like_pattern = $wpdb->esc_like( 'image/' ) . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Vendor-specific count, not cacheable
        return (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'attachment' AND post_mime_type LIKE %s AND post_author = %d",
            $like_pattern,
            $vendor_id
        ) );
    }
    
    /**
     * Get vendor optimized image count
     */
    public function get_vendor_optimized_count( $vendor_id ) {
        global $wpdb;
        $like_pattern = $wpdb->esc_like( 'image/' ) . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Vendor-specific count, not cacheable
        return (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p 
             INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
             WHERE p.post_type = 'attachment' 
             AND p.post_mime_type LIKE %s 
             AND p.post_author = %d 
             AND pm.meta_key = '_driop_optimized' 
             AND pm.meta_value = '1'",
            $like_pattern,
            $vendor_id
        ) );
    }
    
    /**
     * Get vendor total savings
     */
    public function get_vendor_savings( $vendor_id ) {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Vendor-specific stats, not cacheable
        $result = $wpdb->get_row( $wpdb->prepare(
            "SELECT SUM(pm1.meta_value) as original, SUM(pm2.meta_value) as optimized 
             FROM {$wpdb->posts} p 
             INNER JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_driop_original_size'
             INNER JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_driop_optimized_size'
             WHERE p.post_type = 'attachment' 
             AND p.post_author = %d",
            $vendor_id
        ) );
        
        if ( $result && $result->original > 0 ) {
            return array(
                'original' => (int) $result->original,
                'optimized' => (int) $result->optimized,
                'saved' => (int) $result->original - (int) $result->optimized,
                'percent' => round( ( ( $result->original - $result->optimized ) / $result->original ) * 100, 1 ),
            );
        }
        
        return array( 'original' => 0, 'optimized' => 0, 'saved' => 0, 'percent' => 0 );
    }
    
    /**
     * Track vendor upload
     */
    public function track_vendor_upload( $attachment_id ) {
        $author_id = get_post_field( 'post_author', $attachment_id );
        if ( $this->is_vendor( $author_id ) ) {
            // Update vendor upload timestamp
            update_user_meta( $author_id, '_driop_last_upload', current_time( 'mysql' ) );
        }
    }
    
    /**
     * Render vendor dashboard widget
     */
    public function render_vendor_dashboard_widget() {
        if ( ! $this->plugin->has_feature( 'vendor_dashboard' ) ) {
            return;
        }
        
        $vendor_id = get_current_user_id();
        if ( ! $this->is_vendor( $vendor_id ) ) {
            return;
        }
        
        $plan = $this->get_vendor_plan( $vendor_id );
        $plans = $this->get_vendor_plan_limits();
        $plan_name = isset( $plans[ $plan ]['name'] ) ? $plans[ $plan ]['name'] : __( 'Basic', 'digital-rise-image-optimizer-for-seo' );
        
        $image_count = $this->get_vendor_image_count( $vendor_id );
        $image_limit = $this->get_vendor_image_limit( $vendor_id, $plan );
        $optimized_count = $this->get_vendor_optimized_count( $vendor_id );
        $savings = $this->get_vendor_savings( $vendor_id );
        
        $usage_percent = $image_limit > 0 ? min( 100, round( ( $image_count / $image_limit ) * 100 ) ) : 0;
        $bar_color = $usage_percent > 80 ? '#dc3545' : ( $usage_percent > 60 ? '#ffc107' : '#28a745' );
        ?>
        <div class="dokan-panel dokan-panel-default" style="margin-bottom: 20px;">
            <div class="dokan-panel-heading"><strong><?php esc_html_e( 'Image Optimization', 'digital-rise-image-optimizer-for-seo' ); ?></strong></div>
            <div class="dokan-panel-body">
                <div style="display: flex; flex-wrap: wrap; gap: 20px;">
                    <div style="flex: 1; min-width: 200px;">
                        <h4 style="margin: 0 0 10px;"><?php esc_html_e( 'Your Plan:', 'digital-rise-image-optimizer-for-seo' ); ?> <span style="color: #0073aa;"><?php echo esc_html( $plan_name ); ?></span></h4>
                        
                        <p style="margin: 5px 0;"><strong><?php esc_html_e( 'Images Used:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> 
                            <?php echo esc_html( $image_count ); ?> / <?php echo $image_limit > 0 ? esc_html( $image_limit ) : '∞'; ?>
                        </p>
                        
                        <?php if ( $image_limit > 0 ) : ?>
                        <div style="background: #e9ecef; border-radius: 4px; height: 10px; margin: 10px 0;">
                            <div style="background: <?php echo esc_attr( $bar_color ); ?>; width: <?php echo esc_attr( $usage_percent ); ?>%; height: 100%; border-radius: 4px;"></div>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <div style="flex: 1; min-width: 200px;">
                        <h4 style="margin: 0 0 10px;"><?php esc_html_e( 'Optimization Stats', 'digital-rise-image-optimizer-for-seo' ); ?></h4>
                        <?php
                        // translators: %d is the number of optimized images
                        ?>
                        <p style="margin: 5px 0;">✅ <?php printf( esc_html__( '%d images optimized', 'digital-rise-image-optimizer-for-seo' ), (int) $optimized_count ); ?></p>
                        <?php
                        // translators: %s is the file size saved (e.g., "5 MB")
                        ?>
                        <p style="margin: 5px 0;">💾 <?php printf( esc_html__( '%s saved', 'digital-rise-image-optimizer-for-seo' ), esc_html( size_format( $savings['saved'] ) ) ); ?></p>
                        <?php if ( $savings['percent'] > 0 ) : ?>
                        <?php
                        // translators: %s is the percentage reduction
                        ?>
                        <p style="margin: 5px 0; color: #28a745;"><strong><?php printf( esc_html__( '%s%% reduction', 'digital-rise-image-optimizer-for-seo' ), esc_html( $savings['percent'] ) ); ?></strong></p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render vendor image stats in settings
     */
    public function render_vendor_image_stats( $current_user ) {
        if ( ! $this->is_vendor( $current_user ) ) {
            return;
        }
        
        $image_count = $this->get_vendor_image_count( $current_user );
        $optimized_count = $this->get_vendor_optimized_count( $current_user );
        ?>
        <div class="dokan-form-group">
            <label class="dokan-w3 dokan-control-label"><?php esc_html_e( 'Image Stats', 'digital-rise-image-optimizer-for-seo' ); ?></label>
            <div class="dokan-w5">
                <p class="dokan-form-control-static">
                    <?php
                    // translators: %1$d is total images, %2$d is optimized images
                    printf( esc_html__( '%1$d total images, %2$d optimized', 'digital-rise-image-optimizer-for-seo' ), (int) $image_count, (int) $optimized_count );
                    ?>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Get all vendors with stats
     */
    public function get_all_vendors_stats() {
        if ( ! $this->is_active() ) {
            return array();
        }
        
        $vendors = get_users( array(
            'role__in' => array( 'seller', 'vendor' ),
            'number' => 100,
        ) );
        
        $stats = array();
        foreach ( $vendors as $vendor ) {
            $stats[] = array(
                'id' => $vendor->ID,
                'name' => $vendor->display_name,
                'email' => $vendor->user_email,
                'plan' => $this->get_vendor_plan( $vendor->ID ),
                'images' => $this->get_vendor_image_count( $vendor->ID ),
                'limit' => $this->get_vendor_image_limit( $vendor->ID ),
                'optimized' => $this->get_vendor_optimized_count( $vendor->ID ),
            );
        }
        
        return $stats;
    }

    public function get_stats() {
        if ( ! $this->is_active() ) {
            return array( 'active' => false );
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Dokan vendor count
        $vendors = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->usermeta} WHERE meta_key = 'dokan_enable_selling' AND meta_value = 'yes'"
        );

        return array(
            'active' => true,
            'total_vendors' => (int) $vendors,
        );
    }
}
