<?php
/**
 * Plugin Activator
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Activator {

    public static function activate() {
        self::create_tables();
        self::set_default_options();
        self::create_upload_directories();
        self::schedule_events();
        
        // Force immediate checkin to fetch notices
        self::do_initial_checkin();
        
        // PHASE 5: Trigger sitemap and robots generation
        do_action( 'driop_activated' );
        
        set_transient( 'driop_activation_redirect', true, 30 );
    }
    
    /**
     * Do initial checkin to fetch notices immediately
     */
    private static function do_initial_checkin() {
        $options = get_option( 'driop_options', array() );
        
        // Generate installation ID if not exists
        if ( empty( $options['installation_id'] ) ) {
            $options['installation_id'] = wp_generate_uuid4();
            update_option( 'driop_options', $options );
        }
        
        $site_data = array(
            'site_url' => home_url(),
            'site_name' => get_bloginfo( 'name' ),
            'admin_email' => get_option( 'admin_email' ),
            'user_email' => isset( $options['user_email'] ) ? $options['user_email'] : '',
            'user_whatsapp' => isset( $options['user_whatsapp'] ) ? $options['user_whatsapp'] : '',
            'wp_version' => get_bloginfo( 'version' ),
            'php_version' => phpversion(),
            'plugin_version' => defined( 'DRIOP_VERSION' ) ? DRIOP_VERSION : '1.0.0',
            'current_plan' => isset( $options['current_plan'] ) ? $options['current_plan'] : 'free',
            'license_key' => isset( $options['license_key'] ) ? $options['license_key'] : '',
            'license_status' => isset( $options['license_status'] ) ? $options['license_status'] : 'inactive',
            'installation_id' => $options['installation_id'],
            'installation_date' => isset( $options['installation_date'] ) ? $options['installation_date'] : current_time( 'mysql' ),
            'woocommerce_active' => class_exists( 'WooCommerce' ),
            'dokan_active' => class_exists( 'WeDevs_Dokan' ) || function_exists( 'dokan' ),
            'multisite' => is_multisite(),
            'locale' => get_locale(),
            'is_first_checkin' => true,
        );
        
        $api_url = defined( 'DRIOP_OWNER_API' ) ? DRIOP_OWNER_API : 'https://digital-rise-solutions.com/wp-json/driop-license/v1/';
        
        $response = wp_remote_post( $api_url . 'checkin', array(
            'timeout' => 15,
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body' => wp_json_encode( $site_data ),
        ));
        
        if ( ! is_wp_error( $response ) ) {
            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );
            
            if ( $data && isset( $data['notices'] ) && is_array( $data['notices'] ) ) {
                global $wpdb;
                $table = $wpdb->prefix . 'driop_remote_notices';
                
                foreach ( $data['notices'] as $notice ) {
                    $notice_id = isset( $notice['notice_id'] ) ? $notice['notice_id'] : '';
                    $notice_content = isset( $notice['notice_content'] ) ? $notice['notice_content'] : '';
                    
                    if ( empty( $notice_id ) || empty( $notice_content ) ) {
                        continue;
                    }
                    
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check during activation
                    $existing = $wpdb->get_var( $wpdb->prepare(
                        "SELECT id FROM {$wpdb->prefix}driop_remote_notices WHERE notice_id = %s",
                        $notice_id
                    ));
                    
                    if ( ! $existing ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert during activation
                        $wpdb->insert( $table, array(
                            'notice_id' => sanitize_text_field( $notice_id ),
                            'notice_type' => sanitize_text_field( isset( $notice['notice_type'] ) ? $notice['notice_type'] : 'info' ),
                            'notice_title' => sanitize_text_field( isset( $notice['notice_title'] ) ? $notice['notice_title'] : '' ),
                            'notice_content' => wp_kses_post( $notice_content ),
                            'notice_link' => esc_url_raw( isset( $notice['notice_link'] ) ? $notice['notice_link'] : '' ),
                            'is_dismissed' => 0,
                        ));
                    }
                }
            }
        }
    }

    private static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Optimization log
        $table1 = $wpdb->prefix . 'driop_optimization_log';
        $sql = "CREATE TABLE IF NOT EXISTS $table1 (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            attachment_id bigint(20) unsigned NOT NULL,
            user_id bigint(20) unsigned NOT NULL,
            original_size bigint(20) unsigned NOT NULL,
            optimized_size bigint(20) unsigned NOT NULL,
            savings_percent float NOT NULL,
            optimization_type varchar(50) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY attachment_id (attachment_id),
            KEY user_id (user_id)
        ) $charset_collate;";
        
        // User image count
        $table2 = $wpdb->prefix . 'driop_user_image_count';
        $sql .= "CREATE TABLE IF NOT EXISTS $table2 (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            user_id bigint(20) unsigned NOT NULL,
            image_count int(11) NOT NULL DEFAULT 0,
            last_updated datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY user_id (user_id)
        ) $charset_collate;";
        
        // Remote notices
        $table3 = $wpdb->prefix . 'driop_remote_notices';
        $sql .= "CREATE TABLE IF NOT EXISTS $table3 (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            notice_id varchar(100) NOT NULL,
            notice_type varchar(50) NOT NULL,
            notice_title varchar(255) DEFAULT NULL,
            notice_content text NOT NULL,
            notice_link varchar(500) DEFAULT NULL,
            is_dismissed tinyint(1) NOT NULL DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY notice_id (notice_id)
        ) $charset_collate;";
        
        // Keywords
        $table4 = $wpdb->prefix . 'driop_keywords';
        $sql .= "CREATE TABLE IF NOT EXISTS $table4 (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            keyword varchar(255) NOT NULL,
            usage_count int(11) NOT NULL DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY keyword (keyword)
        ) $charset_collate;";
        
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
        
        update_option( 'driop_db_version', DRIOP_VERSION );
    }

    private static function set_default_options() {
        $existing = get_option( 'driop_options', array() );
        
        $defaults = array(
            'current_plan' => 'free',
            'license_key' => '',
            'license_status' => 'inactive',
            'license_expires' => '',
            'max_width' => 1920,
            'max_height' => 1080,
            'target_file_size' => 100,
            'compression_quality' => 85,
            'auto_compress' => false, // Changed to false - manual compression
            'max_images_per_user' => 0,
            'max_images_per_product' => 10,
            'max_images_per_vendor' => 50,
            'keywords' => '',
            'seo_pattern' => '{site_name} {content_title} {keyword}',
            'auto_rename' => true,
            'auto_alt' => true,
            'auto_description' => true,
            'convert_to_webp' => true,
            'remote_connection' => true,
            'remote_api_key' => wp_generate_password( 32, false ),
            'show_attribution' => true,
            'wc_integration' => true,
            'dokan_integration' => true,
            'installation_date' => current_time( 'mysql' ),
            'installation_id' => wp_generate_uuid4(),
            'user_email' => '',
            'user_whatsapp' => '',
        );
        
        // Merge with existing - PRESERVE existing values (including license!)
        if ( ! empty( $existing ) && is_array( $existing ) ) {
            // Keep existing license and settings
            $merged = array_merge( $defaults, $existing );
            
            // Ensure installation_id is never regenerated if exists
            if ( ! empty( $existing['installation_id'] ) ) {
                $merged['installation_id'] = $existing['installation_id'];
            }
            
            // Ensure license is preserved
            if ( ! empty( $existing['license_key'] ) ) {
                $merged['license_key'] = $existing['license_key'];
                $merged['license_status'] = $existing['license_status'] ?? 'active';
                $merged['license_expires'] = $existing['license_expires'] ?? '';
                $merged['current_plan'] = $existing['current_plan'] ?? 'free';
            }
            
            update_option( 'driop_options', $merged );
        } else {
            // Fresh install
            update_option( 'driop_options', $defaults );
        }
    }

    private static function create_upload_directories() {
        $upload_dir = wp_upload_dir();
        $driop_dir = $upload_dir['basedir'] . '/driop-optimized';
        
        if ( ! file_exists( $driop_dir ) ) {
            wp_mkdir_p( $driop_dir );
            file_put_contents( $driop_dir . '/index.php', '<?php // Silence is golden' );
            file_put_contents( $driop_dir . '/.htaccess', 'Options -Indexes' );
        }
    }

    private static function schedule_events() {
        if ( ! wp_next_scheduled( 'driop_remote_checkin' ) ) {
            wp_schedule_event( time(), 'daily', 'driop_remote_checkin' );
        }
        if ( ! wp_next_scheduled( 'driop_license_check' ) ) {
            wp_schedule_event( time(), 'daily', 'driop_license_check' );
        }
    }
}
