<?php
/**
 * Admin Dashboard
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Admin {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        
        // Add AJAX handler for review submission
        add_action( 'wp_ajax_driop_submit_review', array( $this, 'handle_review_submission' ) );
    }

    public function enqueue_styles( $hook ) {
        if ( strpos( $hook, 'driop' ) === false ) return;
        wp_enqueue_style( 'driop-admin', DRIOP_PLUGIN_URL . 'admin/css/driop-admin.css', array(), DRIOP_VERSION );
    }

    public function enqueue_scripts( $hook ) {
        if ( strpos( $hook, 'driop' ) === false ) return;
        wp_enqueue_script( 'driop-admin', DRIOP_PLUGIN_URL . 'admin/js/driop-admin.js', array( 'jquery' ), DRIOP_VERSION, true );
        wp_localize_script( 'driop-admin', 'driop_admin', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'driop_admin_nonce' ),
            'strings' => array(
                'optimizing' => __( 'Optimizing...', 'digital-rise-image-optimizer-for-seo' ),
                'complete' => __( 'Complete!', 'digital-rise-image-optimizer-for-seo' ),
                'error' => __( 'Error occurred', 'digital-rise-image-optimizer-for-seo' ),
            ),
        ) );
    }

    public function add_admin_menu() {
        add_menu_page( __( 'Image Optimizer for SEO', 'digital-rise-image-optimizer-for-seo' ), __( 'Image Optimizer', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-dashboard', array( $this, 'render_dashboard' ), 'dashicons-format-image', 65 );
        add_submenu_page( 'driop-dashboard', __( 'Dashboard', 'digital-rise-image-optimizer-for-seo' ), __( 'Dashboard', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-dashboard', array( $this, 'render_dashboard' ) );
        add_submenu_page( 'driop-dashboard', __( 'Image Optimization', 'digital-rise-image-optimizer-for-seo' ), __( 'Optimization', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-optimization', array( $this, 'render_optimization' ) );
        add_submenu_page( 'driop-dashboard', __( 'Bulk Optimize', 'digital-rise-image-optimizer-for-seo' ), __( 'Bulk Optimize', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-bulk', array( $this, 'render_bulk_optimize' ) );
        add_submenu_page( 'driop-dashboard', __( 'Keywords', 'digital-rise-image-optimizer-for-seo' ), __( 'Keywords', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-keywords', array( $this, 'render_keywords' ) );
        add_submenu_page( 'driop-dashboard', __( 'SEO Titles & Meta', 'digital-rise-image-optimizer-for-seo' ), __( 'SEO Titles', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-page-seo', array( $this, 'render_page_seo' ) );
        add_submenu_page( 'driop-dashboard', __( 'Plans', 'digital-rise-image-optimizer-for-seo' ), __( 'Plans', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-plans', array( $this, 'render_plans' ) );
        add_submenu_page( 'driop-dashboard', __( 'License', 'digital-rise-image-optimizer-for-seo' ), __( 'License', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-license', array( $this, 'render_license' ) );
        add_submenu_page( 'driop-dashboard', __( 'Settings', 'digital-rise-image-optimizer-for-seo' ), __( 'Settings', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-settings', array( $this, 'render_settings' ) );
        // Hidden page for viewing pending reviews (if emails fail)
        add_submenu_page( null, __( 'Pending Reviews', 'digital-rise-image-optimizer-for-seo' ), __( 'Pending Reviews', 'digital-rise-image-optimizer-for-seo' ), 'manage_options', 'driop-reviews', array( $this, 'render_reviews' ) );
    }

    public function register_settings() {
        register_setting( 'driop_options_group', 'driop_options', array(
            'sanitize_callback' => array( $this, 'sanitize_options' ),
        ) );
    }

    public function sanitize_options( $options ) {
        if ( ! is_array( $options ) ) {
            return array();
        }
        
        $sanitized = array();
        foreach ( $options as $key => $value ) {
            if ( is_bool( $value ) ) {
                $sanitized[ $key ] = (bool) $value;
            } elseif ( is_numeric( $value ) ) {
                $sanitized[ $key ] = absint( $value );
            } elseif ( is_array( $value ) ) {
                $sanitized[ $key ] = array_map( 'sanitize_text_field', $value );
            } else {
                $sanitized[ $key ] = sanitize_text_field( $value );
            }
        }
        return $sanitized;
    }

    public function display_admin_notices() {
        if ( get_transient( 'driop_activation_redirect' ) ) {
            delete_transient( 'driop_activation_redirect' );
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Redirect check only, no data processing
            if ( ! isset( $_GET['activate-multi'] ) ) {
                wp_safe_redirect( admin_url( 'admin.php?page=driop-dashboard' ) );
                exit;
            }
        }

        // FIX #2: Show YOUR branded notices on ALL admin pages
        // WordPress notices stay normal, YOUR notices get branded box
        $remote = new DRIOP_Remote( $this->plugin );
        $notices = $remote->get_active_notices();
        
        if ( ! empty( $notices ) && is_array( $notices ) ) {
            foreach ( $notices as $notice ) {
                if ( ! is_array( $notice ) || empty( $notice['notice_content'] ) ) {
                    continue;
                }
                
                $type = isset( $notice['notice_type'] ) && in_array( $notice['notice_type'], array( 'info', 'warning', 'error', 'success', 'promo' ) ) ? $notice['notice_type'] : 'info';
                $notice_id = isset( $notice['notice_id'] ) ? $notice['notice_id'] : 'notice-' . md5( $notice['notice_content'] );
                
                // Border color based on type
                $border_colors = array(
                    'promo' => '#ffd700',
                    'error' => '#dc3545',
                    'warning' => '#ffc107',
                    'success' => '#4CAF50',
                    'info' => '#2196F3',
                );
                $border_color = isset( $border_colors[ $type ] ) ? $border_colors[ $type ] : '#2196F3';
                
                // YOUR BRANDED NOTICE BOX - Only for YOUR notices from owner panel
                ?>
                <div class="notice is-dismissible driop-remote-notice driop-branded-notice" data-notice-id="<?php echo esc_attr( $notice_id ); ?>" style="background: linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%); border: none; border-left: 4px solid <?php echo esc_attr( $border_color ); ?>; color: white; padding: 15px 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); margin: 5px 15px 15px 0;">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <?php
                        // Show logo
                        $logo_url = DRIOP_LOGO_URL;
                        if ( ! empty( $logo_url ) && @getimagesize( $logo_url ) ) : ?>
                            <img src="<?php echo esc_url( $logo_url ); ?>" alt="Digital Rise" style="height: 40px; border-radius: 6px; background: white; padding: 8px;">
                        <?php else : ?>
                            <div style="width: 40px; height: 40px; background: white; border-radius: 6px; display: flex; align-items: center; justify-content: center; font-size: 18px; font-weight: bold; color: #1e3a5f;">DR</div>
                        <?php endif; ?>
                        
                        <div style="flex: 1;">
                            <?php if ( ! empty( $notice['notice_title'] ) ) : ?>
                                <strong style="font-size: 16px; display: block; margin-bottom: 5px;"><?php echo esc_html( $notice['notice_title'] ); ?></strong>
                            <?php endif; ?>
                            <div style="font-size: 14px; opacity: 0.95;">
                                <?php echo wp_kses_post( $notice['notice_content'] ); ?>
                                <?php if ( ! empty( $notice['notice_link'] ) ) : ?>
                                    <a href="<?php echo esc_url( $notice['notice_link'] ); ?>" target="_blank" style="color: #ffd700; font-weight: 600; text-decoration: underline; margin-left: 10px;"><?php esc_html_e( 'Learn more →', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <?php if ( $type === 'promo' ) : ?>
                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button" style="background: #ffd700; color: #1e3a5f; border: none; font-weight: bold; padding: 10px 20px; white-space: nowrap;">
                                ⭐ <?php esc_html_e( 'Upgrade Now', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </a>
                        <?php endif; ?>
                        
                        <span class="driop-plan-badge driop-plan-<?php echo esc_attr( $this->plugin->current_plan ); ?>" style="<?php echo $this->plugin->current_plan === 'free' ? 'background: #ffd700; color: #333;' : ''; ?> padding: 6px 12px; border-radius: 20px; font-size: 12px; font-weight: 600; text-transform: uppercase;">
                            <?php echo esc_html( ucfirst( $this->plugin->current_plan ) ); ?>
                        </span>
                    </div>
                </div>
                <?php
            }
        }
    }

    private function render_header( $title ) {
        $logo_url = DRIOP_LOGO_URL;
        ?>
        <div class="driop-header">
            <div class="driop-header-logo">
                <?php if ( ! empty( $logo_url ) && @getimagesize( $logo_url ) ) : ?>
                    <img src="<?php echo esc_url( $logo_url ); ?>" alt="Digital Rise" height="50">
                <?php else : ?>
                    <span class="driop-logo-text">Image Optimizer for SEO</span>
                <?php endif; ?>
            </div>
            <div class="driop-header-info" style="flex: 1;">
                <h1><?php echo esc_html( $title ); ?></h1>
                <span class="driop-plan-badge driop-plan-<?php echo esc_attr( $this->plugin->current_plan ); ?>">
                    <?php echo esc_html( ucfirst( $this->plugin->current_plan ) ); ?>
                </span>
            </div>
            <?php if ( $this->plugin->current_plan === 'free' ) : ?>
                <div class="driop-header-actions" style="display: flex; gap: 10px; align-items: center;">
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button button-primary button-hero" style="background: #ffd700; border-color: #ffd700; color: #333; font-weight: 600; text-shadow: none;">
                        ⭐ <?php esc_html_e( 'Upgrade to Pro', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </a>
                    <button type="button" id="driop-review-btn" class="button button-secondary" style="background: #0073aa; color: white; border-color: #0073aa; font-weight: 600; padding: 8px 15px; height: auto;">
                        <span class="dashicons dashicons-star-filled" style="color: #ffd700; margin-top: 3px;"></span> <?php esc_html_e( 'Review Plugin', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </button>
                </div>
            <?php else : ?>
                <div class="driop-header-actions" style="display: flex; gap: 10px; align-items: center;">
                    <button type="button" id="driop-review-btn" class="button button-primary" style="background: #0073aa; color: white; border-color: #0073aa; font-weight: 600; padding: 8px 15px; height: auto;">
                        <span class="dashicons dashicons-star-filled" style="color: #ffd700; margin-top: 3px;"></span> <?php esc_html_e( 'Review Plugin', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </button>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Review Modal -->
        <div id="driop-review-modal" style="display: none; position: fixed; z-index: 999999; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.6);">
            <div style="background-color: #fefefe; margin: 5% auto; padding: 0; border-radius: 8px; width: 90%; max-width: 500px; box-shadow: 0 4px 20px rgba(0,0,0,0.3);">
                <!-- Header -->
                <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px 8px 0 0;">
                    <h2 style="margin: 0; color: white; display: flex; align-items: center;">
                        <span class="dashicons dashicons-star-filled" style="font-size: 24px; margin-right: 10px;"></span>
                        <?php esc_html_e( 'Review Our Plugin', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </h2>
                    <p style="margin: 10px 0 0 0; opacity: 0.9;"><?php esc_html_e( 'Your feedback helps us improve!', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                </div>
                
                <!-- Body -->
                <div style="padding: 30px;">
                    <!-- Email Configuration Notice -->
                    <div style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin-bottom: 25px; border-radius: 4px;">
                        <p style="margin: 0; color: #856404; font-size: 14px; line-height: 1.6;">
                            <strong>ℹ️ <?php esc_html_e( 'Note:', 'digital-rise-image-optimizer-for-seo' ); ?></strong>
                            <?php esc_html_e( 'If email delivery is not configured on your server, please send your review directly to:', 'digital-rise-image-optimizer-for-seo' ); ?>
                            <br>
                            <a href="mailto:seo.oussama@gmail.com" style="color: #667eea; font-weight: 600; text-decoration: none;">
                                📧 seo.oussama@gmail.com
                            </a>
                        </p>
                    </div>
                    
                    <form id="driop-review-form">
                        <!-- Star Rating -->
                        <div style="margin-bottom: 25px; text-align: center;">
                            <label style="display: block; margin-bottom: 15px; font-weight: 600; font-size: 16px; color: #333;">
                                <?php esc_html_e( 'Rate your experience:', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </label>
                            <div class="driop-star-rating" style="font-size: 40px; cursor: pointer;">
                                <span class="driop-star" data-rating="1">☆</span>
                                <span class="driop-star" data-rating="2">☆</span>
                                <span class="driop-star" data-rating="3">☆</span>
                                <span class="driop-star" data-rating="4">☆</span>
                                <span class="driop-star" data-rating="5">☆</span>
                            </div>
                            <input type="hidden" id="driop-rating" name="rating" value="0">
                        </div>
                        
                        <!-- Review Text -->
                        <div style="margin-bottom: 20px;">
                            <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #333;">
                                <?php esc_html_e( 'Your Review:', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </label>
                            <textarea name="review" id="driop-review-text" rows="5" style="width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;" placeholder="<?php esc_attr_e( 'Tell us about your experience...', 'digital-rise-image-optimizer-for-seo' ); ?>" required></textarea>
                        </div>
                        
                        <!-- Contact Info (Optional) -->
                        <div style="margin-bottom: 20px;">
                            <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #333;">
                                <?php esc_html_e( 'Email (optional):', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </label>
                            <input type="email" name="email" id="driop-review-email" style="width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px;" placeholder="your@email.com">
                        </div>
                        
                        <div style="margin-bottom: 20px;">
                            <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #333;">
                                <?php esc_html_e( 'Phone (optional):', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </label>
                            <input type="tel" name="phone" id="driop-review-phone" style="width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px;" placeholder="+1234567890">
                        </div>
                        
                        <!-- Buttons -->
                        <div style="display: flex; gap: 10px; margin-top: 25px;">
                            <button type="submit" class="button button-primary" style="flex: 1; height: 40px; font-size: 15px;">
                                <span class="dashicons dashicons-yes" style="margin-top: 3px;"></span> <?php esc_html_e( 'Submit Review', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </button>
                            <button type="button" id="driop-copy-review" class="button" style="height: 40px;" title="<?php esc_attr_e( 'Copy review to send manually', 'digital-rise-image-optimizer-for-seo' ); ?>">
                                <span class="dashicons dashicons-clipboard" style="margin-top: 3px;"></span>
                            </button>
                            <button type="button" class="button driop-close-modal" style="flex: 0 0 auto; height: 40px;">
                                <?php esc_html_e( 'Cancel', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </button>
                        </div>
                    </form>
                    
                    <!-- Success Message -->
                    <div id="driop-review-success" style="display: none; text-align: center; padding: 20px;">
                        <span class="dashicons dashicons-yes-alt" style="font-size: 60px; color: #46b450;"></span>
                        <h3 style="color: #46b450; margin: 15px 0;"><?php esc_html_e( 'Thank You!', 'digital-rise-image-optimizer-for-seo' ); ?></h3>
                        <p><?php esc_html_e( 'Your review has been sent successfully.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                        <button type="button" class="button button-primary driop-close-modal" style="margin-top: 15px;">
                            <?php esc_html_e( 'Close', 'digital-rise-image-optimizer-for-seo' ); ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
            .driop-star { 
                color: #ddd; 
                transition: color 0.2s; 
                display: inline-block;
                margin: 0 3px;
            }
            .driop-star:hover,
            .driop-star.active { 
                color: #ffd700; 
            }
            .driop-star.hovered {
                color: #ffd700;
            }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            var selectedRating = 0;
            
            // Open modal
            $('#driop-review-btn').on('click', function() {
                $('#driop-review-modal').fadeIn(300);
            });
            
            // Close modal
            $('.driop-close-modal, #driop-review-modal').on('click', function(e) {
                if (e.target === this) {
                    $('#driop-review-modal').fadeOut(300);
                    setTimeout(function() {
                        $('#driop-review-form')[0].reset();
                        $('#driop-review-success').hide();
                        $('#driop-review-form').show();
                        $('.driop-star').removeClass('active');
                        selectedRating = 0;
                    }, 300);
                }
            });
            
            // Star rating
            $('.driop-star').on('mouseenter', function() {
                var rating = $(this).data('rating');
                $('.driop-star').each(function() {
                    if ($(this).data('rating') <= rating) {
                        $(this).addClass('hovered');
                    } else {
                        $(this).removeClass('hovered');
                    }
                });
            });
            
            $('.driop-star-rating').on('mouseleave', function() {
                $('.driop-star').removeClass('hovered');
            });
            
            $('.driop-star').on('click', function() {
                selectedRating = $(this).data('rating');
                $('#driop-rating').val(selectedRating);
                
                $('.driop-star').each(function() {
                    if ($(this).data('rating') <= selectedRating) {
                        $(this).addClass('active').text('★');
                    } else {
                        $(this).removeClass('active').text('☆');
                    }
                });
            });
            
            // Copy review to clipboard
            $('#driop-copy-review').on('click', function() {
                var stars = '';
                for (var i = 0; i < selectedRating; i++) stars += '★';
                for (var i = selectedRating; i < 5; i++) stars += '☆';
                
                var reviewText = 'PLUGIN REVIEW\n\n';
                reviewText += 'Rating: ' + stars + ' (' + selectedRating + '/5)\n\n';
                reviewText += 'Review:\n' + $('#driop-review-text').val() + '\n\n';
                reviewText += 'Website: <?php echo esc_js( home_url() ); ?>\n';
                
                var email = $('#driop-review-email').val();
                if (email) reviewText += 'Email: ' + email + '\n';
                
                var phone = $('#driop-review-phone').val();
                if (phone) reviewText += 'Phone: ' + phone + '\n';
                
                reviewText += '\nPlease send to: seo.oussama@gmail.com';
                
                // Copy to clipboard
                if (navigator.clipboard && navigator.clipboard.writeText) {
                    navigator.clipboard.writeText(reviewText).then(function() {
                        var $btn = $('#driop-copy-review');
                        var originalHtml = $btn.html();
                        $btn.html('<span class="dashicons dashicons-yes" style="margin-top: 3px; color: #46b450;"></span>');
                        setTimeout(function() {
                            $btn.html(originalHtml);
                        }, 2000);
                        alert('<?php esc_html_e( 'Review copied! You can paste it in an email to seo.oussama@gmail.com', 'digital-rise-image-optimizer-for-seo' ); ?>');
                    }).catch(function() {
                        // Fallback for older browsers
                        copyToClipboardFallback(reviewText);
                    });
                } else {
                    // Fallback for older browsers
                    copyToClipboardFallback(reviewText);
                }
            });
            
            // Fallback copy method
            function copyToClipboardFallback(text) {
                var $temp = $('<textarea>');
                $('body').append($temp);
                $temp.val(text).select();
                try {
                    document.execCommand('copy');
                    alert('<?php esc_html_e( 'Review copied! You can paste it in an email to seo.oussama@gmail.com', 'digital-rise-image-optimizer-for-seo' ); ?>');
                } catch (err) {
                    alert('<?php esc_html_e( 'Could not copy. Please copy manually.', 'digital-rise-image-optimizer-for-seo' ); ?>');
                }
                $temp.remove();
            }
            
            // Submit review
            $('#driop-review-form').on('submit', function(e) {
                e.preventDefault();
                
                if (selectedRating === 0) {
                    alert('<?php esc_html_e( 'Please select a star rating!', 'digital-rise-image-optimizer-for-seo' ); ?>');
                    return;
                }
                
                var $btn = $(this).find('button[type="submit"]');
                $btn.prop('disabled', true).text('<?php esc_html_e( 'Sending...', 'digital-rise-image-optimizer-for-seo' ); ?>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'driop_submit_review',
                        nonce: '<?php echo esc_js( wp_create_nonce( 'driop_review_nonce' ) ); ?>',
                        rating: selectedRating,
                        review: $('#driop-review-text').val(),
                        email: $('#driop-review-email').val(),
                        phone: $('#driop-review-phone').val(),
                        site_url: '<?php echo esc_js( home_url() ); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#driop-review-form').fadeOut(200, function() {
                                $('#driop-review-success').fadeIn(200);
                            });
                        } else {
                            alert(response.data || '<?php esc_html_e( 'Error sending review. Please try again.', 'digital-rise-image-optimizer-for-seo' ); ?>');
                            $btn.prop('disabled', false).html('<span class="dashicons dashicons-yes" style="margin-top: 3px;"></span> <?php esc_html_e( 'Submit Review', 'digital-rise-image-optimizer-for-seo' ); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php esc_html_e( 'Error sending review. Please try again.', 'digital-rise-image-optimizer-for-seo' ); ?>');
                        $btn.prop('disabled', false).html('<span class="dashicons dashicons-yes" style="margin-top: 3px;"></span> <?php esc_html_e( 'Submit Review', 'digital-rise-image-optimizer-for-seo' ); ?>');
                    }
                });
            });
        });
        </script>
        <?php
    }

    public function render_dashboard() {
        // Handle sitemap cache clear
        if ( isset( $_POST['driop_clear_sitemap_cache'] ) && check_admin_referer( 'driop_clear_sitemap_nonce' ) ) {
            // Clear all sitemap caches
            delete_transient( 'driop_sitemap_index_cache' );
            delete_transient( 'driop_sitemap_pages_cache' );
            delete_transient( 'driop_sitemap_posts_cache' );
            delete_transient( 'driop_sitemap_categories_cache' );
            delete_transient( 'driop_sitemap_tags_cache' );
            delete_transient( 'driop_sitemap_products_cache' );
            delete_transient( 'driop_sitemap_product-categories_cache' );
            delete_transient( 'driop_sitemap_product-tags_cache' );
            delete_transient( 'driop_sitemap_authors_cache' );
            delete_transient( 'driop_sitemap_vendors_cache' );
            delete_transient( 'driop_sitemap_vendor-products_cache' );
            delete_transient( 'driop_sitemap_images_cache' );
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Sitemap cache cleared! All sitemaps will regenerate with updated image URLs.', 'digital-rise-image-optimizer-for-seo' ) . '</p></div>';
        }
        
        // Handle contact form save
        if ( isset( $_POST['driop_save_contact'] ) && check_admin_referer( 'driop_contact_nonce' ) ) {
            $options = $this->plugin->options;
            $options['user_email'] = isset( $_POST['user_email'] ) ? sanitize_email( wp_unslash( $_POST['user_email'] ) ) : '';
            $options['user_whatsapp'] = isset( $_POST['user_whatsapp'] ) ? sanitize_text_field( wp_unslash( $_POST['user_whatsapp'] ) ) : '';
            $this->plugin->save_options( $options );
            
            // Trigger immediate sync
            $remote = new DRIOP_Remote( $this->plugin );
            $remote->do_checkin();
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Contact info saved and synced!', 'digital-rise-image-optimizer-for-seo' ) . '</p></div>';
        }
        
        // Auto-sync on page load (once per hour)
        $last_auto_sync = get_transient( 'driop_auto_sync' );
        if ( ! $last_auto_sync ) {
            $remote = new DRIOP_Remote( $this->plugin );
            $remote->do_checkin();
            set_transient( 'driop_auto_sync', time(), HOUR_IN_SECONDS );
        }
        
        $optimizer = new DRIOP_Image_Optimizer( $this->plugin );
        $stats = $optimizer->get_stats();
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        $wc = new DRIOP_WooCommerce( $this->plugin );
        $dokan = new DRIOP_Dokan( $this->plugin );
        $user_email = $this->plugin->get_option( 'user_email', '' );
        $user_whatsapp = $this->plugin->get_option( 'user_whatsapp', '' );
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Dashboard', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <div class="driop-dashboard-grid">
                <!-- Contact Info Card - FIRST -->
                <div class="driop-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #fff;">
                    <h2 style="color: #fff;">📧 <?php esc_html_e( 'Your Contact Info', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <form method="post">
                        <?php wp_nonce_field( 'driop_contact_nonce' ); ?>
                        <p>
                            <label style="color: #fff;"><?php esc_html_e( 'Email:', 'digital-rise-image-optimizer-for-seo' ); ?></label><br>
                            <input type="email" name="user_email" value="<?php echo esc_attr( $user_email ); ?>" style="width: 100%; padding: 8px;" placeholder="your@email.com">
                        </p>
                        <p>
                            <label style="color: #fff;"><?php esc_html_e( 'WhatsApp:', 'digital-rise-image-optimizer-for-seo' ); ?></label><br>
                            <input type="text" name="user_whatsapp" value="<?php echo esc_attr( $user_whatsapp ); ?>" style="width: 100%; padding: 8px;" placeholder="+216 XX XXX XXX">
                        </p>
                        <p>
                            <button type="submit" name="driop_save_contact" class="button" style="background: #fff; color: #667eea; border: none; font-weight: bold;">
                                <?php esc_html_e( 'Save & Sync', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </button>
                        </p>
                    </form>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Optimization Stats', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <div class="driop-stats">
                        <div class="driop-stat">
                            <span class="driop-stat-num"><?php echo esc_html( $stats['total_images'] ); ?></span>
                            <span class="driop-stat-label"><?php esc_html_e( 'Total Images', 'digital-rise-image-optimizer-for-seo' ); ?></span>
                        </div>
                        <div class="driop-stat">
                            <span class="driop-stat-num"><?php echo esc_html( $stats['webp_count'] ); ?></span>
                            <span class="driop-stat-label"><?php esc_html_e( 'WebP', 'digital-rise-image-optimizer-for-seo' ); ?></span>
                        </div>
                        <div class="driop-stat">
                            <span class="driop-stat-num"><?php echo esc_html( $stats['optimized'] ); ?></span>
                            <span class="driop-stat-label"><?php esc_html_e( 'Optimized', 'digital-rise-image-optimizer-for-seo' ); ?></span>
                        </div>
                    </div>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Image Status', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <div class="driop-progress-wrap">
                        <div class="driop-progress-bar">
                            <div class="driop-progress-fill" style="width: <?php echo esc_attr( $progress['percentage'] ); ?>%"></div>
                        </div>
                        <?php
                        // translators: %1$d is the number of optimized images, %2$d is the total number of images
                        ?>
                        <p><?php printf( esc_html__( '%1$d of %2$d images optimized', 'digital-rise-image-optimizer-for-seo' ), (int) $progress['optimized'], (int) $progress['total'] ); ?></p>
                    </div>
                    <?php if ( $progress['remaining'] > 0 ) : ?>
                        <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-bulk' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Optimize All', 'digital-rise-image-optimizer-for-seo' ); ?></a></p>
                    <?php endif; ?>
                </div>
                
                <div class="driop-card">
                    <h2>🗺️ <?php esc_html_e( 'Sitemap', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <p><?php esc_html_e( 'Your sitemap is automatically updated when images are optimized.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    <p style="margin-bottom: 15px;">
                        <a href="<?php echo esc_url( home_url( '/sitemap.xml' ) ); ?>" target="_blank" class="button">
                            <?php esc_html_e( 'View Sitemap', 'digital-rise-image-optimizer-for-seo' ); ?>
                        </a>
                    </p>
                    <form method="post" style="display: inline;">
                        <?php wp_nonce_field( 'driop_clear_sitemap_nonce' ); ?>
                        <button type="submit" name="driop_clear_sitemap_cache" class="button">
                            🔄 <?php esc_html_e( 'Refresh Sitemap Now', 'digital-rise-image-optimizer-for-seo' ); ?>
                        </button>
                    </form>
                    <p class="description" style="margin-top: 10px;">
                        <?php esc_html_e( 'Clear cache and regenerate sitemap with latest image URLs.', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </p>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'WooCommerce', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <?php $wc_stats = $wc->get_stats(); ?>
                    <?php if ( $wc_stats['active'] ) : ?>
                        <p class="driop-status-active"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e( 'Active', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                        <?php
                        // translators: %d is the number of products
                        ?>
                        <p><?php printf( esc_html__( 'Products: %d', 'digital-rise-image-optimizer-for-seo' ), (int) $wc_stats['total_products'] ); ?></p>
                    <?php else : ?>
                        <p class="driop-status-inactive"><?php esc_html_e( 'Not installed', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    <?php endif; ?>
                </div>

                <div class="driop-card <?php echo ! $dokan_stats['active'] ? 'driop-locked-feature' : ''; ?>">
                    <h2><?php esc_html_e( 'Dokan', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <?php $dokan_stats = $dokan->get_stats(); ?>
                    <?php if ( $dokan_stats['active'] ) : ?>
                        <p class="driop-status-active"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e( 'Active', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                        <?php
                        // translators: %d is the number of vendors
                        ?>
                        <p><?php printf( esc_html__( 'Vendors: %d', 'digital-rise-image-optimizer-for-seo' ), (int) $dokan_stats['total_vendors'] ); ?></p>
                    <?php else : ?>
                        <p class="driop-status-inactive"><span class="dashicons dashicons-lock"></span> <?php esc_html_e( 'Dokan not installed', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                        <p><?php esc_html_e( 'Install Dokan to enable multi-vendor image management and per-vendor limits.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                        <p>
                            <a href="<?php echo esc_url( admin_url( 'plugin-install.php?s=dokan&tab=search&type=term' ) ); ?>" class="button"><?php esc_html_e( 'Install Dokan', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-plans' ) ); ?>" class="button button-primary"><?php esc_html_e( 'View Pro Features', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                        </p>
                    <?php endif; ?>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Quick Actions', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-bulk' ) ); ?>" class="button"><?php esc_html_e( 'Bulk Optimize', 'digital-rise-image-optimizer-for-seo' ); ?></a></p>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-keywords' ) ); ?>" class="button"><?php esc_html_e( 'Manage Keywords', 'digital-rise-image-optimizer-for-seo' ); ?></a></p>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-settings' ) ); ?>" class="button"><?php esc_html_e( 'Settings', 'digital-rise-image-optimizer-for-seo' ); ?></a></p>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Connection Status', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <?php 
                    $last_checkin = $this->plugin->get_option( 'last_checkin' );
                    $remote = new DRIOP_Remote( $this->plugin );
                    $notices = $remote->get_active_notices();
                    ?>
                    <p><strong><?php esc_html_e( 'Last Sync:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> 
                        <?php echo $last_checkin ? esc_html( human_time_diff( strtotime( $last_checkin ) ) ) . ' ago' : esc_html__( 'Never', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </p>
                    <p><strong><?php esc_html_e( 'Active Notices:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> <?php echo count( $notices ); ?></p>
                    <p><strong><?php esc_html_e( 'Status:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> 
                        <span style="color: green;">✅ <?php esc_html_e( 'Auto-sync enabled', 'digital-rise-image-optimizer-for-seo' ); ?></span>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }

    public function render_optimization() {
        if ( isset( $_POST['driop_save_optimization'] ) && check_admin_referer( 'driop_optimization_nonce' ) ) {
            $options = $this->plugin->options;
            $options['max_width'] = isset( $_POST['max_width'] ) ? absint( $_POST['max_width'] ) : 2048;
            $options['max_height'] = isset( $_POST['max_height'] ) ? absint( $_POST['max_height'] ) : 2048;
            $options['target_file_size'] = isset( $_POST['target_file_size'] ) ? absint( $_POST['target_file_size'] ) : 100;
            $options['compression_quality'] = isset( $_POST['compression_quality'] ) ? min( 100, max( 1, absint( $_POST['compression_quality'] ) ) ) : 82;
            $options['auto_compress'] = isset( $_POST['auto_compress'] );
            $options['max_images_per_user'] = isset( $_POST['max_images_per_user'] ) ? absint( $_POST['max_images_per_user'] ) : 0;
            $options['max_images_per_product'] = isset( $_POST['max_images_per_product'] ) ? absint( $_POST['max_images_per_product'] ) : 10;
            $options['max_images_per_vendor'] = isset( $_POST['max_images_per_vendor'] ) ? absint( $_POST['max_images_per_vendor'] ) : 50;
            $this->plugin->save_options( $options );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-image-optimizer-for-seo' ) . '</p></div>';
        }
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Image Optimization', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_optimization_nonce' ); ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Size Settings', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Max Width (px)', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td><input type="number" name="max_width" value="<?php echo esc_attr( $this->plugin->get_option( 'max_width', 1920 ) ); ?>" min="100" max="10000"></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Max Height (px)', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td><input type="number" name="max_height" value="<?php echo esc_attr( $this->plugin->get_option( 'max_height', 1080 ) ); ?>" min="100" max="10000"></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Target File Size (KB)', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="number" name="target_file_size" value="<?php echo esc_attr( $this->plugin->get_option( 'target_file_size', 100 ) ); ?>" min="10" max="500">
                                <p class="description"><?php esc_html_e( 'Images will be compressed to this target size (default: 100KB)', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Compression Quality', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="range" name="compression_quality" id="compression_quality" value="<?php echo esc_attr( $this->plugin->get_option( 'compression_quality', 82 ) ); ?>" min="1" max="100">
                                <span id="quality_value"><?php echo esc_html( $this->plugin->get_option( 'compression_quality', 82 ) ); ?>%</span>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Auto Compress', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td><label><input type="checkbox" name="auto_compress" <?php checked( $this->plugin->get_option( 'auto_compress', true ) ); ?>> <?php esc_html_e( 'Automatically compress on upload', 'digital-rise-image-optimizer-for-seo' ); ?></label></td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Upload Limits', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <?php if ( ! $this->plugin->has_feature( 'per_user_limits' ) ) : ?>
                        <div class="driop-pro-notice"><span class="dashicons dashicons-lock"></span> <?php esc_html_e( 'Upload limits require Pro plan.', 'digital-rise-image-optimizer-for-seo' ); ?></div>
                    <?php endif; ?>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per User', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td><input type="number" name="max_images_per_user" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_user', 0 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_user_limits' ) ); ?>><p class="description"><?php esc_html_e( '0 = unlimited', 'digital-rise-image-optimizer-for-seo' ); ?></p></td>
                        </tr>
                        <?php if ( class_exists( 'WooCommerce' ) ) : ?>
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per Product', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td><input type="number" name="max_images_per_product" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_product', 10 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_product_limits' ) ); ?>></td>
                        </tr>
                        <?php endif; ?>
                        <?php if ( class_exists( 'WeDevs_Dokan' ) || class_exists( 'Dokan_Pro' ) ) : ?>
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per Vendor', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td><input type="number" name="max_images_per_vendor" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_vendor', 50 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_vendor_limits' ) ); ?>></td>
                        </tr>
                        <?php endif; ?>
                    </table>
                </div>

                <p class="submit"><input type="submit" name="driop_save_optimization" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-image-optimizer-for-seo' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    public function render_bulk_optimize() {
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Bulk Optimize', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <div class="driop-card">
                <h2><?php esc_html_e( 'Optimize All Images', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                <p><?php esc_html_e( 'This will convert images to WebP and apply SEO to all your images.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                
                <div class="driop-bulk-progress">
                    <div class="driop-progress-bar">
                        <div class="driop-progress-fill" id="bulk-progress-fill" style="width: <?php echo esc_attr( $progress['percentage'] ); ?>%"></div>
                    </div>
                    <p id="bulk-progress-text">
                        <?php
                        // translators: %1$d is optimized images, %2$d is total images, %3$d is remaining images
                        printf( esc_html__( '%1$d of %2$d images optimized (%3$d remaining)', 'digital-rise-image-optimizer-for-seo' ), (int) $progress['optimized'], (int) $progress['total'], (int) $progress['remaining'] );
                        ?>
                    </p>
                </div>

                <table class="form-table">
                    <tr>
                        <th><?php esc_html_e( 'Options', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                        <td>
                            <p><label><input type="checkbox" id="apply_seo" checked> <?php esc_html_e( 'Apply SEO (ALT text, title, description)', 'digital-rise-image-optimizer-for-seo' ); ?></label></p>
                            <p><label><input type="checkbox" id="reoptimize_all"> <?php esc_html_e( 'Re-optimize ALL images (including previously optimized)', 'digital-rise-image-optimizer-for-seo' ); ?></label></p>
                        </td>
                    </tr>
                </table>

                <p>
                    <button type="button" id="start-bulk-optimize" class="button button-primary button-hero">
                        <?php esc_html_e( 'Start Bulk Optimization', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </button>
                    <button type="button" id="stop-bulk-optimize" class="button button-secondary" style="display:none;">
                        <?php esc_html_e( 'Stop', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </button>
                    <button type="button" id="reset-optimization" class="button">
                        <?php esc_html_e( 'Reset All (Mark as Unoptimized)', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </button>
                </p>

                <div id="bulk-log" class="driop-bulk-log" style="display:none;"></div>
            </div>
        </div>
        <?php
    }

    public function render_keywords() {
        $keywords = new DRIOP_Keywords( $this->plugin );
        
        if ( isset( $_POST['driop_save_keywords'] ) && check_admin_referer( 'driop_keywords_nonce' ) ) {
            // FIX #4: Use sanitize_textarea_field (preserves line breaks) like old version
            $keywords_input = isset( $_POST['keywords'] ) ? sanitize_textarea_field( wp_unslash( $_POST['keywords'] ) ) : '';
            $keywords->save_keywords( $keywords_input );
            
            // Save default pattern
            $this->plugin->update_option( 'image_seo_pattern', isset( $_POST['image_seo_pattern'] ) ? sanitize_text_field( wp_unslash( $_POST['image_seo_pattern'] ) ) : '{site_name} {keyword}' );
            
            // PHASE 3 FIX #6: Save content-type-specific patterns
            $content_types = array( 'post', 'page', 'product' );
            foreach ( $content_types as $type ) {
                $key = 'image_seo_pattern_' . $type;
                if ( isset( $_POST[ $key ] ) ) {
                    $this->plugin->update_option( $key, sanitize_text_field( wp_unslash( $_POST[ $key ] ) ) );
                }
            }
            
            $this->plugin->update_option( 'auto_rename', isset( $_POST['auto_rename'] ) );
            $this->plugin->update_option( 'auto_alt', isset( $_POST['auto_alt'] ) );
            $this->plugin->update_option( 'auto_title', isset( $_POST['auto_title'] ) );
            $this->plugin->update_option( 'auto_description', isset( $_POST['auto_description'] ) );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-image-optimizer-for-seo' ) . '</p></div>';
        }
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Keywords Management', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <?php if ( ! $this->plugin->has_feature( 'custom_keywords' ) ) : ?>
                <div class="driop-card">
                    <div class="driop-pro-notice">
                        <span class="dashicons dashicons-lock"></span> 
                        <?php esc_html_e( 'Custom keywords require Pro plan. Free plan uses basic patterns.', 'digital-rise-image-optimizer-for-seo' ); ?>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button"><?php esc_html_e( 'Upgrade', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                    </div>
                </div>
            <?php endif; ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_keywords_nonce' ); ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Keywords', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <p><?php esc_html_e( 'Enter keywords for image SEO. One keyword phrase per line.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    <textarea name="keywords" rows="10" class="large-text code" <?php disabled( ! $this->plugin->has_feature( 'custom_keywords' ) ); ?> placeholder="<?php esc_attr_e( "Example:\nreal estate tunisia\nbuy apartment kairouan\nluxury villa sousse", 'digital-rise-image-optimizer-for-seo' ); ?>"><?php echo esc_textarea( $this->plugin->get_option( 'keywords', '' ) ); ?></textarea>
                    <?php
                    // translators: %d is the number of keywords
                    ?>
                    <p class="description"><?php printf( esc_html__( 'Total keywords: %d (one per line recommended)', 'digital-rise-image-optimizer-for-seo' ), (int) $keywords->get_count() ); ?></p>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Image SEO Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <p><?php esc_html_e( 'Define how image names, titles, ALT text, and descriptions are generated. Maximum 7 words will be used.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="image_seo_pattern" value="<?php echo esc_attr( $this->plugin->get_option( 'image_seo_pattern', '{site_name} {keyword}' ) ); ?>" class="large-text" id="image-seo-pattern">
                                <p class="description"><?php esc_html_e( 'This pattern will be applied to: Image filename, Title, ALT text, Caption, Description', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Available Variables', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <div class="driop-variable-buttons">
                                    <?php 
                                    $seo = new DRIOP_SEO( $this->plugin );
                                    $vars = $seo->get_available_variables();
                                    foreach ( $vars as $var => $desc ) : 
                                    ?>
                                        <button type="button" class="button driop-insert-var" data-var="<?php echo esc_attr( $var ); ?>" title="<?php echo esc_attr( $desc ); ?>">
                                            <?php echo esc_html( $var ); ?>
                                        </button>
                                    <?php endforeach; ?>
                                </div>
                                <p class="description" style="margin-top: 10px;">
                                    <strong><?php esc_html_e( 'Click a variable to insert it. You can also add static text.', 'digital-rise-image-optimizer-for-seo' ); ?></strong><br>
                                    <?php esc_html_e( 'Example: "{site_name} {keyword} best quality" → "My Store Real Estate Tunisia Best Quality"', 'digital-rise-image-optimizer-for-seo' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Word Limit', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <strong><?php esc_html_e( '7 words maximum', 'digital-rise-image-optimizer-for-seo' ); ?></strong>
                                <p class="description"><?php esc_html_e( 'Google recommends short, descriptive image names and ALT text. The pattern result will be trimmed to 7 words.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <?php if ( $this->plugin->has_feature( 'content_type_patterns' ) ) : ?>
                <div class="driop-card">
                    <h2>🎯 <?php esc_html_e( 'Content-Type-Specific Patterns', 'digital-rise-image-optimizer-for-seo' ); ?> 
                        <span class="driop-pro-badge">PRO</span>
                    </h2>
                    <p><?php esc_html_e( 'Define different SEO patterns for different content types. If not set, the default pattern above will be used.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Blog Posts', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="image_seo_pattern_post" value="<?php echo esc_attr( $this->plugin->get_option( 'image_seo_pattern_post', '' ) ); ?>" class="large-text" placeholder="<?php esc_attr_e( 'e.g., {site_name} {content_title} {category}', 'digital-rise-image-optimizer-for-seo' ); ?>">
                                <p class="description"><?php esc_html_e( 'Pattern for images in blog posts', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Pages', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="image_seo_pattern_page" value="<?php echo esc_attr( $this->plugin->get_option( 'image_seo_pattern_page', '' ) ); ?>" class="large-text" placeholder="<?php esc_attr_e( 'e.g., {site_name} {content_title}', 'digital-rise-image-optimizer-for-seo' ); ?>">
                                <p class="description"><?php esc_html_e( 'Pattern for images in pages', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Products (WooCommerce)', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="image_seo_pattern_product" value="<?php echo esc_attr( $this->plugin->get_option( 'image_seo_pattern_product', '' ) ); ?>" class="large-text" placeholder="<?php esc_attr_e( 'e.g., {content_title} {keyword} {site_name}', 'digital-rise-image-optimizer-for-seo' ); ?>">
                                <p class="description"><?php esc_html_e( 'Pattern for product images', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
                <?php else : ?>
                <div class="driop-card driop-locked-feature">
                    <h2>🎯 <?php esc_html_e( 'Content-Type-Specific Patterns', 'digital-rise-image-optimizer-for-seo' ); ?> 
                        <span class="driop-pro-badge">PRO</span>
                    </h2>
                    <div class="driop-pro-notice">
                        <span class="dashicons dashicons-lock"></span> 
                        <?php esc_html_e( 'Define different SEO patterns for posts, pages, and products. Upgrade to Pro or Agency to unlock this feature.', 'digital-rise-image-optimizer-for-seo' ); ?>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Upgrade Now', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                    </div>
                </div>
                <?php endif; ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Auto Features', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'On Upload', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="auto_rename" <?php checked( $this->plugin->get_option( 'auto_rename', true ) ); ?>> <?php esc_html_e( 'Auto rename images (SEO-friendly filenames)', 'digital-rise-image-optimizer-for-seo' ); ?></label><br>
                                <label><input type="checkbox" name="auto_alt" <?php checked( $this->plugin->get_option( 'auto_alt', true ) ); ?>> <?php esc_html_e( 'Auto generate ALT text', 'digital-rise-image-optimizer-for-seo' ); ?></label><br>
                                <label><input type="checkbox" name="auto_title" <?php checked( $this->plugin->get_option( 'auto_title', true ) ); ?>> <?php esc_html_e( 'Auto generate Title', 'digital-rise-image-optimizer-for-seo' ); ?></label><br>
                                <label><input type="checkbox" name="auto_description" <?php checked( $this->plugin->get_option( 'auto_description', true ) ); ?>> <?php esc_html_e( 'Auto generate Caption & Description', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                            </td>
                        </tr>
                    </table>
                </div>

                <script>
                jQuery(document).ready(function($) {
                    $('.driop-insert-var').on('click', function() {
                        var varText = $(this).data('var');
                        var $input = $('#image-seo-pattern');
                        var currentVal = $input.val();
                        $input.val(currentVal + (currentVal ? ' ' : '') + varText);
                        $input.focus();
                    });
                });
                </script>
                <style>
                .driop-variable-buttons { display: flex; flex-wrap: wrap; gap: 5px; }
                .driop-variable-buttons .button { font-family: monospace; }
                </style>

                <p class="submit"><input type="submit" name="driop_save_keywords" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-image-optimizer-for-seo' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    public function render_plans() {
        $plans = new DRIOP_Plans( $this->plugin );
        
        // Handle refresh prices
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Simple cache clear, no data modification
        if ( isset( $_GET['refresh_prices'] ) ) {
            $plans->clear_price_cache();
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Prices refreshed from server!', 'digital-rise-image-optimizer-for-seo' ) . '</p></div>';
        }
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Plans & Features', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <?php if ( $this->plugin->current_plan === 'free' ) : ?>
            <!-- CUSTOM BRANDED UPGRADE NOTICE - ONLY FOR YOUR PLUGIN -->
            <div class="driop-card" style="background: linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%); color: white; padding: 25px; margin-bottom: 25px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; align-items: center; gap: 20px;">
                    <?php
                    $logo_url = DRIOP_LOGO_URL;
                    if ( ! empty( $logo_url ) && @getimagesize( $logo_url ) ) : ?>
                        <img src="<?php echo esc_url( $logo_url ); ?>" alt="Digital Rise" style="height: 60px; border-radius: 8px; background: white; padding: 10px;">
                    <?php else : ?>
                        <div style="width: 60px; height: 60px; background: white; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 24px; font-weight: bold; color: #1e3a5f;">DR</div>
                    <?php endif; ?>
                    
                    <div style="flex: 1;">
                        <h2 style="color: white; margin: 0 0 8px 0; font-size: 24px;">
                            🚀 <?php esc_html_e( 'Upgrade to Pro!', 'digital-rise-image-optimizer-for-seo' ); ?>
                        </h2>
                        <p style="margin: 0; font-size: 16px; opacity: 0.95;">
                            <?php esc_html_e( 'Unlock custom keywords, advanced SEO features, content-type patterns, and more.', 'digital-rise-image-optimizer-for-seo' ); ?>
                        </p>
                    </div>
                    
                    <div>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button button-hero" style="background: #ffd700; color: #1e3a5f; border: none; font-weight: bold; font-size: 16px; padding: 15px 30px; box-shadow: 0 4px 10px rgba(255, 215, 0, 0.3); transition: all 0.3s;" onmouseover="this.style.background='#ffed4e'; this.style.transform='translateY(-2px)'; this.style.boxShadow='0 6px 15px rgba(255, 215, 0, 0.5)';" onmouseout="this.style.background='#ffd700'; this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 10px rgba(255, 215, 0, 0.3)';">
                            ⭐ <?php esc_html_e( 'Upgrade Now', 'digital-rise-image-optimizer-for-seo' ); ?>
                        </a>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <p style="margin-bottom: 20px;">
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-plans&refresh_prices=1' ) ); ?>" class="button">
                    <span class="dashicons dashicons-update" style="margin-top: 3px;"></span> <?php esc_html_e( 'Refresh Prices', 'digital-rise-image-optimizer-for-seo' ); ?>
                </a>
            </p>
            
            <div class="driop-plans-grid">
                <?php foreach ( $plans->get_plans() as $plan_key => $plan ) : ?>
                    <div class="driop-plan-card <?php echo $this->plugin->current_plan === $plan_key ? 'driop-plan-current' : ''; ?>">
                        <h3><?php echo esc_html( $plan['name'] ); ?></h3>
                        <div class="driop-plan-price"><?php echo esc_html( $plan['price'] ); ?></div>
                        <?php if ( ! empty( $plan['price_note'] ) ) : ?>
                            <div class="driop-plan-price-note" style="color: #666; font-size: 14px; margin-top: -10px; margin-bottom: 15px;"><?php echo esc_html( $plan['price_note'] ); ?></div>
                        <?php endif; ?>
                        <p><?php echo esc_html( $plan['description'] ); ?></p>
                        <ul>
                            <?php foreach ( $plan['features'] as $feature ) : ?>
                                <li><span class="dashicons dashicons-yes"></span> <?php echo esc_html( $feature ); ?></li>
                            <?php endforeach; ?>
                            <?php if ( isset( $plan['limitations'] ) ) : ?>
                                <?php foreach ( $plan['limitations'] as $limitation ) : ?>
                                    <li class="limitation"><span class="dashicons dashicons-no"></span> <?php echo esc_html( $limitation ); ?></li>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </ul>
                        <?php if ( $this->plugin->current_plan === $plan_key ) : ?>
                            <span class="button disabled"><?php esc_html_e( 'Current Plan', 'digital-rise-image-optimizer-for-seo' ); ?></span>
                        <?php elseif ( $plan_key !== 'free' ) : ?>
                            <a href="<?php echo esc_url( $plans->get_purchase_url() ); ?>" target="_blank" class="button button-primary"><?php esc_html_e( 'Upgrade', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }

    public function render_license() {
        $license = new DRIOP_License( $this->plugin );
        $status = $license->get_status();
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'License & Upgrade', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <div class="driop-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
                <h2 style="color: white;"><?php esc_html_e( 'Upgrade Your Plan', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                <p style="font-size: 16px;"><?php esc_html_e( 'Unlock all features with Pro or Developer plan. Easy payment via USDT/Crypto.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" rel="noopener" class="button button-hero" style="background: #fff; color: #667eea; border: none; font-weight: bold; font-size: 16px; padding: 12px 30px;">
                    <span class="dashicons dashicons-cart" style="margin-top: 3px;"></span> <?php esc_html_e( 'Buy License Now', 'digital-rise-image-optimizer-for-seo' ); ?>
                </a>
                <p style="margin-top: 15px; opacity: 0.9;"><?php esc_html_e( 'After payment, contact us to receive your license key:', 'digital-rise-image-optimizer-for-seo' ); ?> <a href="mailto:<?php echo esc_attr( DRIOP_SUPPORT_EMAIL ); ?>" style="color: white; font-weight: bold;"><?php echo esc_html( DRIOP_SUPPORT_EMAIL ); ?></a></p>
            </div>

            <div class="driop-card">
                <h2><?php esc_html_e( 'License Activation', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                
                <div class="driop-license-status">
                    <p><strong><?php esc_html_e( 'Status:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> 
                        <span class="driop-status-<?php echo esc_attr( $status['status'] ); ?>">
                            <?php echo esc_html( ucfirst( $status['status'] ) ); ?>
                        </span>
                    </p>
                    <p><strong><?php esc_html_e( 'Plan:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> <?php echo esc_html( ucfirst( $status['plan'] ) ); ?></p>
                    <?php if ( $status['key'] ) : ?>
                        <p><strong><?php esc_html_e( 'Key:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> <?php echo esc_html( $license->get_masked_key() ); ?></p>
                    <?php endif; ?>
                    <?php if ( $status['expires'] ) : ?>
                        <p><strong><?php esc_html_e( 'Expires:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> <?php echo esc_html( $status['expires'] ); ?></p>
                    <?php endif; ?>
                </div>

                <form id="license-form">
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'License Key', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="license_key" id="license_key" value="" class="regular-text" placeholder="<?php esc_attr_e( 'Enter your license key', 'digital-rise-image-optimizer-for-seo' ); ?>">
                            </td>
                        </tr>
                    </table>
                    <p>
                        <button type="button" id="activate-license" class="button button-primary"><?php esc_html_e( 'Activate License', 'digital-rise-image-optimizer-for-seo' ); ?></button>
                        <?php if ( $status['key'] ) : ?>
                            <button type="button" id="deactivate-license" class="button"><?php esc_html_e( 'Deactivate', 'digital-rise-image-optimizer-for-seo' ); ?></button>
                        <?php endif; ?>
                    </p>
                </form>
                <div id="license-message"></div>
            </div>

            <div class="driop-card driop-affiliate-card">
                <h2><?php esc_html_e( 'Support Development', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                <p><?php esc_html_e( 'If you find this plugin useful, please consider supporting its development.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" rel="noopener" class="button button-primary button-hero" style="background: #2ecc71; border-color: #27ae60;"><?php esc_html_e( 'Donate with USDT', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                <p style="margin-top: 15px;"><strong><?php esc_html_e( 'Need RedotPay for easy crypto payments?', 'digital-rise-image-optimizer-for-seo' ); ?></strong></p>
                <a href="https://url.hk/i/en/4bq59" target="_blank" rel="noopener" class="button"><?php esc_html_e( 'Get RedotPay', 'digital-rise-image-optimizer-for-seo' ); ?></a>
            </div>
        </div>
        <?php
    }

    public function render_settings() {
        if ( isset( $_POST['driop_save_settings'] ) && check_admin_referer( 'driop_settings_nonce' ) ) {
            $options = $this->plugin->options;
            $options['remote_connection'] = isset( $_POST['remote_connection'] );
            $options['show_attribution'] = isset( $_POST['show_attribution'] );
            $options['wc_integration'] = isset( $_POST['wc_integration'] );
            $options['dokan_integration'] = isset( $_POST['dokan_integration'] );
            $options['confirm_before_optimize'] = isset( $_POST['confirm_before_optimize'] );
            $options['convert_to_webp'] = isset( $_POST['convert_to_webp'] );
            $options['css_minify_enabled'] = isset( $_POST['css_minify_enabled'] );
            $options['js_minify_enabled'] = isset( $_POST['js_minify_enabled'] );
            $options['minify_exclude'] = isset( $_POST['minify_exclude'] ) ? sanitize_textarea_field( wp_unslash( $_POST['minify_exclude'] ) ) : '';
            $options['user_email'] = isset( $_POST['user_email'] ) ? sanitize_email( wp_unslash( $_POST['user_email'] ) ) : '';
            $options['user_whatsapp'] = isset( $_POST['user_whatsapp'] ) ? sanitize_text_field( wp_unslash( $_POST['user_whatsapp'] ) ) : '';
            $this->plugin->save_options( $options );
            
            // Sync contact info with server (like dashboard button)
            $remote = new DRIOP_Remote( $this->plugin );
            $remote->do_checkin();
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved and synced!', 'digital-rise-image-optimizer-for-seo' ) . '</p></div>';
        }
        
        // Handle cache clear
        if ( isset( $_POST['driop_clear_cache'] ) && check_admin_referer( 'driop_settings_nonce' ) ) {
            $minifier = new DRIOP_Minifier( $this->plugin );
            $minifier->clear_cache();
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Cache cleared.', 'digital-rise-image-optimizer-for-seo' ) . '</p></div>';
        }
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Settings', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_settings_nonce' ); ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Image Optimization Behavior', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Confirm Before Optimizing', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="confirm_before_optimize" <?php checked( $this->plugin->get_option( 'confirm_before_optimize', false ) ); ?>> <?php esc_html_e( 'Ask before automatically optimizing images', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                                <p class="description"><?php esc_html_e( 'When enabled, images will NOT be auto-optimized on upload. Use Bulk Optimize to process images manually.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Convert to WebP', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <?php 
                                $optimizer = new DRIOP_Image_Optimizer( $this->plugin );
                                $webp_supported = $optimizer->supports_webp();
                                ?>
                                <label>
                                    <input type="checkbox" name="convert_to_webp" <?php checked( $this->plugin->get_option( 'convert_to_webp', true ) ); ?> <?php disabled( ! $webp_supported ); ?>>
                                    <?php esc_html_e( 'Convert all uploaded images to WebP format', 'digital-rise-image-optimizer-for-seo' ); ?>
                                </label>
                                <?php if ( $webp_supported ) : ?>
                                    <p class="description" style="color: green;">✓ <?php esc_html_e( 'WebP is supported on your server.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                                <?php else : ?>
                                    <p class="description" style="color: red;">✗ <?php esc_html_e( 'WebP is NOT supported. Please update GD library with WebP support.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                                <?php endif; ?>
                                <p class="description"><?php esc_html_e( 'WebP provides 25-34% smaller file sizes compared to JPEG. Your image URLs will look like: yoursite.com/image-keyword.webp', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'CSS & JS Optimization', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Minify CSS', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="css_minify_enabled" <?php checked( $this->plugin->get_option( 'css_minify_enabled', false ) ); ?>> <?php esc_html_e( 'Enable CSS minification', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                                <p class="description"><?php esc_html_e( 'Removes whitespace and comments from CSS files to reduce file size.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Minify JavaScript', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="js_minify_enabled" <?php checked( $this->plugin->get_option( 'js_minify_enabled', false ) ); ?>> <?php esc_html_e( 'Enable JavaScript minification', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                                <p class="description"><?php esc_html_e( 'Removes whitespace and comments from JS files. Use with caution.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Exclude Handles', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="minify_exclude" rows="4" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'minify_exclude', '' ) ); ?></textarea>
                                <p class="description"><?php esc_html_e( 'Enter script/style handles to exclude from minification (one per line). jQuery core is always excluded.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Cache', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <?php 
                                $minifier = new DRIOP_Minifier( $this->plugin );
                                $cache_size = $minifier->get_cache_size();
                                // translators: %s is the cache size (e.g., "5 MB")
                                ?>
                                <p><?php printf( esc_html__( 'Cache size: %s', 'digital-rise-image-optimizer-for-seo' ), esc_html( size_format( $cache_size ) ) ); ?></p>
                                <button type="submit" name="driop_clear_cache" class="button"><?php esc_html_e( 'Clear Cache', 'digital-rise-image-optimizer-for-seo' ); ?></button>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Your Contact Information', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <p><?php esc_html_e( 'Optional: Provide your contact information to receive updates and support.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Email Address', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="email" name="user_email" value="<?php echo esc_attr( $this->plugin->get_option( 'user_email', '' ) ); ?>" class="regular-text" placeholder="your@email.com">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'WhatsApp Number', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="user_whatsapp" value="<?php echo esc_attr( $this->plugin->get_option( 'user_whatsapp', '' ) ); ?>" class="regular-text" placeholder="+1234567890">
                                <p class="description"><?php esc_html_e( 'Include country code (e.g., +1 for USA)', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'General', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Remote Connection', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="remote_connection" <?php checked( $this->plugin->get_option( 'remote_connection', true ) ); ?>> <?php esc_html_e( 'Enable connection to Digital Rise Solutions for license validation', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                                <p class="description"><?php esc_html_e( 'This is optional. Some features require this connection.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Attribution', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="show_attribution" <?php checked( $this->plugin->get_option( 'show_attribution', true ) ); ?> <?php disabled( ! $this->plugin->has_feature( 'disable_attribution' ) && $this->plugin->current_plan === 'free' ); ?>> <?php esc_html_e( 'Show attribution link', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                                <?php if ( ! $this->plugin->has_feature( 'disable_attribution' ) ) : ?>
                                    <p class="description"><?php esc_html_e( 'Upgrade to Pro to disable attribution.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Integrations', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'WooCommerce', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td><label><input type="checkbox" name="wc_integration" <?php checked( $this->plugin->get_option( 'wc_integration', true ) ); ?>> <?php esc_html_e( 'Enable WooCommerce integration', 'digital-rise-image-optimizer-for-seo' ); ?></label></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Dokan', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td><label><input type="checkbox" name="dokan_integration" <?php checked( $this->plugin->get_option( 'dokan_integration', true ) ); ?>> <?php esc_html_e( 'Enable Dokan integration', 'digital-rise-image-optimizer-for-seo' ); ?></label></td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'System Info', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr><th><?php esc_html_e( 'Plugin Version', 'digital-rise-image-optimizer-for-seo' ); ?></th><td><?php echo esc_html( DRIOP_VERSION ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'WordPress', 'digital-rise-image-optimizer-for-seo' ); ?></th><td><?php echo esc_html( get_bloginfo( 'version' ) ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'PHP', 'digital-rise-image-optimizer-for-seo' ); ?></th><td><?php echo esc_html( phpversion() ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'GD Library', 'digital-rise-image-optimizer-for-seo' ); ?></th><td><?php echo function_exists( 'imagecreatetruecolor' ) ? '✓' : '✗'; ?></td></tr>
                        <tr><th><?php esc_html_e( 'Installation ID', 'digital-rise-image-optimizer-for-seo' ); ?></th><td><code><?php echo esc_html( $this->plugin->get_option( 'installation_id' ) ); ?></code></td></tr>
                    </table>
                </div>
                
                <div class="driop-card" style="background: linear-gradient(135deg, #2ecc71, #27ae60); color: #fff;">
                    <h2 style="color: #fff;"><?php esc_html_e( 'Support This Plugin', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <p><?php esc_html_e( 'If you find this plugin useful, please consider making a donation to support continued development.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" rel="noopener" class="button button-hero" style="background: #fff; color: #27ae60; border: none;"><?php esc_html_e( 'Donate with USDT', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                </div>

                <p class="submit"><input type="submit" name="driop_save_settings" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-image-optimizer-for-seo' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    // AJAX Handlers
    public function ajax_dismiss_notice() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        $notice = isset( $_POST['notice'] ) ? sanitize_text_field( wp_unslash( $_POST['notice'] ) ) : '';
        if ( $notice === 'upgrade_banner' ) {
            update_option( 'driop_upgrade_dismissed', true );
        } else {
            $remote = new DRIOP_Remote( $this->plugin );
            $remote->dismiss_notice( $notice );
        }
        wp_send_json_success();
    }

    public function ajax_optimize_image() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        $attachment_id = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;
        
        $optimizer = new DRIOP_Image_Optimizer( $this->plugin );
        $result = $optimizer->optimize_existing( $attachment_id );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result['message'] );
        }
    }

    public function ajax_bulk_optimize() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        $apply_seo = isset( $_POST['apply_seo'] ) && $_POST['apply_seo'] === 'true';
        $reoptimize = isset( $_POST['reoptimize'] ) && $_POST['reoptimize'] === 'true';
        
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $result = $bulk->optimize_batch( 5, $apply_seo, $reoptimize );
        
        wp_send_json_success( $result );
    }

    public function ajax_reset_optimization() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $bulk->reset_optimization_status();
        
        wp_send_json_success( array( 'message' => __( 'All images marked as unoptimized.', 'digital-rise-image-optimizer-for-seo' ) ) );
    }

    public function ajax_activate_license() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        $license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';
        
        $license = new DRIOP_License( $this->plugin );
        $result = $license->activate_license( $license_key );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result['message'] );
        }
    }

    public function ajax_deactivate_license() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $license = new DRIOP_License( $this->plugin );
        $result = $license->deactivate_license();
        
        wp_send_json_success( $result );
    }

    public function ajax_get_optimization_progress() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        
        wp_send_json_success( $progress );
    }

    public function ajax_sync_notices() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $remote = new DRIOP_Remote( $this->plugin );
        $remote->do_checkin();
        
        $notices = $remote->get_active_notices();
        
        wp_send_json_success( array( 
            'message' => __( 'Notices synced successfully.', 'digital-rise-image-optimizer-for-seo' ),
            'count' => count( $notices ),
        ));
    }

    public function render_page_seo() {
        // Handle save
        if ( isset( $_POST['driop_save_page_seo'] ) && check_admin_referer( 'driop_page_seo_nonce' ) ) {
            $this->plugin->update_option( 'page_seo_enabled', isset( $_POST['page_seo_enabled'] ) );
            
            // Posts patterns
            $this->plugin->update_option( 'seo_title_post', isset( $_POST['seo_title_post'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_post'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_post', isset( $_POST['seo_meta_post'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_post'] ) ) : '' );
            
            // Pages patterns
            $this->plugin->update_option( 'seo_title_page', isset( $_POST['seo_title_page'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_page'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_page', isset( $_POST['seo_meta_page'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_page'] ) ) : '' );
            
            // Products patterns (WooCommerce)
            $this->plugin->update_option( 'seo_title_product', isset( $_POST['seo_title_product'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_product'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_product', isset( $_POST['seo_meta_product'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_product'] ) ) : '' );
            
            // Categories patterns
            $this->plugin->update_option( 'seo_title_category', isset( $_POST['seo_title_category'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_category'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_category', isset( $_POST['seo_meta_category'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_category'] ) ) : '' );
            
            // Tags patterns
            $this->plugin->update_option( 'seo_title_tag', isset( $_POST['seo_title_tag'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_tag'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_tag', isset( $_POST['seo_meta_tag'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_tag'] ) ) : '' );
            
            // Product categories patterns (WooCommerce)
            $this->plugin->update_option( 'seo_title_product_cat', isset( $_POST['seo_title_product_cat'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_product_cat'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_product_cat', isset( $_POST['seo_meta_product_cat'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_product_cat'] ) ) : '' );
            
            // Homepage patterns
            $this->plugin->update_option( 'seo_title_home', isset( $_POST['seo_title_home'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_home'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_home', isset( $_POST['seo_meta_home'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_home'] ) ) : '' );
            
            // Archive patterns
            $this->plugin->update_option( 'seo_title_archive', isset( $_POST['seo_title_archive'] ) ? sanitize_text_field( wp_unslash( $_POST['seo_title_archive'] ) ) : '' );
            $this->plugin->update_option( 'seo_meta_archive', isset( $_POST['seo_meta_archive'] ) ? sanitize_textarea_field( wp_unslash( $_POST['seo_meta_archive'] ) ) : '' );
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-image-optimizer-for-seo' ) . '</p></div>';
        }
        
        $page_seo = new DRIOP_Page_SEO( $this->plugin );
        $variables = $page_seo->get_available_variables();
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'SEO Titles & Meta Descriptions', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_page_seo_nonce' ); ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'General Settings', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Enable Page SEO', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="page_seo_enabled" <?php checked( $this->plugin->get_option( 'page_seo_enabled', true ) ); ?>>
                                    <?php esc_html_e( 'Automatically generate SEO title and meta description', 'digital-rise-image-optimizer-for-seo' ); ?>
                                </label>
                                <p class="description"><?php esc_html_e( 'When enabled, the patterns below will be used unless overridden on individual content.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <style>
                    .driop-quick-vars-settings { 
                        margin-top: 8px; 
                        padding: 10px; 
                        background: #f9f9f9; 
                        border-left: 3px solid #2271b1; 
                        border-radius: 3px;
                    }
                    .driop-quick-vars-settings small { 
                        color: #646970; 
                        font-weight: 600; 
                        display: block; 
                        margin-bottom: 5px;
                    }
                    .driop-quick-vars-settings code { 
                        cursor: pointer; 
                        padding: 4px 8px; 
                        background: #fff; 
                        border-radius: 3px; 
                        margin: 3px; 
                        display: inline-block; 
                        color: #1d2327; 
                        border: 1px solid #ddd; 
                        transition: all 0.2s;
                        font-size: 12px;
                    }
                    .driop-quick-vars-settings code:hover { 
                        background: #2271b1; 
                        color: #fff; 
                        border-color: #2271b1; 
                        transform: translateY(-1px);
                        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                    }
                    .driop-quick-vars-settings code.spacer-sep { 
                        background: #e3f2fd; 
                        border-color: #2196f3;
                    }
                    .driop-quick-vars-settings code.spacer-sep:hover { 
                        background: #2196f3; 
                        color: #fff;
                    }
                </style>
                
                <script>
                jQuery(document).ready(function($) {
                    // Handle variable insertion in settings page
                    var lastFocusedField = null;
                    
                    // Track which field is focused
                    $('input[type="text"], textarea').on('focus', function() {
                        lastFocusedField = this;
                    });
                    
                    // Handle variable click
                    $(document).on('click', '.driop-var-insert', function(e) {
                        e.preventDefault();
                        var varText = $(this).data('var');
                        
                        if (lastFocusedField) {
                            var $field = $(lastFocusedField);
                            var field = lastFocusedField;
                            var currentVal = $field.val() || '';
                            var pos = field.selectionStart !== undefined ? field.selectionStart : currentVal.length;
                            
                            var newVal = currentVal.substring(0, pos) + varText + currentVal.substring(pos);
                            $field.val(newVal);
                            
                            // Update cursor position
                            $field.focus();
                            if (field.setSelectionRange) {
                                field.setSelectionRange(pos + varText.length, pos + varText.length);
                            }
                        }
                    });
                });
                </script>
                
                <!-- Homepage -->
                <div class="driop-card">
                    <h2><span class="dashicons dashicons-admin-home"></span> <?php esc_html_e( 'Homepage', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_home" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_home', '{site_name} - {site_tagline}' ) ); ?>" class="large-text">
                                <div class="driop-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                                    <code class="driop-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="driop-var-insert" data-var="{site_tagline}">{site_tagline}</code>
                                    <code class="driop-var-insert" data-var="{year}">{year}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="seo_meta_home" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_home', '{site_tagline}' ) ); ?></textarea>
                                <div class="driop-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                                    <code class="driop-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="driop-var-insert" data-var="{site_tagline}">{site_tagline}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Posts -->
                <div class="driop-card">
                    <h2><span class="dashicons dashicons-admin-post"></span> <?php esc_html_e( 'Posts', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_post" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_post', '{title} | {site_name}' ) ); ?>" class="large-text">
                                <div class="driop-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                                    <code class="driop-var-insert" data-var="{title}">{title}</code>
                                    <code class="driop-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="driop-var-insert" data-var="{category}">{category}</code>
                                    <code class="driop-var-insert" data-var="{author}">{author}</code>
                                    <code class="driop-var-insert" data-var="{date}">{date}</code>
                                    <code class="driop-var-insert" data-var="{year}">{year}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="seo_meta_post" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_post', '{excerpt}' ) ); ?></textarea>
                                <div class="driop-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                                    <code class="driop-var-insert" data-var="{excerpt}">{excerpt}</code>
                                    <code class="driop-var-insert" data-var="{title}">{title}</code>
                                    <code class="driop-var-insert" data-var="{category}">{category}</code>
                                    <code class="driop-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Pages -->
                <div class="driop-card">
                    <h2><span class="dashicons dashicons-admin-page"></span> <?php esc_html_e( 'Pages', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_page" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_page', '{title} | {site_name}' ) ); ?>" class="large-text">
                                <div class="driop-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                                    <code class="driop-var-insert" data-var="{title}">{title}</code>
                                    <code class="driop-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="driop-var-insert" data-var="{parent_title}">{parent_title}</code>
                                    <code class="driop-var-insert" data-var="{year}">{year}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="seo_meta_page" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_page', '{excerpt}' ) ); ?></textarea>
                                <div class="driop-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                                    <code class="driop-var-insert" data-var="{excerpt}">{excerpt}</code>
                                    <code class="driop-var-insert" data-var="{title}">{title}</code>
                                    <code class="driop-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>

                <?php if ( class_exists( 'WooCommerce' ) ) : ?>
                <!-- Products -->
                <div class="driop-card">
                    <h2><span class="dashicons dashicons-cart"></span> <?php esc_html_e( 'Products (WooCommerce)', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_product" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_product', '{title} - Buy Online | {site_name}' ) ); ?>" class="large-text">
                                <div class="driop-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                                    <code class="driop-var-insert" data-var="{title}">{title}</code>
                                    <code class="driop-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="driop-var-insert" data-var="{product_price}">{product_price}</code>
                                    <code class="driop-var-insert" data-var="{product_sku}">{product_sku}</code>
                                    <code class="driop-var-insert" data-var="{category}">{category}</code>
                                    <code class="driop-var-insert" data-var="{year}">{year}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="seo_meta_product" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_product', 'Buy {title} at best price. {excerpt}' ) ); ?></textarea>
                                <div class="driop-quick-vars-settings">
                                    <small><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                                    <code class="driop-var-insert" data-var="{excerpt}">{excerpt}</code>
                                    <code class="driop-var-insert" data-var="{title}">{title}</code>
                                    <code class="driop-var-insert" data-var="{product_price}">{product_price}</code>
                                    <code class="driop-var-insert" data-var="{category}">{category}</code>
                                    <code class="driop-var-insert" data-var="{site_name}">{site_name}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{spacer}">{spacer}</code>
                                    <code class="driop-var-insert spacer-sep" data-var="{separator}">{separator}</code>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <!-- Product Categories -->
                <div class="driop-card">
                    <h2><span class="dashicons dashicons-category"></span> <?php esc_html_e( 'Product Categories', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_product_cat" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_product_cat', '{title} - Shop {count} Products | {site_name}' ) ); ?>" class="large-text">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="seo_meta_product_cat" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_product_cat', 'Browse our {title} collection. {count} products available at {site_name}.' ) ); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>
                <?php endif; ?>

                <!-- Categories -->
                <div class="driop-card">
                    <h2><span class="dashicons dashicons-category"></span> <?php esc_html_e( 'Categories', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_category" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_category', '{title} Archives | {site_name}' ) ); ?>" class="large-text">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="seo_meta_category" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_category', 'Browse all posts in {title}. {count} articles available.' ) ); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Tags -->
                <div class="driop-card">
                    <h2><span class="dashicons dashicons-tag"></span> <?php esc_html_e( 'Tags', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_tag" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_tag', 'Posts Tagged "{title}" | {site_name}' ) ); ?>" class="large-text">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="seo_meta_tag" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_tag', 'All posts tagged with {title}. {count} articles available.' ) ); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Archives -->
                <div class="driop-card">
                    <h2><span class="dashicons dashicons-archive"></span> <?php esc_html_e( 'Date Archives', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Title Pattern', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <input type="text" name="seo_title_archive" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_title_archive', 'Archives: {date} | {site_name}' ) ); ?>" class="large-text">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            <td>
                                <textarea name="seo_meta_archive" rows="2" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'seo_meta_archive', 'Archive of posts from {date} on {site_name}.' ) ); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Available Variables', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <p><?php esc_html_e( 'Use these variables in your patterns. They will be replaced with actual values.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    
                    <table class="widefat striped" style="max-width: 800px;">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Variable', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                                <th><?php esc_html_e( 'Description', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $variables as $var => $desc ) : ?>
                                <tr>
                                    <td><code><?php echo esc_html( $var ); ?></code></td>
                                    <td><?php echo esc_html( $desc ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'How It Works', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                    <ol>
                        <li><strong><?php esc_html_e( 'Default patterns:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> <?php esc_html_e( 'Set above for each content type, applied automatically.', 'digital-rise-image-optimizer-for-seo' ); ?></li>
                        <li><strong><?php esc_html_e( 'Custom override:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> <?php esc_html_e( 'Edit any post, page, or product and scroll down to find the "SEO Settings" meta box to override.', 'digital-rise-image-optimizer-for-seo' ); ?></li>
                        <li><strong><?php esc_html_e( 'Priority:', 'digital-rise-image-optimizer-for-seo' ); ?></strong> <?php esc_html_e( 'Custom values always take precedence over default patterns.', 'digital-rise-image-optimizer-for-seo' ); ?></li>
                    </ol>
                </div>

                <p class="submit"><input type="submit" name="driop_save_page_seo" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-image-optimizer-for-seo' ); ?>"></p>
            </form>
        </div>
        <?php
    }
    
    
    /**
     * Render pending reviews page (hidden, accessed via direct URL)
     */
    public function render_reviews() {
        $reviews = get_option( 'driop_pending_reviews', array() );
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Pending Reviews', 'digital-rise-image-optimizer-for-seo' ) ); ?>
            
            <div class="driop-card">
                <h2><?php esc_html_e( 'Reviews Pending Email Delivery', 'digital-rise-image-optimizer-for-seo' ); ?></h2>
                <p><?php esc_html_e( 'These reviews were submitted but email delivery failed. You can manually send them to seo.oussama@gmail.com', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                
                <?php if ( empty( $reviews ) ) : ?>
                    <p style="background: #d4edda; padding: 15px; border-left: 4px solid #28a745; border-radius: 4px;">
                        ✅ <?php esc_html_e( 'No pending reviews. All reviews were delivered successfully!', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </p>
                <?php else : ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Date', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                                <th><?php esc_html_e( 'Rating', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                                <th><?php esc_html_e( 'Review', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                                <th><?php esc_html_e( 'Website', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                                <th><?php esc_html_e( 'Contact', 'digital-rise-image-optimizer-for-seo' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $reviews as $review ) : ?>
                                <tr>
                                    <td><?php echo esc_html( $review['date'] ); ?></td>
                                    <td><?php echo esc_html( str_repeat( '★', $review['rating'] ) . str_repeat( '☆', 5 - $review['rating'] ) ); ?></td>
                                    <td><?php echo esc_html( wp_trim_words( $review['review'], 20 ) ); ?></td>
                                    <td><a href="<?php echo esc_url( $review['site_url'] ); ?>" target="_blank"><?php echo esc_html( $review['site_url'] ); ?></a></td>
                                    <td>
                                        <?php if ( ! empty( $review['email'] ) ) : ?>
                                            📧 <a href="mailto:<?php echo esc_attr( $review['email'] ); ?>"><?php echo esc_html( $review['email'] ); ?></a><br>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $review['phone'] ) ) : ?>
                                            📞 <?php echo esc_html( $review['phone'] ); ?>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <p style="margin-top: 20px;">
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-reviews&action=clear' ) ); ?>" class="button button-secondary" onclick="return confirm('<?php esc_attr_e( 'Clear all pending reviews?', 'digital-rise-image-optimizer-for-seo' ); ?>');">
                            <?php esc_html_e( 'Clear All', 'digital-rise-image-optimizer-for-seo' ); ?>
                        </a>
                    </p>
                <?php endif; ?>
            </div>
        </div>
        <?php
        
        // Handle clear action
        if ( isset( $_GET['action'] ) && $_GET['action'] === 'clear' ) {
            delete_option( 'driop_pending_reviews' );
            wp_safe_redirect( admin_url( 'admin.php?page=driop-reviews' ) );
            exit;
        }
    }
    
    /**
     * Handle review submission via AJAX
     */
    public function handle_review_submission() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'driop_review_nonce' ) ) {
            wp_send_json_error( __( 'Security check failed', 'digital-rise-image-optimizer-for-seo' ) );
        }
        
        // Get review data
        $rating = isset( $_POST['rating'] ) ? intval( $_POST['rating'] ) : 0;
        $review = isset( $_POST['review'] ) ? sanitize_textarea_field( wp_unslash( $_POST['review'] ) ) : '';
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        $phone = isset( $_POST['phone'] ) ? sanitize_text_field( wp_unslash( $_POST['phone'] ) ) : '';
        $site_url = isset( $_POST['site_url'] ) ? esc_url_raw( wp_unslash( $_POST['site_url'] ) ) : '';
        
        // Validate
        if ( $rating < 1 || $rating > 5 || empty( $review ) ) {
            wp_send_json_error( __( 'Please provide a rating and review', 'digital-rise-image-optimizer-for-seo' ) );
        }
        
        // Prepare email
        $to = 'seo.oussama@gmail.com';
        $subject = sprintf( 
            '[%s] Plugin Review - %d Stars', 
            wp_parse_url( $site_url, PHP_URL_HOST ), 
            $rating 
        );
        
        // Create star display
        $stars = str_repeat( '★', $rating ) . str_repeat( '☆', 5 - $rating );
        
        // Build HTML email
        $message = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; border-radius: 8px 8px 0 0; text-align: center; }
        .stars { font-size: 32px; color: #ffd700; margin: 15px 0; }
        .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 8px 8px; }
        .review-box { background: white; padding: 20px; border-left: 4px solid #667eea; margin: 20px 0; border-radius: 4px; }
        .info-row { padding: 10px 0; border-bottom: 1px solid #eee; }
        .label { font-weight: bold; color: #667eea; display: inline-block; width: 120px; }
        .footer { text-align: center; color: #999; margin-top: 30px; font-size: 12px; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1 style="margin: 0; color: white;">📧 New Plugin Review</h1>
            <div class="stars">' . esc_html( $stars ) . '</div>
            <p style="margin: 0; opacity: 0.9;">' . esc_html( $rating ) . ' out of 5 stars</p>
        </div>
        
        <div class="content">
            <h2 style="color: #667eea; margin-top: 0;">Review Details</h2>
            
            <div class="review-box">
                <p style="margin: 0; font-size: 16px; line-height: 1.8;">' . nl2br( esc_html( $review ) ) . '</p>
            </div>
            
            <h3 style="color: #667eea;">Contact Information</h3>
            <div class="info-row">
                <span class="label">🌐 Website:</span>
                <a href="' . esc_url( $site_url ) . '" style="color: #667eea; text-decoration: none;">' . esc_html( $site_url ) . '</a>
            </div>';
            
        if ( ! empty( $email ) ) {
            $message .= '
            <div class="info-row">
                <span class="label">📧 Email:</span>
                <a href="mailto:' . esc_attr( $email ) . '" style="color: #667eea; text-decoration: none;">' . esc_html( $email ) . '</a>
            </div>';
        }
        
        if ( ! empty( $phone ) ) {
            $message .= '
            <div class="info-row">
                <span class="label">📞 Phone:</span>
                <span>' . esc_html( $phone ) . '</span>
            </div>';
        }
        
        $message .= '
            <div class="footer">
                <p>Sent from Digital Rise Image Optimizer for SEO Plugin</p>
                <p>Received: ' . esc_html( current_time( 'F j, Y \a\t g:i a' ) ) . '</p>
            </div>
        </div>
    </div>
</body>
</html>';
        
        // Email headers
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_bloginfo( 'name' ) . ' <' . get_bloginfo( 'admin_email' ) . '>',
        );
        
        if ( ! empty( $email ) ) {
            $headers[] = 'Reply-To: ' . $email;
        }
        
        // Try to send email with wp_mail first
        $sent = wp_mail( $to, $subject, $message, $headers );
        
        // If wp_mail fails, try alternative method using PHP mail()
        if ( ! $sent ) {
            // Create simple text version as fallback
            $text_message = "New Plugin Review\n\n";
            $text_message .= "Rating: " . $stars . " (" . $rating . "/5)\n\n";
            $text_message .= "Review:\n" . $review . "\n\n";
            $text_message .= "Website: " . $site_url . "\n";
            if ( ! empty( $email ) ) {
                $text_message .= "Email: " . $email . "\n";
            }
            if ( ! empty( $phone ) ) {
                $text_message .= "Phone: " . $phone . "\n";
            }
            $text_message .= "\nReceived: " . current_time( 'F j, Y \a\t g:i a' );
            
            // Build headers for PHP mail
            $simple_headers = "From: " . get_bloginfo( 'name' ) . " <" . get_bloginfo( 'admin_email' ) . ">\r\n";
            $simple_headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
            if ( ! empty( $email ) ) {
                $simple_headers .= "Reply-To: " . $email . "\r\n";
            }
            
            // Try PHP mail as fallback
            $sent = mail( $to, $subject, $text_message, $simple_headers );
        }
        
        // If still failed, save to database as backup
        if ( ! $sent ) {
            $review_data = array(
                'rating' => $rating,
                'review' => $review,
                'email' => $email,
                'phone' => $phone,
                'site_url' => $site_url,
                'date' => current_time( 'mysql' ),
            );
            
            // Save to options table
            $reviews = get_option( 'driop_pending_reviews', array() );
            $reviews[] = $review_data;
            update_option( 'driop_pending_reviews', $reviews );
            
            // Still return success to user but log the issue
            error_log( 'DRIOP Review: Email failed but saved to database. Review from: ' . $site_url );
            wp_send_json_success( __( 'Thank you! Your review has been recorded and will be sent shortly.', 'digital-rise-image-optimizer-for-seo' ) );
        }
        
        if ( $sent ) {
            wp_send_json_success( __( 'Thank you for your review!', 'digital-rise-image-optimizer-for-seo' ) );
        } else {
            wp_send_json_error( __( 'Failed to send review. Please try again.', 'digital-rise-image-optimizer-for-seo' ) );
        }
    }
}
