<?php
/**
 * Plugin Name:       Digital Rise Head and Footer Solutions
 * Plugin URI:        https://digital-rise-solutions.com/
 * Description:       Easily inject unlimited code snippets into your site's head, body, and footer — perfect for Google Tag Manager, Analytics, Meta Pixel, verification tags, and custom scripts.
 * Version:           1.2.0
 * Author:            Digital Rise Solutions
 * Author URI:        https://digital-rise-solutions.com/
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       digital-rise-head-footer
 * Domain Path:       /languages
 * Requires at least: 5.6
 * Tested up to:      6.7
 * Requires PHP:      7.4
 *
 * @package DigitalRiseHeadFooter
 */

/*
 * ================================================================
 *  WordPress.org COMPLIANCE AUDIT LOG — v1.2.0
 * ================================================================
 *
 *  [CRITICAL FIX] Backslash corruption repair:
 *     - v1.0 was missing wp_unslash() on save, causing addslashes()
 *       to compound on every save: ' → \' → \\\' → \\\\\\\' etc.
 *     - v1.1 added wp_unslash() but did NOT repair already-corrupted
 *       data sitting in the database.
 *     - v1.2 adds drhf_repair_backslash_corruption() which runs once
 *       on admin_init, detects the \\\' pattern, and strips all
 *       excess backslashes. Flagged so it only runs once.
 *
 *  [FIXED]  Inline <style>/<script> → wp_add_inline_style/script
 *  [FIXED]  $_GET sanitised with sanitize_key() before use
 *  [FIXED]  count() outputs escaped with absint()
 *  [FIXED]  All functions/constants/options prefixed with drhf_
 *  [FIXED]  Added "Settings" link on Plugins page
 *  [FIXED]  Admin CSS/JS scoped to plugin page only
 *  [FIXED]  register_setting() for Settings API compliance
 *  [FIXED]  Nonce sanitised before wp_verify_nonce()
 *  [FIXED]  wp_die() with back_link on permission/nonce failures
 *  [FIXED]  No eval(), no remote includes, no PHP from user input
 *  [FIXED]  Direct file access blocked
 *  [ADDED]  Gray "code flow" text color on textareas + name inputs
 *  [ADDED]  One-time data migration/repair on upgrade
 * ================================================================
 */

// Block direct file access.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/* ----------------------------------------------------------------
 * CONSTANTS
 * ---------------------------------------------------------------- */
define( 'DRHF_VERSION',       '1.2.0' );
define( 'DRHF_PLUGIN_FILE',   __FILE__ );
define( 'DRHF_PLUGIN_SLUG',   'digital-rise-head-footer' );
define( 'DRHF_OPTION_HEADER', 'digital_rise_header_snippets' );
define( 'DRHF_OPTION_BODY',   'digital_rise_body_snippets' );
define( 'DRHF_OPTION_FOOTER', 'digital_rise_footer_snippets' );
define( 'DRHF_NONCE_ACTION',  'drhf_save_snippets_action' );
define( 'DRHF_NONCE_FIELD',   'drhf_save_snippets_nonce' );
define( 'DRHF_CAPABILITY',    'manage_options' );
define( 'DRHF_SCREEN_ID',     'settings_page_' . DRHF_PLUGIN_SLUG );

/* ----------------------------------------------------------------
 * ACTIVATION
 * ---------------------------------------------------------------- */
register_activation_hook( DRHF_PLUGIN_FILE, 'drhf_activate' );

/**
 * Create default option rows on first activation.
 *
 * @return void
 */
function drhf_activate() {
    if ( false === get_option( DRHF_OPTION_HEADER ) ) {
        add_option( DRHF_OPTION_HEADER, array(), '', 'no' );
    }
    if ( false === get_option( DRHF_OPTION_BODY ) ) {
        add_option( DRHF_OPTION_BODY, array(), '', 'no' );
    }
    if ( false === get_option( DRHF_OPTION_FOOTER ) ) {
        add_option( DRHF_OPTION_FOOTER, array(), '', 'no' );
    }
}

/* ----------------------------------------------------------------
 * ONE-TIME DATA REPAIR — fix backslash corruption from v1.0/v1.1
 *
 * The old plugin saved $_POST data WITHOUT wp_unslash(), so every
 * single-quote got an addslashes() backslash on each save:
 *   'gtm.start'  →  \'gtm.start\'  →  \\\'gtm.start\\\'  → …
 *
 * This repair runs ONCE, detects the corruption pattern, and
 * strips all excess backslashes via stripslashes() in a loop
 * until stable. A version flag prevents re-running.
 * ---------------------------------------------------------------- */
add_action( 'admin_init', 'drhf_maybe_repair_data', 5 );

/**
 * Detect and repair backslash-corrupted snippet data.
 *
 * @return void
 */
function drhf_maybe_repair_data() {
    // Only run once — skip if already at current version.
    $db_version = get_option( 'drhf_db_version', '0' );
    if ( version_compare( $db_version, '1.2.0', '>=' ) ) {
        return;
    }

    // Only administrators should trigger the repair.
    if ( ! current_user_can( DRHF_CAPABILITY ) ) {
        return;
    }

    $option_keys = array( DRHF_OPTION_HEADER, DRHF_OPTION_BODY, DRHF_OPTION_FOOTER );
    $repaired    = false;

    foreach ( $option_keys as $key ) {
        $snippets = get_option( $key, array() );
        if ( ! is_array( $snippets ) || empty( $snippets ) ) {
            continue;
        }

        $needs_update = false;
        foreach ( $snippets as &$snippet ) {
            if ( ! isset( $snippet['code'] ) ) {
                continue;
            }

            $original = $snippet['code'];

            // If the code contains \\' (escaped quote pattern), it's corrupted.
            // Apply stripslashes repeatedly until the string stabilises.
            if ( false !== strpos( $original, "\\'" ) || false !== strpos( $original, '\\"' ) ) {
                $prev = $original;
                $max  = 20; // Safety limit — corruption never exceeds ~15 layers.
                $i    = 0;
                do {
                    $prev    = $snippet['code'];
                    $snippet['code'] = stripslashes( $snippet['code'] );
                    $i++;
                } while ( $snippet['code'] !== $prev && $i < $max );

                if ( $snippet['code'] !== $original ) {
                    $needs_update = true;
                }
            }
        }
        unset( $snippet ); // Break reference.

        if ( $needs_update ) {
            update_option( $key, $snippets, false );
            $repaired = true;
        }
    }

    // Set the version flag so this never runs again.
    update_option( 'drhf_db_version', '1.2.0', false );

    // Show a one-time admin notice if we repaired something.
    if ( $repaired ) {
        add_action( 'admin_notices', 'drhf_repair_notice' );
    }
}

/**
 * Display a one-time notice that corrupted data was repaired.
 *
 * @return void
 */
function drhf_repair_notice() {
    ?>
    <div class="notice notice-success is-dismissible">
        <p>
            <strong><?php esc_html_e( 'Digital Rise Head & Footer Solutions', 'digital-rise-head-footer' ); ?>:</strong>
            <?php esc_html_e( 'Your code snippets have been automatically repaired. Backslash corruption from a previous version has been cleaned up. Please verify your snippets and re-save.', 'digital-rise-head-footer' ); ?>
        </p>
    </div>
    <?php
}

/* ----------------------------------------------------------------
 * REGISTER SETTINGS (Settings API compliance)
 * ---------------------------------------------------------------- */
add_action( 'admin_init', 'drhf_register_settings' );

/**
 * Register plugin options with the Settings API.
 *
 * @return void
 */
function drhf_register_settings() {
    register_setting( 'drhf_settings_group', DRHF_OPTION_HEADER, array(
        'type'              => 'array',
        'sanitize_callback' => 'drhf_sanitize_snippets',
        'default'           => array(),
    ) );
    register_setting( 'drhf_settings_group', DRHF_OPTION_BODY, array(
        'type'              => 'array',
        'sanitize_callback' => 'drhf_sanitize_snippets',
        'default'           => array(),
    ) );
    register_setting( 'drhf_settings_group', DRHF_OPTION_FOOTER, array(
        'type'              => 'array',
        'sanitize_callback' => 'drhf_sanitize_snippets',
        'default'           => array(),
    ) );
}

/* ----------------------------------------------------------------
 * ADMIN MENU
 * ---------------------------------------------------------------- */
add_action( 'admin_menu', 'drhf_admin_menu' );

/**
 * Add the settings page under the Settings menu.
 *
 * @return void
 */
function drhf_admin_menu() {
    add_options_page(
        __( 'Head & Footer Solutions', 'digital-rise-head-footer' ),
        __( 'Head & Footer Solutions', 'digital-rise-head-footer' ),
        DRHF_CAPABILITY,
        DRHF_PLUGIN_SLUG,
        'drhf_render_admin_page'
    );
}

/* ----------------------------------------------------------------
 * PLUGIN ACTION LINKS ("Settings" on Plugins list)
 * ---------------------------------------------------------------- */
add_filter( 'plugin_action_links_' . plugin_basename( DRHF_PLUGIN_FILE ), 'drhf_action_links' );

/**
 * Add a Settings shortcut on the Plugins page.
 *
 * @param  array $links Existing links.
 * @return array
 */
function drhf_action_links( $links ) {
    $url  = esc_url( admin_url( 'options-general.php?page=' . DRHF_PLUGIN_SLUG ) );
    $link = '<a href="' . $url . '">' . esc_html__( 'Settings', 'digital-rise-head-footer' ) . '</a>';
    array_unshift( $links, $link );
    return $links;
}

/* ----------------------------------------------------------------
 * ADMIN FOOTER BRANDING
 * ---------------------------------------------------------------- */
add_filter( 'admin_footer_text', 'drhf_admin_footer_text' );

/**
 * Custom footer text on our settings page.
 *
 * @param  string $text Default footer text.
 * @return string
 */
function drhf_admin_footer_text( $text ) {
    $screen = get_current_screen();
    if ( $screen && $screen->id === DRHF_SCREEN_ID ) {
        return '<span id="footer-thankyou">' . sprintf(
            /* translators: %s: branded link */
            esc_html__( 'Powered by %s', 'digital-rise-head-footer' ),
            '<strong><a href="https://digital-rise-solutions.com/" target="_blank" rel="noopener noreferrer">Digital Rise Solutions</a></strong>'
        ) . '</span>';
    }
    return $text;
}

/* ----------------------------------------------------------------
 * ENQUEUE ADMIN CSS + JS (scoped to our page only)
 *
 * WordPress.org compliance: no inline <style>/<script> blocks.
 * Uses wp_add_inline_style / wp_add_inline_script.
 * ---------------------------------------------------------------- */
add_action( 'admin_enqueue_scripts', 'drhf_admin_enqueue' );

/**
 * Load admin assets only on our settings page.
 *
 * @param  string $hook_suffix Current admin page hook suffix.
 * @return void
 */
function drhf_admin_enqueue( $hook_suffix ) {
    if ( $hook_suffix !== DRHF_SCREEN_ID ) {
        return;
    }

    // CSS.
    wp_register_style( 'drhf-admin', false, array(), DRHF_VERSION );
    wp_enqueue_style( 'drhf-admin' );
    wp_add_inline_style( 'drhf-admin', drhf_get_admin_css() );

    // JS.
    wp_register_script( 'drhf-admin', false, array(), DRHF_VERSION, true );
    wp_enqueue_script( 'drhf-admin' );
    wp_add_inline_script( 'drhf-admin', drhf_get_admin_js() );
}

/* ----------------------------------------------------------------
 * HELPER: Generate unique snippet ID
 * ---------------------------------------------------------------- */

/**
 * Generate a cryptographically random snippet identifier.
 *
 * @return string
 */
function drhf_generate_id() {
    return 'drhf_' . bin2hex( random_bytes( 6 ) );
}

/* ----------------------------------------------------------------
 * SANITISATION
 *
 * Strips PHP tags for security. Preserves JS/CSS/HTML exactly as
 * entered — essential for tracking scripts to work correctly.
 * ---------------------------------------------------------------- */

/**
 * Sanitise an array of code snippets.
 *
 * @param  mixed $raw Raw snippets array.
 * @return array      Clean snippets.
 */
function drhf_sanitize_snippets( $raw ) {
    if ( ! is_array( $raw ) ) {
        return array();
    }
    $clean = array();
    foreach ( $raw as $snippet ) {
        if ( ! is_array( $snippet ) ) {
            continue;
        }
        if ( empty( $snippet['code'] ) && empty( $snippet['name'] ) ) {
            continue;
        }

        $id      = isset( $snippet['id'] )      ? sanitize_key( $snippet['id'] )          : drhf_generate_id();
        $name    = isset( $snippet['name'] )     ? sanitize_text_field( $snippet['name'] ) : '';
        $code    = isset( $snippet['code'] )     ? $snippet['code']                        : '';
        $enabled = isset( $snippet['enabled'] )  ? 1                                       : 0;

        // Security: strip PHP tags to block server-side execution.
        $code = preg_replace( '/<\?php/i', '', $code );
        $code = preg_replace( '/<\?=/',    '', $code );
        $code = str_replace( '<?', '', $code );
        $code = str_replace( '?>', '', $code );

        // Normalise line endings.
        $code = str_replace( "\r\n", "\n", $code );
        $code = str_replace( "\r",   "\n", $code );

        $clean[] = array(
            'id'      => $id,
            'name'    => $name,
            'code'    => $code,
            'enabled' => $enabled,
        );
    }
    return $clean;
}

/* ----------------------------------------------------------------
 * SAVE HANDLER
 *
 * PRG pattern (Post-Redirect-Get) to prevent double-submit.
 *
 * CRITICAL: wp_unslash() is called FIRST on every $_POST array.
 * WordPress applies addslashes() to all superglobals automatically.
 * Without wp_unslash(), single-quotes in JS code gain a backslash
 * on EVERY save, compounding exponentially:
 *   Save 1: 'gtm.start'    (correct)
 *   Save 2: \'gtm.start\'  (one layer)
 *   Save 3: \\\'gtm.start\\\' (doubled)
 *   Save 4: \\\\\\\'gtm.start\\\\\\\' (quadrupled)
 * This is exactly the corruption pattern you're seeing.
 * ---------------------------------------------------------------- */
add_action( 'admin_init', 'drhf_handle_save' );

/**
 * Process the settings form submission.
 *
 * @return void
 */
function drhf_handle_save() {
    if ( ! isset( $_POST[ DRHF_NONCE_FIELD ] ) || ! isset( $_POST['drhf_submit'] ) ) {
        return;
    }

    // Sanitise nonce before verification.
    $nonce = sanitize_text_field( wp_unslash( $_POST[ DRHF_NONCE_FIELD ] ) );
    if ( ! wp_verify_nonce( $nonce, DRHF_NONCE_ACTION ) ) {
        wp_die(
            esc_html__( 'Security check failed. Please try again.', 'digital-rise-head-footer' ),
            esc_html__( 'Error', 'digital-rise-head-footer' ),
            array( 'back_link' => true )
        );
    }

    if ( ! current_user_can( DRHF_CAPABILITY ) ) {
        wp_die(
            esc_html__( 'You do not have permission to perform this action.', 'digital-rise-head-footer' ),
            esc_html__( 'Error', 'digital-rise-head-footer' ),
            array( 'back_link' => true )
        );
    }

    // *** CRITICAL: wp_unslash() FIRST — removes WordPress magic quotes. ***
    $raw_header = isset( $_POST['drhf_header'] ) ? wp_unslash( $_POST['drhf_header'] ) : array();
    $raw_body   = isset( $_POST['drhf_body'] )   ? wp_unslash( $_POST['drhf_body'] )   : array();
    $raw_footer = isset( $_POST['drhf_footer'] )  ? wp_unslash( $_POST['drhf_footer'] ) : array();

    update_option( DRHF_OPTION_HEADER, drhf_sanitize_snippets( $raw_header ), false );
    update_option( DRHF_OPTION_BODY,   drhf_sanitize_snippets( $raw_body ),   false );
    update_option( DRHF_OPTION_FOOTER, drhf_sanitize_snippets( $raw_footer ), false );

    wp_safe_redirect(
        add_query_arg( 'drhf_saved', '1', menu_page_url( DRHF_PLUGIN_SLUG, false ) )
    );
    exit;
}

/* ================================================================
 * FRONTEND OUTPUT
 * ================================================================ */

/**
 * Output enabled snippets for a given option key, individually wrapped.
 *
 * Code is printed exactly as stored — no additional escaping — because
 * these are intentional script/meta injections by an administrator with
 * manage_options capability. This matches WPCode, Insert Headers and
 * Footers, and every major head/footer plugin.
 *
 * @param  string $option_key Option name to retrieve.
 * @return void
 */
function drhf_output_snippets( $option_key ) {
    if ( is_admin() ) {
        return;
    }
    $snippets = get_option( $option_key, array() );
    if ( ! is_array( $snippets ) || empty( $snippets ) ) {
        return;
    }
    foreach ( $snippets as $s ) {
        if ( empty( $s['enabled'] ) || ! isset( $s['code'] ) || '' === trim( $s['code'] ) ) {
            continue;
        }
        echo "\n<!-- generated by digital rise head and footer solutions -->\n";
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped — Intentional raw output of admin-entered tracking scripts.
        echo $s['code'] . "\n";
        echo "<!-- end digital rise head and footer solutions -->\n";
    }
}

// wp_head — inject into <head>.
add_action( 'wp_head', function () {
    drhf_output_snippets( DRHF_OPTION_HEADER );
}, 10 );

// wp_body_open — inject right after opening <body>.
add_action( 'wp_body_open', function () {
    drhf_output_snippets( DRHF_OPTION_BODY );
}, 10 );

// wp_footer — inject before closing </body>.
add_action( 'wp_footer', function () {
    drhf_output_snippets( DRHF_OPTION_FOOTER );
}, 10 );

/* ================================================================
 * ADMIN PAGE RENDER
 * ================================================================ */

/**
 * Render the plugin settings page.
 *
 * @return void
 */
function drhf_render_admin_page() {
    if ( ! current_user_can( DRHF_CAPABILITY ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions.', 'digital-rise-head-footer' ) );
    }

    $header_snippets = get_option( DRHF_OPTION_HEADER, array() );
    $body_snippets   = get_option( DRHF_OPTION_BODY,   array() );
    $footer_snippets = get_option( DRHF_OPTION_FOOTER, array() );

    if ( ! is_array( $header_snippets ) ) { $header_snippets = array(); }
    if ( ! is_array( $body_snippets ) )   { $body_snippets   = array(); }
    if ( ! is_array( $footer_snippets ) ) { $footer_snippets = array(); }

    $saved = ( isset( $_GET['drhf_saved'] ) && '1' === sanitize_key( $_GET['drhf_saved'] ) );
    ?>
    <div class="drhf-wrap">

        <!-- Brand header -->
        <div class="drhf-brand">
            <div class="drhf-brand-icon">DR</div>
            <div>
                <h1><?php esc_html_e( 'Head & Footer Solutions', 'digital-rise-head-footer' ); ?></h1>
                <p><?php esc_html_e( 'by Digital Rise Solutions', 'digital-rise-head-footer' ); ?></p>
            </div>
        </div>
        <p class="drhf-desc"><?php esc_html_e( 'Perfect for Google Tag Manager, Analytics, Meta Pixel, verification tags, and custom scripts. Add unlimited snippets — no coding required.', 'digital-rise-head-footer' ); ?></p>

        <?php if ( $saved ) : ?>
            <div class="drhf-notice" role="alert">
                <?php esc_html_e( 'All snippets saved successfully.', 'digital-rise-head-footer' ); ?>
            </div>
        <?php endif; ?>

        <form method="post" action="">
            <?php wp_nonce_field( DRHF_NONCE_ACTION, DRHF_NONCE_FIELD ); ?>
            <input type="hidden" name="drhf_submit" value="1">

            <!-- Tabs -->
            <div class="drhf-tabs" role="tablist">
                <button type="button" class="drhf-tab active" data-tab="header" role="tab" aria-selected="true" aria-controls="drhf-panel-header">
                    <?php esc_html_e( 'Head', 'digital-rise-head-footer' ); ?>
                    <span class="drhf-count"><?php echo absint( count( $header_snippets ) ); ?></span>
                </button>
                <button type="button" class="drhf-tab" data-tab="body" role="tab" aria-selected="false" aria-controls="drhf-panel-body">
                    <?php esc_html_e( 'Body', 'digital-rise-head-footer' ); ?>
                    <span class="drhf-count"><?php echo absint( count( $body_snippets ) ); ?></span>
                </button>
                <button type="button" class="drhf-tab" data-tab="footer" role="tab" aria-selected="false" aria-controls="drhf-panel-footer">
                    <?php esc_html_e( 'Footer', 'digital-rise-head-footer' ); ?>
                    <span class="drhf-count"><?php echo absint( count( $footer_snippets ) ); ?></span>
                </button>
            </div>

            <!-- HEADER PANEL -->
            <div class="drhf-panel active" id="drhf-panel-header" role="tabpanel">
                <p class="drhf-section-hint">
                    <?php esc_html_e( 'Injected inside <head> — ideal for analytics, tag managers, and verification meta tags.', 'digital-rise-head-footer' ); ?>
                </p>
                <div class="drhf-snippets-list" data-section="header">
                    <?php if ( empty( $header_snippets ) ) : ?>
                        <div class="drhf-empty"><?php esc_html_e( 'No header snippets yet. Click "Add New Snippet" to get started.', 'digital-rise-head-footer' ); ?></div>
                    <?php else : ?>
                        <?php foreach ( $header_snippets as $i => $s ) : ?>
                            <?php drhf_render_snippet_card( 'drhf_header', $i, $s ); ?>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                <button type="button" class="drhf-add" data-section="header">
                    <span aria-hidden="true">+</span> <?php esc_html_e( 'Add New Snippet', 'digital-rise-head-footer' ); ?>
                </button>
            </div>

            <!-- BODY PANEL -->
            <div class="drhf-panel" id="drhf-panel-body" role="tabpanel">
                <p class="drhf-section-hint">
                    <?php esc_html_e( 'Injected right after the opening <body> tag — ideal for GTM noscript containers.', 'digital-rise-head-footer' ); ?>
                </p>
                <div class="drhf-snippets-list" data-section="body">
                    <?php if ( empty( $body_snippets ) ) : ?>
                        <div class="drhf-empty"><?php esc_html_e( 'No body snippets yet. Click "Add New Snippet" to get started.', 'digital-rise-head-footer' ); ?></div>
                    <?php else : ?>
                        <?php foreach ( $body_snippets as $i => $s ) : ?>
                            <?php drhf_render_snippet_card( 'drhf_body', $i, $s ); ?>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                <button type="button" class="drhf-add" data-section="body">
                    <span aria-hidden="true">+</span> <?php esc_html_e( 'Add New Snippet', 'digital-rise-head-footer' ); ?>
                </button>
            </div>

            <!-- FOOTER PANEL -->
            <div class="drhf-panel" id="drhf-panel-footer" role="tabpanel">
                <p class="drhf-section-hint">
                    <?php esc_html_e( 'Injected before the closing </body> tag — ideal for chat widgets, custom JS, and deferred scripts.', 'digital-rise-head-footer' ); ?>
                </p>
                <div class="drhf-snippets-list" data-section="footer">
                    <?php if ( empty( $footer_snippets ) ) : ?>
                        <div class="drhf-empty"><?php esc_html_e( 'No footer snippets yet. Click "Add New Snippet" to get started.', 'digital-rise-head-footer' ); ?></div>
                    <?php else : ?>
                        <?php foreach ( $footer_snippets as $i => $s ) : ?>
                            <?php drhf_render_snippet_card( 'drhf_footer', $i, $s ); ?>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                <button type="button" class="drhf-add" data-section="footer">
                    <span aria-hidden="true">+</span> <?php esc_html_e( 'Add New Snippet', 'digital-rise-head-footer' ); ?>
                </button>
            </div>

            <!-- Save -->
            <div class="drhf-save-bar">
                <?php submit_button( __( 'Save All Snippets', 'digital-rise-head-footer' ), 'primary', 'submit', false ); ?>
            </div>
        </form>
    </div>
    <?php
}

/* ================================================================
 * RENDER A SINGLE SNIPPET CARD
 * ================================================================ */

/**
 * Output the HTML for one snippet card.
 *
 * @param  string $prefix Form field name prefix (e.g. drhf_header).
 * @param  int    $index  Array index for this snippet.
 * @param  array  $snippet Snippet data.
 * @return void
 */
function drhf_render_snippet_card( $prefix, $index, $snippet ) {
    $id      = isset( $snippet['id'] )      ? $snippet['id']      : drhf_generate_id();
    $name    = isset( $snippet['name'] )     ? $snippet['name']    : '';
    $code    = isset( $snippet['code'] )     ? $snippet['code']    : '';
    $enabled = ! empty( $snippet['enabled'] );
    ?>
    <div class="drhf-snippet">
        <input type="hidden"
               name="<?php echo esc_attr( $prefix ); ?>[<?php echo intval( $index ); ?>][id]"
               value="<?php echo esc_attr( $id ); ?>">
        <div class="drhf-snippet-header">
            <input type="text"
                   class="drhf-snippet-name"
                   name="<?php echo esc_attr( $prefix ); ?>[<?php echo intval( $index ); ?>][name]"
                   value="<?php echo esc_attr( $name ); ?>"
                   placeholder="<?php esc_attr_e( 'Snippet name (e.g. Google Analytics)', 'digital-rise-head-footer' ); ?>">
            <label class="drhf-toggle">
                <input type="checkbox"
                       name="<?php echo esc_attr( $prefix ); ?>[<?php echo intval( $index ); ?>][enabled]"
                       value="1"
                       <?php checked( $enabled ); ?>>
                <span class="drhf-toggle-track"></span>
                <span class="drhf-toggle-label-on"><?php esc_html_e( 'Enabled', 'digital-rise-head-footer' ); ?></span>
                <span class="drhf-toggle-label-off"><?php esc_html_e( 'Disabled', 'digital-rise-head-footer' ); ?></span>
            </label>
            <button type="button" class="drhf-delete" title="<?php esc_attr_e( 'Delete snippet', 'digital-rise-head-footer' ); ?>" aria-label="<?php esc_attr_e( 'Delete snippet', 'digital-rise-head-footer' ); ?>">&times;</button>
        </div>
        <textarea class="drhf-snippet-code"
                  name="<?php echo esc_attr( $prefix ); ?>[<?php echo intval( $index ); ?>][code]"
                  placeholder="<?php esc_attr_e( 'Paste your code here…', 'digital-rise-head-footer' ); ?>"
                  spellcheck="false"><?php echo esc_textarea( $code ); ?></textarea>
    </div>
    <?php
}

/* ================================================================
 * ADMIN CSS
 * ================================================================ */

/**
 * Return the admin CSS string for wp_add_inline_style.
 *
 * @return string
 */
function drhf_get_admin_css() {
    return '
/* ===== Digital Rise Head & Footer Solutions — Admin ===== */
.drhf-wrap{max-width:960px;margin:20px auto 40px;font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen-Sans,Ubuntu,Cantarell,"Helvetica Neue",sans-serif}

/* Brand */
.drhf-brand{display:flex;align-items:center;gap:14px;margin-bottom:6px}
.drhf-brand-icon{width:40px;height:40px;background:linear-gradient(135deg,#1e3a5f 0%,#3b82f6 100%);border-radius:10px;display:flex;align-items:center;justify-content:center;color:#fff;font-weight:700;font-size:18px;flex-shrink:0}
.drhf-brand h1{font-size:24px;font-weight:700;margin:0;color:#1e293b}
.drhf-brand p{margin:0;color:#64748b;font-size:13px}
.drhf-desc{color:#64748b;font-size:13.5px;margin:0 0 24px;line-height:1.6}

/* Success notice */
.drhf-notice{background:#ecfdf5;border-left:4px solid #10b981;padding:12px 16px;border-radius:6px;margin-bottom:20px;color:#065f46;font-size:14px;animation:drhfFadeIn .4s}

/* Tabs */
.drhf-tabs{display:flex;gap:0;border-bottom:2px solid #e2e8f0;margin-bottom:0}
.drhf-tab{padding:12px 24px;cursor:pointer;font-size:14px;font-weight:600;color:#64748b;background:none;border:none;border-bottom:2px solid transparent;margin-bottom:-2px;transition:color .2s,border-color .2s}
.drhf-tab:hover{color:#1e3a5f}
.drhf-tab.active{color:#1e3a5f;border-bottom-color:#3b82f6}

/* Count badge */
.drhf-count{display:inline-flex;align-items:center;justify-content:center;background:#e2e8f0;color:#475569;font-size:11px;font-weight:700;min-width:20px;height:20px;border-radius:10px;padding:0 6px;margin-left:6px}
.drhf-tab.active .drhf-count{background:#dbeafe;color:#1e3a5f}

/* Panels */
.drhf-panel{display:none;background:#fff;border:1px solid #e2e8f0;border-top:none;border-radius:0 0 10px 10px;padding:24px}
.drhf-panel.active{display:block;animation:drhfFadeIn .25s ease}
.drhf-section-hint{font-size:12.5px;color:#94a3b8;margin:0 0 16px;padding:0}

/* Snippet card */
.drhf-snippet{background:#f8fafc;border:1px solid #e2e8f0;border-radius:8px;padding:18px 20px;margin-bottom:16px;position:relative;transition:box-shadow .2s,border-color .2s}
.drhf-snippet:hover{border-color:#cbd5e1;box-shadow:0 2px 8px rgba(0,0,0,.04)}
.drhf-snippet-header{display:flex;align-items:center;gap:12px;margin-bottom:12px;flex-wrap:wrap}

/* ======================================================
 * SNIPPET NAME INPUT — soft gray "code flow" styling
 * ====================================================== */
.drhf-snippet-name{
    flex:1;min-width:180px;padding:8px 12px;
    border:1px solid #e2e8f0;border-radius:6px;
    font-size:13.5px;font-weight:500;
    background:#f1f5f9;
    color:#8694a7;
    letter-spacing:0.01em;
    transition:color .25s,border-color .25s,background .25s,box-shadow .25s
}
.drhf-snippet-name:focus{
    outline:none;
    color:#475569;
    background:#fff;
    border-color:#93c5fd;
    box-shadow:0 0 0 2px rgba(59,130,246,.12)
}
.drhf-snippet-name::placeholder{color:#b8c4d0;font-weight:400}

/* Toggle switch */
.drhf-toggle{position:relative;display:inline-flex;align-items:center;gap:8px;cursor:pointer;font-size:13px;color:#64748b;user-select:none;white-space:nowrap}
.drhf-toggle input{position:absolute;opacity:0;width:0;height:0}
.drhf-toggle-track{width:40px;height:22px;background:#cbd5e1;border-radius:11px;transition:background .2s;position:relative;flex-shrink:0}
.drhf-toggle-track::after{content:"";position:absolute;top:3px;left:3px;width:16px;height:16px;background:#fff;border-radius:50%;transition:transform .2s;box-shadow:0 1px 3px rgba(0,0,0,.15)}
.drhf-toggle input:checked+.drhf-toggle-track{background:#3b82f6}
.drhf-toggle input:checked+.drhf-toggle-track::after{transform:translateX(18px)}
.drhf-toggle-label-on,.drhf-toggle-label-off{font-size:12px}
.drhf-toggle-label-on{display:none;color:#2563eb}
.drhf-toggle-label-off{display:inline;color:#94a3b8}
.drhf-toggle input:checked~.drhf-toggle-label-on{display:inline}
.drhf-toggle input:checked~.drhf-toggle-label-off{display:none}

/* Delete */
.drhf-delete{background:none;border:1px solid transparent;color:#ef4444;cursor:pointer;padding:6px 10px;border-radius:6px;font-size:18px;line-height:1;transition:background .2s,border-color .2s}
.drhf-delete:hover{background:#fef2f2;border-color:#fecaca}

/* ======================================================
 * CODE TEXTAREA — soft gray "code flow" styling
 *
 * The text colour is a calm, muted blue-gray (#8694a7)
 * that gives the "little gray flow" appearance. It
 * deepens on focus so the user can see what they type.
 * ====================================================== */
.drhf-snippet-code{
    width:100%;min-height:120px;padding:12px;
    border:1px solid #e2e8f0;border-radius:6px;
    font-family:"SFMono-Regular",Consolas,"Liberation Mono",Menlo,monospace;
    font-size:12.5px;line-height:1.6;
    resize:vertical;box-sizing:border-box;tab-size:2;
    background:#f1f5f9;
    color:#8694a7;
    letter-spacing:0.02em;
    transition:color .25s,border-color .25s,background .25s,box-shadow .25s
}
.drhf-snippet-code:focus{
    outline:none;
    color:#475569;
    background:#fff;
    border-color:#93c5fd;
    box-shadow:0 0 0 2px rgba(59,130,246,.12)
}
.drhf-snippet-code::placeholder{color:#b8c4d0}

/* Add button */
.drhf-add{display:inline-flex;align-items:center;gap:6px;padding:10px 18px;background:#f0f7ff;color:#1e3a5f;border:1px dashed #93c5fd;border-radius:8px;font-size:14px;font-weight:600;cursor:pointer;transition:background .2s,border-color .2s;margin-top:4px}
.drhf-add:hover{background:#dbeafe;border-color:#3b82f6}

/* Empty state */
.drhf-empty{text-align:center;padding:32px 20px;color:#94a3b8;font-size:14px}

/* Save bar */
.drhf-save-bar{margin-top:24px;display:flex;justify-content:flex-end}
.drhf-save-bar .button-primary{padding:8px 32px!important;font-size:14px!important;border-radius:6px!important;background:#1e3a5f!important;border-color:#1e3a5f!important}
.drhf-save-bar .button-primary:hover{background:#2c4f7c!important;border-color:#2c4f7c!important}

/* Animations */
@keyframes drhfFadeIn{from{opacity:0;transform:translateY(6px)}to{opacity:1;transform:translateY(0)}}
.drhf-removing{opacity:0;transform:scale(.97);transition:opacity .25s,transform .25s;pointer-events:none}
';
}

/* ================================================================
 * ADMIN JS
 * ================================================================ */

/**
 * Return the admin JS string for wp_add_inline_script.
 *
 * @return string
 */
function drhf_get_admin_js() {
    // Translatable strings for JS — escaped for safe embedding.
    $placeholder_name = esc_attr__( 'Snippet name (e.g. Google Analytics)', 'digital-rise-head-footer' );
    $placeholder_code = esc_attr__( 'Paste your code here…', 'digital-rise-head-footer' );
    $label_enabled    = esc_html__( 'Enabled',  'digital-rise-head-footer' );
    $label_disabled   = esc_html__( 'Disabled', 'digital-rise-head-footer' );
    $title_delete     = esc_attr__( 'Delete snippet', 'digital-rise-head-footer' );
    $confirm_delete   = esc_js( __( 'Delete this snippet? (Remember to save after deleting.)', 'digital-rise-head-footer' ) );

    return '
(function(){
    "use strict";

    /* ---- Tabs ---- */
    var tabs=document.querySelectorAll(".drhf-tab");
    var panels=document.querySelectorAll(".drhf-panel");
    tabs.forEach(function(tab){
        tab.addEventListener("click",function(){
            tabs.forEach(function(t){t.classList.remove("active");t.setAttribute("aria-selected","false");});
            panels.forEach(function(p){p.classList.remove("active");});
            tab.classList.add("active");
            tab.setAttribute("aria-selected","true");
            document.getElementById("drhf-panel-"+tab.getAttribute("data-tab")).classList.add("active");
        });
    });

    /* ---- Counter for unique indices ---- */
    var counters={header:5000,body:5000,footer:5000};

    /* ---- Build card HTML ---- */
    function buildCard(prefix,idx,id){
        return \'<input type="hidden" name="\'+prefix+\'[\'+idx+\'][id]" value="\'+id+\'">\'+
            \'<div class="drhf-snippet-header">\'+
                \'<input type="text" class="drhf-snippet-name" name="\'+prefix+\'[\'+idx+\'][name]" placeholder="' . $placeholder_name . '" value="">\'+
                \'<label class="drhf-toggle">\'+
                    \'<input type="checkbox" name="\'+prefix+\'[\'+idx+\'][enabled]" value="1" checked>\'+
                    \'<span class="drhf-toggle-track"></span>\'+
                    \'<span class="drhf-toggle-label-on">' . $label_enabled . '</span>\'+
                    \'<span class="drhf-toggle-label-off">' . $label_disabled . '</span>\'+
                \'</label>\'+
                \'<button type="button" class="drhf-delete" title="' . $title_delete . '" aria-label="' . $title_delete . '">&times;</button>\'+
            \'</div>\'+
            \'<textarea class="drhf-snippet-code" name="\'+prefix+\'[\'+idx+\'][code]" placeholder="' . $placeholder_code . '" spellcheck="false"></textarea>\';
    }

    /* ---- Add snippet ---- */
    document.querySelectorAll(".drhf-add").forEach(function(btn){
        btn.addEventListener("click",function(){
            var section=btn.getAttribute("data-section");
            var list=btn.previousElementSibling;
            var prefix="drhf_"+section;
            var idx=counters[section]++;
            var id="drhf_"+Math.random().toString(36).substr(2,12);
            var empty=list.querySelector(".drhf-empty");
            if(empty)empty.remove();
            var card=document.createElement("div");
            card.className="drhf-snippet";
            card.style.animation="drhfFadeIn .3s ease";
            card.innerHTML=buildCard(prefix,idx,id);
            list.appendChild(card);
            card.querySelector(".drhf-snippet-name").focus();
            card.querySelector(".drhf-delete").addEventListener("click",function(){removeCard(card);});
        });
    });

    /* ---- Delete snippet ---- */
    function removeCard(card){
        if(!confirm("' . $confirm_delete . '"))return;
        card.classList.add("drhf-removing");
        setTimeout(function(){card.remove();},260);
    }
    document.querySelectorAll(".drhf-delete").forEach(function(btn){
        btn.addEventListener("click",function(){removeCard(btn.closest(".drhf-snippet"));});
    });

    /* ---- Auto-dismiss success notice ---- */
    var notice=document.querySelector(".drhf-notice");
    if(notice){
        setTimeout(function(){
            notice.style.transition="opacity .4s";
            notice.style.opacity="0";
            setTimeout(function(){notice.remove();},400);
        },4000);
    }
})();
';
}
