<?php
/**
 * Watermark System with Size Control
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Watermark {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Apply watermark to uploaded image
     */
    public function apply_watermark( $metadata, $attachment_id ) {
        if ( ! $this->plugin->get_option( 'watermark_enabled', false ) ) {
            return $metadata;
        }

        if ( ! isset( $metadata['file'] ) ) {
            return $metadata;
        }

        $upload_dir = wp_upload_dir();
        $file_path = $upload_dir['basedir'] . '/' . $metadata['file'];

        $this->add_watermark_to_file( $file_path );

        // Apply to large and medium sizes
        if ( isset( $metadata['sizes'] ) && is_array( $metadata['sizes'] ) ) {
            $file_dir = dirname( $file_path );
            $sizes_to_watermark = array( 'large', 'medium_large', 'medium' );

            foreach ( $sizes_to_watermark as $size ) {
                if ( isset( $metadata['sizes'][ $size ] ) ) {
                    $size_path = $file_dir . '/' . $metadata['sizes'][ $size ]['file'];
                    if ( file_exists( $size_path ) ) {
                        $this->add_watermark_to_file( $size_path );
                    }
                }
            }
        }

        update_post_meta( $attachment_id, '_driop_watermarked', true );

        return $metadata;
    }

    /**
     * Add watermark to specific file
     */
    public function add_watermark_to_file( $file_path ) {
        if ( ! file_exists( $file_path ) ) {
            return false;
        }

        $image_info = getimagesize( $file_path );
        if ( ! $image_info ) {
            return false;
        }

        $width = $image_info[0];
        $height = $image_info[1];
        $mime = $image_info['mime'];

        // Create image resource
        $image = $this->create_image_resource( $file_path, $mime );
        if ( ! $image ) {
            return false;
        }

        // Get watermark text
        $watermark_type = $this->plugin->get_option( 'watermark_type', 'site_name' );
        
        if ( $watermark_type === 'site_name' || ! $this->plugin->has_feature( 'custom_watermark' ) ) {
            $text = get_bloginfo( 'name' );
        } else {
            $text = $this->plugin->get_option( 'watermark_text', get_bloginfo( 'name' ) );
        }

        if ( empty( $text ) ) {
            $text = get_bloginfo( 'name' );
        }

        // Calculate font size
        $size_type = $this->plugin->get_option( 'watermark_size_type', 'percentage' );
        
        if ( $size_type === 'percentage' ) {
            $percentage = $this->plugin->get_option( 'watermark_size_percentage', 15 );
            $font_size = max( 10, ( min( $width, $height ) * $percentage ) / 100 );
        } else {
            $font_size = $this->plugin->get_option( 'watermark_font_size', 24 );
        }

        // Get opacity (0-100 to 0-127)
        $opacity = $this->plugin->get_option( 'watermark_opacity', 50 );
        $alpha = (int) ( 127 - ( $opacity * 1.27 ) );

        // Colors
        $white = imagecolorallocatealpha( $image, 255, 255, 255, $alpha );
        $black = imagecolorallocatealpha( $image, 0, 0, 0, min( 127, $alpha + 30 ) );

        // Use built-in font (scaled by repeating for larger sizes)
        $builtin_font = 5; // Largest built-in font
        $char_width = imagefontwidth( $builtin_font );
        $char_height = imagefontheight( $builtin_font );
        
        // Calculate scale factor for larger text
        $scale = max( 1, $font_size / $char_height );
        
        // For larger watermarks, we'll use imagestring with scaling
        $text_width = strlen( $text ) * $char_width * $scale;
        $text_height = $char_height * $scale;

        // Position
        $position = $this->plugin->get_option( 'watermark_position', 'bottom-right' );
        $padding = 20;

        switch ( $position ) {
            case 'top-left':
                $x = $padding;
                $y = $padding;
                break;
            case 'top-right':
                $x = $width - $text_width - $padding;
                $y = $padding;
                break;
            case 'bottom-left':
                $x = $padding;
                $y = $height - $text_height - $padding;
                break;
            case 'bottom-right':
            default:
                $x = $width - $text_width - $padding;
                $y = $height - $text_height - $padding;
                break;
            case 'center':
                $x = ( $width - $text_width ) / 2;
                $y = ( $height - $text_height ) / 2;
                break;
        }

        $x = max( $padding, $x );
        $y = max( $padding, $y );

        // If scale > 1, create a larger text image and merge
        if ( $scale > 1.5 ) {
            $this->add_scaled_text( $image, $text, $x, $y, $scale, $white, $black, $builtin_font );
        } else {
            // Shadow
            imagestring( $image, $builtin_font, $x + 2, $y + 2, $text, $black );
            // Main text
            imagestring( $image, $builtin_font, $x, $y, $text, $white );
        }

        // Save
        $quality = $this->plugin->get_option( 'compression_quality', 82 );
        $this->save_image( $image, $file_path, $mime, $quality );

        imagedestroy( $image );

        return true;
    }

    /**
     * Add scaled text for larger watermarks
     */
    private function add_scaled_text( $image, $text, $x, $y, $scale, $white, $black, $font ) {
        $char_width = imagefontwidth( $font );
        $char_height = imagefontheight( $font );
        
        // Create temporary image for text
        $text_width = strlen( $text ) * $char_width;
        $text_height = $char_height;
        
        $text_img = imagecreatetruecolor( $text_width, $text_height );
        $transparent = imagecolorallocatealpha( $text_img, 0, 0, 0, 127 );
        imagefill( $text_img, 0, 0, $transparent );
        imagesavealpha( $text_img, true );
        imagealphablending( $text_img, false );
        
        // Draw text on temp image
        $temp_white = imagecolorallocate( $text_img, 255, 255, 255 );
        imagestring( $text_img, $font, 0, 0, $text, $temp_white );
        
        // Scale and merge
        $new_width = (int) ( $text_width * $scale );
        $new_height = (int) ( $text_height * $scale );
        
        $scaled_img = imagecreatetruecolor( $new_width, $new_height );
        $transparent2 = imagecolorallocatealpha( $scaled_img, 0, 0, 0, 127 );
        imagefill( $scaled_img, 0, 0, $transparent2 );
        imagesavealpha( $scaled_img, true );
        imagealphablending( $scaled_img, false );
        
        imagecopyresampled( $scaled_img, $text_img, 0, 0, 0, 0, $new_width, $new_height, $text_width, $text_height );
        
        // Merge onto main image with opacity
        $opacity = $this->plugin->get_option( 'watermark_opacity', 50 );
        imagecopymerge( $image, $scaled_img, $x, $y, 0, 0, $new_width, $new_height, $opacity );
        
        imagedestroy( $text_img );
        imagedestroy( $scaled_img );
    }

    /**
     * Create image resource
     */
    private function create_image_resource( $file_path, $mime ) {
        switch ( $mime ) {
            case 'image/jpeg':
                return imagecreatefromjpeg( $file_path );
            case 'image/png':
                $img = imagecreatefrompng( $file_path );
                imagealphablending( $img, true );
                return $img;
            case 'image/gif':
                return imagecreatefromgif( $file_path );
            case 'image/webp':
                if ( function_exists( 'imagecreatefromwebp' ) ) {
                    return imagecreatefromwebp( $file_path );
                }
                return false;
            default:
                return false;
        }
    }

    /**
     * Save image
     */
    private function save_image( $image, $file_path, $mime, $quality ) {
        switch ( $mime ) {
            case 'image/jpeg':
                imagejpeg( $image, $file_path, $quality );
                break;
            case 'image/png':
                imagesavealpha( $image, true );
                $png_quality = (int) ( ( 100 - $quality ) / 11.1 );
                imagepng( $image, $file_path, $png_quality );
                break;
            case 'image/gif':
                imagegif( $image, $file_path );
                break;
            case 'image/webp':
                if ( function_exists( 'imagewebp' ) ) {
                    imagewebp( $image, $file_path, $quality );
                }
                break;
        }
    }

    /**
     * Generate watermark preview (base64)
     */
    public function generate_preview() {
        // Check if GD is available
        if ( ! function_exists( 'imagecreatetruecolor' ) ) {
            return $this->get_placeholder_image();
        }
        
        $width = 400;
        $height = 300;
        
        // Create base image
        $image = imagecreatetruecolor( $width, $height );
        if ( ! $image ) {
            return $this->get_placeholder_image();
        }
        
        // Fill with gradient background
        for ( $i = 0; $i < $height; $i++ ) {
            $r = min( 255, 70 + intval( $i * 0.3 ) );
            $g = min( 255, 120 + intval( $i * 0.3 ) );
            $b = min( 255, 170 + intval( $i * 0.2 ) );
            $color = imagecolorallocate( $image, $r, $g, $b );
            imageline( $image, 0, $i, $width, $i, $color );
        }
        
        // Add decorative circles
        $light = imagecolorallocate( $image, 255, 255, 255 );
        imagefilledellipse( $image, 100, 100, 80, 80, imagecolorallocatealpha( $image, 255, 255, 255, 110 ) );
        imagefilledellipse( $image, 320, 220, 60, 60, imagecolorallocatealpha( $image, 255, 255, 255, 110 ) );

        // Get watermark text
        $watermark_type = $this->plugin->get_option( 'watermark_type', 'site_name' );
        if ( $watermark_type === 'site_name' ) {
            $text = get_bloginfo( 'name' );
        } else {
            $text = $this->plugin->get_option( 'watermark_text', '' );
        }
        
        if ( empty( $text ) ) {
            $text = get_bloginfo( 'name' );
        }
        if ( empty( $text ) ) {
            $text = 'Your Site Name';
        }

        // Get opacity
        $opacity = intval( $this->plugin->get_option( 'watermark_opacity', 50 ) );
        $alpha = max( 0, min( 127, intval( 127 - ( $opacity * 1.27 ) ) ) );
        
        // Colors
        $white = imagecolorallocatealpha( $image, 255, 255, 255, $alpha );
        $shadow = imagecolorallocatealpha( $image, 0, 0, 0, min( 127, $alpha + 50 ) );
        
        // Font settings
        $font = 5; // Largest built-in font
        $char_width = imagefontwidth( $font );
        $char_height = imagefontheight( $font );
        $text_width = strlen( $text ) * $char_width;
        $text_height = $char_height;
        
        // Get position
        $position = $this->plugin->get_option( 'watermark_position', 'bottom-right' );
        $padding = 20;

        switch ( $position ) {
            case 'top-left':
                $x = $padding;
                $y = $padding;
                break;
            case 'top-right':
                $x = $width - $text_width - $padding;
                $y = $padding;
                break;
            case 'bottom-left':
                $x = $padding;
                $y = $height - $text_height - $padding;
                break;
            case 'center':
                $x = ( $width - $text_width ) / 2;
                $y = ( $height - $text_height ) / 2;
                break;
            case 'bottom-right':
            default:
                $x = $width - $text_width - $padding;
                $y = $height - $text_height - $padding;
                break;
        }

        $x = max( $padding, intval( $x ) );
        $y = max( $padding, intval( $y ) );

        // Draw shadow
        imagestring( $image, $font, $x + 2, $y + 2, $text, $shadow );
        // Draw text
        imagestring( $image, $font, $x, $y, $text, $white );

        // Output as PNG base64
        ob_start();
        imagepng( $image );
        $png_data = ob_get_clean();
        imagedestroy( $image );

        if ( empty( $png_data ) ) {
            return $this->get_placeholder_image();
        }

        return 'data:image/png;base64,' . base64_encode( $png_data );
    }
    
    /**
     * Get placeholder image when GD fails
     */
    private function get_placeholder_image() {
        $svg = '<svg xmlns="http://www.w3.org/2000/svg" width="400" height="300" viewBox="0 0 400 300">
            <defs>
                <linearGradient id="bg" x1="0%" y1="0%" x2="0%" y2="100%">
                    <stop offset="0%" style="stop-color:#4a90a4"/>
                    <stop offset="100%" style="stop-color:#2d5f6e"/>
                </linearGradient>
            </defs>
            <rect fill="url(#bg)" width="400" height="300"/>
            <text x="200" y="140" font-family="Arial,sans-serif" font-size="18" fill="white" text-anchor="middle">Watermark Preview</text>
            <text x="200" y="170" font-family="Arial,sans-serif" font-size="12" fill="rgba(255,255,255,0.7)" text-anchor="middle">GD Library required for live preview</text>
        </svg>';
        return 'data:image/svg+xml;base64,' . base64_encode( $svg );
    }
}
