<?php
/**
 * Plugin Auto-Updater
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Updater {

    private $plugin;
    private $plugin_slug;
    private $plugin_basename;
    private $version;
    private $cache_key;
    private $cache_expiration;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        $this->plugin_slug = 'digital-rise-image-optimizer-pro';
        $this->plugin_basename = DRIOP_PLUGIN_BASENAME;
        $this->version = DRIOP_VERSION;
        $this->cache_key = 'driop_update_check';
        $this->cache_expiration = 12 * HOUR_IN_SECONDS;
    }

    /**
     * Check for plugin updates
     */
    public function check_for_update( $transient ) {
        if ( empty( $transient->checked ) ) {
            return $transient;
        }

        $remote_info = $this->get_remote_info();

        if ( $remote_info && isset( $remote_info['version'] ) ) {
            if ( version_compare( $this->version, $remote_info['version'], '<' ) ) {
                $plugin_data = array(
                    'slug' => $this->plugin_slug,
                    'plugin' => $this->plugin_basename,
                    'new_version' => $remote_info['version'],
                    'url' => isset( $remote_info['homepage'] ) ? $remote_info['homepage'] : DRIOP_OWNER_URL,
                    'package' => isset( $remote_info['download_url'] ) ? $remote_info['download_url'] : '',
                    'icons' => isset( $remote_info['icons'] ) ? $remote_info['icons'] : array(),
                    'banners' => isset( $remote_info['banners'] ) ? $remote_info['banners'] : array(),
                    'tested' => isset( $remote_info['tested'] ) ? $remote_info['tested'] : '',
                    'requires_php' => isset( $remote_info['requires_php'] ) ? $remote_info['requires_php'] : '7.4',
                    'requires' => isset( $remote_info['requires'] ) ? $remote_info['requires'] : '5.8',
                );

                $transient->response[ $this->plugin_basename ] = (object) $plugin_data;
            } else {
                // No update available
                $transient->no_update[ $this->plugin_basename ] = (object) array(
                    'slug' => $this->plugin_slug,
                    'plugin' => $this->plugin_basename,
                    'new_version' => $this->version,
                );
            }
        }

        return $transient;
    }

    /**
     * Plugin info for WordPress updates screen
     */
    public function plugin_info( $result, $action, $args ) {
        if ( $action !== 'plugin_information' ) {
            return $result;
        }

        if ( ! isset( $args->slug ) || $args->slug !== $this->plugin_slug ) {
            return $result;
        }

        $remote_info = $this->get_remote_info();

        if ( ! $remote_info ) {
            return $result;
        }

        $plugin_info = new stdClass();
        $plugin_info->name = isset( $remote_info['name'] ) ? $remote_info['name'] : 'Image Optimizer for SEO';
        $plugin_info->slug = $this->plugin_slug;
        $plugin_info->version = isset( $remote_info['version'] ) ? $remote_info['version'] : $this->version;
        $plugin_info->author = isset( $remote_info['author'] ) ? $remote_info['author'] : '<a href="https://digital-rise-solutions.com/">Digital Rise Solutions</a>';
        $plugin_info->author_profile = isset( $remote_info['author_profile'] ) ? $remote_info['author_profile'] : 'https://digital-rise-solutions.com/';
        $plugin_info->homepage = isset( $remote_info['homepage'] ) ? $remote_info['homepage'] : DRIOP_OWNER_URL;
        $plugin_info->requires = isset( $remote_info['requires'] ) ? $remote_info['requires'] : '5.8';
        $plugin_info->tested = isset( $remote_info['tested'] ) ? $remote_info['tested'] : '6.4';
        $plugin_info->requires_php = isset( $remote_info['requires_php'] ) ? $remote_info['requires_php'] : '7.4';
        $plugin_info->last_updated = isset( $remote_info['last_updated'] ) ? $remote_info['last_updated'] : '';
        $plugin_info->sections = isset( $remote_info['sections'] ) ? $remote_info['sections'] : array(
            'description' => 'Professional image optimization, SEO automation, and watermarking for WordPress.',
            'changelog' => isset( $remote_info['changelog'] ) ? $remote_info['changelog'] : '',
        );
        $plugin_info->download_link = isset( $remote_info['download_url'] ) ? $remote_info['download_url'] : '';
        $plugin_info->banners = isset( $remote_info['banners'] ) ? $remote_info['banners'] : array();
        $plugin_info->icons = isset( $remote_info['icons'] ) ? $remote_info['icons'] : array();

        return $plugin_info;
    }

    /**
     * Get remote update info
     */
    private function get_remote_info() {
        $cached = get_transient( $this->cache_key );
        
        if ( $cached !== false ) {
            return $cached;
        }

        $response = wp_remote_get( DRIOP_UPDATE_API . 'info', array(
            'timeout' => 10,
            'headers' => array(
                'Accept' => 'application/json',
                'X-DRIOP-Site' => home_url(),
                'X-DRIOP-Version' => $this->version,
                'X-DRIOP-License' => $this->plugin->get_option( 'license_key', '' ),
            ),
        ));

        if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
            return false;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( empty( $data ) ) {
            return false;
        }

        set_transient( $this->cache_key, $data, $this->cache_expiration );

        return $data;
    }

    /**
     * After update hook
     */
    public function after_update( $upgrader_object, $options ) {
        if ( $options['action'] !== 'update' || $options['type'] !== 'plugin' ) {
            return;
        }

        if ( isset( $options['plugins'] ) && in_array( $this->plugin_basename, $options['plugins'] ) ) {
            delete_transient( $this->cache_key );
            
            // Run any necessary migrations
            $this->run_migrations();
        }
    }

    /**
     * Run migrations after update
     */
    private function run_migrations() {
        $db_version = get_option( 'driop_db_version', '0.0.0' );
        
        // Add migration logic here for future versions
        // if ( version_compare( $db_version, '1.1.0', '<' ) ) {
        //     // Migration code
        // }
        
        update_option( 'driop_db_version', DRIOP_VERSION );
    }

    /**
     * Clear update cache (can be called manually)
     */
    public function clear_cache() {
        delete_transient( $this->cache_key );
    }
}