<?php
/**
 * Image SEO Automation
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_SEO {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Auto-set image meta - FIXED to work properly
     */
    public function auto_set_image_meta( $metadata, $attachment_id ) {
        // Get attachment
        $attachment = get_post( $attachment_id );
        
        if ( ! $attachment || $attachment->post_type !== 'attachment' ) {
            return $metadata;
        }
        
        // Check if it's an image
        if ( strpos( get_post_mime_type( $attachment_id ), 'image/' ) !== 0 ) {
            return $metadata;
        }

        // Get filename for fallback
        $file_path = get_attached_file( $attachment_id );
        $filename = $file_path ? pathinfo( $file_path, PATHINFO_FILENAME ) : '';
        $filename = str_replace( array( '-', '_' ), ' ', $filename );
        $filename = ucwords( trim( $filename ) );

        // Generate SEO text
        $seo_text = $this->generate_seo_text( $attachment_id, $filename );
        
        // Make sure we have something to use
        if ( empty( $seo_text ) ) {
            $seo_text = $filename ? $filename : get_bloginfo( 'name' ) . ' Image';
        }

        // Update title (post_title)
        if ( $this->plugin->get_option( 'auto_rename', true ) ) {
            wp_update_post( array(
                'ID' => $attachment_id,
                'post_title' => $seo_text,
            ) );
        }

        // Update ALT text - THIS IS THE KEY FIX
        if ( $this->plugin->get_option( 'auto_alt', true ) ) {
            delete_post_meta( $attachment_id, '_wp_attachment_image_alt' );
            add_post_meta( $attachment_id, '_wp_attachment_image_alt', $seo_text, true );
        }

        // Update caption and description
        if ( $this->plugin->get_option( 'auto_description', true ) ) {
            $description = $this->generate_description( $attachment_id, $filename );
            wp_update_post( array(
                'ID' => $attachment_id,
                'post_excerpt' => $description, // Caption
                'post_content' => $description, // Description
            ) );
        }

        // Mark as processed
        update_post_meta( $attachment_id, '_driop_seo_processed', '1' );
        update_post_meta( $attachment_id, '_driop_seo_text', $seo_text );

        return $metadata;
    }

    /**
     * Generate SEO text from pattern
     */
    public function generate_seo_text( $attachment_id, $filename ) {
        $pattern = $this->plugin->get_option( 'seo_pattern', '{site_name} {content_title} {keyword}' );
        
        // Get keyword
        $keywords = new DRIOP_Keywords( $this->plugin );
        $keyword = $keywords->get_random_keyword();
        
        // Get parent post if exists
        $parent_id = wp_get_post_parent_id( $attachment_id );
        $parent = $parent_id ? get_post( $parent_id ) : null;
        $content_title = '';
        $category = '';

        if ( $parent ) {
            $content_title = $parent->post_title;
            
            // Get category
            if ( $parent->post_type === 'product' && taxonomy_exists( 'product_cat' ) ) {
                $categories = get_the_terms( $parent->ID, 'product_cat' );
                if ( $categories && ! is_wp_error( $categories ) ) {
                    $category = $categories[0]->name;
                }
            } elseif ( $parent->post_type === 'post' ) {
                $categories = get_the_category( $parent->ID );
                if ( $categories && ! empty( $categories ) ) {
                    $category = $categories[0]->name;
                }
            }
        }

        $replacements = array(
            '{site_name}' => get_bloginfo( 'name' ),
            '{content_title}' => $content_title,
            '{keyword}' => $keyword,
            '{filename}' => $filename,
            '{date}' => date_i18n( get_option( 'date_format' ) ),
            '{category}' => $category,
        );

        $result = str_replace( array_keys( $replacements ), array_values( $replacements ), $pattern );
        
        // Clean up multiple spaces
        $result = preg_replace( '/\s{2,}/', ' ', $result );
        $result = trim( $result );

        // Fallback
        if ( empty( $result ) || $result === $pattern ) {
            $parts = array_filter( array( get_bloginfo( 'name' ), $keyword, $filename ) );
            $result = implode( ' ', $parts );
        }

        return $result;
    }

    /**
     * Generate description
     */
    public function generate_description( $attachment_id, $filename ) {
        $parent_id = wp_get_post_parent_id( $attachment_id );
        $parent = $parent_id ? get_post( $parent_id ) : null;
        $site_name = get_bloginfo( 'name' );
        
        $keywords = new DRIOP_Keywords( $this->plugin );
        $keyword = $keywords->get_random_keyword();
        
        $parts = array_filter( array(
            ucwords( $filename ),
            $parent ? $parent->post_title : '',
            $keyword,
            $site_name,
        ) );
        
        return implode( ' - ', $parts );
    }
    
    /**
     * Apply SEO to existing image
     */
    public function apply_seo_to_image( $attachment_id ) {
        $file_path = get_attached_file( $attachment_id );
        $filename = $file_path ? pathinfo( $file_path, PATHINFO_FILENAME ) : '';
        $filename = str_replace( array( '-', '_' ), ' ', $filename );
        $filename = ucwords( trim( $filename ) );
        
        $seo_text = $this->generate_seo_text( $attachment_id, $filename );
        
        if ( empty( $seo_text ) ) {
            $seo_text = $filename ? $filename : get_bloginfo( 'name' ) . ' Image';
        }
        
        // Update title
        wp_update_post( array(
            'ID' => $attachment_id,
            'post_title' => $seo_text,
        ) );
        
        // Update ALT
        delete_post_meta( $attachment_id, '_wp_attachment_image_alt' );
        add_post_meta( $attachment_id, '_wp_attachment_image_alt', $seo_text, true );
        
        // Update description
        $description = $this->generate_description( $attachment_id, $filename );
        wp_update_post( array(
            'ID' => $attachment_id,
            'post_excerpt' => $description,
            'post_content' => $description,
        ) );
        
        update_post_meta( $attachment_id, '_driop_seo_processed', '1' );
        update_post_meta( $attachment_id, '_driop_seo_text', $seo_text );
        
        return $seo_text;
    }

    /**
     * Get SEO stats
     */
    public function get_seo_stats() {
        global $wpdb;

        $total = $wpdb->get_var(
            "SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = 'attachment' AND post_mime_type LIKE 'image/%'"
        );

        $with_alt = $wpdb->get_var(
            "SELECT COUNT(DISTINCT p.ID) FROM $wpdb->posts p
            INNER JOIN $wpdb->postmeta pm ON p.ID = pm.post_id
            WHERE p.post_type = 'attachment' AND p.post_mime_type LIKE 'image/%'
            AND pm.meta_key = '_wp_attachment_image_alt' AND pm.meta_value != ''"
        );

        $processed = $wpdb->get_var(
            "SELECT COUNT(DISTINCT p.ID) FROM $wpdb->posts p
            INNER JOIN $wpdb->postmeta pm ON p.ID = pm.post_id
            WHERE p.post_type = 'attachment' AND p.post_mime_type LIKE 'image/%'
            AND pm.meta_key = '_driop_seo_processed' AND pm.meta_value = '1'"
        );

        return array(
            'total' => (int) $total,
            'with_alt' => (int) $with_alt,
            'without_alt' => (int) $total - (int) $with_alt,
            'processed' => (int) $processed,
            'alt_percentage' => $total > 0 ? round( ( $with_alt / $total ) * 100, 1 ) : 0,
        );
    }
}
