<?php
/**
 * Page SEO - Title and Meta Description for Pages, Posts, Products, Categories, Tags
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Page_SEO {

    private $plugin;
    
    // Available dynamic variables
    private $variables = array(
        '{title}' => 'Page/Post Title',
        '{site_name}' => 'Site Name',
        '{site_tagline}' => 'Site Tagline',
        '{excerpt}' => 'Excerpt (first 160 chars)',
        '{category}' => 'Primary Category',
        '{tags}' => 'Tags (comma separated)',
        '{author}' => 'Author Name',
        '{date}' => 'Publish Date',
        '{year}' => 'Current Year',
        '{month}' => 'Current Month',
        '{product_price}' => 'Product Price (WooCommerce)',
        '{product_sku}' => 'Product SKU (WooCommerce)',
        '{keyword}' => 'Random Keyword from List',
        '{parent_title}' => 'Parent Page/Category Title',
        '{count}' => 'Number of Items (for archives)',
    );

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Get available variables for display
     */
    public function get_available_variables() {
        return $this->variables;
    }

    /**
     * Add meta boxes to posts, pages, products
     */
    public function add_meta_boxes() {
        $post_types = array( 'post', 'page', 'product' );
        
        foreach ( $post_types as $post_type ) {
            add_meta_box(
                'driop_page_seo',
                __( 'SEO Settings - Image Optimizer', 'digital-rise-solutions' ),
                array( $this, 'render_meta_box' ),
                $post_type,
                'normal',
                'high'
            );
        }
    }

    /**
     * Render meta box
     */
    public function render_meta_box( $post ) {
        wp_nonce_field( 'driop_page_seo_nonce', 'driop_page_seo_nonce' );
        
        $seo_title = get_post_meta( $post->ID, '_driop_seo_title', true );
        $seo_description = get_post_meta( $post->ID, '_driop_seo_description', true );
        
        $default_title = $this->plugin->get_option( 'default_title_pattern', '{title} | {site_name}' );
        $default_desc = $this->plugin->get_option( 'default_meta_pattern', '{excerpt}' );
        ?>
        <div class="driop-seo-metabox">
            <style>
                .driop-seo-metabox { padding: 15px 0; }
                .driop-seo-field { margin-bottom: 20px; }
                .driop-seo-field label { display: block; font-weight: 600; margin-bottom: 5px; }
                .driop-seo-field input[type="text"], .driop-seo-field textarea { width: 100%; }
                .driop-seo-field textarea { height: 80px; }
                .driop-seo-variables { background: #f0f0f1; padding: 10px 15px; border-radius: 4px; margin-top: 15px; }
                .driop-seo-variables code { background: #fff; padding: 2px 6px; margin: 2px; display: inline-block; cursor: pointer; border-radius: 3px; }
                .driop-seo-variables code:hover { background: #2271b1; color: #fff; }
                .driop-seo-preview { background: #fff; border: 1px solid #ddd; padding: 15px; margin-top: 15px; border-radius: 4px; }
                .driop-seo-preview-title { color: #1a0dab; font-size: 18px; margin-bottom: 5px; }
                .driop-seo-preview-url { color: #006621; font-size: 14px; margin-bottom: 5px; }
                .driop-seo-preview-desc { color: #545454; font-size: 14px; line-height: 1.4; }
                .driop-char-count { font-size: 12px; color: #666; margin-top: 5px; }
                .driop-char-count.warning { color: #d63638; }
            </style>
            
            <div class="driop-seo-field">
                <label for="driop_seo_title"><?php esc_html_e( 'SEO Title', 'digital-rise-solutions' ); ?></label>
                <input type="text" id="driop_seo_title" name="driop_seo_title" value="<?php echo esc_attr( $seo_title ); ?>" placeholder="<?php echo esc_attr( $default_title ); ?>">
                <div class="driop-char-count" id="title-char-count">0 / 60 characters</div>
            </div>
            
            <div class="driop-seo-field">
                <label for="driop_seo_description"><?php esc_html_e( 'Meta Description', 'digital-rise-solutions' ); ?></label>
                <textarea id="driop_seo_description" name="driop_seo_description" placeholder="<?php echo esc_attr( $default_desc ); ?>"><?php echo esc_textarea( $seo_description ); ?></textarea>
                <div class="driop-char-count" id="desc-char-count">0 / 160 characters</div>
            </div>
            
            <div class="driop-seo-variables">
                <strong><?php esc_html_e( 'Available Variables (click to insert):', 'digital-rise-solutions' ); ?></strong><br>
                <?php foreach ( $this->variables as $var => $label ) : ?>
                    <code class="driop-insert-var" data-var="<?php echo esc_attr( $var ); ?>" title="<?php echo esc_attr( $label ); ?>"><?php echo esc_html( $var ); ?></code>
                <?php endforeach; ?>
            </div>
            
            <div class="driop-seo-preview">
                <strong><?php esc_html_e( 'Search Preview:', 'digital-rise-solutions' ); ?></strong>
                <div class="driop-seo-preview-title" id="seo-preview-title"><?php echo esc_html( $post->post_title ); ?></div>
                <div class="driop-seo-preview-url"><?php echo esc_url( get_permalink( $post->ID ) ); ?></div>
                <div class="driop-seo-preview-desc" id="seo-preview-desc"><?php echo esc_html( wp_trim_words( $post->post_excerpt ?: $post->post_content, 25 ) ); ?></div>
            </div>
            
            <script>
            jQuery(document).ready(function($) {
                function updateCharCount() {
                    var titleLen = $('#driop_seo_title').val().length;
                    var descLen = $('#driop_seo_description').val().length;
                    
                    $('#title-char-count').text(titleLen + ' / 60 characters').toggleClass('warning', titleLen > 60);
                    $('#desc-char-count').text(descLen + ' / 160 characters').toggleClass('warning', descLen > 160);
                    
                    // Update preview
                    var title = $('#driop_seo_title').val() || '<?php echo esc_js( $post->post_title ); ?>';
                    var desc = $('#driop_seo_description').val() || '<?php echo esc_js( wp_trim_words( $post->post_excerpt ?: strip_tags( $post->post_content ), 25 ) ); ?>';
                    
                    $('#seo-preview-title').text(title.replace(/\{[^}]+\}/g, '...'));
                    $('#seo-preview-desc').text(desc.replace(/\{[^}]+\}/g, '...'));
                }
                
                $('#driop_seo_title, #driop_seo_description').on('input', updateCharCount);
                updateCharCount();
                
                // Insert variable on click
                var lastFocused = null;
                $('#driop_seo_title, #driop_seo_description').on('focus', function() {
                    lastFocused = this;
                });
                
                $('.driop-insert-var').on('click', function() {
                    var varText = $(this).data('var');
                    if (lastFocused) {
                        var pos = lastFocused.selectionStart;
                        var val = $(lastFocused).val();
                        $(lastFocused).val(val.substring(0, pos) + varText + val.substring(pos));
                        lastFocused.focus();
                        lastFocused.setSelectionRange(pos + varText.length, pos + varText.length);
                        updateCharCount();
                    }
                });
            });
            </script>
        </div>
        <?php
    }

    /**
     * Save meta box data
     */
    public function save_meta_box( $post_id ) {
        if ( ! isset( $_POST['driop_page_seo_nonce'] ) || ! wp_verify_nonce( $_POST['driop_page_seo_nonce'], 'driop_page_seo_nonce' ) ) {
            return;
        }
        
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }
        
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }
        
        if ( isset( $_POST['driop_seo_title'] ) ) {
            update_post_meta( $post_id, '_driop_seo_title', sanitize_text_field( $_POST['driop_seo_title'] ) );
        }
        
        if ( isset( $_POST['driop_seo_description'] ) ) {
            update_post_meta( $post_id, '_driop_seo_description', sanitize_textarea_field( $_POST['driop_seo_description'] ) );
        }
    }

    /**
     * Add fields to taxonomy add form
     */
    public function add_term_fields( $taxonomy ) {
        ?>
        <div class="form-field">
            <label for="driop_term_seo_title"><?php esc_html_e( 'SEO Title', 'digital-rise-solutions' ); ?></label>
            <input type="text" name="driop_term_seo_title" id="driop_term_seo_title" value="">
            <p class="description"><?php esc_html_e( 'Custom title for search engines. Use variables like {title}, {site_name}, {count}', 'digital-rise-solutions' ); ?></p>
        </div>
        <div class="form-field">
            <label for="driop_term_seo_description"><?php esc_html_e( 'Meta Description', 'digital-rise-solutions' ); ?></label>
            <textarea name="driop_term_seo_description" id="driop_term_seo_description" rows="3"></textarea>
            <p class="description"><?php esc_html_e( 'Custom description for search engines (max 160 characters recommended)', 'digital-rise-solutions' ); ?></p>
        </div>
        <?php
    }

    /**
     * Add fields to taxonomy edit form
     */
    public function edit_term_fields( $term ) {
        $seo_title = get_term_meta( $term->term_id, '_driop_seo_title', true );
        $seo_description = get_term_meta( $term->term_id, '_driop_seo_description', true );
        ?>
        <tr class="form-field">
            <th scope="row"><label for="driop_term_seo_title"><?php esc_html_e( 'SEO Title', 'digital-rise-solutions' ); ?></label></th>
            <td>
                <input type="text" name="driop_term_seo_title" id="driop_term_seo_title" value="<?php echo esc_attr( $seo_title ); ?>">
                <p class="description"><?php esc_html_e( 'Custom title for search engines. Variables: {title}, {site_name}, {count}', 'digital-rise-solutions' ); ?></p>
            </td>
        </tr>
        <tr class="form-field">
            <th scope="row"><label for="driop_term_seo_description"><?php esc_html_e( 'Meta Description', 'digital-rise-solutions' ); ?></label></th>
            <td>
                <textarea name="driop_term_seo_description" id="driop_term_seo_description" rows="3"><?php echo esc_textarea( $seo_description ); ?></textarea>
                <p class="description"><?php esc_html_e( 'Custom description for search engines (max 160 characters)', 'digital-rise-solutions' ); ?></p>
            </td>
        </tr>
        <?php
    }

    /**
     * Save term meta
     */
    public function save_term_meta( $term_id, $tt_id, $taxonomy ) {
        if ( isset( $_POST['driop_term_seo_title'] ) ) {
            update_term_meta( $term_id, '_driop_seo_title', sanitize_text_field( $_POST['driop_term_seo_title'] ) );
        }
        if ( isset( $_POST['driop_term_seo_description'] ) ) {
            update_term_meta( $term_id, '_driop_seo_description', sanitize_textarea_field( $_POST['driop_term_seo_description'] ) );
        }
    }

    /**
     * Output meta tags in head
     */
    public function output_meta_tags() {
        $description = $this->get_meta_description();
        
        if ( ! empty( $description ) ) {
            echo '<meta name="description" content="' . esc_attr( $description ) . '">' . "\n";
        }
        
        // Open Graph
        $title = $this->get_seo_title();
        if ( ! empty( $title ) ) {
            echo '<meta property="og:title" content="' . esc_attr( $title ) . '">' . "\n";
        }
        if ( ! empty( $description ) ) {
            echo '<meta property="og:description" content="' . esc_attr( $description ) . '">' . "\n";
        }
        
        // Twitter
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        if ( ! empty( $title ) ) {
            echo '<meta name="twitter:title" content="' . esc_attr( $title ) . '">' . "\n";
        }
        if ( ! empty( $description ) ) {
            echo '<meta name="twitter:description" content="' . esc_attr( $description ) . '">' . "\n";
        }
    }

    /**
     * Filter document title
     */
    public function filter_title( $title ) {
        $seo_title = $this->get_seo_title();
        return ! empty( $seo_title ) ? $seo_title : $title;
    }

    /**
     * Filter title parts
     */
    public function filter_title_parts( $title_parts ) {
        $seo_title = $this->get_seo_title();
        if ( ! empty( $seo_title ) ) {
            return array( 'title' => $seo_title );
        }
        return $title_parts;
    }

    /**
     * Get SEO title for current page
     */
    public function get_seo_title() {
        $pattern = '';
        
        if ( is_singular() ) {
            global $post;
            $pattern = get_post_meta( $post->ID, '_driop_seo_title', true );
            if ( empty( $pattern ) ) {
                $pattern = $this->plugin->get_option( 'default_title_pattern', '{title} | {site_name}' );
            }
        } elseif ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_title', true );
            if ( empty( $pattern ) ) {
                $pattern = '{title} | {site_name}';
            }
        } elseif ( is_home() || is_front_page() ) {
            $pattern = get_bloginfo( 'name' ) . ' | ' . get_bloginfo( 'description' );
            return $pattern;
        }
        
        return $this->parse_pattern( $pattern );
    }

    /**
     * Get meta description for current page
     */
    public function get_meta_description() {
        $pattern = '';
        
        if ( is_singular() ) {
            global $post;
            $pattern = get_post_meta( $post->ID, '_driop_seo_description', true );
            if ( empty( $pattern ) ) {
                $pattern = $this->plugin->get_option( 'default_meta_pattern', '{excerpt}' );
            }
        } elseif ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_description', true );
            if ( empty( $pattern ) ) {
                $pattern = term_description( $term->term_id );
            }
        } elseif ( is_home() || is_front_page() ) {
            $pattern = get_bloginfo( 'description' );
            return wp_trim_words( $pattern, 25 );
        }
        
        $description = $this->parse_pattern( $pattern );
        return wp_trim_words( strip_tags( $description ), 30 );
    }

    /**
     * Parse pattern and replace variables
     */
    public function parse_pattern( $pattern ) {
        if ( empty( $pattern ) ) {
            return '';
        }
        
        $replacements = array(
            '{site_name}' => get_bloginfo( 'name' ),
            '{site_tagline}' => get_bloginfo( 'description' ),
            '{year}' => date( 'Y' ),
            '{month}' => date( 'F' ),
        );
        
        // Get keyword
        $keywords = new DRIOP_Keywords( $this->plugin );
        $replacements['{keyword}'] = $keywords->get_random_keyword();
        
        if ( is_singular() ) {
            global $post;
            $replacements['{title}'] = get_the_title( $post->ID );
            $replacements['{excerpt}'] = $this->get_excerpt( $post );
            $replacements['{author}'] = get_the_author_meta( 'display_name', $post->post_author );
            $replacements['{date}'] = get_the_date( '', $post );
            
            // Categories
            $categories = get_the_category( $post->ID );
            $replacements['{category}'] = ! empty( $categories ) ? $categories[0]->name : '';
            
            // Tags
            $tags = get_the_tags( $post->ID );
            $replacements['{tags}'] = ! empty( $tags ) ? implode( ', ', wp_list_pluck( $tags, 'name' ) ) : '';
            
            // Parent
            $parent_id = $post->post_parent;
            $replacements['{parent_title}'] = $parent_id ? get_the_title( $parent_id ) : '';
            
            // WooCommerce
            if ( $post->post_type === 'product' && function_exists( 'wc_get_product' ) ) {
                $product = wc_get_product( $post->ID );
                if ( $product ) {
                    $replacements['{product_price}'] = $product->get_price_html();
                    $replacements['{product_sku}'] = $product->get_sku();
                    
                    // Product categories
                    $product_cats = get_the_terms( $post->ID, 'product_cat' );
                    if ( $product_cats && ! is_wp_error( $product_cats ) ) {
                        $replacements['{category}'] = $product_cats[0]->name;
                    }
                }
            }
        } elseif ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            $replacements['{title}'] = $term->name;
            $replacements['{excerpt}'] = $term->description;
            $replacements['{count}'] = $term->count;
            
            // Parent term
            if ( $term->parent ) {
                $parent = get_term( $term->parent );
                $replacements['{parent_title}'] = $parent ? $parent->name : '';
            } else {
                $replacements['{parent_title}'] = '';
            }
        }
        
        $result = str_replace( array_keys( $replacements ), array_values( $replacements ), $pattern );
        
        // Clean up empty variables
        $result = preg_replace( '/\{[^}]+\}/', '', $result );
        $result = preg_replace( '/\s{2,}/', ' ', $result );
        $result = trim( $result, ' |,-' );
        
        return $result;
    }

    /**
     * Get excerpt for post
     */
    private function get_excerpt( $post ) {
        if ( ! empty( $post->post_excerpt ) ) {
            return $post->post_excerpt;
        }
        
        $content = strip_shortcodes( $post->post_content );
        $content = strip_tags( $content );
        $content = str_replace( array( "\r", "\n", "\t" ), ' ', $content );
        
        return wp_trim_words( $content, 25 );
    }
}