<?php
/**
 * Image Optimizer - Compresses images and converts to WebP
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Image_Optimizer {

    private $plugin;
    private $target_size = 102400; // 100KB in bytes

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        $this->target_size = $this->plugin->get_option( 'target_file_size', 50 ) * 1024;
    }

    /**
     * Handle upload - Convert to WebP, compress, and rename
     */
    public function handle_upload( $upload, $context = 'upload' ) {
        if ( ! isset( $upload['type'] ) || strpos( $upload['type'], 'image/' ) !== 0 ) {
            return $upload;
        }

        // Skip if already WebP and conversion is enabled
        $convert_to_webp = $this->plugin->get_option( 'convert_to_webp', true );
        
        // Check if GD supports WebP
        if ( $convert_to_webp && ! $this->supports_webp() ) {
            $convert_to_webp = false;
        }

        // Always resize if needed
        $max_width = $this->plugin->get_option( 'max_width', 1920 );
        $max_height = $this->plugin->get_option( 'max_height', 1080 );
        $this->resize_image( $upload['file'], $max_width, $max_height );

        // Convert to WebP if enabled and not already WebP
        if ( $convert_to_webp && $upload['type'] !== 'image/webp' ) {
            $upload = $this->convert_to_webp( $upload );
        }

        // Always compress to target size (50KB)
        $this->compress_to_target( $upload['file'] );

        // Rename if enabled
        if ( $this->plugin->get_option( 'auto_rename', true ) ) {
            $upload = $this->rename_image( $upload );
        }

        return $upload;
    }

    /**
     * Check if server supports WebP
     */
    public function supports_webp() {
        if ( ! function_exists( 'imagecreatefromwebp' ) || ! function_exists( 'imagewebp' ) ) {
            return false;
        }
        
        // Check if GD was compiled with WebP support
        if ( function_exists( 'gd_info' ) ) {
            $gd_info = gd_info();
            if ( isset( $gd_info['WebP Support'] ) && $gd_info['WebP Support'] ) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Convert image to WebP format
     */
    public function convert_to_webp( $upload ) {
        $file_path = $upload['file'];
        
        if ( ! file_exists( $file_path ) ) {
            return $upload;
        }

        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return $upload;
        }

        $mime = $image_info['mime'];
        
        // Skip if already WebP or GIF (preserve animations)
        if ( $mime === 'image/webp' || $mime === 'image/gif' ) {
            return $upload;
        }

        // Create image resource
        $source = $this->create_image_resource( $file_path, $mime );
        if ( ! $source ) {
            return $upload;
        }

        // Get path info
        $path_info = pathinfo( $file_path );
        $webp_path = $path_info['dirname'] . '/' . $path_info['filename'] . '.webp';

        // Handle transparency for PNG
        if ( $mime === 'image/png' ) {
            imagepalettetotruecolor( $source );
            imagealphablending( $source, true );
            imagesavealpha( $source, true );
        }

        // Get compression quality
        $quality = $this->plugin->get_option( 'compression_quality', 82 );

        // Convert to WebP
        $success = imagewebp( $source, $webp_path, $quality );
        imagedestroy( $source );

        if ( $success && file_exists( $webp_path ) ) {
            // Delete original file
            @unlink( $file_path );
            
            // Update upload info
            $upload['file'] = $webp_path;
            $upload['url'] = preg_replace( '/\.(jpe?g|png|gif)$/i', '.webp', $upload['url'] );
            $upload['type'] = 'image/webp';
            
            return $upload;
        }

        return $upload;
    }

    /**
     * Optimize attachment after upload
     */
    public function optimize_attachment( $metadata, $attachment_id ) {
        if ( ! isset( $metadata['file'] ) ) {
            return $metadata;
        }

        $upload_dir = wp_upload_dir();
        $file_path = $upload_dir['basedir'] . '/' . $metadata['file'];
        
        $original_size = file_exists( $file_path ) ? filesize( $file_path ) : 0;
        
        // Check if we should convert to WebP
        $convert_to_webp = $this->plugin->get_option( 'convert_to_webp', true ) && $this->supports_webp();
        
        // Get mime type
        $image_info = @getimagesize( $file_path );
        $mime = $image_info ? $image_info['mime'] : '';
        
        // Convert main image to WebP if not already
        if ( $convert_to_webp && $mime !== 'image/webp' && $mime !== 'image/gif' ) {
            $new_path = $this->convert_file_to_webp( $file_path );
            if ( $new_path && $new_path !== $file_path ) {
                // Update metadata
                $metadata['file'] = str_replace( $upload_dir['basedir'] . '/', '', $new_path );
                $file_path = $new_path;
                
                // Update attachment
                $this->update_attachment_webp( $attachment_id, $metadata );
            }
        }
        
        // Compress main image to target
        $this->compress_to_target( $file_path );
        
        $optimized_size = file_exists( $file_path ) ? filesize( $file_path ) : 0;
        
        // Optimize and convert thumbnails
        if ( isset( $metadata['sizes'] ) && is_array( $metadata['sizes'] ) ) {
            $file_dir = dirname( $file_path );
            
            foreach ( $metadata['sizes'] as $size => $size_data ) {
                $size_path = $file_dir . '/' . $size_data['file'];
                
                if ( file_exists( $size_path ) ) {
                    // Convert thumbnail to WebP
                    if ( $convert_to_webp ) {
                        $thumb_info = @getimagesize( $size_path );
                        $thumb_mime = $thumb_info ? $thumb_info['mime'] : '';
                        
                        if ( $thumb_mime !== 'image/webp' && $thumb_mime !== 'image/gif' ) {
                            $new_thumb_path = $this->convert_file_to_webp( $size_path );
                            if ( $new_thumb_path && $new_thumb_path !== $size_path ) {
                                $metadata['sizes'][ $size ]['file'] = basename( $new_thumb_path );
                                $metadata['sizes'][ $size ]['mime-type'] = 'image/webp';
                                $size_path = $new_thumb_path;
                            }
                        }
                    }
                    
                    // Compress thumbnail
                    $this->compress_to_target( $size_path );
                }
            }
        }

        // Log optimization
        $this->log_optimization( $attachment_id, $original_size, $optimized_size );

        // Update user count
        $user_id = get_current_user_id();
        if ( $user_id ) {
            $this->increment_user_image_count( $user_id );
        }

        return $metadata;
    }

    /**
     * Convert a file to WebP
     */
    private function convert_file_to_webp( $file_path ) {
        if ( ! file_exists( $file_path ) ) {
            return $file_path;
        }

        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return $file_path;
        }

        $mime = $image_info['mime'];
        
        // Skip if already WebP or GIF
        if ( $mime === 'image/webp' || $mime === 'image/gif' ) {
            return $file_path;
        }

        $source = $this->create_image_resource( $file_path, $mime );
        if ( ! $source ) {
            return $file_path;
        }

        $path_info = pathinfo( $file_path );
        $webp_path = $path_info['dirname'] . '/' . $path_info['filename'] . '.webp';

        // Handle transparency for PNG
        if ( $mime === 'image/png' ) {
            imagepalettetotruecolor( $source );
            imagealphablending( $source, true );
            imagesavealpha( $source, true );
        }

        $quality = $this->plugin->get_option( 'compression_quality', 82 );
        $success = imagewebp( $source, $webp_path, $quality );
        imagedestroy( $source );

        if ( $success && file_exists( $webp_path ) ) {
            @unlink( $file_path );
            return $webp_path;
        }

        return $file_path;
    }

    /**
     * Update attachment after WebP conversion
     */
    private function update_attachment_webp( $attachment_id, $metadata ) {
        $upload_dir = wp_upload_dir();
        $new_file = $upload_dir['basedir'] . '/' . $metadata['file'];
        
        // Update attached file
        update_attached_file( $attachment_id, $new_file );
        
        // Update post mime type
        wp_update_post( array(
            'ID' => $attachment_id,
            'post_mime_type' => 'image/webp',
        ));
        
        // Update GUID
        $new_url = $upload_dir['baseurl'] . '/' . $metadata['file'];
        global $wpdb;
        $wpdb->update( 
            $wpdb->posts, 
            array( 'guid' => $new_url ), 
            array( 'ID' => $attachment_id ) 
        );
    }

    /**
     * Compress image to target size (50KB)
     */
    public function compress_to_target( $file_path, $max_iterations = 10 ) {
        if ( ! file_exists( $file_path ) ) {
            return false;
        }

        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return false;
        }

        $mime = $image_info['mime'];
        $current_size = filesize( $file_path );
        
        // Already under target
        if ( $current_size <= $this->target_size ) {
            return true;
        }

        // Start with configured quality
        $quality = $this->plugin->get_option( 'compression_quality', 82 );
        $iteration = 0;

        while ( $current_size > $this->target_size && $iteration < $max_iterations && $quality > 10 ) {
            $this->compress_image( $file_path, $quality );
            clearstatcache( true, $file_path );
            $current_size = filesize( $file_path );
            
            // Reduce quality more aggressively if still too large
            if ( $current_size > $this->target_size * 2 ) {
                $quality -= 15;
            } elseif ( $current_size > $this->target_size * 1.5 ) {
                $quality -= 10;
            } else {
                $quality -= 5;
            }
            
            $quality = max( 10, $quality );
            $iteration++;
        }

        // If still too large, resize dimensions
        if ( $current_size > $this->target_size ) {
            $this->reduce_dimensions( $file_path );
            $this->compress_image( $file_path, max( 20, $quality ) );
        }

        return true;
    }

    /**
     * Compress image with specific quality
     */
    public function compress_image( $file_path, $quality = 82 ) {
        if ( ! file_exists( $file_path ) ) {
            return false;
        }

        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return false;
        }

        $mime = $image_info['mime'];

        switch ( $mime ) {
            case 'image/jpeg':
                $image = @imagecreatefromjpeg( $file_path );
                if ( $image ) {
                    imagejpeg( $image, $file_path, $quality );
                    imagedestroy( $image );
                    return true;
                }
                break;
                
            case 'image/png':
                $image = @imagecreatefrompng( $file_path );
                if ( $image ) {
                    imagealphablending( $image, false );
                    imagesavealpha( $image, true );
                    $png_quality = (int) ( ( 100 - $quality ) / 11.1 );
                    imagepng( $image, $file_path, $png_quality );
                    imagedestroy( $image );
                    return true;
                }
                break;
                
            case 'image/webp':
                if ( function_exists( 'imagecreatefromwebp' ) && function_exists( 'imagewebp' ) ) {
                    $image = @imagecreatefromwebp( $file_path );
                    if ( $image ) {
                        imagewebp( $image, $file_path, $quality );
                        imagedestroy( $image );
                        return true;
                    }
                }
                break;

            case 'image/gif':
                // GIFs generally can't be compressed much
                return true;
        }

        return false;
    }

    /**
     * Resize image to max dimensions
     */
    public function resize_image( $file_path, $max_width, $max_height ) {
        if ( ! file_exists( $file_path ) ) {
            return false;
        }

        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return false;
        }

        $width = $image_info[0];
        $height = $image_info[1];
        $mime = $image_info['mime'];

        if ( $width <= $max_width && $height <= $max_height ) {
            return true;
        }

        $ratio = min( $max_width / $width, $max_height / $height );
        $new_width = (int) ( $width * $ratio );
        $new_height = (int) ( $height * $ratio );

        $source = $this->create_image_resource( $file_path, $mime );
        if ( ! $source ) {
            return false;
        }

        $destination = imagecreatetruecolor( $new_width, $new_height );

        // Preserve transparency for PNG and WebP
        if ( $mime === 'image/png' || $mime === 'image/webp' ) {
            imagealphablending( $destination, false );
            imagesavealpha( $destination, true );
            $transparent = imagecolorallocatealpha( $destination, 0, 0, 0, 127 );
            imagefilledrectangle( $destination, 0, 0, $new_width, $new_height, $transparent );
        }

        imagecopyresampled( $destination, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height );

        $this->save_image( $destination, $file_path, $mime );

        imagedestroy( $source );
        imagedestroy( $destination );

        return true;
    }

    /**
     * Reduce dimensions to help with compression
     */
    private function reduce_dimensions( $file_path ) {
        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return;
        }

        $width = $image_info[0];
        $height = $image_info[1];

        // Reduce by 25%
        $new_width = (int) ( $width * 0.75 );
        $new_height = (int) ( $height * 0.75 );

        $this->resize_image( $file_path, $new_width, $new_height );
    }

    /**
     * Create image resource from file
     */
    private function create_image_resource( $file_path, $mime ) {
        switch ( $mime ) {
            case 'image/jpeg':
                return @imagecreatefromjpeg( $file_path );
            case 'image/png':
                $img = @imagecreatefrompng( $file_path );
                if ( $img ) {
                    imagepalettetotruecolor( $img );
                }
                return $img;
            case 'image/gif':
                return @imagecreatefromgif( $file_path );
            case 'image/webp':
                if ( function_exists( 'imagecreatefromwebp' ) ) {
                    return @imagecreatefromwebp( $file_path );
                }
                return false;
            default:
                return false;
        }
    }

    /**
     * Save image to file
     */
    private function save_image( $image, $file_path, $mime, $quality = 82 ) {
        switch ( $mime ) {
            case 'image/jpeg':
                imagejpeg( $image, $file_path, $quality );
                break;
            case 'image/png':
                imagesavealpha( $image, true );
                $png_quality = (int) ( ( 100 - $quality ) / 11.1 );
                imagepng( $image, $file_path, $png_quality );
                break;
            case 'image/gif':
                imagegif( $image, $file_path );
                break;
            case 'image/webp':
                if ( function_exists( 'imagewebp' ) ) {
                    imagewebp( $image, $file_path, $quality );
                }
                break;
        }
    }

    /**
     * Rename image based on SEO pattern
     */
    public function rename_image( $upload ) {
        $keywords = new DRIOP_Keywords( $this->plugin );
        $keyword = $keywords->get_random_keyword();
        
        $pattern = $this->plugin->get_option( 'seo_pattern', '{site_name} {content_title} {keyword}' );
        
        $replacements = array(
            '{site_name}' => sanitize_title( get_bloginfo( 'name' ) ),
            '{content_title}' => '',
            '{keyword}' => sanitize_title( $keyword ),
            '{date}' => date( 'Y-m-d' ),
            '{timestamp}' => time(),
        );

        // Get content title from context
        global $post;
        if ( $post ) {
            $replacements['{content_title}'] = sanitize_title( $post->post_title );
        }

        $new_name = str_replace( array_keys( $replacements ), array_values( $replacements ), $pattern );
        $new_name = preg_replace( '/\s+/', '-', $new_name );
        $new_name = preg_replace( '/-+/', '-', $new_name );
        $new_name = trim( $new_name, '-' );
        
        if ( empty( $new_name ) ) {
            $new_name = 'image-' . time();
        }

        $path_info = pathinfo( $upload['file'] );
        $extension = isset( $path_info['extension'] ) ? strtolower( $path_info['extension'] ) : 'webp';
        
        // Make filename unique but keep it SEO-friendly
        $new_filename = $new_name . '-' . substr( uniqid(), -6 ) . '.' . $extension;
        $new_path = $path_info['dirname'] . '/' . $new_filename;
        
        if ( rename( $upload['file'], $new_path ) ) {
            $upload['file'] = $new_path;
            $upload['url'] = str_replace( basename( $upload['url'] ), $new_filename, $upload['url'] );
        }

        return $upload;
    }

    /**
     * Get user image count
     */
    public function get_user_image_count( $user_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'driop_user_image_count';
        $count = $wpdb->get_var( $wpdb->prepare( "SELECT image_count FROM $table WHERE user_id = %d", $user_id ) );
        return $count ? (int) $count : 0;
    }

    /**
     * Increment user image count
     */
    public function increment_user_image_count( $user_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'driop_user_image_count';
        $wpdb->query( $wpdb->prepare(
            "INSERT INTO $table (user_id, image_count) VALUES (%d, 1) ON DUPLICATE KEY UPDATE image_count = image_count + 1",
            $user_id
        ) );
    }

    /**
     * Log optimization
     */
    private function log_optimization( $attachment_id, $original_size, $optimized_size ) {
        global $wpdb;
        
        $table = $wpdb->prefix . 'driop_optimization_log';
        $savings = $original_size > 0 ? ( ( $original_size - $optimized_size ) / $original_size ) * 100 : 0;
        
        $wpdb->insert( $table, array(
            'attachment_id' => $attachment_id,
            'user_id' => get_current_user_id(),
            'original_size' => $original_size,
            'optimized_size' => $optimized_size,
            'savings_percent' => $savings,
            'optimization_type' => 'auto',
        ), array( '%d', '%d', '%d', '%d', '%f', '%s' ) );
        
        update_post_meta( $attachment_id, '_driop_original_size', $original_size );
        update_post_meta( $attachment_id, '_driop_optimized_size', $optimized_size );
        update_post_meta( $attachment_id, '_driop_savings_percent', $savings );
        update_post_meta( $attachment_id, '_driop_optimized', true );
    }

    /**
     * Get optimization stats
     */
    public function get_stats() {
        global $wpdb;
        
        $table = $wpdb->prefix . 'driop_optimization_log';
        
        // Check if table exists
        if ( $wpdb->get_var( "SHOW TABLES LIKE '$table'" ) !== $table ) {
            return array(
                'total_images' => 0,
                'total_original' => 0,
                'total_optimized' => 0,
                'total_savings' => 0,
                'avg_savings' => 0,
            );
        }
        
        $stats = $wpdb->get_row(
            "SELECT COUNT(*) as total_images, SUM(original_size) as total_original, 
             SUM(optimized_size) as total_optimized, AVG(savings_percent) as avg_savings FROM $table"
        );

        return array(
            'total_images' => $stats ? (int) $stats->total_images : 0,
            'total_original' => $stats ? (int) $stats->total_original : 0,
            'total_optimized' => $stats ? (int) $stats->total_optimized : 0,
            'total_savings' => $stats ? (int) $stats->total_original - (int) $stats->total_optimized : 0,
            'avg_savings' => $stats ? round( (float) $stats->avg_savings, 2 ) : 0,
        );
    }

    /**
     * Optimize existing image
     */
    public function optimize_existing( $attachment_id, $apply_watermark = false ) {
        $file_path = get_attached_file( $attachment_id );
        
        if ( ! $file_path || ! file_exists( $file_path ) ) {
            return array( 'success' => false, 'message' => __( 'Image file not found.', 'digital-rise-solutions' ) );
        }

        $original_size = filesize( $file_path );
        $convert_to_webp = $this->plugin->get_option( 'convert_to_webp', true ) && $this->supports_webp();
        
        // Resize if needed
        $max_width = $this->plugin->get_option( 'max_width', 1920 );
        $max_height = $this->plugin->get_option( 'max_height', 1080 );
        $this->resize_image( $file_path, $max_width, $max_height );
        
        // Convert to WebP if enabled
        if ( $convert_to_webp ) {
            $image_info = @getimagesize( $file_path );
            $mime = $image_info ? $image_info['mime'] : '';
            
            if ( $mime !== 'image/webp' && $mime !== 'image/gif' ) {
                $new_path = $this->convert_file_to_webp( $file_path );
                if ( $new_path && $new_path !== $file_path ) {
                    // Update attachment
                    $upload_dir = wp_upload_dir();
                    $new_file_rel = str_replace( $upload_dir['basedir'] . '/', '', $new_path );
                    
                    update_attached_file( $attachment_id, $new_path );
                    wp_update_post( array(
                        'ID' => $attachment_id,
                        'post_mime_type' => 'image/webp',
                    ));
                    
                    $file_path = $new_path;
                }
            }
        }
        
        // Compress to target
        $this->compress_to_target( $file_path );
        
        // Apply watermark if requested
        if ( $apply_watermark && $this->plugin->get_option( 'watermark_enabled', false ) ) {
            $watermark = new DRIOP_Watermark( $this->plugin );
            $watermark->add_watermark_to_file( $file_path );
        }
        
        clearstatcache( true, $file_path );
        $optimized_size = filesize( $file_path );
        
        $this->log_optimization( $attachment_id, $original_size, $optimized_size );
        
        $savings = $original_size > 0 ? ( ( $original_size - $optimized_size ) / $original_size ) * 100 : 0;

        return array(
            'success' => true,
            'original_size' => $original_size,
            'optimized_size' => $optimized_size,
            'savings' => round( $savings, 2 ),
            'message' => sprintf( __( 'Optimized! Saved %s%%.', 'digital-rise-solutions' ), round( $savings, 2 ) ),
        );
    }
}
