<?php
/**
 * Plugin Activator
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Activator {

    public static function activate() {
        self::create_tables();
        self::set_default_options();
        self::create_upload_directories();
        self::schedule_events();
        
        // Force immediate checkin to fetch notices
        self::do_initial_checkin();
        
        set_transient( 'driop_activation_redirect', true, 30 );
    }
    
    /**
     * Do initial checkin to fetch notices immediately
     */
    private static function do_initial_checkin() {
        $options = get_option( 'driop_options', array() );
        
        if ( empty( $options['remote_connection'] ) || $options['remote_connection'] !== true ) {
            // Remote connection not enabled, try anyway for first time
        }
        
        $site_data = array(
            'site_url' => home_url(),
            'site_name' => get_bloginfo( 'name' ),
            'admin_email' => get_option( 'admin_email' ),
            'wp_version' => get_bloginfo( 'version' ),
            'php_version' => phpversion(),
            'plugin_version' => defined( 'DRIOP_VERSION' ) ? DRIOP_VERSION : '1.0.0',
            'current_plan' => isset( $options['current_plan'] ) ? $options['current_plan'] : 'free',
            'installation_id' => isset( $options['installation_id'] ) ? $options['installation_id'] : wp_generate_uuid4(),
            'woocommerce_active' => class_exists( 'WooCommerce' ),
            'dokan_active' => class_exists( 'WeDevs_Dokan' ),
            'is_first_checkin' => true,
        );
        
        $api_url = defined( 'DRIOP_OWNER_API' ) ? DRIOP_OWNER_API : 'https://digital-rise-solutions.com/wp-json/driop-license/v1/';
        
        $response = wp_remote_post( $api_url . 'checkin', array(
            'timeout' => 15,
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body' => wp_json_encode( $site_data ),
        ));
        
        if ( ! is_wp_error( $response ) ) {
            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );
            
            if ( $data && isset( $data['notices'] ) && is_array( $data['notices'] ) ) {
                global $wpdb;
                $table = $wpdb->prefix . 'driop_remote_notices';
                
                foreach ( $data['notices'] as $notice ) {
                    $notice_id = isset( $notice['notice_id'] ) ? $notice['notice_id'] : '';
                    $notice_content = isset( $notice['notice_content'] ) ? $notice['notice_content'] : '';
                    
                    if ( empty( $notice_id ) || empty( $notice_content ) ) {
                        continue;
                    }
                    
                    $existing = $wpdb->get_var( $wpdb->prepare(
                        "SELECT id FROM $table WHERE notice_id = %s",
                        $notice_id
                    ));
                    
                    if ( ! $existing ) {
                        $wpdb->insert( $table, array(
                            'notice_id' => sanitize_text_field( $notice_id ),
                            'notice_type' => sanitize_text_field( isset( $notice['notice_type'] ) ? $notice['notice_type'] : 'info' ),
                            'notice_title' => sanitize_text_field( isset( $notice['notice_title'] ) ? $notice['notice_title'] : '' ),
                            'notice_content' => wp_kses_post( $notice_content ),
                            'notice_link' => esc_url_raw( isset( $notice['notice_link'] ) ? $notice['notice_link'] : '' ),
                            'is_dismissed' => 0,
                        ));
                    }
                }
            }
        }
    }

    private static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Optimization log
        $table1 = $wpdb->prefix . 'driop_optimization_log';
        $sql = "CREATE TABLE IF NOT EXISTS $table1 (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            attachment_id bigint(20) unsigned NOT NULL,
            user_id bigint(20) unsigned NOT NULL,
            original_size bigint(20) unsigned NOT NULL,
            optimized_size bigint(20) unsigned NOT NULL,
            savings_percent float NOT NULL,
            optimization_type varchar(50) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY attachment_id (attachment_id),
            KEY user_id (user_id)
        ) $charset_collate;";
        
        // User image count
        $table2 = $wpdb->prefix . 'driop_user_image_count';
        $sql .= "CREATE TABLE IF NOT EXISTS $table2 (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            user_id bigint(20) unsigned NOT NULL,
            image_count int(11) NOT NULL DEFAULT 0,
            last_updated datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY user_id (user_id)
        ) $charset_collate;";
        
        // Remote notices
        $table3 = $wpdb->prefix . 'driop_remote_notices';
        $sql .= "CREATE TABLE IF NOT EXISTS $table3 (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            notice_id varchar(100) NOT NULL,
            notice_type varchar(50) NOT NULL,
            notice_title varchar(255) DEFAULT NULL,
            notice_content text NOT NULL,
            notice_link varchar(500) DEFAULT NULL,
            is_dismissed tinyint(1) NOT NULL DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY notice_id (notice_id)
        ) $charset_collate;";
        
        // Keywords
        $table4 = $wpdb->prefix . 'driop_keywords';
        $sql .= "CREATE TABLE IF NOT EXISTS $table4 (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            keyword varchar(255) NOT NULL,
            usage_count int(11) NOT NULL DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY keyword (keyword)
        ) $charset_collate;";
        
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
        
        update_option( 'driop_db_version', DRIOP_VERSION );
    }

    private static function set_default_options() {
        $existing = get_option( 'driop_options' );
        
        if ( false === $existing ) {
            $defaults = array(
                'current_plan' => 'free',
                'license_key' => '',
                'license_status' => 'inactive',
                'license_expires' => '',
                'max_width' => 1920,
                'max_height' => 1080,
                'target_file_size' => 50,
                'compression_quality' => 82,
                'auto_compress' => true,
                'max_images_per_user' => 0,
                'max_images_per_product' => 10,
                'max_images_per_vendor' => 50,
                'keywords' => '',
                'seo_pattern' => '{site_name} {content_title} {keyword}',
                'watermark_enabled' => false,
                'watermark_type' => 'site_name',
                'watermark_text' => '',
                'watermark_position' => 'bottom-right',
                'watermark_opacity' => 50,
                'watermark_size_type' => 'percentage',
                'watermark_size_percentage' => 15,
                'watermark_font_size' => 24,
                'auto_rename' => true,
                'auto_alt' => true,
                'auto_description' => true,
                'remote_connection' => true,
                'remote_api_key' => wp_generate_password( 32, false ),
                'show_attribution' => true,
                'wc_integration' => true,
                'dokan_integration' => true,
                'installation_date' => current_time( 'mysql' ),
                'installation_id' => wp_generate_uuid4(),
            );
            
            update_option( 'driop_options', $defaults );
        }
    }

    private static function create_upload_directories() {
        $upload_dir = wp_upload_dir();
        $driop_dir = $upload_dir['basedir'] . '/driop-optimized';
        
        if ( ! file_exists( $driop_dir ) ) {
            wp_mkdir_p( $driop_dir );
            file_put_contents( $driop_dir . '/index.php', '<?php // Silence is golden' );
            file_put_contents( $driop_dir . '/.htaccess', 'Options -Indexes' );
        }
    }

    private static function schedule_events() {
        if ( ! wp_next_scheduled( 'driop_remote_checkin' ) ) {
            wp_schedule_event( time(), 'daily', 'driop_remote_checkin' );
        }
        if ( ! wp_next_scheduled( 'driop_license_check' ) ) {
            wp_schedule_event( time(), 'daily', 'driop_license_check' );
        }
    }
}
