/* Digital Rise Image Optimizer Pro - Admin JS */
(function($) {
    'use strict';

    $(document).ready(function() {
        // Range sliders
        $('#compression_quality').on('input', function() {
            $('#quality_value').text($(this).val() + '%');
        });
        $('#watermark_opacity').on('input', function() {
            $('#opacity_value').text($(this).val() + '%');
        });
        $('#watermark_size_percentage').on('input', function() {
            $('#size_percentage_value').text($(this).val() + '%');
        });

        // Watermark size type toggle
        $('#watermark_size_type').on('change', function() {
            if ($(this).val() === 'percentage') {
                $('.watermark-size-percentage').show();
                $('.watermark-size-fixed').hide();
            } else {
                $('.watermark-size-percentage').hide();
                $('.watermark-size-fixed').show();
            }
        }).trigger('change');

        // License activation
        $('#activate-license').on('click', function() {
            var key = $('#license_key').val();
            if (!key) {
                alert('Please enter a license key');
                return;
            }

            var $btn = $(this);
            $btn.prop('disabled', true).text(driop_admin.strings.optimizing);

            $.ajax({
                url: driop_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'driop_activate_license',
                    license_key: key,
                    nonce: driop_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $('#license-message').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                        setTimeout(function() { location.reload(); }, 1500);
                    } else {
                        $('#license-message').html('<div class="notice notice-error"><p>' + response.data + '</p></div>');
                    }
                },
                error: function() {
                    $('#license-message').html('<div class="notice notice-error"><p>Error occurred</p></div>');
                },
                complete: function() {
                    $btn.prop('disabled', false).text('Activate License');
                }
            });
        });

        // License deactivation
        $('#deactivate-license').on('click', function() {
            if (!confirm('Are you sure you want to deactivate your license?')) return;

            $.ajax({
                url: driop_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'driop_deactivate_license',
                    nonce: driop_admin.nonce
                },
                success: function() {
                    location.reload();
                }
            });
        });

        // Bulk optimization
        var bulkRunning = false;

        $('#start-bulk-optimize').on('click', function() {
            bulkRunning = true;
            $(this).hide();
            $('#stop-bulk-optimize').show();
            $('#bulk-log').show().html('');
            $('#reset-optimization').prop('disabled', true);
            runBulkOptimize();
        });

        $('#stop-bulk-optimize').on('click', function() {
            bulkRunning = false;
            $(this).hide();
            $('#start-bulk-optimize').show();
            $('#reset-optimization').prop('disabled', false);
            logMessage('Stopped by user', 'info');
        });
        
        // Reset optimization
        $('#reset-optimization').on('click', function() {
            if (!confirm('This will mark all images as unoptimized. Are you sure?')) return;
            
            var $btn = $(this);
            $btn.prop('disabled', true);
            
            $.ajax({
                url: driop_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'driop_reset_optimization',
                    nonce: driop_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        location.reload();
                    }
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        });

        function runBulkOptimize() {
            if (!bulkRunning) return;

            var applyWatermark = $('#apply_watermark').is(':checked');
            var applySeo = $('#apply_seo').is(':checked');
            var reoptimize = $('#reoptimize_all').is(':checked');

            $.ajax({
                url: driop_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'driop_bulk_optimize',
                    apply_watermark: applyWatermark,
                    apply_seo: applySeo,
                    reoptimize: reoptimize,
                    nonce: driop_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        
                        // Update progress
                        updateProgress();
                        
                        // Log results
                        if (data.details) {
                            data.details.forEach(function(item) {
                                var msg = item.title + ' - ' + (item.status === 'success' ? 'Saved ' + item.savings + '%' : item.message);
                                logMessage(msg, item.status);
                            });
                        }

                        if (data.completed) {
                            bulkRunning = false;
                            $('#stop-bulk-optimize').hide();
                            $('#start-bulk-optimize').show().prop('disabled', false).text('Optimization Complete - Click to Run Again');
                            $('#reset-optimization').prop('disabled', false);
                            logMessage('All images processed!', 'success');
                        } else {
                            setTimeout(runBulkOptimize, 500);
                        }
                    }
                },
                error: function() {
                    logMessage('Error occurred, retrying...', 'error');
                    setTimeout(runBulkOptimize, 2000);
                }
            });
        }

        function updateProgress() {
            $.ajax({
                url: driop_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'driop_get_optimization_progress',
                    nonce: driop_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        $('#bulk-progress-fill').css('width', data.percentage + '%');
                        $('#bulk-progress-text').text(data.optimized + ' of ' + data.total + ' images optimized (' + data.remaining + ' remaining)');
                    }
                }
            });
        }

        function logMessage(msg, type) {
            var $log = $('#bulk-log');
            var time = new Date().toLocaleTimeString();
            $log.append('<div class="' + type + '">[' + time + '] ' + msg + '</div>');
            $log.scrollTop($log[0].scrollHeight);
        }

        // Dismiss notices
        $(document).on('click', '.driop-upgrade-banner .notice-dismiss', function() {
            $.post(driop_admin.ajax_url, {
                action: 'driop_dismiss_notice',
                notice: 'upgrade_banner',
                nonce: driop_admin.nonce
            });
        });

        $(document).on('click', '.driop-remote-notice .notice-dismiss', function() {
            var noticeId = $(this).closest('.driop-remote-notice').data('notice-id');
            $.post(driop_admin.ajax_url, {
                action: 'driop_dismiss_notice',
                notice: noticeId,
                nonce: driop_admin.nonce
            });
        });

        // Watermark preview refresh
        $('#refresh-preview').on('click', function() {
            var $btn = $(this);
            var $img = $('#watermark-preview');
            
            $btn.prop('disabled', true).text('Loading...');
            
            $.ajax({
                url: driop_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'driop_generate_watermark_preview',
                    nonce: driop_admin.nonce
                },
                success: function(response) {
                    if (response.success && response.data.preview) {
                        $img.attr('src', response.data.preview);
                    }
                },
                complete: function() {
                    $btn.prop('disabled', false).text('Refresh Preview');
                }
            });
        });

        // Sync notices button
        $('#sync-notices').on('click', function() {
            var $btn = $(this);
            $btn.prop('disabled', true).text('Syncing...');
            
            $.ajax({
                url: driop_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'driop_sync_notices',
                    nonce: driop_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data.message + ' (' + response.data.count + ' active notices)');
                        location.reload();
                    } else {
                        alert('Sync failed. Please try again.');
                    }
                },
                error: function() {
                    alert('Connection error. Please check your internet connection.');
                },
                complete: function() {
                    $btn.prop('disabled', false).text('Sync Notices');
                }
            });
        });

        // Auto-refresh preview when watermark settings change
        $('#watermark_type, #watermark_position, #watermark_opacity, #watermark_size_type, #watermark_size_percentage, #watermark_font_size, #watermark_text').on('change', function() {
            // Save current values to options via form submission not needed here
            // Just refresh preview with AJAX would require saving first
        });
    });
})(jQuery);
