<?php
/**
 * Admin Dashboard
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Admin {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    public function enqueue_styles( $hook ) {
        if ( strpos( $hook, 'driop' ) === false ) return;
        wp_enqueue_style( 'driop-admin', DRIOP_PLUGIN_URL . 'admin/css/driop-admin.css', array(), DRIOP_VERSION );
    }

    public function enqueue_scripts( $hook ) {
        if ( strpos( $hook, 'driop' ) === false ) return;
        wp_enqueue_script( 'driop-admin', DRIOP_PLUGIN_URL . 'admin/js/driop-admin.js', array( 'jquery' ), DRIOP_VERSION, true );
        wp_localize_script( 'driop-admin', 'driop_admin', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'driop_admin_nonce' ),
            'strings' => array(
                'optimizing' => __( 'Optimizing...', 'digital-rise-solutions' ),
                'complete' => __( 'Complete!', 'digital-rise-solutions' ),
                'error' => __( 'Error occurred', 'digital-rise-solutions' ),
            ),
        ) );
    }

    public function add_admin_menu() {
        add_menu_page( __( 'Image Optimizer for SEO', 'digital-rise-solutions' ), __( 'Image Optimizer', 'digital-rise-solutions' ), 'manage_options', 'driop-dashboard', array( $this, 'render_dashboard' ), 'dashicons-format-image', 65 );
        add_submenu_page( 'driop-dashboard', __( 'Dashboard', 'digital-rise-solutions' ), __( 'Dashboard', 'digital-rise-solutions' ), 'manage_options', 'driop-dashboard', array( $this, 'render_dashboard' ) );
        add_submenu_page( 'driop-dashboard', __( 'Image Optimization', 'digital-rise-solutions' ), __( 'Optimization', 'digital-rise-solutions' ), 'manage_options', 'driop-optimization', array( $this, 'render_optimization' ) );
        add_submenu_page( 'driop-dashboard', __( 'Bulk Optimize', 'digital-rise-solutions' ), __( 'Bulk Optimize', 'digital-rise-solutions' ), 'manage_options', 'driop-bulk', array( $this, 'render_bulk_optimize' ) );
        add_submenu_page( 'driop-dashboard', __( 'Keywords', 'digital-rise-solutions' ), __( 'Keywords', 'digital-rise-solutions' ), 'manage_options', 'driop-keywords', array( $this, 'render_keywords' ) );
        add_submenu_page( 'driop-dashboard', __( 'SEO Titles & Meta', 'digital-rise-solutions' ), __( 'SEO Titles', 'digital-rise-solutions' ), 'manage_options', 'driop-page-seo', array( $this, 'render_page_seo' ) );
        add_submenu_page( 'driop-dashboard', __( 'Watermark', 'digital-rise-solutions' ), __( 'Watermark', 'digital-rise-solutions' ), 'manage_options', 'driop-watermark', array( $this, 'render_watermark' ) );
        add_submenu_page( 'driop-dashboard', __( 'Plans', 'digital-rise-solutions' ), __( 'Plans', 'digital-rise-solutions' ), 'manage_options', 'driop-plans', array( $this, 'render_plans' ) );
        add_submenu_page( 'driop-dashboard', __( 'License', 'digital-rise-solutions' ), __( 'License', 'digital-rise-solutions' ), 'manage_options', 'driop-license', array( $this, 'render_license' ) );
        add_submenu_page( 'driop-dashboard', __( 'Settings', 'digital-rise-solutions' ), __( 'Settings', 'digital-rise-solutions' ), 'manage_options', 'driop-settings', array( $this, 'render_settings' ) );
    }

    public function register_settings() {
        register_setting( 'driop_options_group', 'driop_options' );
    }

    public function display_admin_notices() {
        if ( get_transient( 'driop_activation_redirect' ) ) {
            delete_transient( 'driop_activation_redirect' );
            if ( ! isset( $_GET['activate-multi'] ) ) {
                wp_safe_redirect( admin_url( 'admin.php?page=driop-dashboard' ) );
                exit;
            }
        }

        // Show upgrade banner for free plan (only on plugin pages)
        $screen = get_current_screen();
        if ( $this->plugin->current_plan === 'free' ) {
            if ( $screen && strpos( $screen->id, 'driop' ) !== false && ! get_option( 'driop_upgrade_dismissed' ) ) {
                ?>
                <div class="notice notice-info is-dismissible driop-upgrade-banner">
                    <p><strong><?php esc_html_e( 'Upgrade to Pro!', 'digital-rise-solutions' ); ?></strong> 
                    <?php esc_html_e( 'Unlock custom keywords, watermark size control, and more.', 'digital-rise-solutions' ); ?>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Upgrade', 'digital-rise-solutions' ); ?></a></p>
                </div>
                <?php
            }
        }

        // Remote notices - SHOW ON ALL ADMIN PAGES
        $remote = new DRIOP_Remote( $this->plugin );
        $notices = $remote->get_active_notices();
        
        if ( ! empty( $notices ) && is_array( $notices ) ) {
            foreach ( $notices as $notice ) {
                if ( ! is_array( $notice ) || empty( $notice['notice_content'] ) ) {
                    continue;
                }
                
                $type = isset( $notice['notice_type'] ) && in_array( $notice['notice_type'], array( 'info', 'warning', 'error', 'success', 'promo' ) ) ? $notice['notice_type'] : 'info';
                
                // Convert promo to info for WordPress notice class
                $wp_type = $type === 'promo' ? 'info' : $type;
                
                $notice_id = isset( $notice['notice_id'] ) ? $notice['notice_id'] : 'notice-' . md5( $notice['notice_content'] );
                ?>
                <div class="notice notice-<?php echo esc_attr( $wp_type ); ?> is-dismissible driop-remote-notice" data-notice-id="<?php echo esc_attr( $notice_id ); ?>" style="border-left-color: <?php echo $type === 'promo' ? '#9c27b0' : ''; ?>">
                    <?php if ( ! empty( $notice['notice_title'] ) ) : ?>
                        <p><strong><?php echo esc_html( $notice['notice_title'] ); ?></strong></p>
                    <?php endif; ?>
                    <p><?php echo wp_kses_post( $notice['notice_content'] ); ?>
                    <?php if ( ! empty( $notice['notice_link'] ) ) : ?>
                        <a href="<?php echo esc_url( $notice['notice_link'] ); ?>" target="_blank"><?php esc_html_e( 'Learn more', 'digital-rise-solutions' ); ?></a>
                    <?php endif; ?>
                    </p>
                </div>
                <?php
            }
        }
    }

    private function render_header( $title ) {
        $logo_url = DRIOP_LOGO_URL;
        ?>
        <div class="driop-header">
            <div class="driop-header-logo">
                <?php if ( ! empty( $logo_url ) && @getimagesize( $logo_url ) ) : ?>
                    <img src="<?php echo esc_url( $logo_url ); ?>" alt="Digital Rise" height="50">
                <?php else : ?>
                    <span class="driop-logo-text">Image Optimizer for SEO</span>
                <?php endif; ?>
            </div>
            <div class="driop-header-info">
                <h1><?php echo esc_html( $title ); ?></h1>
                <span class="driop-plan-badge driop-plan-<?php echo esc_attr( $this->plugin->current_plan ); ?>">
                    <?php echo esc_html( ucfirst( $this->plugin->current_plan ) ); ?>
                </span>
            </div>
        </div>
        <?php
    }

    public function render_dashboard() {
        $optimizer = new DRIOP_Image_Optimizer( $this->plugin );
        $stats = $optimizer->get_stats();
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        $wc = new DRIOP_WooCommerce( $this->plugin );
        $dokan = new DRIOP_Dokan( $this->plugin );
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Dashboard', 'digital-rise-solutions' ) ); ?>
            
            <div class="driop-dashboard-grid">
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Optimization Stats', 'digital-rise-solutions' ); ?></h2>
                    <div class="driop-stats">
                        <div class="driop-stat">
                            <span class="driop-stat-num"><?php echo esc_html( $stats['total_images'] ); ?></span>
                            <span class="driop-stat-label"><?php esc_html_e( 'Optimized', 'digital-rise-solutions' ); ?></span>
                        </div>
                        <div class="driop-stat">
                            <span class="driop-stat-num"><?php echo esc_html( size_format( $stats['total_savings'] ) ); ?></span>
                            <span class="driop-stat-label"><?php esc_html_e( 'Saved', 'digital-rise-solutions' ); ?></span>
                        </div>
                        <div class="driop-stat">
                            <span class="driop-stat-num"><?php echo esc_html( $stats['avg_savings'] ); ?>%</span>
                            <span class="driop-stat-label"><?php esc_html_e( 'Avg. Savings', 'digital-rise-solutions' ); ?></span>
                        </div>
                    </div>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Image Status', 'digital-rise-solutions' ); ?></h2>
                    <div class="driop-progress-wrap">
                        <div class="driop-progress-bar">
                            <div class="driop-progress-fill" style="width: <?php echo esc_attr( $progress['percentage'] ); ?>%"></div>
                        </div>
                        <p><?php printf( esc_html__( '%d of %d images optimized', 'digital-rise-solutions' ), $progress['optimized'], $progress['total'] ); ?></p>
                    </div>
                    <?php if ( $progress['remaining'] > 0 ) : ?>
                        <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-bulk' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Optimize All', 'digital-rise-solutions' ); ?></a></p>
                    <?php endif; ?>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'WooCommerce', 'digital-rise-solutions' ); ?></h2>
                    <?php $wc_stats = $wc->get_stats(); ?>
                    <?php if ( $wc_stats['active'] ) : ?>
                        <p class="driop-status-active"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e( 'Active', 'digital-rise-solutions' ); ?></p>
                        <p><?php printf( esc_html__( 'Products: %d', 'digital-rise-solutions' ), $wc_stats['total_products'] ); ?></p>
                    <?php else : ?>
                        <p class="driop-status-inactive"><?php esc_html_e( 'Not installed', 'digital-rise-solutions' ); ?></p>
                    <?php endif; ?>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Dokan', 'digital-rise-solutions' ); ?></h2>
                    <?php $dokan_stats = $dokan->get_stats(); ?>
                    <?php if ( $dokan_stats['active'] ) : ?>
                        <p class="driop-status-active"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e( 'Active', 'digital-rise-solutions' ); ?></p>
                        <p><?php printf( esc_html__( 'Vendors: %d', 'digital-rise-solutions' ), $dokan_stats['total_vendors'] ); ?></p>
                    <?php else : ?>
                        <p class="driop-status-inactive"><?php esc_html_e( 'Not installed', 'digital-rise-solutions' ); ?></p>
                    <?php endif; ?>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Quick Actions', 'digital-rise-solutions' ); ?></h2>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-bulk' ) ); ?>" class="button"><?php esc_html_e( 'Bulk Optimize', 'digital-rise-solutions' ); ?></a></p>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-keywords' ) ); ?>" class="button"><?php esc_html_e( 'Manage Keywords', 'digital-rise-solutions' ); ?></a></p>
                    <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-watermark' ) ); ?>" class="button"><?php esc_html_e( 'Watermark Settings', 'digital-rise-solutions' ); ?></a></p>
                    <p><button type="button" id="sync-notices" class="button"><?php esc_html_e( 'Sync Notices', 'digital-rise-solutions' ); ?></button></p>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Connection Status', 'digital-rise-solutions' ); ?></h2>
                    <?php 
                    $last_checkin = $this->plugin->get_option( 'last_checkin' );
                    $remote = new DRIOP_Remote( $this->plugin );
                    $notices = $remote->get_active_notices();
                    ?>
                    <p><strong><?php esc_html_e( 'Last Sync:', 'digital-rise-solutions' ); ?></strong> 
                        <?php echo $last_checkin ? esc_html( human_time_diff( strtotime( $last_checkin ) ) ) . ' ago' : esc_html__( 'Never', 'digital-rise-solutions' ); ?>
                    </p>
                    <p><strong><?php esc_html_e( 'Active Notices:', 'digital-rise-solutions' ); ?></strong> <?php echo count( $notices ); ?></p>
                </div>
            </div>
        </div>
        <?php
    }

    public function render_optimization() {
        if ( isset( $_POST['driop_save_optimization'] ) && check_admin_referer( 'driop_optimization_nonce' ) ) {
            $options = $this->plugin->options;
            $options['max_width'] = absint( $_POST['max_width'] );
            $options['max_height'] = absint( $_POST['max_height'] );
            $options['target_file_size'] = absint( $_POST['target_file_size'] );
            $options['compression_quality'] = min( 100, max( 1, absint( $_POST['compression_quality'] ) ) );
            $options['auto_compress'] = isset( $_POST['auto_compress'] );
            $options['max_images_per_user'] = absint( $_POST['max_images_per_user'] );
            $options['max_images_per_product'] = absint( $_POST['max_images_per_product'] );
            $options['max_images_per_vendor'] = absint( $_POST['max_images_per_vendor'] );
            $this->plugin->save_options( $options );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-solutions' ) . '</p></div>';
        }
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Image Optimization', 'digital-rise-solutions' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_optimization_nonce' ); ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Size Settings', 'digital-rise-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Max Width (px)', 'digital-rise-solutions' ); ?></th>
                            <td><input type="number" name="max_width" value="<?php echo esc_attr( $this->plugin->get_option( 'max_width', 1920 ) ); ?>" min="100" max="10000"></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Max Height (px)', 'digital-rise-solutions' ); ?></th>
                            <td><input type="number" name="max_height" value="<?php echo esc_attr( $this->plugin->get_option( 'max_height', 1080 ) ); ?>" min="100" max="10000"></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Target File Size (KB)', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <input type="number" name="target_file_size" value="<?php echo esc_attr( $this->plugin->get_option( 'target_file_size', 50 ) ); ?>" min="10" max="500">
                                <p class="description"><?php esc_html_e( 'Images will be compressed to this target size (default: 50KB)', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Compression Quality', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <input type="range" name="compression_quality" id="compression_quality" value="<?php echo esc_attr( $this->plugin->get_option( 'compression_quality', 82 ) ); ?>" min="1" max="100">
                                <span id="quality_value"><?php echo esc_html( $this->plugin->get_option( 'compression_quality', 82 ) ); ?>%</span>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Auto Compress', 'digital-rise-solutions' ); ?></th>
                            <td><label><input type="checkbox" name="auto_compress" <?php checked( $this->plugin->get_option( 'auto_compress', true ) ); ?>> <?php esc_html_e( 'Automatically compress on upload', 'digital-rise-solutions' ); ?></label></td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Upload Limits', 'digital-rise-solutions' ); ?></h2>
                    <?php if ( ! $this->plugin->has_feature( 'per_user_limits' ) ) : ?>
                        <div class="driop-pro-notice"><span class="dashicons dashicons-lock"></span> <?php esc_html_e( 'Upload limits require Pro plan.', 'digital-rise-solutions' ); ?></div>
                    <?php endif; ?>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per User', 'digital-rise-solutions' ); ?></th>
                            <td><input type="number" name="max_images_per_user" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_user', 0 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_user_limits' ) ); ?>><p class="description"><?php esc_html_e( '0 = unlimited', 'digital-rise-solutions' ); ?></p></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per Product', 'digital-rise-solutions' ); ?></th>
                            <td><input type="number" name="max_images_per_product" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_product', 10 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_product_limits' ) ); ?>></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Max Images Per Vendor', 'digital-rise-solutions' ); ?></th>
                            <td><input type="number" name="max_images_per_vendor" value="<?php echo esc_attr( $this->plugin->get_option( 'max_images_per_vendor', 50 ) ); ?>" min="0" <?php disabled( ! $this->plugin->has_feature( 'per_vendor_limits' ) ); ?>></td>
                        </tr>
                    </table>
                </div>

                <p class="submit"><input type="submit" name="driop_save_optimization" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    public function render_bulk_optimize() {
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Bulk Optimize', 'digital-rise-solutions' ) ); ?>
            
            <div class="driop-card">
                <h2><?php esc_html_e( 'Optimize All Images', 'digital-rise-solutions' ); ?></h2>
                <p><?php esc_html_e( 'This will optimize, add watermarks, and apply SEO to all your images.', 'digital-rise-solutions' ); ?></p>
                
                <div class="driop-bulk-progress">
                    <div class="driop-progress-bar">
                        <div class="driop-progress-fill" id="bulk-progress-fill" style="width: <?php echo esc_attr( $progress['percentage'] ); ?>%"></div>
                    </div>
                    <p id="bulk-progress-text">
                        <?php printf( esc_html__( '%d of %d images optimized (%d remaining)', 'digital-rise-solutions' ), $progress['optimized'], $progress['total'], $progress['remaining'] ); ?>
                    </p>
                </div>

                <table class="form-table">
                    <tr>
                        <th><?php esc_html_e( 'Options', 'digital-rise-solutions' ); ?></th>
                        <td>
                            <p><label><input type="checkbox" id="apply_watermark" checked> <?php esc_html_e( 'Apply watermark to images', 'digital-rise-solutions' ); ?></label></p>
                            <p><label><input type="checkbox" id="apply_seo" checked> <?php esc_html_e( 'Apply SEO (ALT text, title, description)', 'digital-rise-solutions' ); ?></label></p>
                            <p><label><input type="checkbox" id="reoptimize_all"> <?php esc_html_e( 'Re-optimize ALL images (including previously optimized)', 'digital-rise-solutions' ); ?></label></p>
                        </td>
                    </tr>
                </table>

                <p>
                    <button type="button" id="start-bulk-optimize" class="button button-primary button-hero">
                        <?php esc_html_e( 'Start Bulk Optimization', 'digital-rise-solutions' ); ?>
                    </button>
                    <button type="button" id="stop-bulk-optimize" class="button button-secondary" style="display:none;">
                        <?php esc_html_e( 'Stop', 'digital-rise-solutions' ); ?>
                    </button>
                    <button type="button" id="reset-optimization" class="button">
                        <?php esc_html_e( 'Reset All (Mark as Unoptimized)', 'digital-rise-solutions' ); ?>
                    </button>
                </p>

                <div id="bulk-log" class="driop-bulk-log" style="display:none;"></div>
            </div>
        </div>
        <?php
    }

    public function render_keywords() {
        $keywords = new DRIOP_Keywords( $this->plugin );
        
        if ( isset( $_POST['driop_save_keywords'] ) && check_admin_referer( 'driop_keywords_nonce' ) ) {
            $keywords->save_keywords( $_POST['keywords'] );
            $this->plugin->update_option( 'seo_pattern', sanitize_text_field( $_POST['seo_pattern'] ) );
            $this->plugin->update_option( 'auto_rename', isset( $_POST['auto_rename'] ) );
            $this->plugin->update_option( 'auto_alt', isset( $_POST['auto_alt'] ) );
            $this->plugin->update_option( 'auto_description', isset( $_POST['auto_description'] ) );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-solutions' ) . '</p></div>';
        }
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Keywords Management', 'digital-rise-solutions' ) ); ?>
            
            <?php if ( ! $this->plugin->has_feature( 'custom_keywords' ) ) : ?>
                <div class="driop-card">
                    <div class="driop-pro-notice">
                        <span class="dashicons dashicons-lock"></span> 
                        <?php esc_html_e( 'Custom keywords require Pro plan. Free plan uses basic patterns.', 'digital-rise-solutions' ); ?>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button"><?php esc_html_e( 'Upgrade', 'digital-rise-solutions' ); ?></a>
                    </div>
                </div>
            <?php endif; ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_keywords_nonce' ); ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Keywords', 'digital-rise-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Enter keywords for image SEO. One per line or comma-separated.', 'digital-rise-solutions' ); ?></p>
                    <textarea name="keywords" rows="10" class="large-text code" <?php disabled( ! $this->plugin->has_feature( 'custom_keywords' ) ); ?>><?php echo esc_textarea( $this->plugin->get_option( 'keywords', '' ) ); ?></textarea>
                    <p class="description"><?php printf( esc_html__( 'Total keywords: %d', 'digital-rise-solutions' ), $keywords->get_count() ); ?></p>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'SEO Pattern', 'digital-rise-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Pattern', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <input type="text" name="seo_pattern" value="<?php echo esc_attr( $this->plugin->get_option( 'seo_pattern', '{site_name} {content_title} {keyword}' ) ); ?>" class="regular-text">
                                <p class="description"><?php esc_html_e( 'Available: {site_name}, {content_title}, {keyword}, {category}, {date}', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Auto Features', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="auto_rename" <?php checked( $this->plugin->get_option( 'auto_rename', true ) ); ?>> <?php esc_html_e( 'Auto rename images', 'digital-rise-solutions' ); ?></label><br>
                                <label><input type="checkbox" name="auto_alt" <?php checked( $this->plugin->get_option( 'auto_alt', true ) ); ?>> <?php esc_html_e( 'Auto generate ALT text', 'digital-rise-solutions' ); ?></label><br>
                                <label><input type="checkbox" name="auto_description" <?php checked( $this->plugin->get_option( 'auto_description', true ) ); ?>> <?php esc_html_e( 'Auto generate description', 'digital-rise-solutions' ); ?></label>
                            </td>
                        </tr>
                    </table>
                </div>

                <p class="submit"><input type="submit" name="driop_save_keywords" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    public function render_watermark() {
        if ( isset( $_POST['driop_save_watermark'] ) && check_admin_referer( 'driop_watermark_nonce' ) ) {
            $options = $this->plugin->options;
            $options['watermark_enabled'] = isset( $_POST['watermark_enabled'] );
            $options['watermark_type'] = sanitize_text_field( $_POST['watermark_type'] );
            $options['watermark_text'] = sanitize_text_field( $_POST['watermark_text'] );
            $options['watermark_position'] = sanitize_text_field( $_POST['watermark_position'] );
            $options['watermark_opacity'] = min( 100, max( 0, absint( $_POST['watermark_opacity'] ) ) );
            $options['watermark_size_type'] = sanitize_text_field( $_POST['watermark_size_type'] );
            $options['watermark_size_percentage'] = min( 50, max( 5, absint( $_POST['watermark_size_percentage'] ) ) );
            $options['watermark_font_size'] = min( 100, max( 10, absint( $_POST['watermark_font_size'] ) ) );
            $this->plugin->save_options( $options );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-solutions' ) . '</p></div>';
        }

        $watermark = new DRIOP_Watermark( $this->plugin );
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Watermark Settings', 'digital-rise-solutions' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_watermark_nonce' ); ?>
                
                <div class="driop-card-row">
                    <div class="driop-card driop-card-half">
                        <h2><?php esc_html_e( 'Watermark Settings', 'digital-rise-solutions' ); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th><?php esc_html_e( 'Enable Watermark', 'digital-rise-solutions' ); ?></th>
                                <td><label><input type="checkbox" name="watermark_enabled" id="watermark_enabled" <?php checked( $this->plugin->get_option( 'watermark_enabled', false ) ); ?>> <?php esc_html_e( 'Add watermark to images', 'digital-rise-solutions' ); ?></label></td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Watermark Type', 'digital-rise-solutions' ); ?></th>
                                <td>
                                    <select name="watermark_type" id="watermark_type">
                                        <option value="site_name" <?php selected( $this->plugin->get_option( 'watermark_type' ), 'site_name' ); ?>><?php esc_html_e( 'Site Name', 'digital-rise-solutions' ); ?></option>
                                        <option value="custom" <?php selected( $this->plugin->get_option( 'watermark_type' ), 'custom' ); ?> <?php disabled( ! $this->plugin->has_feature( 'custom_watermark' ) ); ?>><?php esc_html_e( 'Custom Text', 'digital-rise-solutions' ); ?> <?php if ( ! $this->plugin->has_feature( 'custom_watermark' ) ) echo '(Pro)'; ?></option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Custom Text', 'digital-rise-solutions' ); ?></th>
                                <td><input type="text" name="watermark_text" id="watermark_text" value="<?php echo esc_attr( $this->plugin->get_option( 'watermark_text', '' ) ); ?>" class="regular-text" <?php disabled( ! $this->plugin->has_feature( 'custom_watermark' ) ); ?>></td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Position', 'digital-rise-solutions' ); ?></th>
                                <td>
                                    <select name="watermark_position" id="watermark_position">
                                        <option value="top-left" <?php selected( $this->plugin->get_option( 'watermark_position' ), 'top-left' ); ?>><?php esc_html_e( 'Top Left', 'digital-rise-solutions' ); ?></option>
                                        <option value="top-right" <?php selected( $this->plugin->get_option( 'watermark_position' ), 'top-right' ); ?>><?php esc_html_e( 'Top Right', 'digital-rise-solutions' ); ?></option>
                                        <option value="center" <?php selected( $this->plugin->get_option( 'watermark_position' ), 'center' ); ?>><?php esc_html_e( 'Center', 'digital-rise-solutions' ); ?></option>
                                        <option value="bottom-left" <?php selected( $this->plugin->get_option( 'watermark_position' ), 'bottom-left' ); ?>><?php esc_html_e( 'Bottom Left', 'digital-rise-solutions' ); ?></option>
                                        <option value="bottom-right" <?php selected( $this->plugin->get_option( 'watermark_position' ), 'bottom-right' ); ?>><?php esc_html_e( 'Bottom Right', 'digital-rise-solutions' ); ?></option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Opacity', 'digital-rise-solutions' ); ?></th>
                                <td>
                                    <input type="range" name="watermark_opacity" id="watermark_opacity" value="<?php echo esc_attr( $this->plugin->get_option( 'watermark_opacity', 50 ) ); ?>" min="0" max="100">
                                    <span id="opacity_value"><?php echo esc_html( $this->plugin->get_option( 'watermark_opacity', 50 ) ); ?>%</span>
                                </td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Size Type', 'digital-rise-solutions' ); ?></th>
                                <td>
                                    <select name="watermark_size_type" id="watermark_size_type" <?php disabled( ! $this->plugin->has_feature( 'watermark_size_control' ) ); ?>>
                                        <option value="percentage" <?php selected( $this->plugin->get_option( 'watermark_size_type' ), 'percentage' ); ?>><?php esc_html_e( 'Percentage of Image', 'digital-rise-solutions' ); ?></option>
                                        <option value="fixed" <?php selected( $this->plugin->get_option( 'watermark_size_type' ), 'fixed' ); ?>><?php esc_html_e( 'Fixed Font Size', 'digital-rise-solutions' ); ?></option>
                                    </select>
                                    <?php if ( ! $this->plugin->has_feature( 'watermark_size_control' ) ) : ?>
                                        <span class="driop-pro-badge">Pro</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <tr class="watermark-size-percentage">
                                <th><?php esc_html_e( 'Size (%)', 'digital-rise-solutions' ); ?></th>
                                <td>
                                    <input type="range" name="watermark_size_percentage" id="watermark_size_percentage" value="<?php echo esc_attr( $this->plugin->get_option( 'watermark_size_percentage', 15 ) ); ?>" min="5" max="50" <?php disabled( ! $this->plugin->has_feature( 'watermark_size_control' ) ); ?>>
                                    <span id="size_percentage_value"><?php echo esc_html( $this->plugin->get_option( 'watermark_size_percentage', 15 ) ); ?>%</span>
                                </td>
                            </tr>
                            <tr class="watermark-size-fixed" style="display:none;">
                                <th><?php esc_html_e( 'Font Size (px)', 'digital-rise-solutions' ); ?></th>
                                <td><input type="number" name="watermark_font_size" value="<?php echo esc_attr( $this->plugin->get_option( 'watermark_font_size', 24 ) ); ?>" min="10" max="100" <?php disabled( ! $this->plugin->has_feature( 'watermark_size_control' ) ); ?>></td>
                            </tr>
                        </table>
                    </div>

                    <div class="driop-card driop-card-half">
                        <h2><?php esc_html_e( 'Preview', 'digital-rise-solutions' ); ?></h2>
                        <div class="driop-watermark-preview">
                            <img src="<?php echo esc_url( $watermark->generate_preview() ); ?>" alt="Preview" id="watermark-preview">
                        </div>
                        <p><button type="button" id="refresh-preview" class="button"><?php esc_html_e( 'Refresh Preview', 'digital-rise-solutions' ); ?></button></p>
                    </div>
                </div>

                <p class="submit"><input type="submit" name="driop_save_watermark" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    public function render_plans() {
        $plans = new DRIOP_Plans( $this->plugin );
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Plans & Features', 'digital-rise-solutions' ) ); ?>
            
            <div class="driop-plans-grid">
                <?php foreach ( $plans->get_plans() as $plan_key => $plan ) : ?>
                    <div class="driop-plan-card <?php echo $this->plugin->current_plan === $plan_key ? 'driop-plan-current' : ''; ?>">
                        <h3><?php echo esc_html( $plan['name'] ); ?></h3>
                        <div class="driop-plan-price"><?php echo esc_html( $plan['price'] ); ?></div>
                        <p><?php echo esc_html( $plan['description'] ); ?></p>
                        <ul>
                            <?php foreach ( $plan['features'] as $feature ) : ?>
                                <li><span class="dashicons dashicons-yes"></span> <?php echo esc_html( $feature ); ?></li>
                            <?php endforeach; ?>
                            <?php if ( isset( $plan['limitations'] ) ) : ?>
                                <?php foreach ( $plan['limitations'] as $limitation ) : ?>
                                    <li class="limitation"><span class="dashicons dashicons-no"></span> <?php echo esc_html( $limitation ); ?></li>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </ul>
                        <?php if ( $this->plugin->current_plan === $plan_key ) : ?>
                            <span class="button disabled"><?php esc_html_e( 'Current Plan', 'digital-rise-solutions' ); ?></span>
                        <?php elseif ( $plan_key !== 'free' ) : ?>
                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Upgrade', 'digital-rise-solutions' ); ?></a>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }

    public function render_license() {
        $license = new DRIOP_License( $this->plugin );
        $status = $license->get_status();
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'License & Upgrade', 'digital-rise-solutions' ) ); ?>
            
            <div class="driop-card">
                <h2><?php esc_html_e( 'How to Upgrade', 'digital-rise-solutions' ); ?></h2>
                <div class="driop-upgrade-info">
                    <p><?php esc_html_e( 'To upgrade your plan, please contact us via email. Payment accepted via USDT.', 'digital-rise-solutions' ); ?></p>
                    <div class="driop-contact-box">
                        <strong><?php esc_html_e( 'Contact:', 'digital-rise-solutions' ); ?></strong>
                        <a href="mailto:<?php echo esc_attr( DRIOP_SUPPORT_EMAIL ); ?>"><?php echo esc_html( DRIOP_SUPPORT_EMAIL ); ?></a>
                    </div>
                    <p><?php esc_html_e( 'After payment, you will receive a license key. Enter it below to activate.', 'digital-rise-solutions' ); ?></p>
                </div>
            </div>

            <div class="driop-card">
                <h2><?php esc_html_e( 'License Activation', 'digital-rise-solutions' ); ?></h2>
                
                <div class="driop-license-status">
                    <p><strong><?php esc_html_e( 'Status:', 'digital-rise-solutions' ); ?></strong> 
                        <span class="driop-status-<?php echo esc_attr( $status['status'] ); ?>">
                            <?php echo esc_html( ucfirst( $status['status'] ) ); ?>
                        </span>
                    </p>
                    <p><strong><?php esc_html_e( 'Plan:', 'digital-rise-solutions' ); ?></strong> <?php echo esc_html( ucfirst( $status['plan'] ) ); ?></p>
                    <?php if ( $status['key'] ) : ?>
                        <p><strong><?php esc_html_e( 'Key:', 'digital-rise-solutions' ); ?></strong> <?php echo esc_html( $license->get_masked_key() ); ?></p>
                    <?php endif; ?>
                    <?php if ( $status['expires'] ) : ?>
                        <p><strong><?php esc_html_e( 'Expires:', 'digital-rise-solutions' ); ?></strong> <?php echo esc_html( $status['expires'] ); ?></p>
                    <?php endif; ?>
                </div>

                <form id="license-form">
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'License Key', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <input type="text" name="license_key" id="license_key" value="" class="regular-text" placeholder="<?php esc_attr_e( 'Enter your license key', 'digital-rise-solutions' ); ?>">
                            </td>
                        </tr>
                    </table>
                    <p>
                        <button type="button" id="activate-license" class="button button-primary"><?php esc_html_e( 'Activate License', 'digital-rise-solutions' ); ?></button>
                        <?php if ( $status['key'] ) : ?>
                            <button type="button" id="deactivate-license" class="button"><?php esc_html_e( 'Deactivate', 'digital-rise-solutions' ); ?></button>
                        <?php endif; ?>
                    </p>
                </form>
                <div id="license-message"></div>
            </div>

            <div class="driop-card driop-affiliate-card">
                <h2><?php esc_html_e( 'Support Development', 'digital-rise-solutions' ); ?></h2>
                <p><?php esc_html_e( 'If you find this plugin useful, please consider supporting its development.', 'digital-rise-solutions' ); ?></p>
                <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" rel="noopener" class="button button-primary button-hero" style="background: #2ecc71; border-color: #27ae60;"><?php esc_html_e( 'Donate with USDT', 'digital-rise-solutions' ); ?></a>
                <p style="margin-top: 15px;"><strong><?php esc_html_e( 'Need RedotPay for easy crypto payments?', 'digital-rise-solutions' ); ?></strong></p>
                <a href="https://url.hk/i/en/4bq59" target="_blank" rel="noopener" class="button"><?php esc_html_e( 'Get RedotPay', 'digital-rise-solutions' ); ?></a>
            </div>
        </div>
        <?php
    }

    public function render_settings() {
        if ( isset( $_POST['driop_save_settings'] ) && check_admin_referer( 'driop_settings_nonce' ) ) {
            $options = $this->plugin->options;
            $options['remote_connection'] = isset( $_POST['remote_connection'] );
            $options['show_attribution'] = isset( $_POST['show_attribution'] );
            $options['wc_integration'] = isset( $_POST['wc_integration'] );
            $options['dokan_integration'] = isset( $_POST['dokan_integration'] );
            $options['confirm_before_optimize'] = isset( $_POST['confirm_before_optimize'] );
            $options['convert_to_webp'] = isset( $_POST['convert_to_webp'] );
            $options['css_minify_enabled'] = isset( $_POST['css_minify_enabled'] );
            $options['js_minify_enabled'] = isset( $_POST['js_minify_enabled'] );
            $options['minify_exclude'] = sanitize_textarea_field( $_POST['minify_exclude'] ?? '' );
            $options['user_email'] = sanitize_email( $_POST['user_email'] ?? '' );
            $options['user_whatsapp'] = sanitize_text_field( $_POST['user_whatsapp'] ?? '' );
            $this->plugin->save_options( $options );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-solutions' ) . '</p></div>';
        }
        
        // Handle cache clear
        if ( isset( $_POST['driop_clear_cache'] ) && check_admin_referer( 'driop_settings_nonce' ) ) {
            $minifier = new DRIOP_Minifier( $this->plugin );
            $minifier->clear_cache();
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Cache cleared.', 'digital-rise-solutions' ) . '</p></div>';
        }
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'Settings', 'digital-rise-solutions' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_settings_nonce' ); ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Image Optimization Behavior', 'digital-rise-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Confirm Before Optimizing', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="confirm_before_optimize" <?php checked( $this->plugin->get_option( 'confirm_before_optimize', false ) ); ?>> <?php esc_html_e( 'Ask before automatically optimizing images', 'digital-rise-solutions' ); ?></label>
                                <p class="description"><?php esc_html_e( 'When enabled, images will NOT be auto-optimized on upload. Use Bulk Optimize to process images manually.', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Convert to WebP', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <?php 
                                $optimizer = new DRIOP_Image_Optimizer( $this->plugin );
                                $webp_supported = $optimizer->supports_webp();
                                ?>
                                <label>
                                    <input type="checkbox" name="convert_to_webp" <?php checked( $this->plugin->get_option( 'convert_to_webp', true ) ); ?> <?php disabled( ! $webp_supported ); ?>>
                                    <?php esc_html_e( 'Convert all uploaded images to WebP format', 'digital-rise-solutions' ); ?>
                                </label>
                                <?php if ( $webp_supported ) : ?>
                                    <p class="description" style="color: green;">✓ <?php esc_html_e( 'WebP is supported on your server.', 'digital-rise-solutions' ); ?></p>
                                <?php else : ?>
                                    <p class="description" style="color: red;">✗ <?php esc_html_e( 'WebP is NOT supported. Please update GD library with WebP support.', 'digital-rise-solutions' ); ?></p>
                                <?php endif; ?>
                                <p class="description"><?php esc_html_e( 'WebP provides 25-34% smaller file sizes compared to JPEG. Your image URLs will look like: yoursite.com/image-keyword.webp', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'CSS & JS Optimization', 'digital-rise-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Minify CSS', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="css_minify_enabled" <?php checked( $this->plugin->get_option( 'css_minify_enabled', false ) ); ?>> <?php esc_html_e( 'Enable CSS minification', 'digital-rise-solutions' ); ?></label>
                                <p class="description"><?php esc_html_e( 'Removes whitespace and comments from CSS files to reduce file size.', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Minify JavaScript', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="js_minify_enabled" <?php checked( $this->plugin->get_option( 'js_minify_enabled', false ) ); ?>> <?php esc_html_e( 'Enable JavaScript minification', 'digital-rise-solutions' ); ?></label>
                                <p class="description"><?php esc_html_e( 'Removes whitespace and comments from JS files. Use with caution.', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Exclude Handles', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <textarea name="minify_exclude" rows="4" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'minify_exclude', '' ) ); ?></textarea>
                                <p class="description"><?php esc_html_e( 'Enter script/style handles to exclude from minification (one per line). jQuery core is always excluded.', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Cache', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <?php 
                                $minifier = new DRIOP_Minifier( $this->plugin );
                                $cache_size = $minifier->get_cache_size();
                                ?>
                                <p><?php printf( esc_html__( 'Cache size: %s', 'digital-rise-solutions' ), size_format( $cache_size ) ); ?></p>
                                <button type="submit" name="driop_clear_cache" class="button"><?php esc_html_e( 'Clear Cache', 'digital-rise-solutions' ); ?></button>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Your Contact Information', 'digital-rise-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Optional: Provide your contact information to receive updates and support.', 'digital-rise-solutions' ); ?></p>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Email Address', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <input type="email" name="user_email" value="<?php echo esc_attr( $this->plugin->get_option( 'user_email', '' ) ); ?>" class="regular-text" placeholder="your@email.com">
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'WhatsApp Number', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <input type="text" name="user_whatsapp" value="<?php echo esc_attr( $this->plugin->get_option( 'user_whatsapp', '' ) ); ?>" class="regular-text" placeholder="+1234567890">
                                <p class="description"><?php esc_html_e( 'Include country code (e.g., +1 for USA)', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'General', 'digital-rise-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Remote Connection', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="remote_connection" <?php checked( $this->plugin->get_option( 'remote_connection', true ) ); ?>> <?php esc_html_e( 'Enable connection to Digital Rise Solutions for license validation', 'digital-rise-solutions' ); ?></label>
                                <p class="description"><?php esc_html_e( 'This is optional. Some features require this connection.', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Attribution', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <label><input type="checkbox" name="show_attribution" <?php checked( $this->plugin->get_option( 'show_attribution', true ) ); ?> <?php disabled( ! $this->plugin->has_feature( 'disable_attribution' ) && $this->plugin->current_plan === 'free' ); ?>> <?php esc_html_e( 'Show attribution link', 'digital-rise-solutions' ); ?></label>
                                <?php if ( ! $this->plugin->has_feature( 'disable_attribution' ) ) : ?>
                                    <p class="description"><?php esc_html_e( 'Upgrade to Pro to disable attribution.', 'digital-rise-solutions' ); ?></p>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Integrations', 'digital-rise-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'WooCommerce', 'digital-rise-solutions' ); ?></th>
                            <td><label><input type="checkbox" name="wc_integration" <?php checked( $this->plugin->get_option( 'wc_integration', true ) ); ?>> <?php esc_html_e( 'Enable WooCommerce integration', 'digital-rise-solutions' ); ?></label></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Dokan', 'digital-rise-solutions' ); ?></th>
                            <td><label><input type="checkbox" name="dokan_integration" <?php checked( $this->plugin->get_option( 'dokan_integration', true ) ); ?>> <?php esc_html_e( 'Enable Dokan integration', 'digital-rise-solutions' ); ?></label></td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'System Info', 'digital-rise-solutions' ); ?></h2>
                    <table class="form-table">
                        <tr><th><?php esc_html_e( 'Plugin Version', 'digital-rise-solutions' ); ?></th><td><?php echo esc_html( DRIOP_VERSION ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'WordPress', 'digital-rise-solutions' ); ?></th><td><?php echo esc_html( get_bloginfo( 'version' ) ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'PHP', 'digital-rise-solutions' ); ?></th><td><?php echo esc_html( phpversion() ); ?></td></tr>
                        <tr><th><?php esc_html_e( 'GD Library', 'digital-rise-solutions' ); ?></th><td><?php echo function_exists( 'imagecreatetruecolor' ) ? '✓' : '✗'; ?></td></tr>
                        <tr><th><?php esc_html_e( 'Installation ID', 'digital-rise-solutions' ); ?></th><td><code><?php echo esc_html( $this->plugin->get_option( 'installation_id' ) ); ?></code></td></tr>
                    </table>
                </div>
                
                <div class="driop-card" style="background: linear-gradient(135deg, #2ecc71, #27ae60); color: #fff;">
                    <h2 style="color: #fff;"><?php esc_html_e( 'Support This Plugin', 'digital-rise-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'If you find this plugin useful, please consider making a donation to support continued development.', 'digital-rise-solutions' ); ?></p>
                    <a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" rel="noopener" class="button button-hero" style="background: #fff; color: #27ae60; border: none;"><?php esc_html_e( 'Donate with USDT', 'digital-rise-solutions' ); ?></a>
                </div>

                <p class="submit"><input type="submit" name="driop_save_settings" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    // AJAX Handlers
    public function ajax_dismiss_notice() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        $notice = isset( $_POST['notice'] ) ? sanitize_text_field( $_POST['notice'] ) : '';
        if ( $notice === 'upgrade_banner' ) {
            update_option( 'driop_upgrade_dismissed', true );
        } else {
            $remote = new DRIOP_Remote( $this->plugin );
            $remote->dismiss_notice( $notice );
        }
        wp_send_json_success();
    }

    public function ajax_optimize_image() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        $attachment_id = absint( $_POST['attachment_id'] );
        $apply_watermark = isset( $_POST['apply_watermark'] ) && $_POST['apply_watermark'] === 'true';
        
        $optimizer = new DRIOP_Image_Optimizer( $this->plugin );
        $result = $optimizer->optimize_existing( $attachment_id, $apply_watermark );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result['message'] );
        }
    }

    public function ajax_bulk_optimize() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        $apply_watermark = isset( $_POST['apply_watermark'] ) && $_POST['apply_watermark'] === 'true';
        $apply_seo = isset( $_POST['apply_seo'] ) && $_POST['apply_seo'] === 'true';
        $reoptimize = isset( $_POST['reoptimize'] ) && $_POST['reoptimize'] === 'true';
        
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $result = $bulk->optimize_batch( 5, $apply_watermark, $apply_seo, $reoptimize );
        
        wp_send_json_success( $result );
    }

    public function ajax_reset_optimization() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $bulk->reset_optimization_status();
        
        wp_send_json_success( array( 'message' => __( 'All images marked as unoptimized.', 'digital-rise-solutions' ) ) );
    }

    public function ajax_activate_license() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        $license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( $_POST['license_key'] ) : '';
        
        $license = new DRIOP_License( $this->plugin );
        $result = $license->activate_license( $license_key );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result['message'] );
        }
    }

    public function ajax_deactivate_license() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $license = new DRIOP_License( $this->plugin );
        $result = $license->deactivate_license();
        
        wp_send_json_success( $result );
    }

    public function ajax_get_optimization_progress() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $bulk = new DRIOP_Bulk_Optimizer( $this->plugin );
        $progress = $bulk->get_progress();
        
        wp_send_json_success( $progress );
    }

    public function ajax_generate_watermark_preview() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $watermark = new DRIOP_Watermark( $this->plugin );
        $preview = $watermark->generate_preview();
        
        wp_send_json_success( array( 'preview' => $preview ) );
    }

    public function ajax_sync_notices() {
        check_ajax_referer( 'driop_admin_nonce', 'nonce' );
        
        $remote = new DRIOP_Remote( $this->plugin );
        $remote->do_checkin();
        
        $notices = $remote->get_active_notices();
        
        wp_send_json_success( array( 
            'message' => __( 'Notices synced successfully.', 'digital-rise-solutions' ),
            'count' => count( $notices ),
        ));
    }

    public function render_page_seo() {
        // Handle save
        if ( isset( $_POST['driop_save_page_seo'] ) && check_admin_referer( 'driop_page_seo_nonce' ) ) {
            $this->plugin->update_option( 'page_seo_enabled', isset( $_POST['page_seo_enabled'] ) );
            $this->plugin->update_option( 'default_title_pattern', sanitize_text_field( $_POST['default_title_pattern'] ) );
            $this->plugin->update_option( 'default_meta_pattern', sanitize_textarea_field( $_POST['default_meta_pattern'] ) );
            
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'digital-rise-solutions' ) . '</p></div>';
        }
        
        $page_seo = new DRIOP_Page_SEO( $this->plugin );
        $variables = $page_seo->get_available_variables();
        ?>
        <div class="wrap driop-wrap">
            <?php $this->render_header( __( 'SEO Titles & Meta Descriptions', 'digital-rise-solutions' ) ); ?>
            
            <form method="post">
                <?php wp_nonce_field( 'driop_page_seo_nonce' ); ?>
                
                <div class="driop-card">
                    <h2><?php esc_html_e( 'Page SEO Settings', 'digital-rise-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Configure automatic SEO titles and meta descriptions for all your content. You can also override these on individual posts, pages, and products.', 'digital-rise-solutions' ); ?></p>
                    
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Enable Page SEO', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="page_seo_enabled" <?php checked( $this->plugin->get_option( 'page_seo_enabled', true ) ); ?>>
                                    <?php esc_html_e( 'Automatically generate SEO title and meta description', 'digital-rise-solutions' ); ?>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Default Title Pattern', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <input type="text" name="default_title_pattern" value="<?php echo esc_attr( $this->plugin->get_option( 'default_title_pattern', '{title} | {site_name}' ) ); ?>" class="large-text">
                                <p class="description"><?php esc_html_e( 'Used when no custom title is set. Recommended: 50-60 characters.', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Default Meta Description', 'digital-rise-solutions' ); ?></th>
                            <td>
                                <textarea name="default_meta_pattern" rows="3" class="large-text"><?php echo esc_textarea( $this->plugin->get_option( 'default_meta_pattern', '{excerpt}' ) ); ?></textarea>
                                <p class="description"><?php esc_html_e( 'Used when no custom description is set. Recommended: 150-160 characters.', 'digital-rise-solutions' ); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'Available Variables', 'digital-rise-solutions' ); ?></h2>
                    <p><?php esc_html_e( 'Use these variables in your patterns. They will be replaced with actual values.', 'digital-rise-solutions' ); ?></p>
                    
                    <table class="widefat striped" style="max-width: 800px;">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Variable', 'digital-rise-solutions' ); ?></th>
                                <th><?php esc_html_e( 'Description', 'digital-rise-solutions' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $variables as $var => $desc ) : ?>
                                <tr>
                                    <td><code><?php echo esc_html( $var ); ?></code></td>
                                    <td><?php echo esc_html( $desc ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="driop-card">
                    <h2><?php esc_html_e( 'How It Works', 'digital-rise-solutions' ); ?></h2>
                    <ol>
                        <li><strong><?php esc_html_e( 'Default patterns:', 'digital-rise-solutions' ); ?></strong> <?php esc_html_e( 'Set above, applied to all content automatically.', 'digital-rise-solutions' ); ?></li>
                        <li><strong><?php esc_html_e( 'Custom override:', 'digital-rise-solutions' ); ?></strong> <?php esc_html_e( 'Edit any post, page, or product and scroll down to find the "SEO Settings" meta box.', 'digital-rise-solutions' ); ?></li>
                        <li><strong><?php esc_html_e( 'Categories & Tags:', 'digital-rise-solutions' ); ?></strong> <?php esc_html_e( 'Edit any category or tag to add custom SEO title and description.', 'digital-rise-solutions' ); ?></li>
                        <li><strong><?php esc_html_e( 'Products:', 'digital-rise-solutions' ); ?></strong> <?php esc_html_e( 'WooCommerce products and categories also support custom SEO settings.', 'digital-rise-solutions' ); ?></li>
                    </ol>
                    <p><strong><?php esc_html_e( 'Example patterns:', 'digital-rise-solutions' ); ?></strong></p>
                    <ul>
                        <li><code>{title} - {category} | {site_name}</code> → "My Product - Electronics | My Store"</li>
                        <li><code>Buy {title} at best price - {site_name}</code> → "Buy iPhone 15 at best price - My Store"</li>
                        <li><code>{title} by {author} - {date}</code> → "How to Code by John - January 2024"</li>
                    </ul>
                </div>

                <p class="submit"><input type="submit" name="driop_save_page_seo" class="button button-primary" value="<?php esc_attr_e( 'Save Settings', 'digital-rise-solutions' ); ?>"></p>
            </form>
        </div>
        <?php
    }
}
