<?php
/**
 * Page SEO - Title and Meta Description for Pages, Posts, Products, Categories, Tags
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Page_SEO {

    private $plugin;
    
    // Available dynamic variables
    private $variables = array(
        '{title}' => 'Page/Post Title',
        '{site_name}' => 'Site Name',
        '{site_tagline}' => 'Site Tagline',
        '{excerpt}' => 'Excerpt (first 160 chars)',
        '{category}' => 'Primary Category',
        '{tags}' => 'Tags (comma separated)',
        '{author}' => 'Author Name',
        '{date}' => 'Publish Date',
        '{year}' => 'Current Year',
        '{month}' => 'Current Month',
        '{product_price}' => 'Product Price (WooCommerce)',
        '{product_sku}' => 'Product SKU (WooCommerce)',
        '{keyword}' => 'Random Keyword from List',
        '{parent_title}' => 'Parent Page/Category Title',
        '{count}' => 'Number of Items (for archives)',
        '{spacer}' => 'Single Space',
        '{separator}' => 'Dash Separator ( - )',
    );

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }
    
    /**
     * Enqueue metabox assets
     */
    public function enqueue_metabox_assets( $hook ) {
        // Only on post/page/product edit screens
        if ( ! in_array( $hook, array( 'post.php', 'post-new.php' ) ) ) {
            return;
        }
        
        // Enqueue media uploader
        wp_enqueue_media();
        
        // Get current screen
        $screen = get_current_screen();
        
        // Add global product page CSS fixes (for WooCommerce)
        if ( $screen && $screen->post_type === 'product' ) {
            wp_add_inline_style( 'wp-admin', '
                /* CRITICAL: Fix white text on product pages */
                body.post-type-product #post-body-content #content,
                body.post-type-product #post-body-content #content_ifr,
                body.post-type-product #postdivrich #content,
                body.post-type-product .wp-editor-area,
                body.post-type-product #content.wp-editor-area,
                body.post-type-product #excerpt,
                body.post-type-product textarea#excerpt,
                body.post-type-product .mce-content-body {
                    color: #1d2327 !important;
                    background: #fff !important;
                }
                
                /* Fix TinyMCE iframe content */
                body.post-type-product #content_ifr {
                    background: #fff !important;
                }
            ' );
        }
        
        // Enqueue inline styles for SEO metabox - VERY SPECIFIC to avoid conflicts
        wp_add_inline_style( 'wp-admin', '
            #driop_page_seo.postbox { background: #fff; }
            #driop_page_seo .driop-seo-metabox { padding: 15px; background: #fff; }
            #driop_page_seo .driop-seo-field { margin-bottom: 20px; clear: both; }
            #driop_page_seo .driop-seo-field label { display: block; font-weight: 600; margin-bottom: 8px; color: #1d2327 !important; font-size: 14px; }
            #driop_page_seo .driop-seo-field input[type="text"],
            #driop_page_seo .driop-seo-field textarea { width: 100%; box-sizing: border-box; color: #1d2327 !important; background: #fff !important; }
            #driop_page_seo .driop-seo-field textarea { height: 80px; resize: vertical; }
            #driop_page_seo .driop-seo-field select { min-width: 150px; color: #1d2327 !important; }
            #driop_page_seo .driop-seo-field .description { color: #646970 !important; font-size: 13px; margin-top: 5px; }
            #driop_page_seo .driop-robots-fields { display: flex; gap: 20px; margin-top: 10px; }
            #driop_page_seo .driop-seo-variables { background: #f0f0f1; padding: 12px 15px; border-radius: 4px; margin-top: 15px; }
            #driop_page_seo .driop-seo-variables strong { color: #1d2327 !important; }
            #driop_page_seo .driop-seo-variables code { background: #fff; padding: 4px 8px; margin: 3px; display: inline-block; cursor: pointer; border-radius: 3px; color: #1d2327 !important; border: 1px solid #ddd; }
            #driop_page_seo .driop-seo-variables code:hover { background: #2271b1; color: #fff !important; border-color: #2271b1; }
            #driop_page_seo .driop-seo-preview { background: #fff; border: 1px solid #ddd; padding: 15px; margin-top: 15px; border-radius: 4px; }
            #driop_page_seo .driop-seo-preview strong { color: #1d2327 !important; }
            #driop_page_seo .driop-seo-preview-title { color: #1a0dab !important; font-size: 18px; margin-bottom: 5px; }
            #driop_page_seo .driop-seo-preview-url { color: #006621 !important; font-size: 14px; margin-bottom: 5px; }
            #driop_page_seo .driop-seo-preview-desc { color: #545454 !important; font-size: 14px; line-height: 1.4; }
            #driop_page_seo .driop-char-count { font-size: 12px; color: #646970 !important; margin-top: 5px; }
            #driop_page_seo .driop-char-count.warning { color: #d63638 !important; }
            #driop_page_seo .driop-og-image-preview img { display: block; }
            #driop_page_seo .driop-social-preview-card h4 { color: #1d2327 !important; }
            #driop_page_seo .driop-social-preview-card { background: #f8f9fa; }
            #driop_page_seo button.button { color: #2271b1 !important; }
            #driop_page_seo .driop-quick-vars { margin-top: 8px; }
            #driop_page_seo .driop-quick-vars code { cursor: pointer; padding: 3px 8px; background: #f0f0f1; border-radius: 3px; margin: 0 4px; display: inline-block; color: #1d2327 !important; border: 1px solid #ddd; transition: all 0.2s; }
            #driop_page_seo .driop-quick-vars code:hover { background: #2271b1; color: #fff !important; border-color: #2271b1; transform: translateY(-1px); }
        ' );
    }

    /**
     * Get available variables for display
     */
    public function get_available_variables() {
        return $this->variables;
    }

    /**
     * Add meta boxes to posts, pages, products
     */
    public function add_meta_boxes() {
        $post_types = array( 'post', 'page', 'product' );
        
        foreach ( $post_types as $post_type ) {
            // For products, use 'low' priority to appear AFTER WooCommerce metabox
            $priority = ( $post_type === 'product' ) ? 'low' : 'high';
            
            add_meta_box(
                'driop_page_seo',
                __( 'SEO Settings - Image Optimizer', 'digital-rise-image-optimizer-for-seo' ),
                array( $this, 'render_meta_box' ),
                $post_type,
                'normal',  // Keep normal position for all
                $priority  // Low priority for products (appears after WooCommerce)
            );
        }
    }

    /**
     * Render meta box
     */
    public function render_meta_box( $post ) {
        wp_nonce_field( 'driop_page_seo_nonce', 'driop_page_seo_nonce' );
        
        $seo_title = get_post_meta( $post->ID, '_driop_seo_title', true );
        $seo_description = get_post_meta( $post->ID, '_driop_seo_description', true );
        // PHASE 4 FIX #9: Robots meta
        $robots_index = get_post_meta( $post->ID, '_driop_robots_index', true );
        $robots_follow = get_post_meta( $post->ID, '_driop_robots_follow', true );
        
        // Set defaults if not set
        if ( '' === $robots_index ) {
            $robots_index = 'index';
        }
        if ( '' === $robots_follow ) {
            $robots_follow = 'follow';
        }
        
        $default_title = $this->plugin->get_option( 'default_title_pattern', '{title} | {site_name}' );
        $default_desc = $this->plugin->get_option( 'default_meta_pattern', '{excerpt}' );
        ?>
        <div class="driop-seo-metabox">
            <div class="driop-seo-field">
                <label for="driop_seo_title"><?php esc_html_e( 'SEO Title', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                <input type="text" id="driop_seo_title" name="driop_seo_title" value="<?php echo esc_attr( $seo_title ); ?>" placeholder="<?php echo esc_attr( $default_title ); ?>">
                <div class="driop-char-count" id="title-char-count">0 / 60 characters</div>
                <div class="driop-quick-vars" style="margin-top: 8px;">
                    <small style="color: #646970;"><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                    <code class="driop-insert-var" data-var="{title}">{title}</code>
                    <code class="driop-insert-var" data-var="{site_name}">{site_name}</code>
                    <?php if ( get_post_type() === 'product' ) : ?>
                        <code class="driop-insert-var" data-var="{product_price}">{product_price}</code>
                        <code class="driop-insert-var" data-var="{product_sku}">{product_sku}</code>
                    <?php endif; ?>
                    <code class="driop-insert-var" data-var="{category}">{category}</code>
                    <code class="driop-insert-var" data-var="{year}">{year}</code>
                    <code class="driop-insert-var" data-var="{spacer}" style="background: #e3f2fd; border-color: #2196f3;">{spacer}</code>
                    <code class="driop-insert-var" data-var="{separator}" style="background: #e3f2fd; border-color: #2196f3;">{separator}</code>
                </div>
            </div>
            
            <div class="driop-seo-field">
                <label for="driop_seo_description"><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                <textarea id="driop_seo_description" name="driop_seo_description" placeholder="<?php echo esc_attr( $default_desc ); ?>"><?php echo esc_textarea( $seo_description ); ?></textarea>
                <div class="driop-char-count" id="desc-char-count">0 / 160 characters</div>
                <div class="driop-quick-vars" style="margin-top: 8px;">
                    <small style="color: #646970;"><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                    <code class="driop-insert-var" data-var="{excerpt}">{excerpt}</code>
                    <code class="driop-insert-var" data-var="{title}">{title}</code>
                    <?php if ( get_post_type() === 'product' ) : ?>
                        <code class="driop-insert-var" data-var="{product_price}">{product_price}</code>
                        <code class="driop-insert-var" data-var="{product_sku}">{product_sku}</code>
                    <?php endif; ?>
                    <code class="driop-insert-var" data-var="{category}">{category}</code>
                    <code class="driop-insert-var" data-var="{site_name}">{site_name}</code>
                    <code class="driop-insert-var" data-var="{spacer}" style="background: #e3f2fd; border-color: #2196f3;">{spacer}</code>
                    <code class="driop-insert-var" data-var="{separator}" style="background: #e3f2fd; border-color: #2196f3;">{separator}</code>
                </div>
            </div>
            
            <!-- SEARCH PREVIEW BOX #1 -->
            <div class="driop-seo-preview driop-search-preview-box" style="background: #fff; border: 1px solid #ddd; padding: 15px; margin: 20px 0; border-radius: 4px;">
                <strong style="display: block; margin-bottom: 10px; color: #1d2327;">🔍 <?php esc_html_e( 'Search Engine Preview:', 'digital-rise-image-optimizer-for-seo' ); ?></strong>
                
                <?php
                // Get all images from this post/product - with error handling
                $post_images = array();
                
                // Only process if post exists and has ID
                if ( ! empty( $post ) && ! empty( $post->ID ) ) {
                    // Featured image
                    if ( has_post_thumbnail( $post->ID ) ) {
                        $thumb_url = get_the_post_thumbnail_url( $post->ID, 'medium' );
                        if ( $thumb_url ) {
                            $post_images[] = array(
                                'url' => $thumb_url,
                                'id' => get_post_thumbnail_id( $post->ID ),
                                'type' => 'featured'
                            );
                        }
                    }
                    
                    // Gallery images (for products) - with try/catch
                    if ( get_post_type( $post->ID ) === 'product' && function_exists( 'wc_get_product' ) ) {
                        try {
                            $product = wc_get_product( $post->ID );
                            if ( $product && is_object( $product ) && method_exists( $product, 'get_gallery_image_ids' ) ) {
                                $gallery_ids = $product->get_gallery_image_ids();
                                if ( is_array( $gallery_ids ) ) {
                                    foreach ( $gallery_ids as $img_id ) {
                                        $img_url = wp_get_attachment_image_url( $img_id, 'medium' );
                                        if ( $img_url ) {
                                            $post_images[] = array(
                                                'url' => $img_url,
                                                'id' => $img_id,
                                                'type' => 'gallery'
                                            );
                                        }
                                    }
                                }
                            }
                        } catch ( Exception $e ) {
                            // Silently fail - product might not be fully initialized
                        }
                    }
                    
                    // Images in content - with safety check
                    if ( ! empty( $post->post_content ) ) {
                        preg_match_all( '/<img[^>]+src=[\'"]([^\'"]+)[\'"][^>]*>/i', $post->post_content, $matches );
                        if ( ! empty( $matches[1] ) && is_array( $matches[1] ) ) {
                            foreach ( array_slice( $matches[1], 0, 5 ) as $img_url ) {
                                if ( ! empty( $img_url ) ) {
                                    $post_images[] = array(
                                        'url' => esc_url( $img_url ),
                                        'id' => 0,
                                        'type' => 'content'
                                    );
                                }
                            }
                        }
                    }
                }
                
                // Get current preview image (saved or random)
                $preview_image = get_post_meta( $post->ID, '_driop_preview_image', true );
                if ( empty( $preview_image ) && ! empty( $post_images ) ) {
                    $preview_image = $post_images[0]['url']; // Default to first image
                }
                ?>
                
                <div style="display: flex; gap: 15px; margin-bottom: 15px;">
                    <!-- Image Preview -->
                    <div id="search-preview-image-container" style="flex-shrink: 0;">
                        <?php if ( $preview_image ) : ?>
                            <img id="search-preview-image" src="<?php echo esc_url( $preview_image ); ?>" style="width: 120px; height: 120px; object-fit: cover; border-radius: 4px; border: 1px solid #ddd;">
                        <?php else : ?>
                            <div id="search-preview-image" style="width: 120px; height: 120px; background: #f0f0f1; border-radius: 4px; border: 1px solid #ddd; display: flex; align-items: center; justify-content: center; color: #666; font-size: 12px; text-align: center;">
                                No image
                            </div>
                        <?php endif; ?>
                        <input type="hidden" id="driop_preview_image" name="driop_preview_image" value="<?php echo esc_attr( $preview_image ); ?>">
                    </div>
                    
                    <!-- Preview Text -->
                    <div style="flex-grow: 1;">
                        <div class="driop-seo-preview-title" id="seo-preview-title" style="color: #1a0dab; font-size: 18px; margin-bottom: 5px; cursor: pointer;"><?php echo esc_html( $post->post_title ); ?></div>
                        <div class="driop-seo-preview-url" style="color: #006621; font-size: 14px; margin-bottom: 5px;"><?php echo esc_url( get_permalink( $post->ID ) ?: home_url( '/' ) ); ?></div>
                        <div class="driop-seo-preview-desc" id="seo-preview-desc" style="color: #545454; font-size: 14px; line-height: 1.4;"><?php echo esc_html( wp_trim_words( $post->post_excerpt ?: $post->post_content, 25 ) ); ?></div>
                    </div>
                </div>
                
                <!-- Image Selector -->
                <?php if ( ! empty( $post_images ) ) : ?>
                <div style="border-top: 1px solid #ddd; padding-top: 10px;">
                    <small style="color: #646970; display: block; margin-bottom: 8px;">
                        <?php esc_html_e( 'Select preview image:', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </small>
                    <div style="display: flex; gap: 8px; flex-wrap: wrap;">
                        <?php foreach ( array_slice( $post_images, 0, 6 ) as $img ) : ?>
                            <img src="<?php echo esc_url( $img['url'] ); ?>" 
                                 class="driop-preview-image-option" 
                                 data-url="<?php echo esc_attr( $img['url'] ); ?>"
                                 style="width: 60px; height: 60px; object-fit: cover; border-radius: 4px; cursor: pointer; border: 2px solid <?php echo $img['url'] === $preview_image ? '#2271b1' : '#ddd'; ?>;"
                                 title="<?php echo esc_attr( ucfirst( $img['type'] ) . ' image' ); ?>">
                        <?php endforeach; ?>
                        <button type="button" class="button button-small driop-choose-preview-image" style="height: 60px;">
                            <?php esc_html_e( 'Choose...', 'digital-rise-image-optimizer-for-seo' ); ?>
                        </button>
                    </div>
                </div>
                <?php else : ?>
                <div style="border-top: 1px solid #ddd; padding-top: 10px;">
                    <button type="button" class="button button-small driop-choose-preview-image">
                        <?php esc_html_e( 'Choose Preview Image', 'digital-rise-image-optimizer-for-seo' ); ?>
                    </button>
                </div>
                <?php endif; ?>
            </div>
            
            <div class="driop-seo-field">
                <label><?php esc_html_e( 'Robots Meta Tags', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                <div class="driop-robots-fields">
                    <div>
                        <label for="driop_robots_index" style="font-weight: normal;">
                            <select name="driop_robots_index" id="driop_robots_index">
                                <option value="index" <?php selected( $robots_index, 'index' ); ?>><?php esc_html_e( 'Index', 'digital-rise-image-optimizer-for-seo' ); ?></option>
                                <option value="noindex" <?php selected( $robots_index, 'noindex' ); ?>><?php esc_html_e( 'No Index', 'digital-rise-image-optimizer-for-seo' ); ?></option>
                            </select>
                        </label>
                        <p class="description"><?php esc_html_e( 'Allow search engines to index this page', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    </div>
                    <div>
                        <label for="driop_robots_follow" style="font-weight: normal;">
                            <select name="driop_robots_follow" id="driop_robots_follow">
                                <option value="follow" <?php selected( $robots_follow, 'follow' ); ?>><?php esc_html_e( 'Follow', 'digital-rise-image-optimizer-for-seo' ); ?></option>
                                <option value="nofollow" <?php selected( $robots_follow, 'nofollow' ); ?>><?php esc_html_e( 'No Follow', 'digital-rise-image-optimizer-for-seo' ); ?></option>
                            </select>
                        </label>
                        <p class="description"><?php esc_html_e( 'Allow search engines to follow links', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    </div>
                </div>
            </div>
            
            <!-- SOCIAL MEDIA PREVIEW (PRO/AGENCY FEATURE) -->
            <?php 
            $has_social_feature = $this->plugin->has_feature( 'social_preview' );
            $og_title = get_post_meta( $post->ID, '_driop_og_title', true );
            $og_description = get_post_meta( $post->ID, '_driop_og_description', true );
            $og_image = get_post_meta( $post->ID, '_driop_og_image', true );
            ?>
            <div class="driop-seo-field <?php echo ! $has_social_feature ? 'driop-locked-feature' : ''; ?>" style="<?php echo ! $has_social_feature ? 'opacity: 0.6; pointer-events: none;' : ''; ?>">
                <label style="display: flex; align-items: center; gap: 10px;">
                    <span style="font-size: 20px;">📱</span>
                    <?php esc_html_e( 'Social Media Appearance', 'digital-rise-image-optimizer-for-seo' ); ?>
                    <?php if ( ! $has_social_feature ) : ?>
                        <span class="driop-pro-badge" style="background: #ffd700; color: #333; padding: 2px 8px; border-radius: 3px; font-size: 11px; font-weight: 600;">PRO</span>
                    <?php endif; ?>
                </label>
                <p class="description" style="margin-bottom: 15px;">
                    <?php esc_html_e( 'Control how your content appears when shared on Facebook, Twitter, LinkedIn, WhatsApp, and other social platforms.', 'digital-rise-image-optimizer-for-seo' ); ?>
                </p>
                
                <div style="margin-bottom: 15px;">
                    <label for="driop_og_title"><?php esc_html_e( 'Social Title', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                    <input type="text" id="driop_og_title" name="driop_og_title" value="<?php echo esc_attr( $og_title ); ?>" placeholder="<?php echo esc_attr( $post->post_title ); ?>" <?php disabled( ! $has_social_feature ); ?>>
                    <p class="description"><?php esc_html_e( 'Leave empty to use SEO title', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    <?php if ( $has_social_feature ) : ?>
                    <div class="driop-quick-vars" style="margin-top: 8px;">
                        <small style="color: #646970;"><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                        <code class="driop-insert-var" data-var="{title}">{title}</code>
                        <code class="driop-insert-var" data-var="{site_name}">{site_name}</code>
                        <?php if ( get_post_type() === 'product' ) : ?>
                            <code class="driop-insert-var" data-var="{product_price}">{product_price}</code>
                        <?php endif; ?>
                        <code class="driop-insert-var" data-var="{spacer}" style="background: #e3f2fd; border-color: #2196f3;">{spacer}</code>
                        <code class="driop-insert-var" data-var="{separator}" style="background: #e3f2fd; border-color: #2196f3;">{separator}</code>
                    </div>
                    <?php endif; ?>
                </div>
                
                <div style="margin-bottom: 15px;">
                    <label for="driop_og_description"><?php esc_html_e( 'Social Description', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                    <textarea id="driop_og_description" name="driop_og_description" rows="3" placeholder="<?php echo esc_attr( wp_trim_words( $post->post_excerpt ? $post->post_excerpt : wp_strip_all_tags( $post->post_content ), 30 ) ); ?>" <?php disabled( ! $has_social_feature ); ?>><?php echo esc_textarea( $og_description ); ?></textarea>
                    <p class="description"><?php esc_html_e( 'Leave empty to use meta description', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                    <?php if ( $has_social_feature ) : ?>
                    <div class="driop-quick-vars" style="margin-top: 8px;">
                        <small style="color: #646970;"><?php esc_html_e( 'Quick insert:', 'digital-rise-image-optimizer-for-seo' ); ?></small>
                        <code class="driop-insert-var" data-var="{excerpt}">{excerpt}</code>
                        <code class="driop-insert-var" data-var="{title}">{title}</code>
                        <?php if ( get_post_type() === 'product' ) : ?>
                            <code class="driop-insert-var" data-var="{product_price}">{product_price}</code>
                        <?php endif; ?>
                        <code class="driop-insert-var" data-var="{spacer}" style="background: #e3f2fd; border-color: #2196f3;">{spacer}</code>
                        <code class="driop-insert-var" data-var="{separator}" style="background: #e3f2fd; border-color: #2196f3;">{separator}</code>
                    </div>
                    <?php endif; ?>
                </div>
                
                <div style="margin-bottom: 15px;">
                    <label><?php esc_html_e( 'Social Image', 'digital-rise-image-optimizer-for-seo' ); ?></label>
                    <div style="display: flex; gap: 15px; align-items: flex-start;">
                        <div>
                            <button type="button" class="button driop-upload-og-image" <?php disabled( ! $has_social_feature ); ?>>
                                <?php esc_html_e( 'Choose Image', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </button>
                            <button type="button" class="button driop-remove-og-image" <?php disabled( ! $has_social_feature ); ?> style="<?php echo empty( $og_image ) ? 'display:none;' : ''; ?>">
                                <?php esc_html_e( 'Remove', 'digital-rise-image-optimizer-for-seo' ); ?>
                            </button>
                            <input type="hidden" id="driop_og_image" name="driop_og_image" value="<?php echo esc_attr( $og_image ); ?>">
                        </div>
                        <div class="driop-og-image-preview" style="<?php echo empty( $og_image ) ? 'display:none;' : ''; ?>">
                            <img src="<?php echo esc_url( $og_image ); ?>" style="max-width: 200px; max-height: 150px; border: 1px solid #ddd; border-radius: 4px;">
                        </div>
                    </div>
                    <p class="description"><?php esc_html_e( 'Recommended: 1200x630px. Leave empty to use featured image.', 'digital-rise-image-optimizer-for-seo' ); ?></p>
                </div>
                
                <?php if ( $has_social_feature ) : ?>
                <div class="driop-social-preview-card" style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin-top: 20px;">
                    <h4 style="margin-top: 0;"><?php esc_html_e( 'Preview:', 'digital-rise-image-optimizer-for-seo' ); ?></h4>
                    
                    <!-- Facebook/LinkedIn Style -->
                    <div style="border: 1px solid #dadde1; border-radius: 8px; overflow: hidden; max-width: 500px; background: white;">
                        <div class="driop-social-preview-image" style="background: #f0f2f5; height: 260px; display: flex; align-items: center; justify-content: center; color: #65676b;">
                            <?php if ( ! empty( $og_image ) ) : ?>
                                <img src="<?php echo esc_url( $og_image ); ?>" style="width: 100%; height: 100%; object-fit: cover;">
                            <?php else : ?>
                                <span>🖼️ <?php esc_html_e( 'Image Preview', 'digital-rise-image-optimizer-for-seo' ); ?></span>
                            <?php endif; ?>
                        </div>
                        <div style="padding: 12px; border-top: 1px solid #dadde1;">
                            <div class="driop-social-preview-title" style="font-size: 16px; font-weight: 600; color: #1c1e21; margin-bottom: 5px;">
                                <?php echo esc_html( ! empty( $og_title ) ? $og_title : $post->post_title ); ?>
                            </div>
                            <div class="driop-social-preview-desc" style="font-size: 14px; color: #65676b; line-height: 1.4;">
                                <?php echo esc_html( ! empty( $og_description ) ? wp_trim_words( $og_description, 20 ) : wp_trim_words( $post->post_excerpt ? $post->post_excerpt : wp_strip_all_tags( $post->post_content ), 20 ) ); ?>
                            </div>
                        </div>
                    </div>
                </div>
                <?php else : ?>
                <div style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 12px; margin-top: 15px; border-radius: 0 4px 4px 0;">
                    <strong>🔒 <?php esc_html_e( 'Upgrade to Pro', 'digital-rise-image-optimizer-for-seo' ); ?></strong><br>
                    <?php esc_html_e( 'Get control over social media appearance, increase clicks and engagement!', 'digital-rise-image-optimizer-for-seo' ); ?>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=driop-license' ) ); ?>" class="button button-primary" style="margin-left: 10px;"><?php esc_html_e( 'Upgrade Now', 'digital-rise-image-optimizer-for-seo' ); ?></a>
                </div>
                <?php endif; ?>
            </div>
            
            <div class="driop-seo-variables">
                <strong><?php esc_html_e( 'Available Variables (click to insert):', 'digital-rise-image-optimizer-for-seo' ); ?></strong><br>
                <?php foreach ( $this->variables as $var => $label ) : ?>
                    <code class="driop-insert-var" data-var="<?php echo esc_attr( $var ); ?>" title="<?php echo esc_attr( $label ); ?>"><?php echo esc_html( $var ); ?></code>
                <?php endforeach; ?>
            </div>
            
            <script>
            console.log('DRIOP: Script tag loaded!');
            console.log('DRIOP: jQuery available?', typeof jQuery);
            
            jQuery(document).ready(function($) {
                console.log('DRIOP: Document ready fired!');
                
                // Track last focused field GLOBALLY (outside setTimeout for event delegation to access)
                var lastFocused = null;
                console.log('DRIOP: lastFocused initialized');
                
                // Use a timeout to ensure WooCommerce has finished loading
                setTimeout(function() {
                    console.log('DRIOP: setTimeout fired (500ms passed)');
                    
                    // Variable replacement values - safely escaped
                    var variableValues = {
                        '{title}': '<?php echo esc_js( $post->post_title ); ?>',
                        '{site_name}': '<?php echo esc_js( get_bloginfo( 'name' ) ); ?>',
                        '{site_tagline}': '<?php echo esc_js( get_bloginfo( 'description' ) ); ?>',
                        '{excerpt}': '<?php echo esc_js( wp_trim_words( $post->post_excerpt ? $post->post_excerpt : wp_strip_all_tags( $post->post_content ), 25 ) ); ?>',
                        '{category}': '<?php $cats = get_the_category( $post->ID ); echo $cats ? esc_js( $cats[0]->name ) : ''; ?>',
                        '{tags}': '<?php $tags = get_the_tags( $post->ID ); echo $tags ? esc_js( implode( ', ', wp_list_pluck( array_slice( $tags, 0, 3 ), 'name' ) ) ) : ''; ?>',
                        '{author}': '<?php echo esc_js( get_the_author_meta( 'display_name', $post->post_author ) ); ?>',
                        '{date}': '<?php echo esc_js( get_the_date( '', $post->ID ) ); ?>',
                        '{year}': '<?php echo esc_js( gmdate( 'Y' ) ); ?>',
                        '{month}': '<?php echo esc_js( gmdate( 'F' ) ); ?>',
                        '{product_price}': '<?php if ( function_exists( 'wc_get_product' ) ) { try { $prod = wc_get_product( $post->ID ); if ( $prod ) { $price = $prod->get_price(); $currency = get_woocommerce_currency_symbol(); echo esc_js( $currency . ' ' . number_format( (float) $price, 3 ) ); } } catch (Exception $e) { echo ''; } } ?>',
                        '{product_sku}': '<?php if ( function_exists( 'wc_get_product' ) ) { try { $prod = wc_get_product( $post->ID ); echo $prod ? esc_js( $prod->get_sku() ) : ''; } catch (Exception $e) { echo ''; } } ?>',
                        '{keyword}': 'Your Keyword',
                        '{parent_title}': '<?php echo $post->post_parent ? esc_js( get_the_title( $post->post_parent ) ) : ''; ?>',
                        '{count}': '',
                        '{spacer}': ' ',
                        '{separator}': ' - '
                    };
                    console.log('DRIOP: variableValues initialized:', Object.keys(variableValues).length, 'variables');
                    
                // Decode HTML entities (like &#x62f; to Arabic د)
                function decodeEntities(text) {
                    var textarea = document.createElement('textarea');
                    textarea.innerHTML = text;
                    return textarea.value;
                }
                    
                function replaceVariables(text) {
                    var result = text;
                    for (var variable in variableValues) {
                        if (variableValues.hasOwnProperty(variable)) {
                            var value = variableValues[variable];
                            if (value) {
                                // Decode HTML entities before replacing
                                value = decodeEntities(value);
                                result = result.replace(new RegExp(variable.replace(/[{}]/g, '\\$&'), 'g'), value);
                            }
                        }
                    }
                    // Remove any remaining unreplaced variables
                    result = result.replace(/\{[^}]+\}/g, '');
                    return result;
                }
                
                function updateCharCount() {
                    // FIX #3: Count ACTUAL text length (after variable replacement), not pattern length
                    var titleInput = $('#driop_seo_title').val() || '<?php echo esc_js( $post->post_title ); ?>';
                    var descInput = $('#driop_seo_description').val() || '<?php echo esc_js( wp_trim_words( $post->post_excerpt ? $post->post_excerpt : wp_strip_all_tags( $post->post_content ), 25 ) ); ?>';
                    
                    // Replace variables to get actual text
                    var titleActual = replaceVariables(titleInput);
                    var descActual = replaceVariables(descInput);
                    
                    // Count actual character length (what search engines see)
                    var titleLen = titleActual.length;
                    var descLen = descActual.length;
                    
                    $('#title-char-count').text(titleLen + ' / 60 characters').toggleClass('warning', titleLen > 60);
                    $('#desc-char-count').text(descLen + ' / 160 characters').toggleClass('warning', descLen > 160);
                    
                    // Update BOTH preview boxes with actual text
                    $('#seo-preview-title, .driop-seo-preview-title-2').text(titleActual);
                    $('#seo-preview-desc, .driop-seo-preview-desc-2').text(descActual);
                }
                
                // SEARCH PREVIEW IMAGE: Select from thumbnails
                $('.driop-preview-image-option').on('click', function() {
                    var imageUrl = $(this).data('url');
                    $('#driop_preview_image').val(imageUrl);
                    
                    // Update preview
                    var $img = $('#search-preview-image');
                    if ($img.is('img')) {
                        $img.attr('src', imageUrl);
                    } else {
                        $('#search-preview-image-container').html('<img id="search-preview-image" src="' + imageUrl + '" style="width: 120px; height: 120px; object-fit: cover; border-radius: 4px; border: 1px solid #ddd;">');
                    }
                    
                    // Update borders
                    $('.driop-preview-image-option').css('border-color', '#ddd');
                    $(this).css('border-color', '#2271b1');
                });
                
                // SEARCH PREVIEW IMAGE: Choose from media library
                var previewImageUploader;
                $('.driop-choose-preview-image').on('click', function(e) {
                    e.preventDefault();
                    
                    if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
                        alert('<?php esc_html_e( 'Media uploader not available. Please refresh the page.', 'digital-rise-image-optimizer-for-seo' ); ?>');
                        return;
                    }
                    
                    if (previewImageUploader) {
                        previewImageUploader.open();
                        return;
                    }
                    
                    previewImageUploader = wp.media({
                        title: '<?php esc_html_e( 'Choose Preview Image', 'digital-rise-image-optimizer-for-seo' ); ?>',
                        button: {
                            text: '<?php esc_html_e( 'Use this image', 'digital-rise-image-optimizer-for-seo' ); ?>'
                        },
                        multiple: false
                    });
                    
                    previewImageUploader.on('select', function() {
                        var attachment = previewImageUploader.state().get('selection').first().toJSON();
                        $('#driop_preview_image').val(attachment.url);
                        
                        // Update preview
                        var $img = $('#search-preview-image');
                        if ($img.is('img')) {
                            $img.attr('src', attachment.url);
                        } else {
                            $('#search-preview-image-container').html('<img id="search-preview-image" src="' + attachment.url + '" style="width: 120px; height: 120px; object-fit: cover; border-radius: 4px; border: 1px solid #ddd;">');
                        }
                        
                        // Update borders
                        $('.driop-preview-image-option').css('border-color', '#ddd');
                    });
                    
                    previewImageUploader.open();
                });
                
                $('#driop_seo_title, #driop_seo_description').on('input', updateCharCount);
                updateCharCount();
                
                // Track focused field for variable insertion (update the global variable)
                $('#driop_seo_title, #driop_seo_description, #driop_og_title, #driop_og_description').on('focus', function() {
                    lastFocused = this;
                });
                
                // NOTE: Variable insertion click handler is outside setTimeout using event delegation
                
                // SOCIAL MEDIA: Media uploader
                var ogImageUploader;
                $('.driop-upload-og-image').on('click', function(e) {
                    e.preventDefault();
                    
                    // Check if wp.media is available
                    if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
                        alert('<?php esc_html_e( 'Media uploader not available. Please refresh the page.', 'digital-rise-image-optimizer-for-seo' ); ?>');
                        return;
                    }
                    
                    if (ogImageUploader) {
                        ogImageUploader.open();
                        return;
                    }
                    
                    ogImageUploader = wp.media({
                        title: '<?php esc_html_e( 'Choose Social Image', 'digital-rise-image-optimizer-for-seo' ); ?>',
                        button: {
                            text: '<?php esc_html_e( 'Use this image', 'digital-rise-image-optimizer-for-seo' ); ?>'
                        },
                        multiple: false
                    });
                    
                    ogImageUploader.on('select', function() {
                        var attachment = ogImageUploader.state().get('selection').first().toJSON();
                        $('#driop_og_image').val(attachment.url);
                        $('.driop-og-image-preview img').attr('src', attachment.url);
                        $('.driop-og-image-preview').show();
                        $('.driop-remove-og-image').show();
                        $('.driop-social-preview-image').html('<img src="' + attachment.url + '" style="width: 100%; height: 100%; object-fit: cover;">');
                        updateSocialPreview(); // Update preview after image selection
                    });
                    
                    ogImageUploader.open();
                });
                
                // Remove social image
                $('.driop-remove-og-image').on('click', function(e) {
                    e.preventDefault();
                    $('#driop_og_image').val('');
                    $('.driop-og-image-preview').hide();
                    $(this).hide();
                    $('.driop-social-preview-image').html('<span>🖼️ <?php esc_html_e( 'Image Preview', 'digital-rise-image-optimizer-for-seo' ); ?></span>');
                });
                
                // Update social preview on input
                function updateSocialPreview() {
                    var title = $('#driop_og_title').val() || '<?php echo esc_js( $post->post_title ); ?>';
                    var desc = $('#driop_og_description').val() || '<?php echo esc_js( wp_trim_words( $post->post_excerpt ? $post->post_excerpt : wp_strip_all_tags( $post->post_content ), 20 ) ); ?>';
                    
                    // Replace variables in social preview too
                    var titleActual = replaceVariables(title);
                    var descActual = replaceVariables(desc);
                    
                    $('.driop-social-preview-title').text(titleActual);
                    $('.driop-social-preview-desc').text(descActual);
                }
                
                $('#driop_og_title, #driop_og_description').on('input', updateSocialPreview);
                updateSocialPreview(); // Initialize on load
                
                }, 500); // Wait 500ms for WooCommerce to finish loading
                
                // Use event delegation for variable insertion (works even after WooCommerce loads)
                $(document).on('click', '.driop-insert-var', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    
                    console.log('DRIOP: Variable clicked!', this);
                    
                    var varText = $(this).data('var');
                    console.log('DRIOP: Variable text:', varText);
                    
                    if (!varText) {
                        console.error('DRIOP: No variable text found!');
                        return;
                    }
                    
                    var $activeField = null;
                    
                    // Strategy 1: Use currently focused field
                    $activeField = $('#driop_seo_title, #driop_seo_description, #driop_og_title, #driop_og_description').filter(':focus');
                    console.log('DRIOP: Strategy 1 (focused field):', $activeField.length, $activeField.attr('id'));
                    
                    // Strategy 2: Use last focused field (from global variable)
                    if ($activeField.length === 0 && typeof lastFocused !== 'undefined' && lastFocused && $(lastFocused).length) {
                        $activeField = $(lastFocused);
                        console.log('DRIOP: Strategy 2 (lastFocused):', $activeField.length, $activeField.attr('id'));
                    }
                    
                    // Strategy 3: Find field by looking up the DOM tree
                    if ($activeField.length === 0) {
                        console.log('DRIOP: Trying Strategy 3 - looking up DOM tree');
                        
                        // Get the parent of this button
                        var $parent = $(this).parent();
                        console.log('DRIOP: Parent element:', $parent.prop('tagName'), $parent.attr('class'));
                        
                        // Look for driop-quick-vars class
                        var $quickVars = $(this).closest('.driop-quick-vars');
                        console.log('DRIOP: Found quick-vars:', $quickVars.length);
                        
                        if ($quickVars.length > 0) {
                            // Get the parent field container
                            var $fieldContainer = $quickVars.closest('.driop-seo-field');
                            console.log('DRIOP: Found field container:', $fieldContainer.length);
                            
                            if ($fieldContainer.length > 0) {
                                // Find input or textarea in this container
                                $activeField = $fieldContainer.find('input[type="text"], textarea').first();
                                console.log('DRIOP: Strategy 3A (field container):', $activeField.length, $activeField.attr('id'));
                            }
                            
                            // Alternative: Look for siblings
                            if ($activeField.length === 0) {
                                $activeField = $quickVars.siblings('input[type="text"], textarea').first();
                                console.log('DRIOP: Strategy 3B (siblings):', $activeField.length, $activeField.attr('id'));
                            }
                            
                            // Alternative: Look before quick-vars
                            if ($activeField.length === 0) {
                                $activeField = $quickVars.prevAll('input[type="text"], textarea').first();
                                console.log('DRIOP: Strategy 3C (prevAll):', $activeField.length, $activeField.attr('id'));
                            }
                        }
                    }
                    
                    // Strategy 4: Default to SEO title
                    if ($activeField.length === 0) {
                        $activeField = $('#driop_seo_title');
                        console.log('DRIOP: Strategy 4 (default to SEO title):', $activeField.length, $activeField.attr('id'));
                    }
                    
                    // Insert the variable
                    if ($activeField.length > 0) {
                        console.log('DRIOP: Target field found:', $activeField.attr('id'));
                        console.log('DRIOP: Current field value:', $activeField.val());
                        
                        var field = $activeField[0];
                        var currentVal = $activeField.val() || '';
                        var pos = field.selectionStart !== undefined ? field.selectionStart : currentVal.length;
                        
                        console.log('DRIOP: Cursor position:', pos);
                        
                        var newVal = currentVal.substring(0, pos) + varText + currentVal.substring(pos);
                        console.log('DRIOP: New value will be:', newVal);
                        
                        $activeField.val(newVal);
                        console.log('DRIOP: Value set, checking:', $activeField.val());
                        
                        // Update cursor position
                        $activeField.focus();
                        if (field.setSelectionRange) {
                            field.setSelectionRange(pos + varText.length, pos + varText.length);
                        }
                        
                        // Update lastFocused
                        if (typeof lastFocused !== 'undefined') {
                            lastFocused = field;
                        }
                        
                        // CRITICAL: Trigger input event to update ALL previews
                        console.log('DRIOP: Triggering input event');
                        $activeField.trigger('input');
                        $activeField.trigger('change');
                        
                        // Also manually call update functions to be sure
                        setTimeout(function() {
                            if (typeof updateCharCount === 'function') {
                                console.log('DRIOP: Calling updateCharCount');
                                updateCharCount();
                            } else {
                                console.warn('DRIOP: updateCharCount not found');
                            }
                            if (typeof updateSocialPreview === 'function') {
                                console.log('DRIOP: Calling updateSocialPreview');
                                updateSocialPreview();
                            } else {
                                console.warn('DRIOP: updateSocialPreview not found');
                            }
                        }, 100);
                        
                        console.log('DRIOP: ✓ Variable insertion complete!');
                    } else {
                        console.error('DRIOP: ✗ No field found to insert into!');
                    }
                });
                
                console.log('DRIOP: ✓ Event delegation handler registered');
            });
            </script>
        </div>
        <?php
    }

    /**
     * Save meta box data
     */
    public function save_meta_box( $post_id ) {
        if ( ! isset( $_POST['driop_page_seo_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['driop_page_seo_nonce'] ) ), 'driop_page_seo_nonce' ) ) {
            return;
        }
        
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }
        
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }
        
        if ( isset( $_POST['driop_seo_title'] ) ) {
            update_post_meta( $post_id, '_driop_seo_title', sanitize_text_field( wp_unslash( $_POST['driop_seo_title'] ) ) );
        }
        
        if ( isset( $_POST['driop_seo_description'] ) ) {
            update_post_meta( $post_id, '_driop_seo_description', sanitize_textarea_field( wp_unslash( $_POST['driop_seo_description'] ) ) );
        }
        
        // Save preview image
        if ( isset( $_POST['driop_preview_image'] ) ) {
            update_post_meta( $post_id, '_driop_preview_image', esc_url_raw( wp_unslash( $_POST['driop_preview_image'] ) ) );
        }
        
        // PHASE 4 FIX #9: Save robots meta
        if ( isset( $_POST['driop_robots_index'] ) ) {
            $robots_index = sanitize_text_field( wp_unslash( $_POST['driop_robots_index'] ) );
            if ( in_array( $robots_index, array( 'index', 'noindex' ), true ) ) {
                update_post_meta( $post_id, '_driop_robots_index', $robots_index );
            }
        }
        
        if ( isset( $_POST['driop_robots_follow'] ) ) {
            $robots_follow = sanitize_text_field( wp_unslash( $_POST['driop_robots_follow'] ) );
            if ( in_array( $robots_follow, array( 'follow', 'nofollow' ), true ) ) {
                update_post_meta( $post_id, '_driop_robots_follow', $robots_follow );
            }
        }
        
        // Save social media fields (Pro/Agency only)
        if ( $this->plugin->has_feature( 'social_preview' ) ) {
            if ( isset( $_POST['driop_og_title'] ) ) {
                update_post_meta( $post_id, '_driop_og_title', sanitize_text_field( wp_unslash( $_POST['driop_og_title'] ) ) );
            }
            
            if ( isset( $_POST['driop_og_description'] ) ) {
                update_post_meta( $post_id, '_driop_og_description', sanitize_textarea_field( wp_unslash( $_POST['driop_og_description'] ) ) );
            }
            
            if ( isset( $_POST['driop_og_image'] ) ) {
                update_post_meta( $post_id, '_driop_og_image', esc_url_raw( wp_unslash( $_POST['driop_og_image'] ) ) );
            }
        }
    }

    /**
     * Add fields to taxonomy add form
     */
    public function add_term_fields( $taxonomy ) {
        ?>
        <div class="form-field">
            <label for="driop_term_seo_title"><?php esc_html_e( 'SEO Title', 'digital-rise-image-optimizer-for-seo' ); ?></label>
            <input type="text" name="driop_term_seo_title" id="driop_term_seo_title" value="">
            <p class="description"><?php esc_html_e( 'Custom title for search engines. Use variables like {title}, {site_name}, {count}', 'digital-rise-image-optimizer-for-seo' ); ?></p>
        </div>
        <div class="form-field">
            <label for="driop_term_seo_description"><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></label>
            <textarea name="driop_term_seo_description" id="driop_term_seo_description" rows="3"></textarea>
            <p class="description"><?php esc_html_e( 'Custom description for search engines (max 160 characters recommended)', 'digital-rise-image-optimizer-for-seo' ); ?></p>
        </div>
        <?php
    }

    /**
     * Add fields to taxonomy edit form
     */
    public function edit_term_fields( $term ) {
        $seo_title = get_term_meta( $term->term_id, '_driop_seo_title', true );
        $seo_description = get_term_meta( $term->term_id, '_driop_seo_description', true );
        ?>
        <tr class="form-field">
            <th scope="row"><label for="driop_term_seo_title"><?php esc_html_e( 'SEO Title', 'digital-rise-image-optimizer-for-seo' ); ?></label></th>
            <td>
                <input type="text" name="driop_term_seo_title" id="driop_term_seo_title" value="<?php echo esc_attr( $seo_title ); ?>">
                <p class="description"><?php esc_html_e( 'Custom title for search engines. Variables: {title}, {site_name}, {count}', 'digital-rise-image-optimizer-for-seo' ); ?></p>
            </td>
        </tr>
        <tr class="form-field">
            <th scope="row"><label for="driop_term_seo_description"><?php esc_html_e( 'Meta Description', 'digital-rise-image-optimizer-for-seo' ); ?></label></th>
            <td>
                <textarea name="driop_term_seo_description" id="driop_term_seo_description" rows="3"><?php echo esc_textarea( $seo_description ); ?></textarea>
                <p class="description"><?php esc_html_e( 'Custom description for search engines (max 160 characters)', 'digital-rise-image-optimizer-for-seo' ); ?></p>
            </td>
        </tr>
        <?php
    }

    /**
     * Save term meta
     */
    public function save_term_meta( $term_id, $tt_id, $taxonomy ) {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified by WordPress core for term save
        if ( isset( $_POST['driop_term_seo_title'] ) ) {
            update_term_meta( $term_id, '_driop_seo_title', sanitize_text_field( wp_unslash( $_POST['driop_term_seo_title'] ) ) );
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified by WordPress core for term save
        if ( isset( $_POST['driop_term_seo_description'] ) ) {
            update_term_meta( $term_id, '_driop_seo_description', sanitize_textarea_field( wp_unslash( $_POST['driop_term_seo_description'] ) ) );
        }
    }

    /**
     * Output meta tags in head
     */
    public function output_meta_tags() {
        $description = $this->get_meta_description();
        
        if ( ! empty( $description ) ) {
            // Decode HTML entities to prevent double-escaping
            echo '<meta name="description" content="' . esc_attr( html_entity_decode( $description, ENT_QUOTES, 'UTF-8' ) ) . '">' . "\n";
        }
        
        // PHASE 4 FIX #9: Robots meta tags
        $robots = $this->get_robots_meta();
        if ( ! empty( $robots ) ) {
            echo '<meta name="robots" content="' . esc_attr( $robots ) . '">' . "\n";
        }
        
        // SOCIAL MEDIA: Get custom or default values
        $og_title = '';
        $og_description = '';
        $og_image = '';
        
        if ( is_singular() ) {
            global $post;
            $og_title = get_post_meta( $post->ID, '_driop_og_title', true );
            $og_description = get_post_meta( $post->ID, '_driop_og_description', true );
            $og_image = get_post_meta( $post->ID, '_driop_og_image', true );
            
            // Parse variables in custom social media fields
            if ( ! empty( $og_title ) ) {
                $og_title = $this->parse_pattern( $og_title );
            }
            if ( ! empty( $og_description ) ) {
                $og_description = $this->parse_pattern( $og_description );
            }
            
            // Fallbacks
            if ( empty( $og_title ) ) {
                $og_title = $this->get_seo_title();
            }
            if ( empty( $og_description ) ) {
                $og_description = $description;
            }
            if ( empty( $og_image ) && has_post_thumbnail( $post->ID ) ) {
                $og_image = get_the_post_thumbnail_url( $post->ID, 'full' );
            }
        } else {
            $og_title = $this->get_seo_title();
            $og_description = $description;
        }
        
        // Open Graph tags - decode HTML entities
        if ( ! empty( $og_title ) ) {
            echo '<meta property="og:title" content="' . esc_attr( html_entity_decode( $og_title, ENT_QUOTES, 'UTF-8' ) ) . '">' . "\n";
        }
        if ( ! empty( $og_description ) ) {
            echo '<meta property="og:description" content="' . esc_attr( html_entity_decode( $og_description, ENT_QUOTES, 'UTF-8' ) ) . '">' . "\n";
        }
        if ( ! empty( $og_image ) ) {
            echo '<meta property="og:image" content="' . esc_url( $og_image ) . '">' . "\n";
            echo '<meta property="og:image:width" content="1200">' . "\n";
            echo '<meta property="og:image:height" content="630">' . "\n";
        }
        echo '<meta property="og:type" content="' . ( is_singular() ? 'article' : 'website' ) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url( get_permalink() ) . '">' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr( get_bloginfo( 'name' ) ) . '">' . "\n";
        
        // Twitter Card tags
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        if ( ! empty( $og_title ) ) {
            echo '<meta name="twitter:title" content="' . esc_attr( $og_title ) . '">' . "\n";
        }
        if ( ! empty( $og_description ) ) {
            echo '<meta name="twitter:description" content="' . esc_attr( $og_description ) . '">' . "\n";
        }
        if ( ! empty( $og_image ) ) {
            echo '<meta name="twitter:image" content="' . esc_url( $og_image ) . '">' . "\n";
        }
    }
    
    /**
     * PHASE 4 FIX #9: Get robots meta content
     */
    private function get_robots_meta() {
        $robots = array();
        
        if ( is_singular() ) {
            global $post;
            $index = get_post_meta( $post->ID, '_driop_robots_index', true );
            $follow = get_post_meta( $post->ID, '_driop_robots_follow', true );
            
            // Default to index, follow if not set
            $index = $index ? $index : 'index';
            $follow = $follow ? $follow : 'follow';
            
            $robots[] = $index;
            $robots[] = $follow;
        }
        
        return ! empty( $robots ) ? implode( ', ', $robots ) : '';
    }

    /**
     * Filter document title
     */
    public function filter_title( $title ) {
        $seo_title = $this->get_seo_title();
        // Decode HTML entities to prevent double-escaping (WordPress will escape again)
        return ! empty( $seo_title ) ? html_entity_decode( $seo_title, ENT_QUOTES, 'UTF-8' ) : $title;
    }

    /**
     * Filter title parts
     */
    public function filter_title_parts( $title_parts ) {
        $seo_title = $this->get_seo_title();
        if ( ! empty( $seo_title ) ) {
            // Decode HTML entities to prevent double-escaping
            return array( 'title' => html_entity_decode( $seo_title, ENT_QUOTES, 'UTF-8' ) );
        }
        return $title_parts;
    }

    /**
     * Get SEO title for current page
     */
    public function get_seo_title() {
        $pattern = '';
        
        if ( is_singular() ) {
            global $post;
            // First check for custom override
            $pattern = get_post_meta( $post->ID, '_driop_seo_title', true );
            if ( empty( $pattern ) ) {
                // Use content-type specific pattern
                $post_type = get_post_type();
                if ( $post_type === 'product' ) {
                    $pattern = $this->plugin->get_option( 'seo_title_product', '{title} - Buy Online | {site_name}' );
                } elseif ( $post_type === 'page' ) {
                    $pattern = $this->plugin->get_option( 'seo_title_page', '{title} | {site_name}' );
                } else {
                    $pattern = $this->plugin->get_option( 'seo_title_post', '{title} | {site_name}' );
                }
            }
        } elseif ( is_category() ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_title', true );
            if ( empty( $pattern ) ) {
                $pattern = $this->plugin->get_option( 'seo_title_category', '{title} Archives | {site_name}' );
            }
        } elseif ( is_tag() ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_title', true );
            if ( empty( $pattern ) ) {
                $pattern = $this->plugin->get_option( 'seo_title_tag', 'Posts Tagged "{title}" | {site_name}' );
            }
        } elseif ( is_tax( 'product_cat' ) ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_title', true );
            if ( empty( $pattern ) ) {
                $pattern = $this->plugin->get_option( 'seo_title_product_cat', '{title} - Shop {count} Products | {site_name}' );
            }
        } elseif ( is_tax() ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_title', true );
            if ( empty( $pattern ) ) {
                $pattern = '{title} | {site_name}';
            }
        } elseif ( is_home() || is_front_page() ) {
            $pattern = $this->plugin->get_option( 'seo_title_home', '{site_name} - {site_tagline}' );
        } elseif ( is_archive() ) {
            $pattern = $this->plugin->get_option( 'seo_title_archive', 'Archives: {date} | {site_name}' );
        }
        
        return $this->parse_pattern( $pattern );
    }

    /**
     * Get meta description for current page
     */
    public function get_meta_description() {
        $pattern = '';
        
        if ( is_singular() ) {
            global $post;
            // First check for custom override
            $pattern = get_post_meta( $post->ID, '_driop_seo_description', true );
            if ( empty( $pattern ) ) {
                // Use content-type specific pattern
                $post_type = get_post_type();
                if ( $post_type === 'product' ) {
                    $pattern = $this->plugin->get_option( 'seo_meta_product', 'Buy {title} at best price. {excerpt}' );
                } elseif ( $post_type === 'page' ) {
                    $pattern = $this->plugin->get_option( 'seo_meta_page', '{excerpt}' );
                } else {
                    $pattern = $this->plugin->get_option( 'seo_meta_post', '{excerpt}' );
                }
            }
        } elseif ( is_category() ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_description', true );
            if ( empty( $pattern ) ) {
                $pattern = $this->plugin->get_option( 'seo_meta_category', 'Browse all posts in {title}. {count} articles available.' );
            }
        } elseif ( is_tag() ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_description', true );
            if ( empty( $pattern ) ) {
                $pattern = $this->plugin->get_option( 'seo_meta_tag', 'All posts tagged with {title}. {count} articles available.' );
            }
        } elseif ( is_tax( 'product_cat' ) ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_description', true );
            if ( empty( $pattern ) ) {
                $pattern = $this->plugin->get_option( 'seo_meta_product_cat', 'Browse our {title} collection. {count} products available at {site_name}.' );
            }
        } elseif ( is_tax() ) {
            $term = get_queried_object();
            $pattern = get_term_meta( $term->term_id, '_driop_seo_description', true );
            if ( empty( $pattern ) ) {
                $pattern = term_description( $term->term_id );
            }
        } elseif ( is_home() || is_front_page() ) {
            $pattern = $this->plugin->get_option( 'seo_meta_home', '{site_tagline}' );
        } elseif ( is_archive() ) {
            $pattern = $this->plugin->get_option( 'seo_meta_archive', 'Archive of posts from {date} on {site_name}.' );
        }
        
        $description = $this->parse_pattern( $pattern );
        return wp_trim_words( wp_strip_all_tags( $description ), 30 );
    }

    /**
     * Parse pattern and replace variables
     */
    public function parse_pattern( $pattern ) {
        if ( empty( $pattern ) ) {
            return '';
        }
        
        $replacements = array(
            '{site_name}' => get_bloginfo( 'name' ),
            '{site_tagline}' => get_bloginfo( 'description' ),
            '{year}' => wp_date( 'Y' ),
            '{month}' => wp_date( 'F' ),
        );
        
        // Get keyword
        $keywords = new DRIOP_Keywords( $this->plugin );
        $replacements['{keyword}'] = $keywords->get_random_keyword();
        
        if ( is_singular() ) {
            global $post;
            $replacements['{title}'] = get_the_title( $post->ID );
            $replacements['{excerpt}'] = $this->get_excerpt( $post );
            $replacements['{author}'] = get_the_author_meta( 'display_name', $post->post_author );
            $replacements['{date}'] = get_the_date( '', $post );
            
            // Categories
            $categories = get_the_category( $post->ID );
            $replacements['{category}'] = ! empty( $categories ) ? $categories[0]->name : '';
            
            // Tags
            $tags = get_the_tags( $post->ID );
            $replacements['{tags}'] = ! empty( $tags ) ? implode( ', ', wp_list_pluck( $tags, 'name' ) ) : '';
            
            // Parent
            $parent_id = $post->post_parent;
            $replacements['{parent_title}'] = $parent_id ? get_the_title( $parent_id ) : '';
            
            // WooCommerce
            if ( $post->post_type === 'product' && function_exists( 'wc_get_product' ) ) {
                $product = wc_get_product( $post->ID );
                if ( $product ) {
                    // Get plain text price without HTML
                    $price = $product->get_price();
                    $currency = get_woocommerce_currency_symbol();
                    $replacements['{product_price}'] = $currency . ' ' . number_format( (float) $price, 3 );
                    $replacements['{product_sku}'] = $product->get_sku();
                    
                    // Product categories
                    $product_cats = get_the_terms( $post->ID, 'product_cat' );
                    if ( $product_cats && ! is_wp_error( $product_cats ) ) {
                        $replacements['{category}'] = $product_cats[0]->name;
                    }
                }
            }
        } elseif ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            $replacements['{title}'] = $term->name;
            $replacements['{excerpt}'] = $term->description;
            $replacements['{count}'] = $term->count;
            
            // Parent term
            if ( $term->parent ) {
                $parent = get_term( $term->parent );
                $replacements['{parent_title}'] = $parent ? $parent->name : '';
            } else {
                $replacements['{parent_title}'] = '';
            }
        }
        
        // Formatting variables (always available)
        $replacements['{spacer}'] = ' ';
        $replacements['{separator}'] = ' - ';
        
        $result = str_replace( array_keys( $replacements ), array_values( $replacements ), $pattern );
        
        // Clean up empty variables
        $result = preg_replace( '/\{[^}]+\}/', '', $result );
        $result = preg_replace( '/\s{2,}/', ' ', $result );
        $result = trim( $result, ' |,-' );
        
        return $result;
    }

    /**
     * Get excerpt for post
     */
    private function get_excerpt( $post ) {
        if ( ! empty( $post->post_excerpt ) ) {
            return $post->post_excerpt;
        }
        
        $content = strip_shortcodes( $post->post_content );
        $content = wp_strip_all_tags( $content );
        $content = str_replace( array( "\r", "\n", "\t" ), ' ', $content );
        
        return wp_trim_words( $content, 25 );
    }
}