<?php
/**
 * CSS and JS Minification
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Minifier {

    private $plugin;
    private $cache_dir;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        $this->cache_dir = WP_CONTENT_DIR . '/cache/driop-minify/';
        
        // Create cache directory if it doesn't exist
        if ( ! file_exists( $this->cache_dir ) ) {
            wp_mkdir_p( $this->cache_dir );
        }
    }

    /**
     * Initialize minification hooks
     */
    public function init() {
        if ( is_admin() || $this->is_page_builder() ) {
            return;
        }
        
        // CSS Minification
        if ( $this->plugin->get_option( 'css_minify_enabled', false ) ) {
            add_filter( 'style_loader_tag', array( $this, 'minify_css' ), 10, 4 );
        }
        
        // JS Minification
        if ( $this->plugin->get_option( 'js_minify_enabled', false ) ) {
            add_filter( 'script_loader_tag', array( $this, 'minify_js' ), 10, 3 );
        }
    }

    /**
     * Check if we're in a page builder
     */
    private function is_page_builder() {
        $builders = array(
            'elementor-preview',
            'fl_builder',
            'ct_builder',
            'vc_editable',
            'preview',
        );
        
        foreach ( $builders as $builder ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking for builder context, no data processing
            if ( isset( $_GET[ $builder ] ) ) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if file should be excluded
     */
    private function is_excluded( $handle ) {
        $excludes = $this->plugin->get_option( 'minify_exclude', '' );
        $exclude_list = array_filter( array_map( 'trim', explode( "\n", $excludes ) ) );
        
        // Always exclude admin and jQuery core
        $exclude_list[] = 'jquery-core';
        $exclude_list[] = 'jquery-migrate';
        $exclude_list[] = 'admin-bar';
        
        return in_array( $handle, $exclude_list );
    }

    /**
     * Minify CSS
     */
    public function minify_css( $html, $handle, $href, $media ) {
        if ( $this->is_excluded( $handle ) ) {
            return $html;
        }
        
        // Only process local files
        if ( strpos( $href, home_url() ) === false && strpos( $href, '/' ) !== 0 ) {
            return $html;
        }
        
        // Get file path
        $file_path = $this->url_to_path( $href );
        
        if ( ! $file_path || ! file_exists( $file_path ) ) {
            return $html;
        }
        
        // Check if already minified
        if ( strpos( $file_path, '.min.css' ) !== false ) {
            return $html;
        }
        
        // Generate cache file
        $cache_file = $this->get_cache_path( $file_path, 'css' );
        $cache_url = $this->get_cache_url( $cache_file );
        
        if ( ! file_exists( $cache_file ) || filemtime( $file_path ) > filemtime( $cache_file ) ) {
            $content = file_get_contents( $file_path );
            $minified = $this->minify_css_content( $content );
            file_put_contents( $cache_file, $minified );
        }
        
        return str_replace( $href, $cache_url, $html );
    }

    /**
     * Minify JS
     */
    public function minify_js( $tag, $handle, $src ) {
        if ( $this->is_excluded( $handle ) ) {
            return $tag;
        }
        
        // Only process local files
        if ( strpos( $src, home_url() ) === false && strpos( $src, '/' ) !== 0 ) {
            return $tag;
        }
        
        // Get file path
        $file_path = $this->url_to_path( $src );
        
        if ( ! $file_path || ! file_exists( $file_path ) ) {
            return $tag;
        }
        
        // Check if already minified
        if ( strpos( $file_path, '.min.js' ) !== false ) {
            return $tag;
        }
        
        // Generate cache file
        $cache_file = $this->get_cache_path( $file_path, 'js' );
        $cache_url = $this->get_cache_url( $cache_file );
        
        if ( ! file_exists( $cache_file ) || filemtime( $file_path ) > filemtime( $cache_file ) ) {
            $content = file_get_contents( $file_path );
            $minified = $this->minify_js_content( $content );
            file_put_contents( $cache_file, $minified );
        }
        
        return str_replace( $src, $cache_url, $tag );
    }

    /**
     * Minify CSS content
     */
    private function minify_css_content( $css ) {
        // Remove comments
        $css = preg_replace( '!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css );
        
        // Remove whitespace
        $css = preg_replace( '/\s+/', ' ', $css );
        
        // Remove spaces around special characters
        $css = preg_replace( '/\s*([\{\}\:\;\,])\s*/', '$1', $css );
        
        // Remove trailing semicolons before closing braces
        $css = str_replace( ';}', '}', $css );
        
        // Remove newlines
        $css = str_replace( array( "\r\n", "\r", "\n" ), '', $css );
        
        return trim( $css );
    }

    /**
     * Minify JS content (basic - safe minification)
     */
    private function minify_js_content( $js ) {
        // Remove single-line comments (but not in strings)
        $js = preg_replace( '#^\s*//.*$#m', '', $js );
        
        // Remove multi-line comments
        $js = preg_replace( '#/\*.*?\*/#s', '', $js );
        
        // Remove unnecessary whitespace
        $js = preg_replace( '/\s+/', ' ', $js );
        
        // Remove spaces around operators (basic, safe)
        $js = preg_replace( '/\s*([=\+\-\*\/\{\}\(\)\[\]\;\,\:\<\>])\s*/', '$1', $js );
        
        // Restore necessary spaces
        $js = preg_replace( '/([a-zA-Z0-9_\$])(return|var|let|const|function|if|else|for|while|do|switch|case|break|continue|new|typeof|instanceof|in|of|throw|try|catch|finally)([^a-zA-Z0-9_\$])/', '$1 $2$3', $js );
        
        return trim( $js );
    }

    /**
     * Convert URL to file path
     */
    private function url_to_path( $url ) {
        // Remove query string
        $url = preg_replace( '/\?.*/', '', $url );
        
        // Handle relative URLs
        if ( strpos( $url, '/' ) === 0 ) {
            return ABSPATH . ltrim( $url, '/' );
        }
        
        // Handle full URLs
        $site_url = site_url();
        if ( strpos( $url, $site_url ) === 0 ) {
            $path = str_replace( $site_url, ABSPATH, $url );
            return $path;
        }
        
        $home_url = home_url();
        if ( strpos( $url, $home_url ) === 0 ) {
            $path = str_replace( $home_url, ABSPATH, $url );
            return $path;
        }
        
        return false;
    }

    /**
     * Get cache file path
     */
    private function get_cache_path( $original_path, $type ) {
        $hash = md5( $original_path . filemtime( $original_path ) );
        $filename = basename( $original_path, '.' . $type ) . '-' . $hash . '.min.' . $type;
        return $this->cache_dir . $filename;
    }

    /**
     * Get cache URL
     */
    private function get_cache_url( $cache_path ) {
        return str_replace( WP_CONTENT_DIR, WP_CONTENT_URL, $cache_path );
    }

    /**
     * Clear cache
     */
    public function clear_cache() {
        $files = glob( $this->cache_dir . '*' );
        foreach ( $files as $file ) {
            if ( is_file( $file ) ) {
                wp_delete_file( $file );
            }
        }
        return true;
    }

    /**
     * Get cache size
     */
    public function get_cache_size() {
        $size = 0;
        $files = glob( $this->cache_dir . '*' );
        foreach ( $files as $file ) {
            if ( is_file( $file ) ) {
                $size += filesize( $file );
            }
        }
        return $size;
    }
}