<?php
/**
 * Image Optimizer - SIMPLE VERSION
 * ONLY: WebP conversion + Rename
 * NO resize, NO compression
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Image_Optimizer {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Handle upload - Rename, Resize, Compress, Convert to WebP
     */
    public function handle_upload( $upload, $context = 'upload' ) {
        // Only process images
        if ( ! isset( $upload['type'] ) || strpos( $upload['type'], 'image/' ) !== 0 ) {
            return $upload;
        }

        // Skip GIFs (animated)
        if ( $upload['type'] === 'image/gif' ) {
            return $upload;
        }
        
        // Check if user wants to confirm before optimizing
        if ( $this->plugin->get_option( 'confirm_before_optimize', false ) ) {
            // User wants to manually optimize - skip auto-optimization
            return $upload;
        }

        $file_path = $upload['file'];
        
        // Step 1: Rename file if enabled
        if ( $this->plugin->get_option( 'auto_rename', true ) ) {
            $upload = $this->rename_file( $upload );
            $file_path = $upload['file'];
        }
        
        // Step 2: Resize if exceeds max dimensions
        $upload = $this->resize_image( $upload );
        $file_path = $upload['file'];
        
        // Step 3: Compress to target size if enabled
        $upload = $this->compress_image( $upload );
        $file_path = $upload['file'];

        // Step 4: Convert to WebP if enabled
        if ( $this->plugin->get_option( 'convert_to_webp', true ) && $this->supports_webp() ) {
            if ( $upload['type'] !== 'image/webp' ) {
                $upload = $this->convert_to_webp_simple( $upload );
            }
        }

        return $upload;
    }

    /**
     * Process attachment metadata - just update meta, no image changes
     */
    public function process_attachment( $metadata, $attachment_id ) {
        if ( empty( $metadata ) || ! is_array( $metadata ) ) {
            return $metadata;
        }

        // Store original info
        $file = get_attached_file( $attachment_id );
        if ( $file && file_exists( $file ) ) {
            update_post_meta( $attachment_id, '_driop_original_size', filesize( $file ) );
            // FIX #2: Set BOTH meta keys to ensure stats count properly
            update_post_meta( $attachment_id, '_driop_processed', 1 );
            update_post_meta( $attachment_id, '_driop_optimized', 1 );
            
            // Clear sitemap cache so new images appear immediately
            delete_transient( 'driop_sitemap_images_cache' );
            delete_transient( 'driop_sitemap_index_cache' );
        }

        // Increment user count
        $user_id = get_current_user_id();
        if ( $user_id ) {
            $this->increment_user_image_count( $user_id );
        }

        return $metadata;
    }

    /**
     * Simple rename - just change filename, no image manipulation
     * Uses simple sequential numbering: site-name-1, site-name-2, etc.
     * MAX 6 WORDS for SEO
     * CHECKS DATABASE to avoid conflicts with existing images
     */
    private function rename_file( $upload ) {
        $file_path = $upload['file'];
        $path_info = pathinfo( $file_path );
        
        // Build SEO-friendly base name (max 6 words)
        $site_name = $this->get_seo_name_max_words( 6 );
        
        if ( empty( $site_name ) ) {
            $site_name = 'image';
        }

        $extension = isset( $path_info['extension'] ) ? strtolower( $path_info['extension'] ) : 'jpg';
        $upload_dir = $path_info['dirname'];
        
        // Find the next available number by checking BOTH filesystem AND database
        $counter = $this->get_next_available_number( $site_name, $extension, $upload_dir );
        
        $new_filename = $site_name . '-' . $counter . '.' . $extension;
        $new_path = $upload_dir . '/' . $new_filename;
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Renaming uploaded file before WordPress processes it
        if ( @rename( $file_path, $new_path ) ) {
            $upload['file'] = $new_path;
            $upload['url'] = str_replace( basename( $upload['url'] ), $new_filename, $upload['url'] );
        }

        return $upload;
    }

    /**
     * Get next available number by checking filesystem AND database
     * This prevents conflicts with existing images
     */
    private function get_next_available_number( $base_name, $extension, $upload_dir ) {
        global $wpdb;
        
        // Get highest number from database for this base name pattern
        $pattern = $base_name . '-%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Finding existing filenames
        $existing = $wpdb->get_col( $wpdb->prepare(
            "SELECT meta_value FROM {$wpdb->postmeta} 
             WHERE meta_key = '_wp_attached_file' 
             AND meta_value LIKE %s",
            '%' . $wpdb->esc_like( $pattern ) . '%'
        ) );
        
        $max_number = 0;
        
        // Find highest number in database
        foreach ( $existing as $file ) {
            $filename = basename( $file );
            if ( preg_match( '/' . preg_quote( $base_name, '/' ) . '-(\d+)/', $filename, $matches ) ) {
                $num = (int) $matches[1];
                if ( $num > $max_number ) {
                    $max_number = $num;
                }
            }
        }
        
        // Also check filesystem to be safe
        $counter = $max_number + 1;
        $new_filename = $base_name . '-' . $counter . '.' . $extension;
        $new_path = $upload_dir . '/' . $new_filename;
        
        while ( file_exists( $new_path ) ) {
            $counter++;
            $new_filename = $base_name . '-' . $counter . '.' . $extension;
            $new_path = $upload_dir . '/' . $new_filename;
        }
        
        return $counter;
    }

    /**
     * Get SEO name with max words limit
     * Combines site name + keyword, limited to max words
     */
    private function get_seo_name_max_words( $max_words = 6 ) {
        $site_name = sanitize_title( get_bloginfo( 'name' ) );
        
        // Get keyword if available
        $keywords = new DRIOP_Keywords( $this->plugin );
        $keyword = $keywords->get_random_keyword();
        
        // Combine site name and keyword
        $parts = array();
        if ( ! empty( $site_name ) ) {
            $parts[] = $site_name;
        }
        if ( ! empty( $keyword ) ) {
            $parts[] = sanitize_title( $keyword );
        }
        
        $full_name = implode( '-', $parts );
        
        // Split into words and limit
        $words = explode( '-', $full_name );
        $words = array_slice( $words, 0, $max_words );
        
        return implode( '-', $words );
    }

    /**
     * Convert to WebP - HIGH QUALITY, preserve dimensions
     */
    private function convert_to_webp_simple( $upload ) {
        $file_path = $upload['file'];
        
        if ( ! file_exists( $file_path ) ) {
            return $upload;
        }

        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return $upload;
        }

        $width = $image_info[0];
        $height = $image_info[1];
        $mime = $image_info['mime'];
        

        // Skip if already WebP or GIF
        if ( $mime === 'image/webp' || $mime === 'image/gif' ) {
            return $upload;
        }

        // Create image resource
        $source = null;
        switch ( $mime ) {
            case 'image/jpeg':
                $source = @imagecreatefromjpeg( $file_path );
                break;
            case 'image/png':
                $source = @imagecreatefrompng( $file_path );
                break;
        }

        if ( ! $source ) {
            return $upload;
        }

        // Verify source dimensions match original
        $source_width = imagesx( $source );
        $source_height = imagesy( $source );

        // Prepare output path
        $path_info = pathinfo( $file_path );
        $webp_path = $path_info['dirname'] . '/' . $path_info['filename'] . '.webp';

        // Handle PNG transparency
        if ( $mime === 'image/png' ) {
            imagepalettetotruecolor( $source );
            imagealphablending( $source, true );
            imagesavealpha( $source, true );
        }

        // Save as WebP with HIGH quality (100 = best quality)
        $quality = 100;
        $success = @imagewebp( $source, $webp_path, $quality );
        
        // Clean up
        imagedestroy( $source );

        if ( $success && file_exists( $webp_path ) ) {
            // Verify WebP dimensions
            $webp_info = @getimagesize( $webp_path );
            if ( $webp_info ) {
                
                // Only use WebP if dimensions match!
                if ( $webp_info[0] === $width && $webp_info[1] === $height ) {
                    // Delete original
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Deleting original file after WebP conversion
                    @unlink( $file_path );
                    
                    // Update upload info
                    $upload['file'] = $webp_path;
                    $upload['url'] = str_replace( $path_info['basename'], $path_info['filename'] . '.webp', $upload['url'] );
                    $upload['type'] = 'image/webp';
                } else {
                    // Dimensions don't match - something went wrong, keep original
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Cleaning up failed WebP conversion
                    @unlink( $webp_path );
                }
            }
        }

        return $upload;
    }
    
    /**
     * Resize image if it exceeds max dimensions
     */
    private function resize_image( $upload ) {
        $max_width = $this->plugin->get_option( 'max_width', 0 );
        $max_height = $this->plugin->get_option( 'max_height', 0 );
        
        // Skip if no limits set
        if ( ! $max_width && ! $max_height ) {
            return $upload;
        }
        
        $file_path = $upload['file'];
        $image_info = @getimagesize( $file_path );
        
        if ( ! $image_info ) {
            return $upload;
        }
        
        $current_width = $image_info[0];
        $current_height = $image_info[1];
        $mime_type = $image_info['mime'];
        
        // Check if resize needed
        $needs_resize = false;
        if ( $max_width > 0 && $current_width > $max_width ) {
            $needs_resize = true;
        }
        if ( $max_height > 0 && $current_height > $max_height ) {
            $needs_resize = true;
        }
        
        if ( ! $needs_resize ) {
            return $upload;
        }
        
        // Calculate new dimensions (maintain aspect ratio)
        $ratio = $current_width / $current_height;
        
        if ( $max_width > 0 && $max_height > 0 ) {
            // Both limits set - use the most restrictive
            if ( $current_width / $max_width > $current_height / $max_height ) {
                $new_width = $max_width;
                $new_height = (int) ( $max_width / $ratio );
            } else {
                $new_height = $max_height;
                $new_width = (int) ( $max_height * $ratio );
            }
        } elseif ( $max_width > 0 ) {
            $new_width = $max_width;
            $new_height = (int) ( $max_width / $ratio );
        } else {
            $new_height = $max_height;
            $new_width = (int) ( $max_height * $ratio );
        }
        
        // Create image resource
        $source = null;
        switch ( $mime_type ) {
            case 'image/jpeg':
                $source = @imagecreatefromjpeg( $file_path );
                break;
            case 'image/png':
                $source = @imagecreatefrompng( $file_path );
                break;
            case 'image/webp':
                $source = @imagecreatefromwebp( $file_path );
                break;
        }
        
        if ( ! $source ) {
            return $upload;
        }
        
        // Create resized image
        $resized = imagecreatetruecolor( $new_width, $new_height );
        
        // Preserve transparency for PNG/WebP
        if ( $mime_type === 'image/png' || $mime_type === 'image/webp' ) {
            imagealphablending( $resized, false );
            imagesavealpha( $resized, true );
            $transparent = imagecolorallocatealpha( $resized, 255, 255, 255, 127 );
            imagefilledrectangle( $resized, 0, 0, $new_width, $new_height, $transparent );
        }
        
        // Resize
        imagecopyresampled( $resized, $source, 0, 0, 0, 0, $new_width, $new_height, $current_width, $current_height );
        
        // Save back to file
        $quality = $this->plugin->get_option( 'compression_quality', 85 );
        $success = false;
        
        switch ( $mime_type ) {
            case 'image/jpeg':
                $success = @imagejpeg( $resized, $file_path, $quality );
                break;
            case 'image/png':
                $png_quality = (int) ( ( 100 - $quality ) / 11.111111 );
                $success = @imagepng( $resized, $file_path, $png_quality );
                break;
            case 'image/webp':
                $success = @imagewebp( $resized, $file_path, $quality );
                break;
        }
        
        imagedestroy( $source );
        imagedestroy( $resized );
        
        return $upload;
    }
    
    /**
     * Compress image to target file size
     */
    private function compress_image( $upload ) {
        if ( ! $this->plugin->get_option( 'auto_compress', false ) ) {
            return $upload;
        }
        
        $target_size = $this->plugin->get_option( 'target_size', 100 ) * 1024; // Convert KB to bytes
        $file_path = $upload['file'];
        
        if ( ! file_exists( $file_path ) ) {
            return $upload;
        }
        
        $current_size = filesize( $file_path );
        
        // Already under target size
        if ( $current_size <= $target_size ) {
            return $upload;
        }
        
        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return $upload;
        }
        
        $mime_type = $image_info['mime'];
        
        // Load image
        $source = null;
        switch ( $mime_type ) {
            case 'image/jpeg':
                $source = @imagecreatefromjpeg( $file_path );
                break;
            case 'image/png':
                $source = @imagecreatefrompng( $file_path );
                break;
            case 'image/webp':
                $source = @imagecreatefromwebp( $file_path );
                break;
        }
        
        if ( ! $source ) {
            return $upload;
        }
        
        // Try different quality levels
        $quality = $this->plugin->get_option( 'compression_quality', 85 );
        $min_quality = 50;
        
        while ( $quality >= $min_quality && $current_size > $target_size ) {
            // Save with current quality
            switch ( $mime_type ) {
                case 'image/jpeg':
                    @imagejpeg( $source, $file_path, $quality );
                    break;
                case 'image/png':
                    $png_quality = (int) ( ( 100 - $quality ) / 11.111111 );
                    @imagepng( $source, $file_path, $png_quality );
                    break;
                case 'image/webp':
                    @imagewebp( $source, $file_path, $quality );
                    break;
            }
            
            clearstatcache( true, $file_path );
            $current_size = filesize( $file_path );
            
            // Reduce quality for next attempt
            $quality -= 5;
        }
        
        imagedestroy( $source );
        
        return $upload;
    }

    /**
     * Check if server supports WebP
     */
    public function supports_webp() {
        if ( ! function_exists( 'imagewebp' ) ) {
            return false;
        }
        
        if ( function_exists( 'gd_info' ) ) {
            $gd_info = gd_info();
            return isset( $gd_info['WebP Support'] ) && $gd_info['WebP Support'];
        }
        
        return false;
    }

    /**
     * Get user image count
     */
    public function get_user_image_count( $user_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'driop_user_image_count';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking custom table
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            return 0;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query
        $count = $wpdb->get_var( $wpdb->prepare( 
            "SELECT image_count FROM {$wpdb->prefix}driop_user_image_count WHERE user_id = %d", 
            $user_id 
        ) );
        
        return $count ? (int) $count : 0;
    }

    /**
     * Increment user image count
     */
    public function increment_user_image_count( $user_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'driop_user_image_count';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking custom table
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
            return;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table insert/update
        $wpdb->query( $wpdb->prepare(
            "INSERT INTO {$wpdb->prefix}driop_user_image_count (user_id, image_count) VALUES (%d, 1) 
             ON DUPLICATE KEY UPDATE image_count = image_count + 1",
            $user_id
        ) );
    }

    /**
     * Get statistics
     */
    public function get_stats() {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Statistics query
        $total_images = $wpdb->get_var( "
            SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_type = 'attachment' 
            AND post_mime_type LIKE 'image/%'
        " );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Statistics query
        $webp_count = $wpdb->get_var( "
            SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_type = 'attachment' 
            AND post_mime_type = 'image/webp'
        " );
        
        // FIX #2: Count images with EITHER meta key for backward compatibility
        // - _driop_optimized: Set during both upload and bulk optimization (now)
        // - _driop_processed: Legacy key from earlier versions
        // Using OR ensures we count all optimized images regardless of when they were processed
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Statistics query
        $optimized = $wpdb->get_var( "
            SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta} 
            WHERE (meta_key = '_driop_optimized' OR meta_key = '_driop_processed') 
            AND meta_value = '1'
        " );
        
        return array(
            'total_images' => (int) $total_images,
            'webp_count' => (int) $webp_count,
            'optimized' => (int) $optimized,
            'pending' => max( 0, (int) $total_images - (int) $optimized ),
        );
    }

    /**
     * Optimize existing image (for bulk optimization)
     * Converts to WebP, renames file, and updates SEO
     * 
     * @param int  $attachment_id Image attachment ID
     * @param bool $reoptimize    Force re-optimization even if already done
     */
    public function optimize_existing( $attachment_id, $reoptimize = false ) {
        // Check if already optimized - skip UNLESS reoptimize is true
        if ( ! $reoptimize && get_post_meta( $attachment_id, '_driop_optimized', true ) ) {
            return array(
                'success' => true,
                'message' => 'Already optimized',
                'savings' => 0,
            );
        }
        
        $file_path = get_attached_file( $attachment_id );
        
        if ( ! $file_path || ! file_exists( $file_path ) ) {
            // Try to get from attachment metadata as fallback
            $metadata = wp_get_attachment_metadata( $attachment_id );
            if ( $metadata && isset( $metadata['file'] ) ) {
                $upload_dir = wp_upload_dir();
                $file_path = $upload_dir['basedir'] . '/' . $metadata['file'];
            }
            
            // Still not found - mark as optimized to skip in future
            if ( ! $file_path || ! file_exists( $file_path ) ) {
                update_post_meta( $attachment_id, '_driop_optimized', 1 );
                return array(
                    'success' => false,
                    'message' => 'File not found',
                    'savings' => 0,
                );
            }
        }

        $original_size = filesize( $file_path );
        $mime_type = get_post_mime_type( $attachment_id );
        
        // Skip GIFs
        if ( $mime_type === 'image/gif' ) {
            update_post_meta( $attachment_id, '_driop_optimized', 1 );
            return array(
                'success' => true,
                'message' => 'GIF skipped',
                'savings' => 0,
            );
        }

        // Step 1: Rename file with SEO name (max 6 words)
        $new_file_path = $this->rename_existing_file( $file_path, $attachment_id );
        if ( $new_file_path && $new_file_path !== $file_path ) {
            $file_path = $new_file_path;
        }

        // Step 2: Convert to WebP if not already
        $mime_type = get_post_mime_type( $attachment_id ); // Refresh mime type
        if ( $mime_type !== 'image/webp' && $this->supports_webp() ) {
            $webp_path = $this->convert_existing_to_webp( $file_path, $attachment_id );
            if ( $webp_path && $webp_path !== $file_path ) {
                $file_path = $webp_path;
            }
        }

        // Calculate savings
        clearstatcache( true, $file_path );
        $new_size = file_exists( $file_path ) ? filesize( $file_path ) : $original_size;
        $savings = $original_size - $new_size;

        // Mark as optimized
        update_post_meta( $attachment_id, '_driop_optimized', 1 );
        update_post_meta( $attachment_id, '_driop_original_size', $original_size );
        update_post_meta( $attachment_id, '_driop_optimized_size', $new_size );

        return array(
            'success' => true,
            'message' => 'Optimized',
            'savings' => $savings,
        );
    }

    /**
     * Rename existing file with SEO name (max 6 words)
     */
    private function rename_existing_file( $file_path, $attachment_id ) {
        $path_info = pathinfo( $file_path );
        $upload_dir = $path_info['dirname'];
        $extension = isset( $path_info['extension'] ) ? strtolower( $path_info['extension'] ) : 'jpg';
        
        // Get SEO name (max 6 words)
        $seo_name = $this->get_seo_name_max_words( 6 );
        
        if ( empty( $seo_name ) ) {
            $seo_name = 'image';
        }
        
        // Find next available number
        $counter = 1;
        $new_filename = $seo_name . '-' . $counter . '.' . $extension;
        $new_path = $upload_dir . '/' . $new_filename;
        
        while ( file_exists( $new_path ) && $new_path !== $file_path ) {
            $counter++;
            $new_filename = $seo_name . '-' . $counter . '.' . $extension;
            $new_path = $upload_dir . '/' . $new_filename;
        }
        
        // Skip if same path
        if ( $new_path === $file_path ) {
            return $file_path;
        }
        
        // Rename file
        // phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Renaming attachment file
        if ( @rename( $file_path, $new_path ) ) {
            // Update attachment file path in database
            update_attached_file( $attachment_id, $new_path );
            
            // Update attachment metadata
            $metadata = wp_get_attachment_metadata( $attachment_id );
            if ( $metadata && isset( $metadata['file'] ) ) {
                $metadata['file'] = str_replace( $path_info['basename'], $new_filename, $metadata['file'] );
                wp_update_attachment_metadata( $attachment_id, $metadata );
            }
            
            // Update GUID
            global $wpdb;
            $old_url = wp_get_attachment_url( $attachment_id );
            $new_url = str_replace( $path_info['basename'], $new_filename, $old_url );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Updating attachment GUID
            $wpdb->update( 
                $wpdb->posts, 
                array( 'guid' => $new_url ), 
                array( 'ID' => $attachment_id ) 
            );
            
            
            return $new_path;
        }
        
        return $file_path;
    }

    /**
     * Convert existing file to WebP
     */
    private function convert_existing_to_webp( $file_path, $attachment_id ) {
        $image_info = @getimagesize( $file_path );
        if ( ! $image_info ) {
            return $file_path;
        }

        $mime = $image_info['mime'];
        $width = $image_info[0];
        $height = $image_info[1];

        // Skip if already WebP or GIF
        if ( $mime === 'image/webp' || $mime === 'image/gif' ) {
            return $file_path;
        }

        // Create image resource
        $source = null;
        switch ( $mime ) {
            case 'image/jpeg':
                $source = @imagecreatefromjpeg( $file_path );
                break;
            case 'image/png':
                $source = @imagecreatefrompng( $file_path );
                break;
        }

        if ( ! $source ) {
            return $file_path;
        }

        // Prepare output path
        $path_info = pathinfo( $file_path );
        $webp_path = $path_info['dirname'] . '/' . $path_info['filename'] . '.webp';

        // Handle PNG transparency
        if ( $mime === 'image/png' ) {
            imagepalettetotruecolor( $source );
            imagealphablending( $source, true );
            imagesavealpha( $source, true );
        }

        // Save as WebP with high quality
        $success = @imagewebp( $source, $webp_path, 100 );
        imagedestroy( $source );

        if ( $success && file_exists( $webp_path ) ) {
            // Verify dimensions match
            $webp_info = @getimagesize( $webp_path );
            if ( $webp_info && $webp_info[0] === $width && $webp_info[1] === $height ) {
                // Delete original
                // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Deleting original after WebP conversion
                @unlink( $file_path );
                
                // Update attachment file path
                update_attached_file( $attachment_id, $webp_path );
                
                // Update mime type
                wp_update_post( array(
                    'ID' => $attachment_id,
                    'post_mime_type' => 'image/webp',
                ) );

                // Update attachment metadata
                $metadata = wp_get_attachment_metadata( $attachment_id );
                if ( $metadata ) {
                    $metadata['file'] = str_replace( $path_info['basename'], $path_info['filename'] . '.webp', $metadata['file'] );
                    wp_update_attachment_metadata( $attachment_id, $metadata );
                }

                return $webp_path;
            } else {
                // Dimensions don't match, delete webp and keep original
                // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Cleaning up failed WebP conversion
                @unlink( $webp_path );
            }
        }

        return $file_path;
    }
}
