<?php
/**
 * Bulk Image Optimizer
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRIOP_Bulk_Optimizer {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;
    }

    /**
     * Get unoptimized images
     */
    public function get_unoptimized_images( $limit = 50 ) {
        global $wpdb;

        $like_pattern = $wpdb->esc_like( 'image/' ) . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Complex query for unoptimized images
        $results = $wpdb->get_results( $wpdb->prepare(
            "SELECT p.ID, p.post_title, p.guid 
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_driop_optimized'
            WHERE p.post_type = 'attachment' 
            AND p.post_mime_type LIKE %s
            AND (pm.meta_value IS NULL OR pm.meta_value = '')
            LIMIT %d",
            $like_pattern,
            $limit
        ), ARRAY_A );

        return $results;
    }
    
    /**
     * Get ALL images for re-optimization
     */
    public function get_all_images( $limit = 50, $offset = 0 ) {
        global $wpdb;

        $like_pattern = $wpdb->esc_like( 'image/' ) . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Batch query for bulk optimization
        $results = $wpdb->get_results( $wpdb->prepare(
            "SELECT p.ID, p.post_title, p.guid 
            FROM {$wpdb->posts} p
            WHERE p.post_type = 'attachment' 
            AND p.post_mime_type LIKE %s
            ORDER BY p.ID ASC
            LIMIT %d OFFSET %d",
            $like_pattern,
            $limit,
            $offset
        ), ARRAY_A );

        return $results;
    }

    /**
     * Get total unoptimized count
     */
    public function get_unoptimized_count() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Count query for optimization progress
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_driop_optimized'
            WHERE p.post_type = 'attachment' 
            AND p.post_mime_type LIKE 'image/%'
            AND (pm.meta_value IS NULL OR pm.meta_value = '')"
        );
    }

    /**
     * Get total images count
     */
    public function get_total_images_count() {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Count query for optimization progress
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'attachment' AND post_mime_type LIKE 'image/%'"
        );
    }

    /**
     * Get optimized count
     */
    public function get_optimized_count() {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Count query for optimization progress
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_driop_optimized' AND meta_value = '1'"
        );
    }

    /**
     * Optimize batch of images - supports both new and re-optimization
     */
    public function optimize_batch( $batch_size = 5, $apply_seo = true, $reoptimize = false ) {
        // Get images based on mode
        if ( $reoptimize ) {
            $offset = get_transient( 'driop_bulk_offset' ) ?: 0;
            $images = $this->get_all_images( $batch_size, $offset );
            set_transient( 'driop_bulk_offset', $offset + $batch_size, HOUR_IN_SECONDS );
        } else {
            $images = $this->get_unoptimized_images( $batch_size );
        }
        
        if ( empty( $images ) ) {
            delete_transient( 'driop_bulk_offset' );
            return array(
                'completed' => true,
                'processed' => 0,
                'remaining' => 0,
                'message' => __( 'All images have been optimized!', 'digital-rise-image-optimizer-for-seo' ),
            );
        }

        $optimizer = new DRIOP_Image_Optimizer( $this->plugin );
        $seo = new DRIOP_SEO( $this->plugin );
        
        $results = array(
            'success' => 0,
            'failed' => 0,
            'details' => array(),
        );

        foreach ( $images as $image ) {
            // Optimize image - pass reoptimize flag
            $result = $optimizer->optimize_existing( $image['ID'], $reoptimize );
            
            // Apply SEO if enabled
            if ( $apply_seo ) {
                $seo->apply_seo_to_image( $image['ID'] );
            }
            
            if ( $result['success'] ) {
                $results['success']++;
                $results['details'][] = array(
                    'id' => $image['ID'],
                    'title' => $image['post_title'],
                    'status' => 'success',
                    'savings' => $result['savings'],
                );
            } else {
                $results['failed']++;
                $results['details'][] = array(
                    'id' => $image['ID'],
                    'title' => $image['post_title'],
                    'status' => 'failed',
                    'message' => $result['message'],
                );
            }
        }

        // Calculate remaining
        if ( $reoptimize ) {
            $total = $this->get_total_images_count();
            $offset = get_transient( 'driop_bulk_offset' ) ?: 0;
            $remaining = max( 0, $total - $offset );
            $completed = $remaining === 0;
        } else {
            $remaining = $this->get_unoptimized_count();
            $completed = $remaining === 0;
        }

        if ( $completed ) {
            delete_transient( 'driop_bulk_offset' );
            
            // Clear sitemap cache when optimization completes
            delete_transient( 'driop_sitemap_images_cache' );
            delete_transient( 'driop_sitemap_index_cache' );
        }

        return array(
            'completed' => $completed,
            'processed' => count( $images ),
            'success' => $results['success'],
            'failed' => $results['failed'],
            'remaining' => $remaining,
            'details' => $results['details'],
            // translators: %1$d is the number of processed images, %2$d is the number of remaining images
            'message' => sprintf(
                __( 'Processed %1$d images. %2$d remaining.', 'digital-rise-image-optimizer-for-seo' ),
                count( $images ),
                $remaining
            ),
        );
    }

    /**
     * Get optimization progress
     */
    public function get_progress() {
        $total = $this->get_total_images_count();
        $optimized = $this->get_optimized_count();
        $remaining = $this->get_unoptimized_count();

        return array(
            'total' => $total,
            'optimized' => $optimized,
            'remaining' => $remaining,
            'percentage' => $total > 0 ? round( ( $optimized / $total ) * 100, 1 ) : 0,
        );
    }

    /**
     * Reset optimization status (for re-optimization)
     */
    public function reset_optimization_status() {
        global $wpdb;
        
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk cleanup
        $wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key = '_driop_optimized'" );
        $wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key = '_driop_watermarked'" );
        $wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key = '_driop_seo_processed'" );
        // phpcs:enable
        delete_transient( 'driop_bulk_offset' );
        
        return true;
    }
    
    /**
     * Reset bulk offset for fresh re-optimization
     */
    public function reset_bulk_offset() {
        delete_transient( 'driop_bulk_offset' );
    }
}
