<?php
/**
 * Plugin Name: Image Optimizer for SEO
 * Plugin URI: https://digital-rise-solutions.com/digital-rise-image-optimizer-pro/
 * Description: Professional image optimization and SEO automation for WordPress, WooCommerce, and Dokan. Converts to WebP and optimizes SEO automatically.
 * Version: 1.0.3
 * Author: Digital Rise Solutions
 * Author URI: https://digital-rise-solutions.com/
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: digital-rise-image-optimizer-for-seo
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 9.5
 *
 * @package Digital_Rise_Image_Optimizer_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'DRIOP_VERSION', '1.0.3' );
define( 'DRIOP_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'DRIOP_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'DRIOP_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
define( 'DRIOP_PLUGIN_FILE', __FILE__ );
define( 'DRIOP_TEXT_DOMAIN', 'digital-rise-image-optimizer-for-seo' );
define( 'DRIOP_OWNER_URL', 'https://digital-rise-solutions.com/' );
define( 'DRIOP_OWNER_API', 'https://digital-rise-solutions.com/wp-json/driop-license/v1/' );
define( 'DRIOP_SUPPORT_EMAIL', 'seo.oussama@gmail.com' );
define( 'DRIOP_LOGO_URL', 'https://digital-rise-solutions.com/wp-content/uploads/2023/06/digital-rise-solutions-digital-marketing-tunisia.webp' );
define( 'DRIOP_MAX_FILE_SIZE', 50 ); // 50KB target compression

/**
 * WooCommerce HPOS Compatibility
 */
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', __FILE__, true );
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'product_block_editor', __FILE__, true );
    }
});

/**
 * Main plugin class
 */
final class Digital_Rise_Image_Optimizer_Pro {

    private static $instance = null;
    public $options = array();
    public $current_plan = 'free';

    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->load_options();
        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    private function load_options() {
        $defaults = array(
            'current_plan' => 'free',
            'license_key' => '',
            'license_status' => 'inactive',
            'license_expires' => '',
            // Image optimization
            'max_width' => 1920,
            'max_height' => 1080,
            'target_file_size' => 100, // 100KB
            'compression_quality' => 82,
            'auto_compress' => true,
            'confirm_before_optimize' => false, // New: ask before optimizing
            'convert_to_webp' => true, // New: convert all images to WebP
            'max_images_per_user' => 0,
            'max_images_per_product' => 10,
            'max_images_per_vendor' => 50,
            // CSS/JS Compression
            'css_minify_enabled' => false,
            'js_minify_enabled' => false,
            'css_combine_enabled' => false,
            'js_combine_enabled' => false,
            'minify_exclude' => '',
            // Keywords
            'keywords' => '',
            'seo_pattern' => '{site_name} {content_title} {keyword}',
            // SEO
            'auto_rename' => true,
            'auto_alt' => true,
            'auto_description' => true,
            // Page SEO
            'page_seo_enabled' => true,
            'default_title_pattern' => '{title} | {site_name}',
            'default_meta_pattern' => '{excerpt}',
            // Remote
            'remote_connection' => true,
            'remote_api_key' => '',
            // User Contact Info
            'user_email' => '',
            'user_whatsapp' => '',
            // Attribution
            'show_attribution' => true,
            // Integrations
            'wc_integration' => true,
            'dokan_integration' => true,
            // Installation
            'installation_id' => '',
            'installation_date' => '',
        );

        $saved_options = get_option( 'driop_options', array() );
        $this->options = wp_parse_args( $saved_options, $defaults );
        $this->current_plan = $this->options['current_plan'];
        
        // Generate installation ID if not exists
        if ( empty( $this->options['installation_id'] ) ) {
            $this->options['installation_id'] = wp_generate_uuid4();
            $this->options['installation_date'] = current_time( 'mysql' );
            update_option( 'driop_options', $this->options );
        }
    }

    private function load_dependencies() {
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-activator.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-deactivator.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-image-optimizer.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-seo.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-page-seo.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-keywords.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-license.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-remote.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-minifier.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-woocommerce.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-dokan.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-plans.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-bulk-optimizer.php';
        // PHASE 5: Sitemap and Robots
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-sitemap.php';
        require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-robots.php';

        if ( is_admin() ) {
            require_once DRIOP_PLUGIN_DIR . 'admin/class-driop-admin.php';
        }

        require_once DRIOP_PLUGIN_DIR . 'public/class-driop-public.php';
    }

    private function set_locale() {
        // WordPress.org handles translations automatically since WP 4.6
        // No need to call load_plugin_textdomain()
    }

    private function define_admin_hooks() {
        if ( ! is_admin() ) {
            return;
        }

        $admin = new DRIOP_Admin( $this );
        
        add_action( 'admin_enqueue_scripts', array( $admin, 'enqueue_styles' ) );
        add_action( 'admin_enqueue_scripts', array( $admin, 'enqueue_scripts' ) );
        add_action( 'admin_menu', array( $admin, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $admin, 'register_settings' ) );
        add_action( 'admin_notices', array( $admin, 'display_admin_notices' ) );
        
        // AJAX handlers
        add_action( 'wp_ajax_driop_dismiss_notice', array( $admin, 'ajax_dismiss_notice' ) );
        add_action( 'wp_ajax_driop_optimize_image', array( $admin, 'ajax_optimize_image' ) );
        add_action( 'wp_ajax_driop_bulk_optimize', array( $admin, 'ajax_bulk_optimize' ) );
        add_action( 'wp_ajax_driop_reset_optimization', array( $admin, 'ajax_reset_optimization' ) );
        add_action( 'wp_ajax_driop_activate_license', array( $admin, 'ajax_activate_license' ) );
        add_action( 'wp_ajax_driop_deactivate_license', array( $admin, 'ajax_deactivate_license' ) );
        add_action( 'wp_ajax_driop_get_optimization_progress', array( $admin, 'ajax_get_optimization_progress' ) );
        add_action( 'wp_ajax_driop_sync_notices', array( $admin, 'ajax_sync_notices' ) );
        
        // Page SEO meta boxes
        $page_seo = new DRIOP_Page_SEO( $this );
        add_action( 'add_meta_boxes', array( $page_seo, 'add_meta_boxes' ) );
        add_action( 'admin_enqueue_scripts', array( $page_seo, 'enqueue_metabox_assets' ) );
        add_action( 'save_post', array( $page_seo, 'save_meta_box' ) );
        add_action( 'edited_term', array( $page_seo, 'save_term_meta' ), 10, 3 );
        add_action( 'created_term', array( $page_seo, 'save_term_meta' ), 10, 3 );
    }

    private function define_public_hooks() {
        $public = new DRIOP_Public( $this );
        $image_optimizer = new DRIOP_Image_Optimizer( $this );
        $seo = new DRIOP_SEO( $this );
        $page_seo = new DRIOP_Page_SEO( $this );
        $remote = new DRIOP_Remote( $this );
        $license = new DRIOP_License( $this );
        $minifier = new DRIOP_Minifier( $this );
        // PHASE 5: Initialize sitemap and robots
        $sitemap = new DRIOP_Sitemap( $this );
        $robots = new DRIOP_Robots( $this );
        
        add_action( 'wp_enqueue_scripts', array( $public, 'enqueue_styles' ) );
        
        // CSS/JS Minification
        add_action( 'wp_loaded', array( $minifier, 'init' ) );
        
        // Plugin action links (Settings, Donate)
        add_filter( 'plugin_action_links_' . DRIOP_PLUGIN_BASENAME, array( $this, 'add_plugin_action_links' ) );
        
        // DISABLE WordPress "scaled" image feature - prevents "-scaled" suffix
        add_filter( 'big_image_size_threshold', '__return_false' );
        
        // Image optimization - WebP conversion and rename only (NO compression)
        add_filter( 'wp_handle_upload', array( $image_optimizer, 'handle_upload' ), 10, 2 );
        add_filter( 'wp_generate_attachment_metadata', array( $image_optimizer, 'process_attachment' ), 10, 2 );
        
        // Image SEO - auto-set ALT, title, description
        add_filter( 'wp_generate_attachment_metadata', array( $seo, 'auto_set_image_meta' ), 30, 2 );
        
        // Page/Post SEO - output meta tags
        if ( $this->options['page_seo_enabled'] ) {
            add_action( 'wp_head', array( $page_seo, 'output_meta_tags' ), 1 );
            add_filter( 'pre_get_document_title', array( $page_seo, 'filter_title' ), 999 );
            add_filter( 'document_title_parts', array( $page_seo, 'filter_title_parts' ), 999 );
        }
        
        // Taxonomy SEO fields
        add_action( 'category_add_form_fields', array( $page_seo, 'add_term_fields' ) );
        add_action( 'category_edit_form_fields', array( $page_seo, 'edit_term_fields' ) );
        add_action( 'post_tag_add_form_fields', array( $page_seo, 'add_term_fields' ) );
        add_action( 'post_tag_edit_form_fields', array( $page_seo, 'edit_term_fields' ) );
        add_action( 'product_cat_add_form_fields', array( $page_seo, 'add_term_fields' ) );
        add_action( 'product_cat_edit_form_fields', array( $page_seo, 'edit_term_fields' ) );
        add_action( 'product_tag_add_form_fields', array( $page_seo, 'add_term_fields' ) );
        add_action( 'product_tag_edit_form_fields', array( $page_seo, 'edit_term_fields' ) );
        
        // Remote check-in
        if ( $this->options['remote_connection'] ) {
            add_action( 'admin_init', array( $remote, 'schedule_checkin' ) );
            add_action( 'driop_remote_checkin', array( $remote, 'do_checkin' ) );
            
            // Do initial checkin on first load
            if ( ! get_transient( 'driop_initial_checkin_done' ) ) {
                add_action( 'admin_init', array( $remote, 'do_checkin' ) );
                set_transient( 'driop_initial_checkin_done', 1, DAY_IN_SECONDS );
            }
        }
        
        // License validation
        add_action( 'admin_init', array( $license, 'maybe_validate_license' ) );
        
        // Attribution
        if ( $this->options['show_attribution'] && $this->current_plan === 'free' ) {
            add_action( 'wp_footer', array( $public, 'output_attribution' ) );
        }
        
        // WooCommerce
        if ( $this->options['wc_integration'] && class_exists( 'WooCommerce' ) ) {
            $wc = new DRIOP_WooCommerce( $this );
            add_action( 'add_meta_boxes', array( $wc, 'add_product_seo_meta_box' ) );
        }
        
        // Dokan
        if ( $this->options['dokan_integration'] && class_exists( 'WeDevs_Dokan' ) ) {
            $dokan = new DRIOP_Dokan( $this );
            add_filter( 'dokan_product_image_upload', array( $dokan, 'check_vendor_image_limit' ), 10, 2 );
        }
    }
    
    /**
     * Add plugin action links
     */
    public function add_plugin_action_links( $links ) {
        $settings_link = '<a href="' . admin_url( 'admin.php?page=driop-dashboard' ) . '">' . __( 'Settings', 'digital-rise-image-optimizer-for-seo' ) . '</a>';
        $donate_link = '<a href="https://digital-rise-solutions.com/pay-easily-with-usdt-trc20-using-redotpay/" target="_blank" style="color: #2ecc71; font-weight: bold;">' . __( 'Donate', 'digital-rise-image-optimizer-for-seo' ) . '</a>';
        
        array_unshift( $links, $settings_link );
        $links[] = $donate_link;
        
        return $links;
    }

    public function get_option( $key, $default = null ) {
        $value = isset( $this->options[ $key ] ) ? $this->options[ $key ] : $default;
        
        // FIX #1 (PRO): For keywords, decode from base64 to preserve line breaks
        if ( $key === 'keywords' && is_string( $value ) && ! empty( $value ) ) {
            // Check if it's base64 encoded (starts with 'b64:')
            if ( strpos( $value, 'b64:' ) === 0 ) {
                $value = base64_decode( substr( $value, 4 ) );
            }
            // Normalize line breaks
            $value = str_replace( array( "\r\n", "\r" ), "\n", $value );
        }
        
        return $value;
    }

    public function update_option( $key, $value ) {
        // FIX #1 (PRO): For keywords, encode in base64 to preserve line breaks permanently
        if ( $key === 'keywords' && is_string( $value ) ) {
            // Normalize line breaks first
            $value = str_replace( array( "\r\n", "\r" ), "\n", $value );
            // Encode in base64 with marker
            $value = 'b64:' . base64_encode( $value );
        }
        
        $this->options[ $key ] = $value;
        update_option( 'driop_options', $this->options );
    }

    public function save_options( $options ) {
        $this->options = wp_parse_args( $options, $this->options );
        update_option( 'driop_options', $this->options );
    }

    public function has_feature( $feature ) {
        $plans = new DRIOP_Plans( $this );
        return $plans->has_feature( $feature );
    }

    public function is_plan( $plan ) {
        return $this->current_plan === $plan;
    }

    public function is_plan_at_least( $plan ) {
        $levels = array( 'free' => 0, 'pro' => 1, 'agency' => 2 );
        $current_level = isset( $levels[ $this->current_plan ] ) ? $levels[ $this->current_plan ] : 0;
        $required_level = isset( $levels[ $plan ] ) ? $levels[ $plan ] : 0;
        return $current_level >= $required_level;
    }
}

function driop_activate() {
    require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-activator.php';
    DRIOP_Activator::activate();
}
register_activation_hook( __FILE__, 'driop_activate' );

function driop_deactivate() {
    require_once DRIOP_PLUGIN_DIR . 'includes/class-driop-deactivator.php';
    DRIOP_Deactivator::deactivate();
}
register_deactivation_hook( __FILE__, 'driop_deactivate' );

function driop() {
    return Digital_Rise_Image_Optimizer_Pro::get_instance();
}

add_action( 'plugins_loaded', 'driop', 0 );
