<?php
/**
 * Database Class - Digital Rise Auto Poster
 * 
 * Simple and reliable duplicate detection using wp_postmeta
 * 
 * @package Digital_Rise_Auto_Poster
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRAP_Database {

    private $wpdb;
    public $campaigns_table;
    public $logs_table;

    public function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->campaigns_table = $wpdb->prefix . 'drap_campaigns';
        $this->logs_table = $wpdb->prefix . 'drap_logs';
    }

    public function create_tables() {
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        $charset = $this->wpdb->get_charset_collate();

        $sql1 = "CREATE TABLE {$this->campaigns_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            title varchar(255) NOT NULL,
            status varchar(20) DEFAULT 'inactive',
            feed_url text NOT NULL,
            post_status varchar(20) DEFAULT 'publish',
            post_author bigint(20) unsigned DEFAULT 1,
            post_category text,
            post_tags text,
            fetch_interval int DEFAULT 60,
            posts_per_fetch int DEFAULT 1,
            last_run datetime DEFAULT NULL,
            next_run datetime DEFAULT NULL,
            total_posts int DEFAULT 0,
            options longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset;";

        $sql2 = "CREATE TABLE {$this->logs_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            campaign_id bigint(20) unsigned NOT NULL,
            level varchar(20) DEFAULT 'info',
            message text NOT NULL,
            details longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY campaign_id (campaign_id)
        ) $charset;";

        dbDelta( $sql1 );
        dbDelta( $sql2 );
    }

    /**
     * Check if URL was already imported
     * Uses MD5 hash stored as post meta key (proven wp-automatic method)
     */
    public function is_duplicate( $url ) {
        $hash = md5( trim( $url ) );

        // Direct query to wp_postmeta
        $found = $this->wpdb->get_var(
            $this->wpdb->prepare(
                "SELECT post_id FROM {$this->wpdb->postmeta} WHERE meta_key = %s LIMIT 1",
                '_drap_url_' . $hash
            )
        );

        if ( $found ) {
            // Verify post still exists and not trashed
            $status = get_post_status( $found );
            if ( $status && $status !== 'trash' ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Mark URL as imported
     */
    public function mark_imported( $post_id, $url ) {
        $hash = md5( trim( $url ) );
        update_post_meta( $post_id, '_drap_url_' . $hash, $url );
        update_post_meta( $post_id, '_drap_source_url', $url );
    }

    /**
     * Get campaign
     */
    public function get_campaign( $id ) {
        return $this->wpdb->get_row(
            $this->wpdb->prepare( "SELECT * FROM {$this->campaigns_table} WHERE id = %d", $id )
        );
    }

    /**
     * Get all campaigns
     */
    public function get_campaigns( $status = '' ) {
        $sql = "SELECT * FROM {$this->campaigns_table}";
        if ( $status ) {
            $sql .= $this->wpdb->prepare( " WHERE status = %s", $status );
        }
        $sql .= " ORDER BY created_at DESC";
        return $this->wpdb->get_results( $sql );
    }

    /**
     * Get due campaigns
     */
    public function get_due_campaigns() {
        return $this->wpdb->get_results(
            $this->wpdb->prepare(
                "SELECT * FROM {$this->campaigns_table} 
                WHERE status = 'active' AND (next_run IS NULL OR next_run <= %s)",
                current_time( 'mysql' )
            )
        );
    }

    /**
     * Insert campaign
     */
    public function insert_campaign( $data ) {
        $defaults = array(
            'title' => '',
            'status' => 'inactive',
            'feed_url' => '',
            'post_status' => 'publish',
            'post_author' => get_current_user_id(),
            'post_category' => '',
            'post_tags' => '',
            'fetch_interval' => 60,
            'posts_per_fetch' => 1,
            'options' => '',
        );

        $data = wp_parse_args( $data, $defaults );

        if ( is_array( $data['post_category'] ) ) {
            $data['post_category'] = serialize( $data['post_category'] );
        }
        if ( is_array( $data['post_tags'] ) ) {
            $data['post_tags'] = serialize( $data['post_tags'] );
        }
        if ( is_array( $data['options'] ) ) {
            $data['options'] = serialize( $data['options'] );
        }

        $data['next_run'] = gmdate( 'Y-m-d H:i:s', time() + ( $data['fetch_interval'] * 60 ) );

        $this->wpdb->insert( $this->campaigns_table, $data );
        return $this->wpdb->insert_id;
    }

    /**
     * Update campaign
     */
    public function update_campaign( $id, $data ) {
        if ( isset( $data['post_category'] ) && is_array( $data['post_category'] ) ) {
            $data['post_category'] = serialize( $data['post_category'] );
        }
        if ( isset( $data['post_tags'] ) && is_array( $data['post_tags'] ) ) {
            $data['post_tags'] = serialize( $data['post_tags'] );
        }
        if ( isset( $data['options'] ) && is_array( $data['options'] ) ) {
            $data['options'] = serialize( $data['options'] );
        }

        return $this->wpdb->update( $this->campaigns_table, $data, array( 'id' => $id ) );
    }

    /**
     * Delete campaign
     */
    public function delete_campaign( $id ) {
        $this->wpdb->delete( $this->logs_table, array( 'campaign_id' => $id ) );
        return $this->wpdb->delete( $this->campaigns_table, array( 'id' => $id ) );
    }

    /**
     * Update after run
     */
    public function update_after_run( $id, $posts_created = 0 ) {
        $campaign = $this->get_campaign( $id );
        if ( ! $campaign ) return;

        $next = gmdate( 'Y-m-d H:i:s', time() + ( $campaign->fetch_interval * 60 ) );

        $this->wpdb->update(
            $this->campaigns_table,
            array(
                'last_run' => current_time( 'mysql' ),
                'next_run' => $next,
                'total_posts' => $campaign->total_posts + $posts_created,
            ),
            array( 'id' => $id )
        );
    }

    /**
     * Log message
     */
    public function log( $campaign_id, $level, $message, $details = array() ) {
        $this->wpdb->insert(
            $this->logs_table,
            array(
                'campaign_id' => $campaign_id,
                'level' => $level,
                'message' => $message,
                'details' => maybe_serialize( $details ),
            )
        );
    }

    /**
     * Get logs
     */
    public function get_logs( $campaign_id, $limit = 50 ) {
        return $this->wpdb->get_results(
            $this->wpdb->prepare(
                "SELECT * FROM {$this->logs_table} WHERE campaign_id = %d ORDER BY created_at DESC LIMIT %d",
                $campaign_id,
                $limit
            )
        );
    }

    /**
     * Get stats
     */
    public function get_stats() {
        $total = $this->wpdb->get_var( "SELECT COUNT(*) FROM {$this->campaigns_table}" );
        $active = $this->wpdb->get_var( "SELECT COUNT(*) FROM {$this->campaigns_table} WHERE status = 'active'" );
        $posts = $this->wpdb->get_var( "SELECT SUM(total_posts) FROM {$this->campaigns_table}" );

        return array(
            'total_campaigns' => (int) $total,
            'active_campaigns' => (int) $active,
            'total_posts' => (int) $posts,
        );
    }
}
