<?php
/**
 * Cron Scheduler - Digital Rise Auto Poster
 * 
 * Reliable WordPress Cron implementation
 * - Runs every 5 minutes to check due campaigns
 * - Supports interval, daily, and weekly scheduling
 * 
 * @package Digital_Rise_Auto_Poster
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRAP_Cron {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;

        // Register cron intervals FIRST (before any scheduling)
        add_filter( 'cron_schedules', array( $this, 'add_cron_intervals' ) );
        
        // Hook for running campaigns
        add_action( 'drap_cron_hook', array( $this, 'run_due_campaigns' ) );
        
        // Ensure cron is scheduled on every page load (admin)
        if ( is_admin() ) {
            add_action( 'admin_init', array( $this, 'ensure_cron_scheduled' ) );
        }
        
        // Also check on frontend (for sites with low admin traffic)
        add_action( 'wp_loaded', array( $this, 'maybe_run_missed_campaigns' ) );
    }

    /**
     * Add custom cron intervals
     */
    public function add_cron_intervals( $schedules ) {
        // Every 5 minutes - more frequent checks
        $schedules['drap_5min'] = array(
            'interval' => 5 * MINUTE_IN_SECONDS,
            'display'  => __( 'Every 5 minutes', 'digital-rise-auto-poster' ),
        );
        
        // Every 15 minutes
        $schedules['drap_15min'] = array(
            'interval' => 15 * MINUTE_IN_SECONDS,
            'display'  => __( 'Every 15 minutes', 'digital-rise-auto-poster' ),
        );
        
        // Legacy support
        $schedules['drap_interval'] = array(
            'interval' => 15 * MINUTE_IN_SECONDS,
            'display'  => __( 'DRAP Interval', 'digital-rise-auto-poster' ),
        );
        
        return $schedules;
    }

    /**
     * Ensure cron is scheduled
     */
    public function ensure_cron_scheduled() {
        $scheduled = wp_next_scheduled( 'drap_cron_hook' );
        
        if ( ! $scheduled ) {
            // Schedule to run every 5 minutes
            wp_schedule_event( time(), 'drap_5min', 'drap_cron_hook' );
            
            // Log that we rescheduled
            error_log( 'DRAP: Cron rescheduled at ' . current_time( 'mysql' ) );
        }
    }

    /**
     * Check and run any missed campaigns on page load
     * This handles cases where WP Cron didn't fire
     */
    public function maybe_run_missed_campaigns() {
        // Only run occasionally to avoid performance issues
        $last_check = get_transient( 'drap_last_cron_check' );
        
        if ( $last_check && ( time() - $last_check ) < 60 ) {
            return; // Only check once per minute max
        }
        
        set_transient( 'drap_last_cron_check', time(), HOUR_IN_SECONDS );
        
        // Check for due campaigns
        $due_campaigns = $this->plugin->db->get_due_campaigns();
        
        if ( ! empty( $due_campaigns ) ) {
            // Run them
            $this->run_due_campaigns();
        }
    }

    /**
     * Run all due campaigns
     */
    public function run_due_campaigns() {
        // Prevent concurrent runs
        if ( get_transient( 'drap_cron_running' ) ) {
            return;
        }
        
        set_transient( 'drap_cron_running', true, 5 * MINUTE_IN_SECONDS );
        
        try {
            $campaigns = $this->plugin->db->get_due_campaigns();
            
            if ( empty( $campaigns ) ) {
                delete_transient( 'drap_cron_running' );
                return;
            }

            foreach ( $campaigns as $campaign ) {
                // Double-check campaign is active
                if ( $campaign->status !== 'active' ) {
                    continue;
                }
                
                // Run the campaign
                $result = $this->run_campaign( $campaign );
                
                // Calculate and set next run time
                $this->set_next_run( $campaign->id );
            }
            
        } catch ( Exception $e ) {
            error_log( 'DRAP Cron Error: ' . $e->getMessage() );
        }
        
        delete_transient( 'drap_cron_running' );
    }

    /**
     * Run a single campaign
     */
    public function run_campaign( $campaign ) {
        $options = maybe_unserialize( $campaign->options );
        if ( ! is_array( $options ) ) {
            $options = array();
        }
        
        // Check max posts limit
        $max_posts = isset( $options['max_posts'] ) ? (int) $options['max_posts'] : 0;
        if ( $max_posts > 0 && $campaign->total_posts >= $max_posts ) {
            // Limit reached
            $this->plugin->db->log( 
                $campaign->id, 
                'info', 
                sprintf( __( 'Max posts limit reached (%d). Skipping.', 'digital-rise-auto-poster' ), $max_posts )
            );
            
            // Pause if option enabled
            if ( ! empty( $options['pause_at_limit'] ) ) {
                $this->plugin->db->update_campaign( $campaign->id, array( 'status' => 'inactive' ) );
                $this->plugin->db->log( $campaign->id, 'info', __( 'Campaign paused (limit reached).', 'digital-rise-auto-poster' ) );
            }
            
            return 0;
        }
        
        // Calculate how many posts we can still create
        $posts_to_fetch = $campaign->posts_per_fetch;
        if ( $max_posts > 0 ) {
            $remaining = $max_posts - $campaign->total_posts;
            $posts_to_fetch = min( $posts_to_fetch, $remaining );
        }
        
        $allow_duplicates = ! empty( $options['allow_duplicates'] );
        $fetch_full_content = ! empty( $options['fetch_full_content'] );

        // Log start
        $this->plugin->db->log( 
            $campaign->id, 
            'info', 
            __( 'Scheduled run starting...', 'digital-rise-auto-poster' )
        );

        // Get items from feed
        $items = $this->plugin->feed->get_items(
            $campaign->feed_url,
            $posts_to_fetch,
            ! $allow_duplicates,
            $fetch_full_content
        );

        if ( is_wp_error( $items ) ) {
            $this->plugin->db->log( $campaign->id, 'error', $items->get_error_message() );
            return 0;
        }

        if ( empty( $items ) ) {
            $this->plugin->db->log( 
                $campaign->id, 
                'info', 
                __( 'Scheduled run: No new items in feed.', 'digital-rise-auto-poster' )
            );
            return 0;
        }

        $created = 0;

        foreach ( $items as $item ) {
            $result = $this->plugin->poster->create( $item, $campaign );
            
            if ( ! is_wp_error( $result ) ) {
                $created++;
                $this->plugin->db->log(
                    $campaign->id,
                    'success',
                    sprintf( __( 'Auto-posted: %s', 'digital-rise-auto-poster' ), $item['title'] ),
                    array( 'post_id' => $result, 'url' => $item['url'] )
                );
            } else {
                $this->plugin->db->log(
                    $campaign->id,
                    'error',
                    sprintf( __( 'Failed: %s', 'digital-rise-auto-poster' ), $result->get_error_message() )
                );
            }
        }

        // Update stats
        $new_total = $campaign->total_posts + $created;
        $update_data = array( 'total_posts' => $new_total );
        
        // Check if we've now hit the limit
        if ( $max_posts > 0 && $new_total >= $max_posts ) {
            if ( ! empty( $options['pause_at_limit'] ) ) {
                $update_data['status'] = 'inactive';
                $this->plugin->db->log( $campaign->id, 'info', __( 'Campaign paused (limit reached).', 'digital-rise-auto-poster' ) );
            }
        }
        
        $this->plugin->db->update_campaign( $campaign->id, $update_data );
        
        if ( $created > 0 ) {
            $this->plugin->db->log(
                $campaign->id,
                'success',
                sprintf( __( 'Scheduled run complete: %d post(s) created. Total: %d', 'digital-rise-auto-poster' ), $created, $new_total )
            );
        }

        return $created;
    }

    /**
     * Set next run time for a campaign
     */
    public function set_next_run( $campaign_id ) {
        $campaign = $this->plugin->db->get_campaign( $campaign_id );
        if ( ! $campaign ) {
            return;
        }

        $options = maybe_unserialize( $campaign->options );
        $schedule_type = isset( $options['schedule_type'] ) ? $options['schedule_type'] : 'interval';
        $schedule_time = isset( $options['schedule_time'] ) ? $options['schedule_time'] : '09:00';
        $schedule_days = isset( $options['schedule_days'] ) ? (array) $options['schedule_days'] : array();

        $next_run = $this->calculate_next_run( $schedule_type, $campaign->fetch_interval, $schedule_time, $schedule_days );

        $this->plugin->db->update_campaign( $campaign_id, array(
            'next_run' => $next_run,
            'last_run' => current_time( 'mysql' ),
        ) );
    }

    /**
     * Schedule a campaign (called when saved)
     */
    public function schedule_campaign( $campaign_id ) {
        $campaign = $this->plugin->db->get_campaign( $campaign_id );
        if ( ! $campaign || $campaign->status !== 'active' ) {
            return;
        }

        $options = maybe_unserialize( $campaign->options );
        $schedule_type = isset( $options['schedule_type'] ) ? $options['schedule_type'] : 'interval';
        $schedule_time = isset( $options['schedule_time'] ) ? $options['schedule_time'] : '09:00';
        $schedule_days = isset( $options['schedule_days'] ) ? (array) $options['schedule_days'] : array();

        $next_run = $this->calculate_next_run( $schedule_type, $campaign->fetch_interval, $schedule_time, $schedule_days );

        $this->plugin->db->update_campaign( $campaign_id, array(
            'next_run' => $next_run,
        ) );

        $this->plugin->db->log( 
            $campaign_id, 
            'info', 
            sprintf( __( 'Scheduled for: %s', 'digital-rise-auto-poster' ), $next_run )
        );
    }

    /**
     * Calculate next run time based on schedule type
     */
    private function calculate_next_run( $type, $interval_minutes, $time, $days ) {
        // Use WordPress timezone
        $now = current_time( 'timestamp' );

        switch ( $type ) {
            case 'daily':
                // Run daily at specified time
                $today_time = strtotime( current_time( 'Y-m-d' ) . ' ' . $time );
                
                if ( $today_time > $now ) {
                    return date( 'Y-m-d H:i:s', $today_time );
                }
                // Tomorrow
                return date( 'Y-m-d H:i:s', $today_time + DAY_IN_SECONDS );

            case 'weekly':
                // Run on specific days
                if ( empty( $days ) ) {
                    $days = array( 'mon', 'wed', 'fri' );
                }

                $day_numbers = array(
                    'sun' => 0, 'mon' => 1, 'tue' => 2, 'wed' => 3,
                    'thu' => 4, 'fri' => 5, 'sat' => 6,
                );

                // Convert selected days to numbers
                $selected_days = array();
                foreach ( $days as $day ) {
                    if ( isset( $day_numbers[ $day ] ) ) {
                        $selected_days[] = $day_numbers[ $day ];
                    }
                }

                if ( empty( $selected_days ) ) {
                    $selected_days = array( 1, 3, 5 ); // Mon, Wed, Fri
                }

                // Find next scheduled day
                for ( $i = 0; $i <= 7; $i++ ) {
                    $check_time = $now + ( $i * DAY_IN_SECONDS );
                    $check_day = (int) date( 'w', $check_time );
                    
                    if ( in_array( $check_day, $selected_days, true ) ) {
                        $run_time = strtotime( date( 'Y-m-d', $check_time ) . ' ' . $time );
                        
                        if ( $run_time > $now ) {
                            return date( 'Y-m-d H:i:s', $run_time );
                        }
                    }
                }
                
                // Fallback
                return date( 'Y-m-d H:i:s', $now + DAY_IN_SECONDS );

            case 'interval':
            default:
                // Every X minutes
                $seconds = max( 15, (int) $interval_minutes ) * MINUTE_IN_SECONDS;
                return date( 'Y-m-d H:i:s', $now + $seconds );
        }
    }

    /**
     * Manual run (Run Now button)
     */
    public function run_now( $campaign_id ) {
        $campaign = $this->plugin->db->get_campaign( $campaign_id );

        if ( ! $campaign ) {
            return new WP_Error( 'not_found', __( 'Campaign not found.', 'digital-rise-auto-poster' ) );
        }

        if ( $campaign->status !== 'active' ) {
            return new WP_Error( 'inactive', __( 'Activate campaign first.', 'digital-rise-auto-poster' ) );
        }

        $options = maybe_unserialize( $campaign->options );
        $allow_duplicates = ! empty( $options['allow_duplicates'] );
        $fetch_full_content = ! empty( $options['fetch_full_content'] );

        // Get items
        $items = $this->plugin->feed->get_items(
            $campaign->feed_url,
            $campaign->posts_per_fetch,
            ! $allow_duplicates,
            $fetch_full_content
        );

        if ( is_wp_error( $items ) ) {
            $this->plugin->db->log( $campaign_id, 'error', $items->get_error_message() );
            return $items;
        }

        $counts = $this->plugin->feed->count_available( $campaign->feed_url );

        if ( empty( $items ) ) {
            return array(
                'success'  => true,
                'no_items' => true,
                'message'  => sprintf(
                    __( 'All %d items imported. No new content.', 'digital-rise-auto-poster' ),
                    $counts['total']
                ),
            );
        }

        $created_posts = array();

        foreach ( $items as $item ) {
            $post_id = $this->plugin->poster->create( $item, $campaign );

            if ( ! is_wp_error( $post_id ) ) {
                $post = get_post( $post_id );
                $created_posts[] = array(
                    'id'    => $post_id,
                    'title' => $post->post_title,
                    'url'   => get_permalink( $post_id ),
                );

                $this->plugin->db->log(
                    $campaign_id,
                    'success',
                    sprintf( __( 'Created: %s', 'digital-rise-auto-poster' ), $post->post_title ),
                    array( 'post_id' => $post_id )
                );
            }
        }

        // Update stats
        $this->plugin->db->update_campaign( $campaign_id, array(
            'total_posts' => $campaign->total_posts + count( $created_posts ),
            'last_run'    => current_time( 'mysql' ),
        ) );

        if ( empty( $created_posts ) ) {
            return array(
                'success'  => true,
                'no_items' => true,
                'message'  => __( 'No posts created.', 'digital-rise-auto-poster' ),
            );
        }

        return array(
            'success'   => true,
            'posts'     => $created_posts,
            'remaining' => max( 0, $counts['available'] - count( $created_posts ) ),
            'message'   => sprintf(
                __( 'Created %d post(s). %d remaining.', 'digital-rise-auto-poster' ),
                count( $created_posts ),
                max( 0, $counts['available'] - count( $created_posts ) )
            ),
        );
    }
}
