<?php
/**
 * Plugin Name:       Digital Rise Auto Poster Solutions
 * Plugin URI:        https://digital-rise-solutions.com/plugins/auto-poster
 * Description:       Professional RSS feed auto-posting solution. Automatically imports and publishes content from RSS feeds with smart duplicate detection, content cleaning, and SEO optimization.
 * Version:           1.0.0
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * Author:            Digital Rise Solutions
 * Author URI:        https://digital-rise-solutions.com
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       digital-rise-auto-poster
 * Domain Path:       /languages
 *
 * @package Digital_Rise_Auto_Poster
 * 
 * Developed by Digital Rise Solutions
 * Team: Oussama Ben Rejab & Haythem Krifi
 * https://digital-rise-solutions.com
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Plugin constants
define( 'DRAP_VERSION', '1.1.0' );
define( 'DRAP_PLUGIN_FILE', __FILE__ );
define( 'DRAP_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'DRAP_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'DRAP_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main plugin class
 */
final class Digital_Rise_Auto_Poster {

    private static $instance = null;

    public $db;
    public $feed;
    public $poster;
    public $cron;

    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->includes();
        $this->init_hooks();
    }

    private function includes() {
        require_once DRAP_PLUGIN_DIR . 'includes/class-drap-database.php';
        require_once DRAP_PLUGIN_DIR . 'includes/class-drap-feed.php';
        require_once DRAP_PLUGIN_DIR . 'includes/class-drap-poster.php';
        require_once DRAP_PLUGIN_DIR . 'includes/class-drap-cron.php';

        if ( is_admin() ) {
            require_once DRAP_PLUGIN_DIR . 'admin/class-drap-admin.php';
        }
    }

    private function init_hooks() {
        register_activation_hook( DRAP_PLUGIN_FILE, array( $this, 'activate' ) );
        register_deactivation_hook( DRAP_PLUGIN_FILE, array( $this, 'deactivate' ) );

        add_action( 'plugins_loaded', array( $this, 'init' ) );
        add_action( 'init', array( $this, 'load_textdomain' ) );
    }

    public function init() {
        $this->db     = new DRAP_Database();
        $this->feed   = new DRAP_Feed( $this->db );
        $this->poster = new DRAP_Poster( $this->db );
        $this->cron   = new DRAP_Cron( $this );

        if ( is_admin() ) {
            new DRAP_Admin( $this );
        }

        // Check for DB updates
        $this->maybe_update();
    }

    public function load_textdomain() {
        load_plugin_textdomain( 'digital-rise-auto-poster', false, dirname( DRAP_PLUGIN_BASENAME ) . '/languages' );
    }

    public function activate() {
        require_once DRAP_PLUGIN_DIR . 'includes/class-drap-database.php';
        $db = new DRAP_Database();
        $db->create_tables();

        add_option( 'drap_version', DRAP_VERSION );

        // Clear any existing cron and reschedule
        wp_clear_scheduled_hook( 'drap_cron_hook' );
        
        // Schedule cron to run every 5 minutes
        if ( ! wp_next_scheduled( 'drap_cron_hook' ) ) {
            wp_schedule_event( time(), 'drap_5min', 'drap_cron_hook' );
        }

        flush_rewrite_rules();
    }

    public function deactivate() {
        wp_clear_scheduled_hook( 'drap_cron_hook' );
        flush_rewrite_rules();
    }

    private function maybe_update() {
        $current = get_option( 'drap_version', '0' );
        if ( version_compare( $current, DRAP_VERSION, '<' ) ) {
            $this->db->create_tables();
            update_option( 'drap_version', DRAP_VERSION );
        }
    }
}

function drap() {
    return Digital_Rise_Auto_Poster::instance();
}

// Initialize
drap();
