<?php
/**
 * Admin Class - Digital Rise Auto Poster
 * 
 * @package Digital_Rise_Auto_Poster
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DRAP_Admin {

    private $plugin;

    public function __construct( $plugin ) {
        $this->plugin = $plugin;

        add_action( 'admin_menu', array( $this, 'add_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        add_action( 'admin_init', array( $this, 'handle_form' ) );

        // AJAX
        add_action( 'wp_ajax_drap_run_now', array( $this, 'ajax_run_now' ) );
        add_action( 'wp_ajax_drap_validate_feed', array( $this, 'ajax_validate_feed' ) );
        add_action( 'wp_ajax_drap_delete', array( $this, 'ajax_delete' ) );
        add_action( 'wp_ajax_drap_toggle', array( $this, 'ajax_toggle' ) );
        add_action( 'wp_ajax_drap_clear_history', array( $this, 'ajax_clear_history' ) );
    }

    public function add_menu() {
        add_menu_page(
            __( 'Auto Poster', 'digital-rise-auto-poster' ),
            __( 'Auto Poster', 'digital-rise-auto-poster' ),
            'manage_options',
            'drap-campaigns',
            array( $this, 'page_campaigns' ),
            'dashicons-rss',
            30
        );

        add_submenu_page(
            'drap-campaigns',
            __( 'Campaigns', 'digital-rise-auto-poster' ),
            __( 'Campaigns', 'digital-rise-auto-poster' ),
            'manage_options',
            'drap-campaigns'
        );

        add_submenu_page(
            'drap-campaigns',
            __( 'Add New', 'digital-rise-auto-poster' ),
            __( 'Add New', 'digital-rise-auto-poster' ),
            'manage_options',
            'drap-edit',
            array( $this, 'page_edit' )
        );

        add_submenu_page(
            'drap-campaigns',
            __( 'Logs', 'digital-rise-auto-poster' ),
            __( 'Logs', 'digital-rise-auto-poster' ),
            'manage_options',
            'drap-logs',
            array( $this, 'page_logs' )
        );
    }

    public function enqueue_scripts( $hook ) {
        if ( strpos( $hook, 'drap' ) === false ) {
            return;
        }

        wp_enqueue_style( 'drap-admin', DRAP_PLUGIN_URL . 'admin/css/admin.css', array(), DRAP_VERSION );
        wp_enqueue_script( 'drap-admin', DRAP_PLUGIN_URL . 'admin/js/admin.js', array( 'jquery' ), DRAP_VERSION, true );

        wp_localize_script( 'drap-admin', 'drap', array(
            'ajax' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'drap_nonce' ),
            'confirm_delete' => __( 'Delete this campaign?', 'digital-rise-auto-poster' ),
            'confirm_clear' => __( 'Clear import history? This will allow re-importing previously imported items.', 'digital-rise-auto-poster' ),
            'running' => __( 'Running...', 'digital-rise-auto-poster' ),
        ) );
    }

    public function handle_form() {
        if ( ! current_user_can( 'manage_options' ) ) return;

        if ( isset( $_POST['drap_save'] ) && wp_verify_nonce( $_POST['_wpnonce'], 'drap_save' ) ) {
            $this->save_campaign();
        }
    }

    private function save_campaign() {
        $id = isset( $_POST['campaign_id'] ) ? absint( $_POST['campaign_id'] ) : 0;

        // Get schedule type
        $schedule_type = sanitize_key( $_POST['schedule_type'] );
        $schedule_days = isset( $_POST['schedule_days'] ) ? array_map( 'sanitize_key', (array) $_POST['schedule_days'] ) : array();
        $schedule_time = sanitize_text_field( $_POST['schedule_time'] );

        $data = array(
            'title' => sanitize_text_field( $_POST['title'] ),
            'status' => sanitize_key( $_POST['status'] ),
            'feed_url' => esc_url_raw( $_POST['feed_url'] ),
            'post_status' => sanitize_key( $_POST['post_status'] ),
            'post_author' => absint( $_POST['post_author'] ),
            'post_category' => isset( $_POST['post_category'] ) ? array_map( 'absint', (array) $_POST['post_category'] ) : array(),
            'post_tags' => sanitize_text_field( $_POST['post_tags'] ),
            'fetch_interval' => max( 15, absint( $_POST['fetch_interval'] ) ),
            'posts_per_fetch' => absint( $_POST['posts_per_fetch'] ),
            'options' => array(
                'import_image' => isset( $_POST['import_image'] ),
                'fetch_full_content' => isset( $_POST['fetch_full_content'] ),
                'convert_webp' => isset( $_POST['convert_webp'] ),
                'image_seo' => isset( $_POST['image_seo'] ),
                'add_source_link' => isset( $_POST['add_source_link'] ),
                'source_link_text' => sanitize_text_field( $_POST['source_link_text'] ),
                'use_original_date' => isset( $_POST['use_original_date'] ),
                'feed_cats_as_tags' => isset( $_POST['feed_cats_as_tags'] ),
                'allow_duplicates' => isset( $_POST['allow_duplicates'] ),
                'max_posts' => absint( $_POST['max_posts'] ),
                'pause_at_limit' => isset( $_POST['pause_at_limit'] ),
                'schedule_type' => $schedule_type,
                'schedule_days' => $schedule_days,
                'schedule_time' => $schedule_time,
            ),
        );

        // Convert tags string to array
        if ( $data['post_tags'] ) {
            $data['post_tags'] = array_map( 'trim', explode( ',', $data['post_tags'] ) );
        }

        if ( $id > 0 ) {
            $this->plugin->db->update_campaign( $id, $data );
            $msg = 'updated';
        } else {
            $id = $this->plugin->db->insert_campaign( $data );
            $msg = 'created';
            $this->plugin->db->log( $id, 'info', __( 'Campaign created', 'digital-rise-auto-poster' ) );
        }

        // Schedule the campaign based on type
        $this->plugin->cron->schedule_campaign( $id );

        wp_redirect( admin_url( 'admin.php?page=drap-edit&id=' . $id . '&msg=' . $msg ) );
        exit;
    }

    // AJAX: Run Now
    public function ajax_run_now() {
        check_ajax_referer( 'drap_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Permission denied' );
        }

        $id = absint( $_POST['campaign_id'] );
        $result = $this->plugin->cron->run_now( $id );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    // AJAX: Validate Feed
    public function ajax_validate_feed() {
        check_ajax_referer( 'drap_nonce', 'nonce' );

        $url = esc_url_raw( $_POST['url'] );
        $result = $this->plugin->feed->validate( $url );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    // AJAX: Delete
    public function ajax_delete() {
        check_ajax_referer( 'drap_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Permission denied' );
        }

        $id = absint( $_POST['campaign_id'] );
        $this->plugin->db->delete_campaign( $id );

        wp_send_json_success();
    }

    // AJAX: Toggle Status
    public function ajax_toggle() {
        check_ajax_referer( 'drap_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Permission denied' );
        }

        $id = absint( $_POST['campaign_id'] );
        $campaign = $this->plugin->db->get_campaign( $id );

        $new_status = $campaign->status === 'active' ? 'inactive' : 'active';
        $this->plugin->db->update_campaign( $id, array( 'status' => $new_status ) );

        wp_send_json_success( array( 'status' => $new_status ) );
    }

    // AJAX: Clear History
    public function ajax_clear_history() {
        check_ajax_referer( 'drap_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Permission denied' );
        }

        $id = absint( $_POST['campaign_id'] );
        $campaign = $this->plugin->db->get_campaign( $id );

        if ( ! $campaign ) {
            wp_send_json_error( 'Campaign not found' );
        }

        global $wpdb;

        // Delete all _drap_url_* meta for posts from this campaign
        $wpdb->query(
            $wpdb->prepare(
                "DELETE pm FROM {$wpdb->postmeta} pm
                INNER JOIN {$wpdb->postmeta} pm2 ON pm.post_id = pm2.post_id
                WHERE pm.meta_key LIKE '_drap_url_%%'
                AND pm2.meta_key = '_drap_campaign_id'
                AND pm2.meta_value = %d",
                $id
            )
        );

        $this->plugin->db->log( $id, 'info', __( 'Import history cleared', 'digital-rise-auto-poster' ) );

        wp_send_json_success( array( 'message' => __( 'History cleared. You can now re-import items.', 'digital-rise-auto-poster' ) ) );
    }

    // Page: Campaigns List
    public function page_campaigns() {
        $campaigns = $this->plugin->db->get_campaigns();
        $stats = $this->plugin->db->get_stats();
        include DRAP_PLUGIN_DIR . 'admin/views/campaigns.php';
    }

    // Page: Edit Campaign
    public function page_edit() {
        $id = isset( $_GET['id'] ) ? absint( $_GET['id'] ) : 0;
        $campaign = $id ? $this->plugin->db->get_campaign( $id ) : null;
        $logs = $id ? $this->plugin->db->get_logs( $id, 20 ) : array();
        include DRAP_PLUGIN_DIR . 'admin/views/edit.php';
    }

    // Page: Logs
    public function page_logs() {
        $campaign_id = isset( $_GET['campaign_id'] ) ? absint( $_GET['campaign_id'] ) : 0;
        $campaigns = $this->plugin->db->get_campaigns();
        $logs = $campaign_id ? $this->plugin->db->get_logs( $campaign_id, 100 ) : array();
        include DRAP_PLUGIN_DIR . 'admin/views/logs.php';
    }
}
